{ ---->>>>COOLDEMO<<<<-------------------------------------------

  Demonstrates use of the COOL MOUSE CURSOR with a
  Turbo Pascal/Turbo Vision 2.0 application.

  Things worthy of note:

    * Only one extra unit needs to be included to make an
      existing Turbo Vision 2.0 application COOL MOUSE-
      ready.

    * The COOL MOUSE CURSOR is written to work with any
      size font your application will use.  This includes
      but is not limited to...
      - 25-line mode  (VGA 8x16 font; EGA 8x14 font)
      - 28-line mode  (VGA 8x14 font)
      - 43-line mode  (EGA 8x8 font)
      - 50-line mode  (VGA 8x8 font)

    * The COOL MOUSE CURSOR automatically senses
      when a mouse is not installed or when the display
      adapter is not an EGA/VGA/SVGA; at those times it
      simply does not install itself.  If a mouse is
      present but the correct adapter type is not, the
      COOL MOUSE CURSOR steps out of the way and lets
      you use the regular text mode mouse block cursor.

    * The COOL MOUSE CURSOR should be deactivated before the
      following actions and reactivated afterward:
      - making Mode/font changes
      - shelling out to DOS
      - using EXEC to run another program
      SetMouseState(msAsleep)  deactivates.
      SetMouseState(msAwake)  reactivates.


  This program demonstrates many useful techniques for
  handling the COOL MOUSE CURSOR, but to really understand
  these techniques you should read the COOL MOUSE CURSOR
  manual.

  ------------------------------------------------------------- }

program COOLDEMO;

uses Objects, App, Drivers, Views, Menus, Editors,
     Memory, Dialogs, MsgBox, StdDlg,
     CoolMou;  { NOTE: This is the only extra unit that
                 needs to be USEd by your program to enable
                 the COOL MOUSE CURSOR }

type
  TAnyApp = object(TApplication)
    constructor Init;
    procedure InitMenuBar; virtual;
    procedure InitDesktop; virtual;
    procedure HandleEvent (var Event: TEvent); virtual;
    function OpenEditor (FileName: FNameStr): PEditWindow;
    procedure OpenWindow;
    procedure ToggleFont;
    procedure ToggleBackground;
  end;


{ DEMO COMMANDS }
const
  cmToggleFont  = 152;
  cmToggleMouse = 153;
  cmOpenWindow  = 154;
  cmToggleBkgd  = 155;


{ CONSTANTS USED TO CHANGE THE BACKGROUND }
const
  NormalBkgdChar  = #176;
  GraphBkgdChar   = #202;
  SolidBkgdChar   = #32;
  SolidColor  = #$1F;
  NormalColor = #$71;


{ UTILITY FUNCTIONS }

{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  FILEEXISTS --
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
function FileExists (FileNm: String): Boolean;
var
  Untyped: File;
begin
  {$I-}
  Assign(Untyped,FileNm);
  FileMode := 0;
  Reset(Untyped);
  Close(Untyped);
  {$I+}
  FileExists := (IOResult = 0) and (FileNm <> '');
end;


{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  DOEDITDIALOG --
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
function DoEditDialog (Dialog: Integer; Info: Pointer): Word; far;
begin
  case Dialog of
    edOutOfMemory:
      DoEditDialog := MessageBox (
        'Not enough memory for this operation.',
        nil, mfError + mfOkButton);
    edReadError:
      DoEditDialog := MessageBox ('Error reading file %s.',
        @Info, mfError + mfOkButton);
    edWriteError:
      DoEditDialog := MessageBox ('Error writing file %s.',
        @Info, mfError + mfOkButton);
    edCreateError:
      DoEditDialog := MessageBox ('Error creating file %s.',
        @Info, mfError + mfOkButton);
    edSaveModify:
      DoEditDialog := cmNo;
    edSaveUntitled:
      DoEditDialog := cmNo;
  end;
end;



{ TANYAPP METHODS }

{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  INIT --

  Demonstrates how to initially make the COOL MOUSE CURSOR
  assume its arrow shape.

  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
constructor TAnyApp.Init;
begin
  MaxHeapSize := 4096;          { This is on account of the edit
                                  window, not the mouse }
  inherited Init;
  EditorDialog := DoEditDialog;

  SetMouseState(msArrow);       { NOTE: This is all that's
                                  required to turn the
                                  block cursor into an arrow! }
end;


{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  OPENWINDOW --

  Allows the user to select and open an editor window.

  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
procedure TAnyApp.OpenWindow;
var
  R: TRect;
  FileDialog: PFileDialog;
  TheFile: FNameStr;
const
  FDOptions: Word = fdOKButton or fdOpenButton;
begin
  TheFile := '*.*';
  New(FileDialog, Init(TheFile, 'Open file', '~F~ile name',
    FDOptions, 1));
  if ExecuteDialog(FileDialog, @TheFile) <> cmCancel then
  begin
    R.Assign(0, 0, 75, 20);
    InsertWindow(New(PEditWindow, Init(R, TheFile, wnNoNumber)));
  end;
end;


{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  INITDESKTOP --

  Initializes the desktop.

  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
procedure TAnyApp.InitDesktop;
begin
  inherited InitDesktop;

  { Change the background pattern and color }
  Desktop^.Background^.Pattern := SolidBkgdChar;
  GetPalette^[1] := SolidColor;
end;


{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  INITMENUBAR --

  Initializes the menu bar.

  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
procedure TAnyApp.InitMenuBar;
var
  R: TRect;
begin
  GetExtent(R);
  R.B.Y := R.A.Y + 1;
  MenuBar := New(PMenubar,Init(R,NewMenu(
    NewSubMenu('~F~ile', hcNoContext, NewMenu(
      NewItem('~O~pen', 'F3', kbF3, cmOpen, hcNoContext,
      NewItem('~O~pen Window', '', 0, cmOpenWindow, hcNoContext,
      NewItem('~D~OS shell', '', 0, cmDosShell, hcNoContext,
      NewLine(
      NewItem('E~x~it', 'Alt+X', kbAltX, cmQuit, hcNoContext,
      nil)))))),
    NewSubMenu('~O~ptions', hcNoContext, NewMenu(
      NewItem('Toggle ~M~ouse', '', 0, cmToggleMouse,
        hcNoContext,
      NewItem('Toggle ~F~ont', '', 0, cmToggleFont, hcNoContext,
      NewItem('Toggle ~B~ackground', '', 0, cmToggleBkgd,
        hcNoContext, nil)))),
    NewSubMenu('~W~indow', hcNoContext, NewMenu(
      StdWindowMenuItems(nil)),
    nil)
  )))));
end;


{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  OPENEDITOR --

  Opens an editor window.

  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
function TAnyApp.OpenEditor;
var
  P: PView;
  R: TRect;
begin
  DeskTop^.GetExtent(R);
  P := Application^.ValidView(New(PEditWindow,
    Init(R, FileName, wnNoNumber)));
  DeskTop^.Insert(P);
  OpenEditor := PEditWindow(P);
end;


{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  TOGGLEFONT --

  This method demonstrates how to change font sizes/screen
  modes in a manner friendly to the COOL MOUSE CURSOR.

  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
procedure TAnyApp.ToggleFont;
begin
  HideMouse;
  SetMouseState(msAsleep); { NOTE: The COOL MOUSE should be
                             deactivated before a mode/
                             font change and reactivated
                             afterward. }

  { Mode/font change }
  if ScreenMode and smFont8x8 = 0 then
    SetScreenMode((ScreenMode or smFont8x8))
  else
    SetScreenMode((ScreenMode and not smFont8x8));

  SetMouseState(msAwake);  { COOL MOUSE reactivated here }
  ShowMouse;
end;


{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  TOGGLEBACKGROUND --

  This procedure demonstrates how to toggle between two
  different backgrounds: (1) a "solid," patternless
  background made up of ASCII code #32, the space
  character; and (2) the traditional Turbo Vision
  2.0 halftone background made up of ASCII code
  #176.

  There's a twist here, though, because the half-
  tone background is a special "steady" version
  that shows no thickening of dots when the COOL MOUSE
  CURSOR crosses it.

  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
procedure TAnyApp.ToggleBackground;
begin
  if DeskTop^.Background^.Pattern <> SolidBkgdChar then
  begin
    { Undo what InstallSteadyBkgd did below }
    RemoveSteadyBkgd;

    { Change the background character }
    DeskTop^.Background^.Pattern := SolidBkgdChar;

    { Change the background color }
    GetPalette^[1] := SolidColor;

    { These two statements together work to redraw the screen }
    DoneMemory;
    Redraw;
  end
  else
  begin
    { Change the background color }
    GetPalette^[1] := NormalColor;

    { Install a steady halftone background and redraw screen }
    InstallSteadyBkgd(NormalBkgdChar, GraphBkgdChar);
  end;
end;


{ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  HANDLEEVENT --

  Demonstrates how to trap and handle DosShell commands in
  a manner friendly to the COOL MOUSE CURSOR.  Also shows
  how to toggle between the arrow and block shapes.

  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - }
procedure TAnyApp.HandleEvent;
const
  ViewFileName = 'COOLDEMO.PAS';

begin

  { NOTE: The following demonstrates how to trap cmDosShell
     commands so that the COOL MOUSE CURSOR can be
     deactivated before shelling out to DOS and then
     reactivated afterward.  The same process should be
     followed before and after using EXEC. }

  if (Event.What = evCommand) and
    (Event.Command = cmDosShell) then
  begin
    SetMouseState(msAsleep);  { Temporarily deactivates mouse }
    DosShell;
    SetMouseState(msAwake);   { Reactivates the mouse--
                                If the mouse was an arrow before
                                going to "sleep" it resumes
                                being an arrow now; if it was
                                a block, it resumes being a
                                block. }

    { Alternatively, you could omit the three statements above
      and substitute the identical procedure provided in the
      COOLMOU unit... }
    (*
    MouseSafeDosShell;         { This statement is equivalent
                                 to the three statements above }
    *)

    ClearEvent(Event);
  end;

  inherited HandleEvent(Event);

  if (Event.What = evCommand) then
  begin
    case Event.Command of
      cmOpenWindow:
        OpenWindow;

      cmOpen:
        if FileExists(ViewFileName) then
          OpenEditor(ViewFileName)
        else
          OpenEditor('');

      cmToggleMouse:
        ToggleMouseShape;    { NOTE: This is all that's
                               required to toggle the
                               COOL MOUSE shape from
                               Block to Arrow and
                               from Arrow to Block. }

      cmToggleFont:
        ToggleFont;

      cmToggleBkgd:
        ToggleBackground;

    end; { case }

    ClearEvent(Event);
  end;
end;



var
  AnyApp: TAnyApp;

begin
  AnyApp.Init;
  AnyApp.Run;
  AnyApp.Done;
end.