/* diskio.c - disk benchmark
 *
 * Author:  Kai Uwe Rommel <rommel@ars.muc.de>
 * Created: Fri Jul 08 1994
 */
 
static char *rcsid =
"$Id: diskio.c,v 1.4 1994/07/11 14:23:00 rommel Exp $";
static char *rcsrev = "$Revision: 1.4 $";

/*
 * $Log: diskio.c,v $
 * Revision 1.4  1994/07/11 14:23:00  rommel
 * changed latency timing
 *
 * Revision 1.3  1994/07/09 13:07:20  rommel
 * changed transfer speed test
 *
 * Revision 1.2  1994/07/08 21:53:05  rommel
 * cleanup
 *
 * Revision 1.1  1994/07/08 21:29:41  rommel
 * Initial revision
 * 
 */

#define INTERVAL 10

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define INCL_DOS
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#define INCL_DOSERRORS
#define INCL_NOPM
#include <os2.h>

#include "diskacc2.h"

int nHandle;
unsigned nSides, nTracks, nSectors;
char *pBuffer;

HEV hSemTimer;
HTIMER hTimer;
QWORD nStart, nStop;

int start_timer(void)
{
  ULONG nPost;
  APIRET rc;

  if ((rc = DosResetEventSem(hSemTimer, &nPost)) && rc != ERROR_ALREADY_RESET)
    return printf("Semaphore error.\n"), -1;

  if(DosAsyncTimer(INTERVAL * 1000, (HSEM) hSemTimer, &hTimer))
    return printf("Timer error.\n"), -1;

  if (DosTmrQueryTime(&nStart))
    return printf("Timer error.\n"), -1;

  nPost = 0;

  return 0;
}

int time_over(void)
{
  ULONG nPost;

  if (DosQueryEventSem(hSemTimer, &nPost))
    return printf("Semaphore error.\n"), 1;

  return nPost;
}

int stop_timer(int accuracy)
{
  ULONG nFreq;

  if (DosTmrQueryTime(&nStop))
    return printf("Timer error.\n"), -1;
  if (DosTmrQueryFreq(&nFreq))
    return printf("Timer error.\n"), -1;

  nFreq = (nFreq + accuracy / 2) / accuracy;

  return (nStop.ulLo - nStart.ulLo) / nFreq;
}

int bench_transfer(void)
{
  int nCnt, nData = 0, nTime;

  printf("Data transfer rate: ");
  fflush(stdout);

  if (start_timer())
    return -1;

  for (nCnt = 0; !time_over(); nCnt++)
  {
    if (DskRead(nHandle, nCnt % nSides, nCnt / nSides, 1, nSectors, pBuffer))
      return printf("Disk read error.\n"), -1;

    nData += nSectors * 512;
  }

  if ((nTime = stop_timer(1024)) == -1)
    return -1;

  printf("%d k/sec\n", nData / nTime);
}

int bench_latency(void)
{
  int nCnt, nSector, nTime;

  printf("Average latency time: ");
  fflush(stdout);

  srand(1);

  if (start_timer())
    return -1;

  for (nCnt = 0; !time_over(); nCnt++)
  {
    nSector = rand() * nSectors / RAND_MAX + 1;

    if (DskRead(nHandle, 0, 0, nSector, 1, pBuffer))
      return printf("Disk read error.\n"), -1;
  }

  if ((nTime = stop_timer(1000)) == -1)
    return -1;

  nTime = nTime * 10 / nCnt;

  printf("%d.%d ms\n", nTime / 10, nTime % 10);

  return nTime;
}

int bench_seek(int nLatency)
{
  int nCnt, nTrack, nTime;

  printf("Average seek time: ");
  fflush(stdout);

  srand(1);

  if (start_timer())
    return -1;

  for (nCnt = 0; !time_over(); nCnt++)
  {
    nTrack = rand() * nTracks / RAND_MAX;

    if (DskRead(nHandle, 0, nTrack, 1, 1, pBuffer))
      return printf("Disk read error.\n"), -1;
  }

  if ((nTime = stop_timer(1000)) == -1)
    return -1;

  nTime = nTime * 10 / nCnt;

  printf("%d.%d ms\n", (nTime - nLatency) / 10, (nTime - nLatency) % 10);
  printf("Average data access time: %d.%d ms\n", nTime / 10, nTime % 10);
}

int bench_disk(int nDisk)
{
  char szName[8];

  sprintf(szName, "$%d:", nDisk);

  if ((nHandle = DskOpen(szName, 0, &nSides, &nTracks, &nSectors)) < 0)
    return printf("\nCannot access disk %d.\n", nDisk), -1;

  printf("\nDisk %d: %d sides, %d cylinders, %d sectors per track = %d MB\n", 
	 nDisk, nSides, nTracks, nSectors,
	 nSides * nTracks * nSectors / 2048);

  if ((pBuffer = malloc(nSectors * 512)) == NULL)
    return printf("\nNot enough memory.\n"), -1;

  bench_transfer();
  bench_seek(bench_latency());

  free(pBuffer);
  DskClose(nHandle);

  return 0;
}

int main(void)
{
  char szVersion[32];
  USHORT nDisks;
  int nCount;

  strcpy(szVersion, rcsrev + sizeof("$Revision: ") - 1);
  *strchr(szVersion, ' ') = 0;

  printf("\nDISKIO - Fixed Disk Benchmark, Version %s"
	 "\n(C) 1994 Kai Uwe Rommel\n", szVersion);

  if (DosPhysicalDisk(INFO_COUNT_PARTITIONABLE_DISKS, &nDisks, sizeof(nDisks), 0, 0))
    return printf("\nCannot determine number of disks.\n"), 1;

  printf("\nNumber of fixed disks: %d\n", nDisks);

  DosCreateEventSem(NULL, &hSemTimer, DC_SEM_SHARED, 0);
  DosSleep(1000);

  for (nCount = 1; nCount <= nDisks; nCount++)
    bench_disk(nCount);
}

/* end of diskio.c */
