// LANUTIL.C -- LANLIB address utility routines
// Copyright (c) 1994 by Allen Brunson  version 1.00  06/30/94

#ifdef __TURBOC__            // If a Borland compiler
#include <mem.h>             // memcpy(), memset()
#else                        // If a Microsoft compiler
#include <memory.h>          // memcpy(), memset()
#endif

#include <stdio.h>           // sprintf()
#include "lanlib.h"          // LANLIB defines


/****************************************************************************/
/*                                                                          */
/***  ipxAddrBrd()                                                        ***/
/*                                                                          */
/****************************************************************************

This procedure sets the node address and immediate address fields of an
IPXADDRFULL structure to FFFFFFFFFFFFh, making it suitable for broadcasting
packets to all addresses on one network segment.  It returns a pointer to
the address.                                                                */

struct IPXADDRFULL *ipxAddrBrd(struct              // Begin ipxAddrBrd()
 IPXADDRFULL *ipxAddr)
  {
    memset(&ipxAddr->node, 0xFF, 6);               // Set node address
    memset(&ipxAddr->immedAddr, 0xFF, 6);          // Set immediate address

    return ipxAddr;                                // Return the pointer
  }                                                // End ipxAddrBrd()


/****************************************************************************/
/*                                                                          */
/***  ipxAddrCmp()                                                        ***/
/*                                                                          */
/****************************************************************************

This procedure compares the network, node, and socket numbers of two
IPXADDRFULL structures.  It returns FALSE if the two addresses are equal,
-1 if the first is less than the second, or 1 if the first is greater than
the second.                                                                 */

int ipxAddrCmp(struct IPXADDRFULL *ipxAddr1,       // Begin ipxAddrCmp()
 struct IPXADDRFULL *ipxAddr2)
  {
    word i;                                        // Loop counter
    byte *addr1;                                   // First pointer
    byte *addr2;                                   // Second pointer

    addr1 = (byte *) ipxAddr1;                     // Point at the two
    addr2 = (byte *) ipxAddr2;                     //  series of bytes

    for (i = 0; i < 12; i++)                       // Loop for each byte
      {
        if (addr1[i] < addr2[i]) return -1;        // If less than
        if (addr1[i] > addr2[i]) return  1;        // If greater than
      }

    return FALSE;                                  // All bytes were the same
  }                                                // End ipxAddrCmp()


/****************************************************************************/
/*                                                                          */
/***  ipxAddrCpy()                                                        ***/
/*                                                                          */
/****************************************************************************

This procedure copies the contents of one IPXADDRFULL structure to another.
It returns a pointer to the destination address.                            */

struct IPXADDRFULL *ipxAddrCpy(struct IPXADDRFULL  // Begin ipxAddrCpy()
 *ipxAddrDst, struct IPXADDRFULL *ipxAddrSrc)
  {
    memcpy(ipxAddrDst, ipxAddrSrc,                 // Copy contents
     sizeof (struct IPXADDRFULL));

    return ipxAddrDst;                             // Return dest pointer
  }                                                // End ipxAddrCpy()


/****************************************************************************/
/*                                                                          */
/***  ipxAddrSocket()                                                     ***/
/*                                                                          */
/****************************************************************************

This procedure accepts a socket number in standard Intel lo-hi format and
a pointer to an IPXADDRFULL structure and puts the socket into the address. */

struct IPXADDRFULL *ipxAddrSocket(struct           // Begin ipxAddrSocket()
 IPXADDRFULL *ipxAddr, word socket)
  {
    socket = ipxByteSwap(socket);                  // Set hi-lo byte order
    ipxAddr->socket = socket;                      // Set the socket number

    return ipxAddr;                                // Return the pointer
  }                                                // End ipxAddrSocket()


/****************************************************************************/
/*                                                                          */
/***  ipxAddrStr()                                                        ***/
/*                                                                          */
/****************************************************************************

This procedure accepts a pointer to a string and a pointer to an IPXADDRFULL
structure and fills the string with a text representation of the network and
node address.  The string should contain at least 22 bytes.  It returns a
pointer to the string.                                                      */

char *ipxAddrStr(char *str,                        // Begin ipxAddrStr()
 struct IPXADDRFULL *ipxAddr)
  {
    sprintf(str,                                   // Format output string

     "%02X%02X%02X%02X:%02X%02X%02X%02X%02X%02X",  // Format string

     ipxAddr->net[0], ipxAddr->net[1],             // Net address
     ipxAddr->net[2], ipxAddr->net[3],

     ipxAddr->node[0], ipxAddr->node[1],           // Node address
     ipxAddr->node[2], ipxAddr->node[3],
     ipxAddr->node[4], ipxAddr->node[5]);

    return str;                                    // Return string pointer
  }                                                // End ipxAddrStr()


/****************************************************************************/
/*                                                                          */
/***  ipxAddrStrLong()                                                    ***/
/*                                                                          */
/****************************************************************************

This procedure accepts a pointer to a string and to an IPXADDRFULL structure
and fills the string with a text representation of all four components of
the address.  The string should contain at least 40 bytes.                  */

char *ipxAddrStrLong(char *str,                    // Begin ipxAddrStrLong()
 struct IPXADDRFULL *ipxAddr)
  {
    sprintf(str,                                   // Format output string

     "%02X%02X%02X%02X:%02X%02X%02X%02X%02X%02X:"  // Format string
     "%04X:%02X%02X%02X%02X%02X%02X",

     ipxAddr->net[0], ipxAddr->net[1],             // Net address
     ipxAddr->net[2], ipxAddr->net[3],

     ipxAddr->node[0], ipxAddr->node[1],           // Node address
     ipxAddr->node[2], ipxAddr->node[3],
     ipxAddr->node[4], ipxAddr->node[5],

     ipxByteSwap(ipxAddr->socket),                 // Socket in lo-hi order

     ipxAddr->immedAddr[0], ipxAddr->immedAddr[1], // Immediate address
     ipxAddr->immedAddr[2], ipxAddr->immedAddr[3],
     ipxAddr->immedAddr[4], ipxAddr->immedAddr[5]);

    return str;                                    // Return string pointer
  }                                                // End ipxAddrStrLong()


/****************************************************************************/
/*                                                                          */
/***  ipxByteSwap()                                                       ***/
/*                                                                          */
/****************************************************************************

This procedure takes a word-sized variable and returns the word-sized
variable with its bytes swapped.  This is useful for converting sockets to
and from Motorola format.                                                   */

word ipxByteSwap(word socket)                      // Begin ipxByteSwap()
  {
    return (socket << 8) + (socket >> 8);          // Flip socket number
  }                                                // End ipxByteSwap()
