/***********************************************************************/
/*  secerase.c  -  Gordon Torrie August 1994 <gordon@torrie.org>       */
/*                                                                     */
/*  compiled using the Microsoft QuickC version 2.5 compiler           */
/*                                                                     */
/*  QCL /Olt /W4 /Za /F 3000 secerase.c                                */
/*                                                                     */
/*  This program demonstrates how to write a file erase program in C.  */
/*  The program operates by first finding how large the specified      */
/*  file is.                                                           */
/*  Then it opens the specified file in read/write binary mode and     */
/*  performs a rewind and write-to-file operation five times.          */
/*  Each time it performs this operation it writes a constant bit-     */
/*  pattern to every byte in the file, using a different bit-pattern   */
/*  on each of the five passes. The bit-patterns used are:             */
/*                                                                     */ 
/*          hex 55                                                     */
/*          hex AA                                                     */
/*          hex DB                                                     */
/*          hex FF                                                     */
/*          hex 00                                                     */
/*                                                                     */ 
/*  Finally, the file is deleted.                                      */ 
/*                                                                     */ 
/*  NOTES:                                                             */ 
/*                                                                     */ 
/*  1. This program has been tested by compiling with the Microsoft    */ 
/*     QuickC version 2.5 compiler under MS-DOS and the resulting      */ 
/*     program run under MS-DOS. Although the program is written in    */ 
/*     ANSI C, there is no guarantee that the program will perform     */ 
/*     the desired operations if compiled with a different compiler    */ 
/*     or when run under a different operating system.                 */ 
/*                                                                     */
/*  History                                                            */
/*  -------                                                            */
/*                                                                     */
/*    3.3 - Added /L switch to report each file name that is           */
/*          deleted and how many bytes of disk space were freed        */
/*          by doing so.                                               */
/*    3.2 - Moved the code that performs the search for the first and  */
/*          and next matching filenames into a new function,           */
/*          get_dpath(). This is simply to isolate the operating       */
/*          system dependant code from the rest of the program.        */
/*    3.1 - Added optional /N switch to prevent the deletion of the    */
/*          file(s). The file(s) are erased.                           */
/*    3.0 - Replaced the fread() of the file with stat() to obtain     */
/*          the file size.                                             */
/*        - Increased the size of W_CNT from 128 to 8192.              */
/*    2.0 - Added code in main() so that one can specify wild-card     */
/*          characters in the filename.                                */
/*        - Added code in main() to check for optional switches.       */
/*        - Added usage() function.                                    */
/*    1.0 - First version. Deletes a single specified file.            */
/*                                                                     */ 
/*  Other similar programs                                             */ 
/*  ----------------------                                             */ 
/*                                                                     */ 
/*  A similar program, DELZ, was published in the October 12, 1985     */ 
/*  issue of PC Magazine.                                              */ 
/*                                                                     */ 
/*  Along with the VMS operating system supplied by Digital Equipment  */ 
/*  Corporation is an assembly-language module, DOD_ERAPAT.MAR, in     */ 
/*  SYS$EXAMPLES. Version 4-001 of this module is dated 15-FEB-1985.   */ 
/***********************************************************************/ 


/***********************************************************************/
/*  Implementation Note                                                */
/*  -------------------                                                */
/*                                                                     */
/*  The execution speed of the program is very dependant on the value  */
/*  of W_CNT. Experimenting with different values and then recording   */
/*  execution time (under MS-DOS) to erase a file 260,419 bytes in     */
/*  size on a fixed disk produced the following results:               */
/*                                                                     */
/*           W_CNT    time                                             */
/*                                                                     */
/*             128    54.5 seconds                                     */
/*             512    54.5                                             */
/*            1024    28.3                                             */
/*            2048    15.2                                             */
/*            4096     8.6                                             */
/*            8192     5.5                                             */
/*                                                                     */
/*  As a point of interest, the time required by the program, with     */
/*  W_CNT = 8192, to delete the same 260,419 byte file on a RAMdisk    */
/*  and on a 3.5-inch floppy disk was 0.8 seconds and 92.9 seconds     */
/*  respectively. For all tests, the files were de-fragmented before   */
/*  conducting the test.                                               */
/***********************************************************************/
#define W_CNT  8192  /*  Number of bytes written at a time  */
#define NUM_PAT  5   /*  Number of erase patterns           */

#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\types.h>
#include <sys\stat.h>

/***********************************************************************/
/*  erase()                                                            */
/*                                                                     */
/*  This routine writes the specified pattern byte to the already      */
/*  open file the specified number of times.                           */
/*                                                                     */
/*  called by: secerase()                                              */
/*  calls to:  fwrite()                                                */
/*             fprintf()                                                */
/***********************************************************************/
int erase(FILE *file, unsigned char pattern, unsigned long ktr)
   {
   size_t write_cnt, cnt;
   int rc;
   unsigned char buf[W_CNT+1];

   for (rc=0; rc<= W_CNT; rc++) buf[rc] = pattern;
   rc = EXIT_SUCCESS;

   while (ktr != 0UL)
      {
      write_cnt = W_CNT;

      if ((unsigned long)write_cnt > ktr) write_cnt = (size_t) ktr;
      cnt = fwrite(buf, 1, write_cnt, file);
      ktr -= (unsigned long) cnt;
      if (cnt == write_cnt) continue;
        /*
          The fwrite() failed to write write_cnt items.
        */
      if (feof(file) == 0) fprintf(stderr, "WARNING:  An error occurred in writing the pattern %02X.\n"
                                           "          Attempting to continue.\n\n", pattern);
      rc = EXIT_FAILURE;
      break;
      }
   return rc;
   }


/***********************************************************************/
/*  secerase()                                                         */
/*                                                                     */
/*  This routine opens the specified file in binary mode.              */
/*  For each pattern byte that is to be written to the file, the       */
/*  routine rewinds the file and calls the erase() routine.            */
/*                                                                     */
/*  called by: main()                                                  */
/*  calls to:  erase()                                                 */
/*             fclose()                                                */
/*             feof()                                                  */
/*             fopen()                                                 */
/*             perror()                                                */
/*             rewind()                                                */
/*             stat()                                                  */
/***********************************************************************/
int secerase( char *path, unsigned long *size )
   {
   unsigned long esize;
   size_t act_cnt;
   FILE *fptr;
   int k, rc;
   unsigned char pat[NUM_PAT];
   struct stat finfo, *finfo_ptr=&finfo;

   rc = EXIT_SUCCESS;
   /*
     These are the erase patterns that will be written to the 
     file. They are written in reverse numeric order (4, 3, 2, 1, and 0).
     Patterns 2, 1 and 0 (in that order) comply with the U.S. DoD 
     specification.
   */
   pat[4] = '\x55'; /* = binary 01010101  */
   pat[3] = '\xAA'; /* = binary 10101010  */
   pat[2] = '\xDB'; /* = binary 11011011  */
   pat[1] = '\xFF'; /* = binary 11111111  */
   pat[0] = '\x00'; /* = binary 00000000  */

   /*  Obtain the size of the specified file.  */
   if (stat(path, finfo_ptr) != 0)
      {
      perror(path);
      return EXIT_FAILURE;
      }

   /*  Attempt to open the specified file.  */
   if ((fptr = fopen(path, "r+b")) == NULL)
      {
      perror(path);
      return EXIT_FAILURE;
      }

   /*
     Now write the erase patterns. 
   */
   for(k=NUM_PAT-1; k>=0; k--)
      {
      rewind(fptr);
      *size = finfo.st_size;
      if (erase(fptr, pat[k], *size) != EXIT_SUCCESS) rc = EXIT_FAILURE;
      }

   /*  Finally, close the file.  */
   fclose(fptr);
   return rc;
   }


/***********************************************************************/
/*  get_dpath ()                                                       */
/*                                                                     */
/*  This function finds the next filename matching the given pattern   */
/*  The pattern is supplied as a full or partial path in the           */
/*  char string path. The filename found plus whatever parts of the    */
/*  path were specified is stored in the char string dpath.            */
/*                                                                     */
/*  Returns: EXIT_SUCCESS if a matching filename was found.            */
/*           EXIT_FAILURE if no filename matching the pattern could    */
/*           be found.                                                 */
/*                                                                     */
/*  Implementation Note                                                */
/*  -------------------                                                */
/*                                                                     */
/*  The following four functions are specific to Microsoft QuickC and  */
/*  are also specific to the DOS operating system.                     */
/*                                                                     */
/*  _dos_findfirst() - find first file that matches the specified      */
/*                     path and attributes. Uses DOS interrupt 0x21    */
/*                     function 0x4E.                                  */
/*                                                                     */
/*  _dos_findnext()  - find next file, if any, that matches the same   */
/*                     path and attributes specified in the last call  */
/*                     to _dos_findfirst(). Uses DOS interrupt 0x21    */
/*                     function 0x4F.                                  */
/*                                                                     */
/*  _makepath()      - creates a single path name from separate        */
/*                     drive, path and filename elements.              */
/*                                                                     */
/*  _splitpath()     - divides the path string into separate drive,    */
/*                     path, file name and file extension strings.     */
/***********************************************************************/
int get_dpath ( char *dpath, char *path)
   {
   static struct find_t dft1, *dft=&dft1;
   static int firstime=0;
   struct path_parts
      {
      char drive  [_MAX_DRIVE];
      char dirctry[_MAX_DIR];
      char fname  [_MAX_FNAME];
      char exten  [_MAX_EXT];
      } path1;

   if (firstime == 0)
      {
      firstime = 1;
      /*  Split the specified path into its component parts  */
      _splitpath(path, path1.drive, path1.dirctry, path1.fname, path1.exten);


      /*  See if we can find the file  */
      if (_dos_findfirst(path, 0, dft) != 0)
         {
         fprintf(stderr, "File not found\n");
         return EXIT_FAILURE;
         }
      else
         {
         /*
           We found the first filename matching the pattern.
           Construct the full path to the file-to-be-deleted.
         */
         strcpy(path1.exten, "");
         _makepath(dpath, path1.drive, path1.dirctry, dft->name, path1.exten);
         return EXIT_SUCCESS;
         }
      }

   else
      {
      /*  Look for another filename matching the pattern  */
      if (_dos_findnext(dft) == 0)
         {
         /*
           We found another filename matching the pattern.
           Split the specified path into its component parts  
           Construct the full path to the file-to-be-deleted.
         */
         _splitpath(path, path1.drive, path1.dirctry, path1.fname, path1.exten);
         strcpy(path1.exten, "");
         _makepath(dpath, path1.drive, path1.dirctry, dft->name, path1.exten);
         return EXIT_SUCCESS;
         }
      }
   return EXIT_FAILURE;
   }


/***********************************************************************/
/***********************************************************************/
int usage( void )
   {
   printf ("\nOverwrites and then deletes one or more files.\n"
           "Although the file might be recovered, the data in it is\n"
           "not recoverable because it has been overwritten.\n\n"
           "SECERASE [drive:][path]filename [/L] [/N] [/P]\n\n"
           "  [drive:][path]filename  Specifies the file(s) to be erased and deleted.\n"
           "                          Specify multiple files by using wildcards.\n"
           "  /L                      Reports the name of each file deleted and how\n"
           "                          many bytes of disk space were freed.\n"
           "  /N                      Erases the file but does not delete it.\n"
           "  /P                      Prompts for confirmation before erasing each file.\n\n");
   return EXIT_SUCCESS;
   }

/***********************************************************************/
/***********************************************************************/
int main (int argc, char *argv[])
   {

   unsigned long dsize, tdsize;
   enum boolean {YES, NO} flag_path, flag_pause, flag_usage, flag_bad, flag_log, flag_delete;
   unsigned int j, num_switches, file_count;
   int i, rc;
   char nc, delpath[_MAX_PATH], *const dpath=delpath, *path, *save;
   char outbuf[81];
  
   flag_pause = NO;     /* YES = prompt for each file found        */
   flag_delete= YES;    /* YES = Will delete file after erasing it */
   flag_log   = NO;     /* YES = report each file deleted          */
   flag_usage = NO;     /* YES = Display usage text only           */
   flag_bad   = NO;     /* YES = Invalid switch encountered        */
   flag_path  = NO;     /* YES = Have found and saved filespec     */
   rc = EXIT_SUCCESS;
   num_switches = 0;
   file_count   = 0;
   tdsize       = 0L;

   for (i=1; i<argc; i++)
      {

      /*
       Is this argument a switch?
      */
      if (*argv[i] != '/') 
         {
         /*
          Yes, have we already encountered a filespec?
         */
         if (flag_path == YES)
            {
            fprintf(stderr, "Too many parameters - %s\n", argv[i]);
            rc = EXIT_FAILURE;
            }
         else
            {
            path = argv[i];
            save = argv[i];
            /*
              Convert the characters in the specified path to 
              upper case.
            */
            for (j=0; j<strlen(save); j++)
               *argv[i]++ = (char) toupper(*path++);
            argv[i] = save;
            path = argv[i];
            flag_path = YES;
            }
         continue;
         }

      /*
        Set flags indicating which switches were specified on the 
        command line.
      */
      nc = (char) toupper(*(argv[i]+ sizeof (**argv)));

      switch (nc)
         {
         case '?': flag_usage=YES;
                   num_switches++;
                   break;

         case 'L': flag_log=YES;
                   num_switches++;
                   break;

         case 'N': flag_delete=NO;
                   num_switches++;
                   break;

         case 'P': flag_pause=YES;
                   num_switches++;
                   break;

         default:  flag_bad = YES;
                   num_switches++;
                   fprintf(stderr, "Invalid switch - %s\n", argv[i]);
                   rc = EXIT_FAILURE;
         }
      }   

   /*
     If the /? switch was specified then display the usage 
     message and do nothing else
   */
   if (flag_usage == YES)
      {
      usage();
      rc = EXIT_FAILURE;
      }

   /*
     If the number of switches specified is one less than argc 
     then no filenames were specified.
   */
   if (++num_switches == (unsigned int) argc)
      {
      fprintf(stderr, "Required parameter missing\n");
      rc = EXIT_FAILURE;
      }

   /* 
      If there were any errors found while parsing the command 
      line then do nothing.
   */
   if (rc != EXIT_SUCCESS) return EXIT_FAILURE;

   /*
     Now find and erase each file matching the pattern in path
   */
   while (get_dpath(dpath, path) == EXIT_SUCCESS)
      {
      if (flag_pause == YES)
         {
         fprintf(stderr, "%s    Erase (Y/N)? ", dpath);
         fflush(stdin);

         if (toupper(fgetchar()) == 'Y')
            {
            rc = secerase(dpath, &dsize);

            if ((flag_delete == YES) && (rc == EXIT_SUCCESS))
               {
               if (remove(dpath) == 0)
                  {
                  file_count++;
                  tdsize += dsize;
                  }

               if (flag_log == YES) printf("%s deleted, %lu bytes freed.\n", dpath, dsize);
               }
            }
         }
      else
         {
         rc = secerase(dpath, &dsize);

         if ((flag_delete == YES) && (rc == EXIT_SUCCESS))
            {
            if (remove(dpath) == 0)
               {
               file_count++;
               tdsize += dsize;
               }

            if (flag_log == YES) printf("%s deleted, %lu bytes freed.\n", dpath, dsize);
            }
         }
      }
   if (flag_log == YES) printf("Total of %u files deleted, %lu bytes freed.\n", file_count, tdsize);
   return rc;
   }

