;============================================================================
; (c) Copyright Elect Software International Inc., 1992, Toronto. Anyone can
; use this code for anything as long as it is not resold as a software
; development resource, as long as the copyright notice isn't removed, as
; long as changes are clearly marked as to authorship, and as long as users
; indemnify Elect from any liability.
; Comments welcome. Henrik Bechmann, CIS:72701,3717; Tel:416-534-8176.
;============================================================================

; Tracer Version 1.03 October 31, 1992.
;
;============================================================================
;                              TRACER DESCRIPTION
;============================================================================
;
; Tracer is an add-in for EventMan (found in EVENTMAN.SC). It traces events
; (EventMan.EventTag) generated by EventMan. It works by in setting
; EventMan.DispatchEventTagProc to "Tracer.TraceEvent", writing the current
; EventMan.EventTag to the tracer window and/or file, and then EXECPROCing
; the original EventMan.DispatchEventTagProc.
;
; Load Tracer by simply issuing Tracer.Constructor() after your application
; has set EventMan.DispatchEventTagProc to its own event dispatcher (or has
; left the default value). EventMan.Constuctor() sets a default handler of
; "-20" for [AltT] in EventMan.Dictionary[] to call the Tracer Menu, so if
; you over-ride EventMan.DispatchEventTagProc with your own, your dispatcher
; should be written to recognize such default events.
;
; Unload Tracer with Tracer.Destructor()
;
; See EDEMO.SC for an example of the use of Tracer.
;
;============================================================================
;                              TRACER INTERFACE
;============================================================================

; Tracer.Constructor()
; Tracer.Destructor()
;
; Tracer.OpenEventTrace()
; Tracer.CloseEventTrace()
; Tracer.OpenEventFile()
; Tracer.CloseEventFile()
; Tracer.SaveAndCloseFile()
;
; Tracer.ControlMenu()
; Tracer.TraceControl()
; Tracer.TraceEvent()
; Tracer.TraceEventToWindow(ObjectTag,EventTag)
;
; Tracer.TraceWindowHandle
; Tracer.TraceWindowIsActive
; Tracer.EventFileSpec
; Tracer.FileWindowHandle
; Tracer.FileWindowIsActive
; Tracer.IsActive
;
;==============================================================================
;                           TRACER IMPLEMENTATION
;==============================================================================

Proc Tracer.Constructor()
   Private
      i

   ;-------------------------------------------------------------------------
   ; Replace default high-level EventMan handler with trace handler...
   ;-------------------------------------------------------------------------
   Tracer_OriginalDispatchProc = EventMan.DispatchEventTagProc
   EventMan.DispatchEventTagProc = "Tracer.TraceEvent"
   ;-------------------------------------------------------------------------
   ; Initialize variables...
   ;-------------------------------------------------------------------------
   Tracer.EventFileSpec = "Events.txt"
   Tracer.FileWindowIsActive = False
   Tracer.TraceWindowIsActive = False
   ;-------------------------------------------------------------------------
   ; Tracelist is used to write trace information to the event tracer canvas...
   ;-------------------------------------------------------------------------
   Array Tracer.TraceList[8]
   For i From 1 to 8
      Tracer.TraceList[i] = ""
   EndFor
   ;-------------------------------------------------------------------------
   ; Control variables for the tracer procs...
   ;-------------------------------------------------------------------------
   Tracer_TraceTrigger = True
   Tracer_TraceMouse = True
   Tracer_TraceKeycode = True
   Tracer_TraceMessage = True
   Tracer_IsRecordOn = False
   Tracer_EventNumber = 0
   Dynarray Tracer_DefaultProcBag[]
   Tracer_DefaultProcBag[-20] = "Tracer.ControlMenu" ; AltT
   EventMan.SetHandlersFrom(Tracer_DefaultProcBag)
   Tracer.IsActive = True

EndProc ; Tracer.Constructor

Proc Tracer.Destructor()
   Message "Quitting."
   If Tracer.FileWindowIsActive Then
      Tracer.SaveAndCloseFile()
   Endif
   If Tracer.TraceWindowIsActive Then
      Tracer.CloseEventTrace()
   Endif
   EventMan.DispatchEventTagProc = Tracer_OriginalDispatchProc
   EventMan.ClearHandlersWith(Tracer_DefaultProcBag)
   Release Vars
      Tracer_OriginalDispatchProc,
      Tracer.EventFileSpec,
      Tracer.TraceWindowHandle,
      Tracer.TraceWindowIsActive,
      Tracer.FileWindowHandle,
      Tracer.FileWindowIsActive,
      Tracer.IsActive,
      Tracer_DefaultProcBag,
      Tracer_TraceProcBag,
      Tracer_FileProcBag,
      Tracer_TraceTrigger,
      Tracer_TraceMouse,
      Tracer_TraceKeycode,
      Tracer_TraceMessage,
      Tracer_IsRecordOn,
      Tracer_EventNumber

EndProc ; Tracer.Destructor

; ---------------------------- TRACER WINDOW --------------------------------

Proc Tracer.OpenEventTrace()
   Private
      WindowBag

   Dynarray WindowBag[]
   WindowBag["Width"] = 38
   WindowBag["Height"] = 10
   If Tracer_IsRecordOn Then
      WindowBag["Title"] = "Event tracer:Record is off"
   Else
      WindowBag["Title"] = "Event tracer:Record is off"
   Endif
   WindowBag["OriginRow"] = 13
   WindowBag["OriginCol"] = 1
   Window Create Attributes WindowBag To Tracer.TraceWindowHandle
   Tracer.TraceWindowIsActive = True

   Dynarray Tracer_TraceProcBag[]
   Tracer_TraceProcBag["Close"] = "Tracer.CloseEventTrace"
   Tracer_TraceProcBag["Up"] = "Tracer.CallControlMenu"
   Tracer_TraceProcBag["ObjectTag"] = StrVal(Tracer.TraceWindowHandle)
   EventMan.SetHandlersFrom(Tracer_TraceProcBag)

   Return 1
EndProc ; Tracer.OpenEventTrace

Proc Tracer.CloseEventTrace()
   Private
      CurrentWindow,
      i

   CurrentWindow = GetWindow()
   Window Select Tracer.TraceWindowHandle
   Window Close
   Tracer.TraceWindowIsActive = False
   EventMan.ClearHandlersWith(Tracer_TraceProcBag)
   For i From 1 to 8
      Tracer.TraceList[i] = ""
   Endfor
   If IsWindow(CurrentWindow) Then
      Window Select CurrentWindow
   Endif
   Return 1
EndProc ; Tracer.CloseEventTrace

;----------------------------- EVENTS.TXT WINDOW ----------------------------

Proc Tracer.OpenEventFile()
   Private
      WindowBag

   If IsFile(Tracer.EventFileSpec) Then
      Editor Open Tracer.EventFileSpec
   Else
      Editor New Tracer.EventFileSpec
   Endif
   Tracer.FileWindowHandle = GetWindow()
   Tracer.FileWindowIsActive = True

   Dynarray WindowBag[]
   WindowBag["Width"] = 38
   WindowBag["Height"] = 10
   WindowBag["OriginRow"] = 13
   WindowBag["OriginCol"] = 1
   Window SetAttributes Tracer.FileWindowHandle From WindowBag

   Dynarray Tracer_FileProcBag[]
   Tracer_FileProcBag["Close"] = "Tracer.CloseEventFile"
   Tracer_FileProcBag["Up"] = "Tracer.CallControlMenu"
   Tracer_FileProcBag[Asc("F2")] = "Tracer.SaveAndCloseFile"
   Tracer_FileProcBag["ObjectTag"] = StrVal(Tracer.FileWindowHandle)
   EventMan.SetHandlersFrom(Tracer_FileProcBag)

   Return 1
EndProc ; Tracer.OpenEventFile

Proc Tracer.CloseEventFile()
   Private
      CurrentWindow,
      Closed
   CurrentWindow = GetWindow()
   Window Select Tracer.FileWindowHandle
   Closed = Tracer!CloseEditorWindow()
   If Closed Then
      Tracer.FileWindowIsActive = False
      EventMan.ClearHandlersWith(Tracer_FileProcBag)
   Endif
   If IsWindow(CurrentWindow) Then
      Window Select CurrentWindow
   Endif
   Return 1
EndProc ; Tracer.CloseEventFile

Proc Tracer.SaveAndCloseFile()
   Private
      CurrentWindow
   CurrentWindow = GetWindow()
   Window Select Tracer.FileWindowHandle
   Do_It!
   Tracer.FileWindowIsActive = False
   EventMan.ClearHandlersWith(Tracer_FileProcBag)
   If IsWindow(CurrentWindow) Then
      Window Select CurrentWindow
   Endif
   Return 1
EndProc ; Tracer.SaveAndCloseFile

;------------------------------ CONTROL MENUS -------------------------------

Proc Tracer.CallControlMenu()
   If EventMan.MouseModeButton = "RIGHT" Then
      Return Tracer.ControlMenu()
   Else
      Return 0
   Endif
EndProc

Proc Tracer.ControlMenu()
   Private
      Command,
      ReturnCode
   ShowPopup "Tracer" Centered
      "Tr~a~ce settings":"Select events to record":"Trace",
      Separator,
      "~S~how traces":"Show traces window":"Show",
      "~V~iew EVENTS.TXT":"View the EVENTS.TXT file":"View",
      Separator,
      "~Q~uit tracer":"Quit the event tracer":"Quit"
   EndMenu
   To Command
   Switch
      Case Command = "Esc":
         ReturnCode = 1
      Case Command = "Trace":
         Tracer.TraceControl()
         ReturnCode = 1
      Case Command = "Show":
         If Not Tracer.TraceWindowIsActive Then
            Echo Off
            Tracer.OpenEventTrace()
            Echo Normal
         Else
            Message "TRACE window already on desktop..."
         Endif
         ReturnCode = 1
      Case Command = "View":
         If Not Tracer.FileWindowIsActive Then
            Echo Off
            Tracer.OpenEventFile()
            Echo Normal
         Else
            Message "EVENTS.TXT already on desktop..."
         Endif
         ReturnCode = 1
      Case Command = "Quit":
         ShowMenu
            "Cancel":"Do NOT close the event tracer",
            "OK":"Close the event tracer"
         To Command
         If Command <> "OK" Then
            ReturnCode = 1
         Else
            Tracer.Destructor()
            ReturnCode = 2
            Message "Tracer is unloaded."
         Endif
   EndSwitch
   Return ReturnCode
EndProc ; Tracer.ControlMenu

Proc Tracer.TraceControl()
   Private
      Control
   ShowDialog "Set trace elements"
      @5,20 Height 12 Width 33
      CheckBoxes
         @1,3 Height 5 Width 25 Tag "Traces"
         "~T~rigger" To Tracer_TraceTrigger,
         "~M~ouse"   To Tracer_TraceMouse,
         "~K~ey"     To Tracer_TraceKeycode,
         "Me~s~sage" To Tracer_TraceMessage,
         "~R~ecord to EVENTS.TXT" To Tracer_IsRecordOn
      PushButton
         @8,2 Width 10 "~O~K" OK
         Default
         Value True
         Tag "OK"
         To Control
      PushButton
         @8,18 Width 10 "~C~ancel" Cancel
         Value True
         Tag "Cancel"
         To Control
   EndDialog
   If Retval Then
      If Tracer.TraceWindowIsActive Then
         Dynarray WindowBag[]
         If Tracer_IsRecordOn Then
            WindowBag["Title"] = "Event tracer:Record is on"
         Else
            WindowBag["Title"] = "Event tracer:Record is off"
         Endif
         Window SetAttributes Tracer.TraceWindowHandle From WindowBag
      Endif
   Endif
   Return 1
EndProc ; Tracer.TraceControl

;------------------------------ UTILITY PROCS -------------------------------

Proc Tracer!CloseEditorWindow()
   Private
      EditorWindow,
      EventBag
   EditorWindow = GetWindow()
   WinClose
   While IsWindow(EditorWindow) And (GetWindow() <> EditorWindow)
      ;--------------------------------------------------------------
      ; MenuChoice() could be used to detect presence of menu, ie. "No"
      ; if the confirmation menu were up, and "Error" or "View" or "File"
      ; or something if not, kind of mushy though, so I prefer the above.
      ;--------------------------------------------------------------
      ; ie. we're on a confirmation menu, so eat events until
      ; accept or deny of winclose is indirectly confirmed. Accept is
      ; detected by absence of EditorWindow; deny is detected by
      ; return to the EditorWindow.
      ;--------------------------------------------------------------
      GetEvent to EventBag
      ExecEvent EventBag
   EndWhile
   If IsWindow(EditorWindow) Then
      Return false
   Else
      Return true
   Endif
EndProc ; Tracer!CloseEditorWindow

; -------------------------- TRACE PROC -------------------------------------

;----------------------------------------------------------------------------
; Requires EventMan.EventType, and EventMan.TriggerTag or EventMan.EventBag
;----------------------------------------------------------------------------
Proc Tracer.TraceEvent()
   Private
      Action,
      Active

   If EventMan.EventTag <> "IDLE" And
      Not (EventMan.EventType = "MOUSE" and EventMan.MouseModeButton = "NONE")
      Then
      Tracer.TraceEventToWindow(EventMan.TargetWindow, EventMan.EventTag)
   Endif
   ExecProc Tracer_OriginalDispatchProc
   Return EventMan.ReturnCode

EndProc ; Tracer.TraceEvent

; ---------------------- WRITE EVENT TO TRACE WINDOW ------------------------

Proc Tracer.TraceEventToWindow(ObjectTag,EventTag)
   Private
      EventTrace,
      i
   If (Not Tracer.TraceWindowIsActive) And
      (Not Tracer_IsRecordOn) Then
      Return
   Endif
   Tracer_EventNumber = Tracer_EventNumber + 1
   EventTrace = Format("W6",Tracer_EventNumber) + " " +
                Format("W6",ObjectTag) + " " + EventTag
   If isAssigned(EventMan.IsWait) Then
      EventTrace = EventTrace + "(" + StrVal(EventMan.TriggerCycle) + ")"
   Endif
   If Tracer.TraceWindowIsActive Then
      SetCanvas Tracer.TraceWindowHandle
      Canvas off
      For i From 2 to 8
         Tracer.TraceList[i-1] = Tracer.TraceList[i]
      EndFor
      Tracer.TraceList[8] = EventTrace
      Clear
      For i From 0 to 7
         @i,0 ?? Tracer.TraceList[i+1]
      EndFor
      Canvas on
   Endif
   If Tracer_IsRecordOn And (Not Tracer.FileWindowIsActive) Then
      Print File  Tracer.EventFileSpec
         EventTrace + "\n"
   Endif
EndProc ; Tracer.TraceEventToWindow