;==============================================================================
; (c) Copyright Elect Software International Inc., 1992, Toronto. Anyone can
; use this code for anything as long as it is not resold as a software
; development resource, as long as the copyright notice isn't removed, as
; long as changes are clearly marked as to authorship, and as long as users
; indemnify Elect from any liability.
; Comments welcome. Henrik Bechmann, CIS:72701,3717; Tel:416-534-8176.
;==============================================================================

;==============================================================================
;                             HELPMAN INTERFACE
;==============================================================================

; Methods:
; HelpMan.Constructor()
; HelpMan.Interact()
; HelpMan.Destructor()

; Properties:
; HelpMan.FormColorBag[]
; HelpPanel.ColorBag[]

;==============================================================================
;                           HELPMAN IMPLEMENTATION
;==============================================================================

Proc HelpMan.Constructor()
   EventMan.Constructor()
   EventMan.SetObjectTagProc = "EventMan.SetIndexedObjectTag"

   HelpMan.Directory = "Generic\\helpman\\data\\"
   HelpMan.CodeDirectory = "Generic\\helpman\\source\\"
   HelpMan.HelpTabName = "help"
   HelpMan.HelpListTabName = "helplist"
   HelpMan.HelpSpecTabName = "helpspec"

   Dynarray HelpMan.FormColorBag[]
   HelpMan.FormColorBag[0] = 8 + 112
   HelpMan.FormColorBag[1] = 8 + 112
   HelpMan.FormColorBag[2] = 8 + 112
   HelpMan.FormColorBag[3] = 8 + 112
   HelpMan.FormColorBag[4] = 8 + 112

   HelpMan_WasListForm = True

   HyperMemo.Constructor()
   HelpTable.Constructor()
   ListTable.Constructor()
   ListForm.Constructor()
   SpecTable.Constructor()
   SpecForm.Constructor()
   HelpPanel.Constructor()
   HelpGen.Constructor()
   HyperMemo.SetFocusProc = "HelpPanel.SetFocus"

   Dynarray HelpMan.HandlerBag[]
   HelpMan.HandlerBag["LeftDown"]       = "HelpMan.SelectWindow"
   HelpMan.HandlerBag["RightDown"]      = "HelpMan.SelectWindow"
   HelpMan.HandlerBag["BothDown"]       = "HelpMan.SelectWindow"
   HelpMan.HandlerBag[Asc("Dos")]       = "EventMan.DisableEvent"
   HelpMan.HandlerBag[Asc("DosBig")]    = "EventMan.DisableEvent"
   HelpMan.HandlerBag[Asc("WinNext")]   = "HelpMan.WinNext"
   EventMan.SetHandlersFrom(HelpMan.HandlerBag) ; sets default handlers

EndProc ; HelpMan.Constructor

Proc HelpMan.Destructor()

   Message "Ending Help System Maintenance..."

   If IsFieldView() Then
      FieldView
   Endif
   If IsFormView() Then
      FormKey
   Endif
   EventMan.ClearHandlersWith(HelpMan.HandlerBag)
   HyperMemo.Destructor()
   HelpTable.Destructor()
   ListTable.Destructor()
   ListForm.Destructor()
   SpecTable.Destructor()
   SpecForm.Destructor()
   HelpPanel.Destructor()
   HelpGen.Destructor()

   Release Vars
      HelpMan.FormColorBag,
      HelpMan.HelpTabName,
      HelpMan_WasListForm,
      HelpMan.Directory,
      HelpMan.CodeDirecory,
      HelpMan.HelpSpecTabName,
      HelpMan.HelpListTabName,
      HelpMan.HandlerBag

   While IsWindow(GetWindow())
      Window Close
   EndWhile
   EventMan.Destructor()
EndProc ; HelpMan.Destructor

Proc HelpMan.SelectWindow()
   Private
      WindowBag,
      EditorBag,
      WindowHandle
   If IsWindow(EventMan.TargetWindow) And
      GetWindow() <> EventMan.TargetWindow Then
      If SysMode() = "CoEdit" Then
         If EventMan.WindowObjectTag[EventMan.TargetWindow] = "ListForm" Then
            Message "Cannot select list form during edit"
            Return 1
         Endif
         If EventMan.WindowObjectTag[GetWindow()] = "HelpPanel" Then
            Window GetAttributes GetWindow() To WindowBag
            Editor Info To EditorBag
            Window Handle Form to WindowHandle
            Window Select WindowHandle
            HelpPanel.UpdateReferences(WindowBag,EditorBag)
            Window Select EventMan.TargetWindow
         Endif
      Endif
      EventMan.ObjectTag = EventMan.WindowObjectTag[EventMan.TargetWindow]
      EventMan.EventTag = "SetFocus"
      ExecProc EventMan.DispatchEventTagProc
      Return EventMan.ReturnCode
   Else
      Return 0
   Endif
EndProc ; HelpMan.SelectWindow

Proc HelpMan.WinNext()
   Private
      TargetWindow,
      WindowBag,
      EditorBag

   If IsWindow(EventMan.ObjectTagWindow["HelpPanel"]) And
      GetWindow() <> EventMan.ObjectTagWindow["HelpPanel"] Then
      HelpPanel.SetFocus()
   Else
      Window Handle Form To TargetWindow
      If Not IsWindow(TargetWindow) Then
         TargetWindow = EventMan.ObjectTagWindow[HelpPanel.FormName]
      Endif
      If TargetWindow <> GetWindow() Then ; This is a memo
         If SysMode() = "CoEdit" Then
            Window GetAttributes GetWindow() To WindowBag
            Editor Info To EditorBag
            Window Select TargetWindow
            HelpPanel.UpdateReferences(WindowBag,EditorBag)
         Endif
         EventMan.ObjectTag = EventMan.WindowObjectTag[TargetWindow]
         EventMan.EventTag = "SetFocus"
         ExecProc EventMan.DispatchEventTagProc
         Return EventMan.ReturnCode
      Else
         Return 1
      Endif
   Endif
EndProc ; HelpMan.WinNext

Proc HelpMan.HideWindow()
   Window Move GetWindow() To 500,500
EndProc ; HelpMan.HideWindow

;--------------------------- HELP MAINTENANCE -------------------------------

Proc HelpMan.Interact()

   ListForm.SetFocus()

   Echo Normal
   Wait WorkSpace
      Proc "EventMan.CatchTrigger"
   EndWait

EndProc ; HelpMan.Interact

Proc HelpMan.Quit()
   Private
      Command
   ShowPopUp "Quit help system?" Centered
      "No":"Do NOT quit editing help Panels":"Cancel",
      "Yes":"Quit editing help Panels":"OK"
   EndMenu
   To Command
   If Command = "OK" Then
      Return 2
   Else
      Return 1
   Endif
EndProc ; HelpMan.Quit

;---------------------------- HELPTABLE -------------------------------------

Proc HelpTable.Constructor()
   View HelpMan.Directory + HelpMan.HelpTabName
   EventMan.WindowObjectTag[GetWindow()] = "HelpTable"
   EventMan.ObjectTagWindow["HelpTable"] = GetWindow()
   HelpMan.HideWindow()
EndProc ; HelpTable.Constructor

Proc HelpTable.Destructor()
   Moveto HelpMan.Directory + HelpMan.HelpTabName
   ClearImage
   Release Vars
      EventMan.WindowObjectTag[EventMan.ObjectTagWindow["HelpTable"]],
      EventMan.ObjectTagWindow["HelpTable"]
EndProc ; HelpTable.Destructor

;---------------------------- LISTTABLE -------------------------------------

Proc ListTable.Constructor()
   Moveto HelpMan.Directory + HelpMan.HelpListTabName
   EventMan.WindowObjectTag[GetWindow()] = "ListTable"
   EventMan.ObjectTagWindow["ListTable"] = GetWindow()
   Dynarray ListTable.HandlerBag[]
   ListTable.HandlerBag[Asc("EditKey")] = "ListTable.Edit"
   ListTable.HandlerBag["ObjectTag"] = "ListTable"
   EventMan.SetHandlersFrom(ListTable.HandlerBag)
EndProc ; ListTable.Constructor

Proc ListTable.Destructor()
   EventMan.ClearHandlersWith(ListTable.HandlerBag)
   Release vars
      ListTable.HandlerBag,
      EventMan.WindowObjectTag[EventMan.ObjectTagWindow["ListTable"]],
      EventMan.ObjectTagWindow["ListTable"]
EndProc ; ListTable.Destructor

Proc ListTable.Edit()
   ExecProc HelpPanel.FormName + ".Edit"
EndProc ; ListTable.Edit

;---------------------------- LISTFORM --------------------------------------

Proc ListForm.Constructor()
   EventMan.ObjectTagWindow["ListForm"] = 0
   Dynarray ListForm.HandlerBag[]
   ListForm.HandlerBag[Asc("Esc")]        = "HelpMan.Quit"
   ListForm.HandlerBag[Asc("F1")]         = "ListForm.CallExpand"
   ListForm.HandlerBag[Asc("EditKey")]    = "ListForm.Edit"
   ListForm.HandlerBag[Asc("FieldView")]  = "ListForm.FieldView"
   ListForm.HandlerBag[Asc("F11")]        = "HelpPanel.Display"
   ListForm.HandlerBag[-108]              = "ListForm.FieldView" ; AltF5
   ListForm.HandlerBag["LeftDoubleDown"]  = "ListForm.CallExpand"
   ListForm.HandlerBag["Close"]           = "HelpMan.Quit"
   ListForm.HandlerBag["MenuTag:Quit"]    = "HelpMan.Quit"
   ListForm.HandlerBag["MenuTag:Specs"]   = "ListForm.Expand"
   ListForm.HandlerBag["MenuTag:Edit"]    = "ListForm.Edit"
   ListForm.HandlerBag["MenuTag:Display"] = "HelpPanel.Display"
   ListForm.HandlerBag["MenuTag:Generate"] = "ListForm.Generate"
   ListForm.HandlerBag["MenuTag:TableBased"] = "ListForm.TableBasedDemo"
   ListForm.HandlerBag["MenuTag:CodeBased"] = "ListForm.CodeBasedDemo"
   ListForm.HandlerBag["MenuTag:RunLib"] = "ListForm.RunLib"
   ListForm.HandlerBag["ArriveRow"]       = "ListForm.ArriveRow"
   ListForm.HandlerBag["SetFocus"]        = "ListForm.SetFocus"
   ListForm.HandlerBag["ObjectTag"]       = "ListForm"
   EventMan.SetHandlersFrom(ListForm.HandlerBag)
EndProc ; ListForm.Constructor

Proc ListForm.Destructor()
   EventMan.ClearHandlersWith(ListForm.HandlerBag)
   Release vars
      ListForm.LibrarySpec,
      ListForm.PanelName,
      ListForm.HandlerBag,
      EventMan.WindowObjectTag[EventMan.ObjectTagWindow["ListForm"]],
      EventMan.ObjectTagWindow["ListForm"]
EndProc ; ListForm.Destructor

Proc ListForm.SetFocus()
   Private
      WindowBag
   If IsWindow(EventMan.ObjectTagWindow["ListForm"]) Then
      Window Select EventMan.ObjectTagWindow["ListForm"]
      If Table() <> HelpMan.Directory + HelpMan.HelpListTabName Then
         DownImage
      Endif
   Else
      Message "Working..."
      Echo off
      If IsFormView() Then
         FormKey
      Endif
      Window Select EventMan.ObjectTagWindow["HelpTable"]
      PickForm 1
      Release Vars
         EventMan.WindowObjectTag[EventMan.ObjectTagWindow["ListForm"]]
      EventMan.ObjectTagWindow["ListForm"] = GetWindow()
      EventMan.WindowObjectTag[GetWindow()] = "ListForm"
;      Dynarray WindowBag[]
;      WindowBag["CanMaximize"] = False
;      WindowBag["CanResize"] = False
;      Window SetAttributes GetWindow() From WindowBag
      Window SetColors GetWindow() From HelpMan.FormColorBag
      DownImage
      Echo normal
      Message ""
   Endif
   If SysMode() = "Main" Then
      ShowPullDown
         "HelpPanel":"View help text":"Display",
         "Specs":"View help Panel specification details":"Specs",
         "Edit":"Edit the help panel specs":"Edit",
;        "Demo":"Demonstrate the two forms of presenting help screens":"Demo"
;           SubMenu
;              "Code based":"Code based hypertext demo":"CodeBased",
;              "Table based":"Table based hypertext demo":"TableBased"
;           EndSubMenu,
         "Generate":"Generate library for help panels":"Generate",
         "RunLib":"Run generated help screens":"RunLib",
         "Quit":"Quit editing help Panels":"Quit"
      EndMenu
      Prompt "[Esc]-Quit Help Maintenance; [F1]-Help Panel detail; " +
                "[ShiftF1]-Help panel"
   Else
      Debug ; Not in main mode on list
   Endif
   Return 1
EndProc ; ListForm.SetFocus

Proc ListForm.CallExpand()
   If Substr(FieldType(),1,1) <> "M" Then
      Return ListForm.Expand()
   Else
      Return HelpPanel.Display()
   Endif
EndProc ; ListForm.CallExpand

Proc ListForm.FieldView()
   If Substr(FieldType(),1,1) = "M" Then
      Return HelpPanel.Display()
   Else
      Return 0
   Endif
EndProc ; ListForm.FieldView

Proc ListForm.Expand()
   If Table() <> HelpMan.Directory + HelpMan.HelpListTabName Then
      DownImage
   Endif
   If Recno() > 0 Then
      SpecForm.SetFocus()
   Else
      Message "No records to expand"
   Endif
   Return 1
EndProc ; ListForm.Expand

Proc ListForm.ArriveRow()
   Private
      WindowBag
   If Table() = HelpMan.Directory + HelpMan.HelpListTabName And
      IsWindow(EventMan.ObjectTagWindow["HelpPanel"]) Then
      Dynarray WindowBag[]
      WindowBag["Title"] = [Help Title]
      Window SetAttributes EventMan.ObjectTagWindow["HelpPanel"] From WindowBag
      If SysMode() = "Main" Then
         HyperMemo.SetStack()
      Endif
   Endif
   Return 0
EndProc ; ListForm.ArriveRow

Proc ListForm.Edit()
   Private
      WasHelpPanel
   HelpMan_WasListForm = True
   WasHelpPanel = (GetWindow() = EventMan.ObjectTagWindow["HelpPanel"])
   If WasHelpPanel Then
      ListForm.SetFocus()
   Endif
   If Recno() = 0 Then
      UpImage
   Endif
   CoEditKey
   SpecForm.SetFocus()
   SpecForm.SetImageRights()
   If WasHelpPanel Then
      HelpPanel.SetFocus()
   Endif
   Return 1
EndProc ; ListForm.Edit()

Proc ListForm.Generate()
   If HelpGen.GetDirName() Then
      FormKey
      HelpGen.GenerateScript()
      ListForm.SetFocus()
   Endif
   Return 1
EndProc ; ListForm.Generate

Proc ListForm.TableBasedDemo()
   FormKey
   HyperMemo.SetFocusProc = "HyperMemo!SetFocus"
   HyperMemo.ShowPanel("HTEXT_BASICS")
   HyperMemo.SetFocusProc = "HelpPanel.SetFocus"
   ListForm.SetFocus()
   Return 1
EndProc ; ListForm.TableBasedDemo

Proc ListForm.CodeBasedDemo()
   Private
      Autolib
   FormKey
   Autolib = "Help\\helpdemo\\helpdemo"
   Hypertext.Constructor()
   Hypertext.ShowPanel("HTEXT_BASICS")
   Hypertext.Destructor()
   ListForm.SetFocus()
   Return 1
EndProc ; ListForm.TableBasedDemo

Proc ListForm.RunLib()
   Private
      Autolib,
      ListForm_ErrorCode,
      ErrorProc
   If ListForm!GetLibSpec() Then
      Autolib = Substr(ListForm.LibrarySpec,1,Len(ListForm.LibrarySpec) - 4)
      ErrorProc = "ListForm!RunLibErrorProc"
      ListForm_ErrorCode = 0
      Hypertext.StartWithPanel(ListForm.PanelName)
      If ListForm_ErrorCode = 34 Then
         Message ListForm.LibrarySpec," is an invalid hypertext help library"
      Endif
      ListForm.SetFocus()
   Endif
   Return 1
EndProc ; ListForm.RunLib

Proc ListForm!RunLibErrorProc()
   ListForm_ErrorCode = ErrorCode()
   Return 1
EndProc ; ListForm!RunLibErrorProc

Proc ListForm!GetLibSpec()
   Private
      ListForm.LibControlVar
   ShowDialog ""
      Proc "ListForm!GetLibProc" Trigger "Accept"
      @5,20 Height 8 Width 41
      @1,3 ?? "Enter library directory and name:"
      Accept @2,3 Width 33 "A64" Picture "*{?,#,-,_,%,&,;?,\\,.,:}" Required
         Tag "Library"
         To ListForm.LibrarySpec
      PushButton @4,5 Width 12 "~O~K"
         OK Default
         Value "OK"
         Tag "OK" To
         ListForm.LibControlVar
      PushButton @4,22 Width 12 "~C~ancel"
         Cancel
         Value "Cancel"
         Tag "Cancel" To
         ListForm.LibControlVar
   EndDialog
   Return Retval
EndProc ; ListForm!GetLibSpec

Proc ListForm!GetLibProc(TriggerType,TagValue,EventValue,ElementValue)
   If Len(ListForm.LibrarySpec) < 4 or
      Upper(Substr(ListForm.LibrarySpec,Len(ListForm.LibrarySpec) - 3,4)) <>
         ".LIB" Then
      ListForm.LibrarySpec = ListForm.LibrarySpec + ".LIB"
      RefreshControl("Library")
   Endif
   Switch
      Case DirExists(ListForm.LibrarySpec) = -1:
         Message "Error in library specification"
         SelectControl "Library"
         Return False
      Case Not IsFile(ListForm.LibrarySpec):
         Message "Library does not exist"
         SelectControl "Library"
         Return False
      Otherwise:
         If Not ListForm!GetPanelName() Then
            SelectControl "Library"
            Return False
         Else
            Return True
         Endif
    EndSwitch
EndProc ; ListForm!GetLibProc

Proc ListForm!GetPanelName()
   Private
      ListForm.PanelControlVar
   ShowDialog ""
      @9,24 Height 8 Width 41
      @1,3 ?? "Enter starting panel name:"
      Accept @2,3 Width 33 "A12" Picture "*{&,.,$,;!,_,#}" Required
         Tag "Panel"
         To ListForm.PanelName
      PushButton @4,5 Width 12 "~O~K"
         OK Default
         Value "OK"
         Tag "OK" To
         ListForm.PanelControlVar
      PushButton @4,22 Width 12 "~C~ancel"
         Cancel
         Value "Cancel"
         Tag "Cancel" To
         ListForm.PanelControlVar
   EndDialog
   Return Retval
EndProc ; ListForm!GetPanelName

;---------------------------- SPECTABLE -------------------------------------

Proc SpecTable.Constructor()
   Moveto HelpMan.Directory + HelpMan.HelpSpecTabName
   EventMan.WindowObjectTag[GetWindow()] = "SpecTable"
   EventMan.ObjectTagWindow["SpecTable"] = GetWindow()
EndProc ; SpecTable.Constructor

Proc SpecTable.Destructor()
   Release Vars
      EventMan.WindowObjectTag[EventMan.ObjectTagWindow["SpecTable"]],
      EventMan.ObjectTagWindow["SpecTable"]
EndProc ; SpecTable.Destructor

;---------------------------- SPECFORM --------------------------------------

Proc SpecForm.Constructor()
   EventMan.ObjectTagWindow["SpecForm"] = 0
   Dynarray SpecForm.HandlerBag[]
   SpecForm.HandlerBag[Asc("Esc")]        = "SpecForm.CallEscape"
   SpecForm.HandlerBag[Asc("F1")]         = "SpecForm.Expand"
   SpecForm.HandlerBag[Asc("Ins")]        = "SpecForm.Insert"
   SpecForm.HandlerBag[Asc("Del")]        = "SpecForm.Delete"
   SpecForm.HandlerBag[Asc("Undo")]       = "SpecForm.Undo"
   SpecForm.HandlerBag[Asc("EditKey")]    = "SpecForm.Edit"
   SpecForm.HandlerBag[Asc("Do_It!")]     = "SpecForm.Commit"
   SpecForm.HandlerBag[Asc("FieldView")]  = "SpecForm.FieldView"
   SpecForm.HandlerBag[Asc("F11")]        = "HelpPanel.Display"
   SpecForm.HandlerBag[-108]              = "SpecForm.FieldView" ; AltF5
   SpecForm.HandlerBag["LeftDoubleDown"]  = "SpecForm.Expand"
   SpecForm.HandlerBag["Close"]           = "ListForm.SetFocus"
   SpecForm.HandlerBag["MenuTag:Quit"]    = "HelpMan.Quit"
   SpecForm.HandlerBag["MenuTag:List"]    = "ListForm.SetFocus"
   SpecForm.HandlerBag["MenuTag:Insert"]  = "SpecForm.Insert"
   SpecForm.HandlerBag["MenuTag:Delete"]  = "SpecForm.Delete"
   SpecForm.HandlerBag["MenuTag:Undo"]    = "SpecForm.Undo"
   SpecForm.HandlerBag["MenuTag:Edit"]    = "SpecForm.Edit"
   SpecForm.HandlerBag["MenuTag:DO-IT!"]  = "SpecForm.Commit"
   SpecForm.HandlerBag["MenuTag:Cancel"]  = "SpecForm.AbandonEdit"
   SpecForm.HandlerBag["MenuTag:Display"] = "HelpPanel.Display"
   SpecForm.HandlerBag["ArriveRow"]       = "SpecForm.ArriveRow"
   SpecForm.HandlerBag["Valcheck"]        = "SpecForm.ResolveNewRef"
   SpecForm.HandlerBag["PostRecord"]      = "SpecForm.CheckDomain"
   SpecForm.HandlerBag["TouchRecord"]     = "SpecForm.TouchRecord"
   SpecForm.HandlerBag["SetFocus"]        = "SpecForm.SetFocus"
   SpecForm.HandlerBag["ObjectTag"]       = "SpecForm"
   EventMan.SetHandlersFrom(SpecForm.HandlerBag)
EndProc ; SpecForm.Constructor

Proc SpecForm.Destructor()
   EventMan.ClearHandlersWith(SpecForm.HandlerBag)
   Release vars
      SpecForm.Record,
      SpecForm.HandlerBag,
      EventMan.WindowObjectTag[EventMan.ObjectTagWindow["SpecForm"]],
      EventMan.ObjectTagWindow["SpecForm"]
EndProc ; SpecForm.Destructor

Proc SpecForm.SetFocus()
   Private
      WindowBag
   If IsWindow(EventMan.ObjectTagWindow["SpecForm"]) Then
      Window Select EventMan.ObjectTagWindow["SpecForm"]
   Else
      Message "Working..."
      Echo off
      If IsFormView() Then
         FormKey
      Endif
      Window Select EventMan.ObjectTagWindow["ListTable"]
      If SysMode() = "CoEdit" And RecNo() = 0 Then
         Ins
      Endif
      PickForm 2
      Release Vars
         EventMan.WindowObjectTag[EventMan.ObjectTagWindow["SpecForm"]]
      EventMan.ObjectTagWindow["SpecForm"] = GetWindow()
      EventMan.WindowObjectTag[GetWindow()] = "SpecForm"
;      Dynarray WindowBag[]
;      WindowBag["CanMaximize"] = False
;      WindowBag["CanResize"] = False
;      Window SetAttributes GetWindow() From WindowBag
      Window SetColors GetWindow() From HelpMan.FormColorBag
      Echo normal
      Message ""
   Endif
   If SysMode() = "Main" Then
      ShowPullDown
         "HelpPanel":"View help text":"Display",
         "List":"View the help Panel list":"List",
         "Edit":"Edit the selected help Panel":"Edit",
         "Quit":"Quit editing help Panels":"Quit"
      EndMenu
      Prompt "[Esc]-previous panel; [ShiftF1]-Help panel"
   Else
      ShowPullDown
         "Insert":"Insert a new help Panel":"Insert",
         "D~e~lete":"Delete the selected help Panel":"Delete",
         "Panel":"Edit help text":"Display",
         "DO-IT!":"Return from edit mode.":"DO-IT!",
         "Cancel":"Cancel edit":"Cancel",
         "Undo":"Undo last change":"Undo"
      EndMenu
      Prompt "[Esc]-Undo last edit change; [F2]-Do-it!"
   Endif
   Return 1
EndProc ; SpecForm.SetFocus

Proc SpecForm.CallEscape()
   If Sysmode() = "Main" Then
      Return ListForm.SetFocus()
   Else
      Return SpecForm.AbandonEdit()
   Endif
EndProc ; SpecForm.CallEscape

Proc SpecForm.Expand()
   If Table() = HelpMan.Directory + HelpMan.HelpSpecTabName Then
      If Recno() > 0 Then
         If Sysmode() = "CoEdit" And
            Field() = "Referenced help panel" Then
            Return 0
         Else
            Code = [Referenced help panel]
            UpImage
            Moveto [Help panel name]
            Locate Code
            DownImage
            Return 1
         Endif
      Endif
   Else
      If Field() = "Help text" Then
         HelpPanel.Display()
      Endif
      Return 1
   Endif
EndProc ; SpecForm.Expand

Proc SpecForm.FieldView()
   If Substr(FieldType(),1,1) = "M" Then
      Return HelpPanel.Display()
   Else
      Return 0
   Endif
EndProc ; SpecForm.FieldView

;------------------------- SPECFORM EDIT PROCS ------------------------------

Proc SpecForm.Edit()
   Private
      WasHelpPanel
   If SysMode() = "Main" Then
      HelpMan_WasListForm = False
      WasHelpPanel = (GetWindow() = EventMan.ObjectTagWindow["HelpPanel"])
      CoEditKey
      SpecForm.SetFocus()
      SpecForm.SetImageRights()
      If WasHelpPanel Then
         HelpPanel.SetFocus()
      Endif
   Endif
   Return 1
EndProc ; SpecForm.Edit()

Proc SpecForm.ArriveRow()
   ListForm.ArriveRow()
   Return SpecForm.SetImageRights()
EndProc ; SpecForm.ArriveRow

Proc SpecForm.SetImageRights()
   If SysMode() = "CoEdit" Then
      If Table() = HelpMan.Directory + HelpMan.HelpListTabName Then
         If RecordStatus("New") Then
            ImageRights
         Else
            ImageRights Update
         Endif
      Endif
   Endif
   Return 0
EndProc ; SpecForm.SetImageRights

Proc SpecForm.ResolveNewRef()
   Private
      ControlVar,
      PanelName,
      CurrentPanel,
      Keyword

   If Field() <> "Referenced help panel" Then
      Return 0
   Endif
   ShowDialog "Warning"
      @10,20 Height 9 Width 45
      @1,1 ?? "The following Panel name does not exist:"
      @3,1 ?? [Referenced help Panel]
      PushButton
         @5,10 Width 10
         "~A~dd"
         OK
         Default
         Value "Add"
         Tag "Add"
         To ControlVar
      PushButton
         @5,25 Width 10
         "~R~eturn"
         Cancel
         Value "Return"
         Tag "Return"
         To ControlVar
   EndDialog
   If RetVal Then
      PanelName = [Referenced help panel]
      KeyWord = [Keyword]
      CtrlBackSpace
      UpImage
      CurrentPanel = [Help Panel name]
      ImageRights
      Ins
      [Help panel name] = PanelName
      Moveto [Help Panel name]
      Locate Indexorder CurrentPanel
      DownImage
      Locate IndexOrder CurrentPanel,Keyword
      Moveto [Referenced help panel]
      [Referenced help panel] = PanelName
      Return 0
   Else
      Return 1
   Endif
EndProc ; SpecForm.ResolveNewRef

Proc SpecForm.TouchRecord()
   If Not IsFieldView() Then
      CopyToArray SpecForm.Record
      If RecordStatus("New") Then
         [Wordwrap] = "Y"
      Endif
   Else
      Echo Off
      FieldView
      CopyToArray SpecForm.Record
      FieldView
      Echo Normal
   Endif
   Return 0
EndProc ; SpecForm.TouchRecord

Proc SpecForm.CheckDomain()
   Private
      OK,
      ReturnCode

   ReturnCode = 0
   If Not (RecordStatus("New") And (Not RecordStatus("Modified"))) Then
      Switch
         Case table() = HelpMan.Directory + HelpMan.HelpListTabName:
            If IsBlank([Help panel name]) Then
               Message "Cannot post with blank panel name field"
               ReturnCode = 1
            Endif
         Case table() = HelpMan.Directory + HelpMan.HelpSpecTabName:
            OK = true
            If Field() = "Referenced help Panel" Then
               If Not IsValid() Then
                  OK = (HelpMan!ResolveNewRef() = 0)
                  If Not OK Then
                     ReturnCode = 1
                  Endif
               Endif
            Endif
            If OK And IsBlank([Keyword]) Then
               OK = False
               Message "Cannot post with blank keyword field"
               ReturnCode = 1
            Endif
            If OK And [Help Panel name] = [Referenced help panel] Then
               Message "Cannot link help Panel to itself"
               ReturnCode = 1
            Endif
            If OK And
               RecordStatus("Modified") And (Not RecordStatus("New")) And
               [Keyword] <> SpecForm.Record["Keyword"] And
               [Status] = "Active" Then
               Message "Cannot change active keyword"
               ReturnCode = 1
            Endif
         Otherwise:
            ReturnCode = 0
      EndSwitch
   Endif
   Return ReturnCode
EndProc ; SpecForm.CheckDomain

Proc SpecForm.PostRecord()
   Private
      OK
   OK = True
   If SysMode() = "CoEdit" And RecordStatus("Modified") Then
      PostRecord NoPost LeaveLocked
      If Not Retval Then
         Message "Key exists -- you can't overwrite master and details"
         OK = False
      Endif
   Endif
   Return OK
EndProc ; SpecForm.PostRecord

Proc SpecForm.Insert()
   If (SpecForm.CheckDomain() = 0) And SpecForm.PostRecord() Then
      ImageRights
      Ins
   Endif
   Return 1
EndProc ; SpecForm.Insert

Proc SpecForm.Delete()
   Private
      Command,
      ReturnCode,
      OK
   If SysMode() <> "CoEdit" Then
      Return 1
   Endif
   ReturnCode = 1
   If (Not IsFieldView()) And (Not RecordStatus("New")) Then
      If Table() = HelpMan.Directory + HelpMan.HelpListTabName Then
         ShowPopup "Delete panel record?" Centered
            "No":"Do NOT delete current Panel record":"No",
            "Yes":"Delete current Panel record":"Yes"
         EndMenu
         To Command
         If Command = "Yes" Then
            OK = SpecForm.IntegrityCheck()
            If OK Then
               Moveto HelpMan.Directory + HelpMan.HelpSpecTabName
               ImageRights
               While Not RecordStatus("New")
                  Del
               EndWhile
               Moveto HelpMan.Directory + HelpMan.HelpListTabName
               ImageRights
               Del
            Else
               Message "Cannot delete -- other panels use this panel"
               ReturnCode = 1
            Endif
         Else
            ReturnCode = 1
         Endif
      Else
         If [Status] = "Active" Then
            Message "Cannot delete an active reference"
            ReturnCode = 1
         Else
            Del
         Endif
      Endif
   Else
      If RecordStatus("New") Then
         ImageRights
         Del
         SpecForm.SetImagerights()
         ReturnCode = 1
      Endif
   Endif
   Return ReturnCode
EndProc ; SpecForm.Delete

Proc SpecForm.IntegrityCheck()
   Private
      PanelName,
      OK
   OK = True
   If Not RecordStatus("New") Then
      If RecordStatus("Modified") Then
         OK = SpecForm.PostRecord()
      Endif
      If OK Then
         UnlockRecord
         PanelName = [Help Panel name]
         FormKey
         Moveto HelpMan.Directory + HelpMan.HelpSpecTabName
         Moveto [Referenced help panel]
         Locate IndexOrder PanelName
         OK = Not Retval
         Moveto HelpMan.Directory + HelpMan.HelpListTabName
         FormKey
      Endif
   Endif
   Return OK
EndProc ; SpecForm.IntegrityCheck

Proc SpecForm.Undo()
   If SysMode() = "CoEdit" Then
      If Not IsFieldView() Then
         ShowPopup "Undo changes?" Centered
            "No":"Do NOT undo change":"No",
            "Yes":"Undo change":"Yes"
         EndMenu
         To Command
         If Command = "Yes" Then
            ImageRights
            Menu {Undo}{Yes}
            SpecForm.SetImageRights()
         Endif
      Else
         Message "End Field View before Undoing."
      Endif
   Endif
   Return 1
EndProc ; SpecForm.Undo

Proc SpecForm.Commit()
   Private
      OK,
      CurrentTable,
      ReturnCode
   ReturnCode = 0
   If SysMode() = "CoEdit" Then
      If Not (RecordStatus("New") And (Not RecordStatus("Modified"))) Then
         OK = (SpecForm.CheckDomain() = 0)
         If OK Then
            OK = SpecForm.PostRecord()
         Endif
      Else
         ImageRights
         Del
         OK = true
      Endif
      If OK Then
         ImageRights
         CurrentTable = Table()
         Do_It!
         If HelpMan_WasListForm Then
            ListForm.SetFocus()
         Else
            Moveto CurrentTable
            SpecForm.SetFocus()
         Endif
         ReturnCode = 1
      Else
         ReturnCode = 1
      Endif
   Endif
   Return ReturnCode
EndProc ; SpecForm.Commit

Proc SpecForm.AbandonEdit()
   Private
      OK
   If RecordStatus("Modified") Then
      ShowPopUp "Abandon edit?" Centered
         "No":"Do NOT undo edit changes":"No",
         "Yes":"Undo edit changes":"Yes"
      EndMenu
      To Command
      If Command = "Yes" Then
         ImageRights
         Menu {Undo}{Yes}
         OK = True
      Else
         OK = False
      Endif
   Else
      OK = True
   Endif
   If OK Then
      Do_It!
      If HelpMan_WasListForm Then
         ListForm.SetFocus()
      Else
         SpecForm.SetFocus()
      Endif
   Endif
   Return 1
EndProc ; SpecForm.AbandonEdit

;----------------------------- PANEL ----------------------------------------

Proc HelpPanel.Constructor()
   EventMan.ObjectTagWindow["HelpPanel"] = 0
   Dynarray HelpPanel.HandlerBag[]
   HelpPanel.HandlerBag[Asc("Do_It!")]        = "HelpPanel.CloseWithSave"
   HelpPanel.HandlerBag[Asc("Enter")]         = "HelpPanel.SelectHotSpot"
   HelpPanel.HandlerBag[Asc("Help")]          = "HelpPanel.SelectHotSpot"
   HelpPanel.HandlerBag[Asc("Esc")]           = "HelpPanel.CallEscape"
   HelpPanel.HandlerBag[19]                   = "HelpPanel.SetHotSpotStart" ; ^S
   HelpPanel.HandlerBag[5]                    = "HelpPanel.SetHotSpotEnd" ; ^E
   HelpPanel.HandlerBag[Asc("FieldView")]     = "HelpMan.WinNext"
   HelpPanel.HandlerBag[Asc("Tab")]           = "HelpPanel.GotoNextHotSpot"
   HelpPanel.HandlerBag[Asc("ReverseTab")]    = "HelpPanel.GotoPrevHotSpot"
   HelpPanel.HandlerBag[-108]                 = "HelpMan.WinNext" ; AltF5
   HelpPanel.HandlerBag["RightDown"]          = "HelpPanel.CallEscape"
   HelpPanel.HandlerBag["LeftDoubleDown"]     = "HelpPanel.SelectHotSpot"
   HelpPanel.HandlerBag["Close"]              = "HelpPanel.Close"
   HelpPanel.HandlerBag["MenuTag:List"]       = "HelpPanel.SelectList"
   HelpPanel.HandlerBag["MenuTag:Specs"]      = "HelpPanel.SelectSpecs"
   HelpPanel.HandlerBag["MenuTag:Edit"]       = "HelpPanel.Edit"
   HelpPanel.HandlerBag["MenuTag:Quit"]       = "HelpMan.Quit"
   HelpPanel.HandlerBag["MenuTag:DO-IT!"]     = "HelpPanel.CloseWithSave"
   HelpPanel.HandlerBag["MenuTag:Start"]      = "HelpPanel.SetHotSpotStart"
   HelpPanel.HandlerBag["MenuTag:End"]        = "HelpPanel.SetHotSpotEnd"
   HelpPanel.HandlerBag["MenuTag:Cancel"]     = "HelpPanel.Close"
   HelpPanel.HandlerBag["MenuTag:Save"]       = "HelpPanel.Save"
   HelpPanel.HandlerBag["MenuTag:CopyToFile"] = "HelpPanel.CopyToFile"
   HelpPanel.HandlerBag["MenuTag:InsertFile"] = "HelpPanel.InsertFile"
   HelpPanel.HandlerBag["MenuTag:WriteBlock"] = "HelpPanel.WriteBlock"
   HelpPanel.HandlerBag["MenuTag:Print"]      = "HelpPanel.Print"
   HelpPanel.HandlerBag["SetFocus"]           = "HelpPanel.SetFocus"
   HelpPanel.HandlerBag["ObjectTag"]          = "HelpPanel"
   EventMan.SetHandlersFrom(HelpPanel.HandlerBag)

   Dynarray HelpPanel.ColorBag[]
   HelpPanel.ColorBag[0] = 8 + 32
   HelpPanel.ColorBag[1] = 8 + 32
   HelpPanel.ColorBag[2] = 8 + 32
   HelpPanel.ColorBag[3] = 8 + 32
   HelpPanel.ColorBag[4] = 8 + 32
EndProc ; HelpPanel.Constructor

Proc HelpPanel.Destructor()
   EventMan.ClearHandlersWith(HelpPanel.HandlerBag)
   Release vars
      HelpPanel.Name,
      HelpPanel.FormName,
      HelpPanel.ColorBag,
      HelpPanel.HandlerBag,
      EventMan.WindowObjectTag[EventMan.ObjectTagWindow["HelpPanel"]],
      EventMan.ObjectTagWindow["HelpPanel"]
EndProc ; HelpPanel.Destructor

Proc HelpPanel.SetFocus()
   Private
      WindowBag
   If IsWindow(EventMan.ObjectTagWindow["HelpPanel"]) Then
      If Table() <> HelpMan.Directory + HelpMan.HelpListTabName Then
         Moveto HelpMan.Directory + HelpMan.HelpListTabName
      Endif
      If HelpPanel.Name <> [Help panel name] Then
         HelpPanel.Name = [Help panel name]
         Echo off
         If SysMode() = "Main" And IsFormView() Then
            HelpPanel.FormName = EventMan.WindowObjectTag[GetWindow()]
            FormKey
         Endif
         Window Select EventMan.ObjectTagWindow["HelpPanel"]
         HelpPanel.SetAttributes()
         Echo normal
      Else
         Window Select EventMan.ObjectTagWindow["HelpPanel"]
      Endif
   Else
      Message "Working..."
      Echo off
      If SysMode() = "Main" And IsFormView() Then
         HelpPanel.FormName = EventMan.WindowObjectTag[GetWindow()]
         Formkey
      Endif
      Moveto HelpMan.Directory + HelpMan.HelpListTabName
      HelpPanel.Name = [Help panel name]
      Moveto [Help text]
      FieldView
      HelpPanel.SetAttributes()
      If SysMode() = "Main" Then
         HyperMemo.SetStack()
      Endif
      Window SetColors GetWindow() From HelpPanel.ColorBag
      Release Vars
         EventMan.WindowObjectTag[EventMan.ObjectTagWindow["HelpPanel"]]
      EventMan.ObjectTagWindow["HelpPanel"] = GetWindow()
      EventMan.WindowObjectTag[GetWindow()] = "HelpPanel"
      Echo Normal
      Message ""
   Endif
   If SysMode() = "Main" Then
      ShowPullDown
         "List":"View the help panel list":"List",
         "Specs":"View the help panel specs":"Specs",
         "Edit":"Edit the selected help Panel":"Edit",
         "Quit":"Quit editing help Panels":"Quit"
      EndMenu
      Prompt "[Esc]-return to help Panel list; [ShiftF1]-Explain"
      HyperMemo.CollectHotSpots()
   Else
      ShowPullDown
         "~S~ave":"Save changes to memo field":"Save",
         "~C~opyToFile":"Copy text to a file":"CopyToFile",
         "~I~nsertFile":"Insert file in text":"InsertFile",
         "~W~riteBlock":"Write marked block to file":"WriteBlock",
         "~P~rint":"Print text":"Print",
         "~H~otspot":"Mark start or end of hotspot":"HotSpot"
            SubMenu
               "Start":"Mark start of hotspot":"Start",
               "End":"Mark end of hotspot":"End"
            EndSubMenu,
         "~D~O-IT!": "Save changes to the help text":"DO-IT!",
         "C~a~ncel":"Cancel edit":"Cancel"
      EndMenu
      Prompt "[F2]-Accept; [Esc]-Reject; [^S]-Start hotspot; [^E]-End hotspot",""
   Endif
   Return 1
EndProc ; HelpPanel.SetFocus

Proc HelpPanel.Edit()
   If HelpPanel.FormName = "ListForm" Then
      Return ListForm.Edit()
   Else
      Return SpecForm.Edit()
   Endif
EndProc ; HelpPanel.Edit

Proc HelpPanel.SetAttributes()
   Private
      WindowBag
   Message "Setting window attributes..."
   If [Wordwrap] = "Y" Then
      Menu {Options}{WordWrap}{Set}
   Else
      Menu {Options}{WordWrap}{Clear}
   Endif
   Dynarray WindowBag[]
   WindowBag["Title"] = [Help title]
   If Not IsBlank([Origin row]) Then
      WindowBag["OriginRow"] = [Origin row]
   Endif
   If Not IsBlank([Origin col]) Then
      WindowBag["OriginCol"] = [Origin col]
   Endif
   If Not IsBlank([Height]) Then
      WindowBag["Height"] = [Height]
   Endif
   If Not IsBlank([Width]) Then
      WindowBag["Width"] = [Width]
   Endif
   Window SetAttributes GetWindow() From WindowBag
EndProc ; HelpPanel.SetAttributes

Proc HelpPanel.SelectList()
   Private
      WindowHandle
   Window Handle Form To WindowHandle
   If Not IsWindow(WindowHandle) Then
      WindowHandle = EventMan.ObjectTagWindow[HelpPanel.FormName]
   Endif
   EventMan.ObjectTag = EventMan.WindowObjectTag[WindowHandle]
   EventMan.EventTag = "SetFocus"
   ExecProc EventMan.DispatchEventTagProc
   If EventMan.WindowObjectTag[GetWindow()] = "ListForm" Then
      Return EventMan.ReturnCode
   Else
      Return ListForm.SetFocus()
   Endif
EndProc ; HelpPanel.SelectList

Proc HelpPanel.SelectSpecs()
   Private
      WindowHandle
   Window Handle Form To WindowHandle
   If Not IsWindow(WindowHandle) Then
      WindowHandle = EventMan.ObjectTagWindow[HelpPanel.FormName]
   Endif
   EventMan.ObjectTag = EventMan.WindowObjectTag[WindowHandle]
   EventMan.EventTag = "SetFocus"
   ExecProc EventMan.DispatchEventTagProc
   If EventMan.WindowObjectTag[GetWindow()] = "SpecForm" Then
      Return EventMan.ReturnCode
   Else
      Return SpecForm.SetFocus()
   Endif
EndProc ; HelpPanel.SelectSpecs

Proc HelpPanel.Display()
   Private
      OK
   Moveto HelpMan.Directory + HelpMan.HelpListTabName
   If IsBlank([Help Panel name]) Then
       Message "Panel name must exist before editing Panel"
   Else
      If SysMode() = "CoEdit" Then
         OK = (SpecForm.CheckDomain() = 0) And SpecForm.PostRecord()
         If OK Then
            HelpPanel.SetFocus()
         Endif
      Else
         HelpPanel.SetFocus()
      Endif
   Endif
   Return 1
EndProc ; HelpPanel.Display

Proc HelpPanel.SelectHotSpot()
   If SysMode() = "Main" Then
      Return HyperMemo.SelectHotSpot()
   Else
      Return 0
   Endif
EndProc ; HelpPanel.SelectHotSpot

Proc HelpPanel.GotoNextHotSpot()
   If SysMode() = "Main" Then
      Return HyperMemo.GotoNextHotSpot()
   Else
      Return 0
   Endif
EndProc ; HelpPanel.GotoNextHotSpot

Proc HelpPanel.GotoPrevHotSpot()
   If SysMode() = "Main" Then
      Return HyperMemo.GotoPrevHotSpot()
   Else
      Return 0
   Endif
EndProc ; HelpPanel.GotoPrevHotSpot

Proc HelpPanel.CallEscape()
   If SysMode() = "Main" Then
      If Not HyperMemo.PrevPanel() Then
         Return HelpPanel.Close()
      Else
         Return 1
      Endif
   Else
      Return HelpPanel.Close()
   Endif
EndProc ; HelpPanel.CallEscape

Proc HelpPanel.Close()
   If HelpPanel.CloseWindow() Then
      If Not IsFormView() Then
         EventMan.ObjectTag = HelpPanel.FormName
      Else
         EventMan.ObjectTag = EventMan.WindowObjectTag[GetWindow()]
      Endif
      EventMan.EventTag = "SetFocus"
      ExecProc EventMan.DispatchEventTagProc
      Return EventMan.ReturnCode
   Else
      Return 1
   Endif
EndProc ; HelpPanel.Close

Proc HelpPanel.CloseWindow()
   Private
      EditorWindow,
      EventBag
   EditorWindow = GetWindow()
   WinClose
   While IsWindow(EditorWindow) And (GetWindow() <> EditorWindow)
      ;--------------------------------------------------------------
      ; MenuChoice() could be used to detect presence of menu, ie. "No"
      ; if the confirmation menu were up, and "Error" or "View" or "File"
      ; or something if not, kind of mushy though, so I prefer the above.
      ;--------------------------------------------------------------
      ; ie. we're on a confirmation menu, so eat events until
      ; accept or deny of winclose is indirectly confirmed. Accept is
      ; detected by absence of EditorWindow; deny is detected by
      ; return to the EditorWindow.
      ;--------------------------------------------------------------
      GetEvent to EventBag
      ExecEvent EventBag
   EndWhile
   If IsWindow(EditorWindow) Then
      Return false
   Else
      Return true
   Endif
EndProc ; HelpPanel.CloseWindow

Proc HelpPanel.CloseWithSave()
   Private
      WindowBag,
      EditorBag
   If SysMode() = "CoEdit" Then
      Window GetAttributes GetWindow() To WindowBag
      Editor Info To EditorBag
      Do_It!
      HelpPanel.UpdateReferences(WindowBag,EditorBag)
   Else
      Do_It!
   Endif
   If IsWindow(GetWindow()) Then
      If Not IsFormView() Then
         EventMan.ObjectTag = HelpPanel.FormName
      Else
         EventMan.ObjectTag = EventMan.WindowObjectTag[GetWindow()]
      Endif
      EventMan.EventTag = "SetFocus"
      ExecProc EventMan.DispatchEventTagProc
      Return EventMan.ReturnCode
   Else
      Return 1
   Endif
EndProc ; HelpPanel.CloseWithSave

Proc HelpPanel.UpdateReferences(WindowBag,EditorBag)
   Private
      EditorBag,
      PanelName,
      HelpText,
      ScanPos,
      LeftMarkPos,
      RightMarkPos,
      Found,
      Scanning,
      Length,
      WorkString,
      HotSpots,
      WasFormView,
      OK

   If Not EditorBag["Changed"] Then
      Return
   Endif
   Message "Working..."
   Dynarray HotSpots[]
   PanelName = [Help Panel name]
   HelpText = [Help text]
   WasFormView = IsFormView()
   If WasFormView Then
      If RecordStatus("Locked") Or
         (RecordStatus("New") And RecordStatus("Modified")) Then
         UnlockRecord
      Endif
      FormKey
   Endif
   Moveto HelpMan.Directory + HelpMan.HelpSpecTabName
   ScanPos = 1
   Scanning = True
   While Scanning
      LeftMarkPos = SearchFrom("",HelpText,ScanPos)
      Found = (LeftMarkPos <> 0)
      Scanning = Found
      If Found Then
         ScanPos = LeftMarkPos
         RightMarkPos = SearchFrom("",HelpText,ScanPos)
         Found = (RightMarkPos <> 0)
         If Found Then
            Length = RightMarkPos - LeftMarkPos + 1 - 2
            OK = (Length <= 32 And Length > 0)
            If OK Then
               WorkString = Substr(HelpText,LeftMarkPos + 1,Length)
               OK = (Substr(WorkString,1,1) <> " ")
               If OK Then
                  OK = (Substr(WorkString,Length,1) <> " ")
                  If OK Then
                     OK = (Search("\010",WorkString) = 0)
                  Endif
               Endif
            Endif
            If OK Then
               Locate PanelName, WorkString
               If Not Retval Then
                  Imagerights
                  Ins
                  [Help Panel name] = PanelName
                  [Keyword] = WorkString
               Endif
               [Status] = "Active"
               ScanPos = RightMarkPos + 1
               HotSpots[WorkString] = True
               UnlockRecord
            Else
               ScanPos = LeftMarkPos + 1
            Endif
         Endif ; Found
      Endif
   EndWhile
   Scan For [Help Panel name] = PanelName
      If Not IsAssigned(HotSpots[[Keyword]]) Then
         [Status] = "Absent"
      EndIf
   EndScan
   Moveto HelpMan.Directory + HelpMan.HelpListTabName
   [Origin row] = WindowBag["OriginRow"]
   [Origin col] = WindowBag["OriginCol"]
   [Height] = WindowBag["Height"]
   [Width] = WindowBag["Width"]
   UnlockRecord
   If WasFormView Then
      FormKey
   Endif
   Message ""
EndProc ; HelpPanel.UpdateReferences

Proc HelpPanel.SetHotSpotStart()
   If SysMode() = "CoEdit" Then
      Editor insert ""
   Endif
   Return 1
EndProc ; HelpPanel.SetHotSpotStart

Proc HelpPanel.SetHotSpotEnd()
   If SysMode() = "CoEdit" Then
      Editor insert ""
   Endif
   Return 1
EndProc ; HelpPanel.SetHotSpotEnd

Proc HelpPanel.Save()
   Private
      WindowBag,
      EditorBag
   Echo off
   Window GetAttributes GetWindow() To WindowBag
   Editor Info To EditorBag
   Menu {File}{Save}
   FieldView
   HelpPanel.UpdateReferences(WindowBag,EditorBag)
   Moveto [Help text]
   FieldView
   Echo Normal
   Return 1
EndProc ; HelpPanel.Save

Proc HelpPanel.CopyToFile()
   Menu {File}{CopyToFile}
   HelpPanel.ProgressThroughMenu()
   Return 1
EndProc ; HelpPanel.CopyToFile

Proc HelpPanel.InsertFile()
   Menu {File}{InsertFile}
   HelpPanel.ProgressThroughMenu()
   Return 1
EndProc ; HelpPanel.InsertFile

Proc HelpPanel.WriteBlock()
   Menu {File}{WriteBlock}
   HelpPanel.ProgressThroughMenu()
   Return 1
EndProc ; HelpPanel.WriteBlock

Proc HelpPanel.Print()
   If PrinterStatus() Then
      Echo off
      Menu {File}{Print}
      Echo Normal
   Else
      Message "Printer not ready"
   Endif
   Return 1
EndProc ; HelpPanel.Print

Proc HelpPanel.ProgressThroughMenu()
   While GetWindow() = 0
      GetEvent To EventBag
      If EventBag["Type"] = "KEY" And
         EventBag["Keycode"] = Asc("Esc") Then
         CtrlBreak
         Quitloop
      Else
         ExecEvent EventBag
      Endif
   EndWhile
EndProc ; HelpPanel.ProgressThroughMenu

;-------------------------- MAINLINE ----------------------------------------

If Not IsAssigned(Librarian.HasControl) Then
   Play "PopHelp"
   HelpMan.Constructor()
   HelpMan.Interact()
   HelpMan.Destructor()
Endif