/* cdrom.c:
 *
 * Basic interaction with the CDROM drive.
 *
 * ----------------------------------------------------------------------
 * This code is (C) Copyright 1993 by Frank Munkert.
 * All rights reserved.
 * This software may be freely distributed and redistributed for
 * non-commercial purposes, provided this notice is included.
 */

#include <clib/exec_protos.h>
#include <clib/alib_protos.h>

#include "cdrom.h"

CDROM *Open_CDROM (char *p_device, int p_scsi_id)
{
  CDROM *cd;
  int i;

  cd = AllocMem (sizeof (CDROM), MEMF_PUBLIC | MEMF_CHIP | MEMF_CLEAR);
  if (!cd)
    return NULL;

  cd->buffers[0] = AllocMem (SCSI_BUFSIZE * SCSI_BUFFERS,
  			     MEMF_PUBLIC | MEMF_CHIP);
  if (!cd->buffers[0]) {
    FreeMem (cd, sizeof (CDROM));
    return NULL;
  }

  for (i=1; i<SCSI_BUFFERS; i++)
    cd->buffers[i] = cd->buffers[0] + i * SCSI_BUFSIZE;

  cd->port = CreatePort (NULL, 0);
  if (!cd->port) {
    FreeMem (cd->buffers[0], SCSI_BUFSIZE * SCSI_BUFFERS);
    FreeMem (cd, sizeof (CDROM));
    return NULL;
  }

  cd->scsireq = CreateStdIO (cd->port);
  if (!cd->scsireq) {
    DeletePort (cd->port);
    FreeMem (cd->buffers[0], SCSI_BUFSIZE * SCSI_BUFFERS);
    FreeMem (cd, sizeof (CDROM));
    return NULL;
  }

  if (OpenDevice ((UBYTE *) p_device, p_scsi_id,
                  (struct IORequest *) cd->scsireq, 0)) {
    DeleteStdIO (cd->scsireq);
    DeletePort (cd->port);
    FreeMem (cd->buffers[0], SCSI_BUFSIZE * SCSI_BUFFERS);
    FreeMem (cd, sizeof (CDROM));
    return NULL;
  }

  for (i=0; i<SCSI_BUFFERS; i++)
    cd->current_sectors[i] = -1;
  
  return cd;
}

int Do_SCSI_Command (CDROM *p_cd, unsigned char *p_buf, long p_buf_length,
		     unsigned char *p_command, int p_length)
{
  p_cd->scsireq->io_Length   = sizeof (struct SCSICmd);
  p_cd->scsireq->io_Data     = (APTR) &p_cd->cmd;
  p_cd->scsireq->io_Command  = HD_SCSICMD;

  p_cd->cmd.scsi_Data        = (UWORD *) p_buf;
  p_cd->cmd.scsi_Length      = p_buf_length;
  p_cd->cmd.scsi_Flags       = SCSIF_AUTOSENSE | SCSIF_READ;
  p_cd->cmd.scsi_SenseData   = (UBYTE *) p_cd->sense;
  p_cd->cmd.scsi_SenseLength = 18;
  p_cd->cmd.scsi_SenseActual = 0;
  p_cd->cmd.scsi_Command     = (UBYTE *) p_command;
  p_cd->cmd.scsi_CmdLength   = p_length;

  DoIO ((struct IORequest *) p_cd->scsireq);
  if (p_cd->cmd.scsi_Status) {
    int i;
    for (i=0; i<SCSI_BUFFERS; i++)
      p_cd->current_sectors[i] = -1;
    return 0;
  } else
    return 1;
}

int Read_Sector_With_Lookahead (CDROM *p_cd, long p_sector, long p_last_sector)
{
  static unsigned char cmd[10] = { 0x28, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
  int status;
  int i;
  static int loc = 0;

  for (i=0; i<SCSI_BUFFERS; i++)
    if (p_cd->current_sectors[i] == p_sector) {
      p_cd->buffer = p_cd->buffers[i];
      return 1;
    }

  cmd[5] = p_sector & 0xff;
  cmd[4] = (p_sector >> 8) & 0xff;
  cmd[3] = (p_sector >> 16) & 0x1f;

  if (p_sector >= p_last_sector) {

    loc++;
    if (loc == SCSI_BUFFERS)
      loc=0;

    cmd[8] = 1;

    status = Do_SCSI_Command (p_cd, p_cd->buffers[loc], SCSI_BUFSIZE,
    			      cmd, sizeof (cmd));
    if (status) {
      p_cd->current_sectors[loc] = p_sector;
      p_cd->buffer = p_cd->buffers[loc];
    }
  
  } else {

    int len = p_last_sector - p_sector + 1;
    
    if (len > SCSI_BUFFERS)
      len = SCSI_BUFFERS;

    cmd[8] = len;
    
    status = Do_SCSI_Command (p_cd, p_cd->buffers[0], SCSI_BUFSIZE * len,
    			      cmd, sizeof (cmd));
    if (status) {
      for (i=0; i<len; i++)
        p_cd->current_sectors[i] = p_sector + i;
      p_cd->buffer = p_cd->buffers[0];
    }

  }

  return status;
}

int Read_Sector (CDROM *p_cd, long p_sector)
{
  return Read_Sector_With_Lookahead (p_cd, p_sector, p_sector);
}

int Test_Unit_Ready (CDROM *p_cd)
{
  static unsigned char cmd[6] = { 0, 0, 0, 0, 0, 0 };

  p_cd->current_sectors[0] = -1;
  return Do_SCSI_Command (p_cd, p_cd->buffers[0], SCSI_BUFSIZE, cmd, 6);
}

void Cleanup_CDROM (CDROM *p_cd)
{
  CloseDevice ((struct IORequest *) p_cd->scsireq);
  DeleteStdIO (p_cd->scsireq);
  DeletePort (p_cd->port);
  FreeMem (p_cd->buffers[0], SCSI_BUFSIZE * SCSI_BUFFERS);
  FreeMem (p_cd, sizeof (CDROM));
}
