/* UDP-related user commands and other functions
 * Copyright 1991 Phil Karn, KA9Q
 *
 * put into one file - DB3FL.920912
 */

#include "global.h"
#include "config.h"
#ifdef UDP
#include "mbuf.h"
#include "netuser.h"
#include "iface.h"
#include "udp.h"
#include "ip.h"
#include "internet.h"
#include "icmp.h"
#include "cmdparse.h"

/* UDP control structures list */
static struct udp_cb *Udps;

static struct mib_entry Udp_mib[] = {
	"",				0,
	"InDatagrams",	0,
	"NoPorts",		0,
	"InErrors",		0,
	"OutDatagrams",	0,
};

/* Look up UDP socket.
 * Return control block pointer or NULLUDP if nonexistant
 * As side effect, move control block to top of list to speed future
 * searches.
 */
static struct udp_cb * near
lookup_udp(struct socket *socket)
{
	struct udp_cb *up, *uplast = NULLUDP;

	for(up = Udps; up != NULLUDP; uplast = up, up = up->next) {
		if(socket->port == up->socket.port
		  && (socket->address == up->socket.address
		  || up->socket.address == INADDR_ANY)) {
			if(uplast != NULLUDP) {
				/* Move to top of list */
				uplast->next = up->next;
				up->next = Udps;
				Udps = up;
			}
			return up;
		}
	}
	return NULLUDP;
}

/* Create a UDP control block for lsocket, so that we can queue
 * incoming datagrams.
 */
struct udp_cb *
open_udp(struct socket *lsocket,void (*r_upcall)())
{
	struct udp_cb *up;

	if((up = lookup_udp(lsocket)) == NULLUDP) {
		up = mxallocw(sizeof (struct udp_cb));

		up->socket.address = lsocket->address;
		up->socket.port = lsocket->port;
		up->r_upcall = r_upcall;

		up->next = Udps;
		Udps = up;
	}
	return up;
}

/* Send a UDP datagram */
int
send_udp(
struct socket *lsocket,		/* Source socket */
struct socket *fsocket,		/* Destination socket */
char tos,					/* Type-of-service for IP */
char ttl,					/* Time-to-live for IP */
struct mbuf *data,			/* Data field, if any */
int16 length,				/* Length of data field */
int16 id,					/* Optional ID field for IP */
char df)					/* Don't Fragment flag for IP */
{
	struct mbuf *bp;
	struct pseudo_header ph;
	struct udp udp;
	int32 laddr = lsocket->address;

	if(length != 0 && data != NULLBUF) {
		trim_mbuf(&data,length);
	} else {
		length = len_p(data);
	}
	length += UDPHDR;

	if(laddr == INADDR_ANY) {
		laddr = locaddr(fsocket->address);
	}
	udp.source = lsocket->port;
	udp.dest = fsocket->port;
	udp.length = length;

	/* Create IP pseudo-header, compute checksum and send it */
	ph.length = length;
	ph.source = laddr;
	ph.dest = fsocket->address;
	ph.protocol = UDP_PTCL;

	udpOutDatagrams++;
	bp = htonudp(&udp,data,&ph);

	ip_send(laddr,fsocket->address,UDP_PTCL,tos,ttl,bp,length,id,df);
	return (int)length;
}

/* Accept a waiting datagram, if available. Returns length of datagram */
int
recv_udp(
struct udp_cb *up,
struct socket *fsocket,		/* Place to stash incoming socket */
struct mbuf **bp)			/* Place to stash data packet */
{
	struct socket sp;
	struct mbuf *buf;
	int16 length;

	if(up == NULLUDP){
		return -1;
	}
	if(up->rcvcnt == 0){
		return -1;
	}
	buf = dequeue(&up->rcvq);
	up->rcvcnt--;

	/* Strip socket header */
	pullup(&buf,(char *)&sp,sizeof(struct socket));

	/* Fill in the user's foreign socket structure, if given */
	if(fsocket != NULLSOCK) {
		fsocket->address = sp.address;
		fsocket->port = sp.port;
	}
	/* Hand data to user */
	length = len_p(buf);

	if(bp != NULLBUFP) {
		*bp = buf;
	} else {
		free_p(buf);
	}
	return (int)length;
}

/* Delete a UDP control block */
int
del_udp(struct udp_cb *conn)
{
	struct mbuf *bp;
	struct udp_cb *up, *udplast = NULLUDP;

	for(up = Udps; up != NULLUDP; udplast = up, up = up->next) {
		if(up == conn) {
			break;
		}
	}
	if(up == NULLUDP) {
		/* Either conn was NULL or not found on list */
		return -1;
	}
	/* Get rid of any pending packets */
	while(up->rcvcnt != 0) {
		bp = up->rcvq;
		up->rcvq = up->rcvq->anext;
		free_p(bp);
		up->rcvcnt--;
	}
	/* Remove from list */
	if(udplast != NULLUDP) {
		udplast->next = up->next;
	} else {
		Udps = up->next;	/* was first on list */
	}
	xfree(up);
	return 0;
}

/* Process an incoming UDP datagram */
void
udp_input(
struct iface *iface,		/* Input interface */
struct ip *ip,				/* IP header */
struct mbuf *bp,			/* UDP header and data */
int rxbroadcast)			/* The only protocol that accepts 'em */
{
	struct pseudo_header ph;
	struct udp udp;
	struct udp_cb *up;
	struct socket lsocket, fsocket;
	struct mbuf *packet;
	int16 length = ip->length - IPLEN - ip->optlen;

	if(bp == NULLBUF) {
		return;
	}
	/* Create pseudo-header and verify checksum */
	ph.source = ip->source;
	ph.dest = ip->dest;
	ph.protocol = ip->protocol;
	ph.length = length;

	/* Peek at header checksum before we extract the header. This
	 * allows us to bypass cksum() if the checksum field was not
	 * set by the sender.
	 */
	if((udp.checksum = udpcksum(bp)) != 0 && cksum(&ph,bp,length) != 0){
		/* Checksum non-zero, and wrong */
		udpInErrors++;
		free_p(bp);
		return;
	}
	/* Extract UDP header in host order */
	if(ntohudp(&udp,&bp) != 0){
		/* Truncated header */
		udpInErrors++;
		free_p(bp);
		return;
	}
	/* If this was a broadcast packet, pretend it was sent to us */
	lsocket.address = (rxbroadcast) ? iface->addr : ip->dest;
	lsocket.port = udp.dest;

	/* See if there's somebody around to read it */
	if((up = lookup_udp(&lsocket)) == NULLUDP){
		/* Nope, return an ICMP message */
		if(!rxbroadcast){
			bp = htonudp(&udp,bp,&ph);
			icmp_output(ip,bp,ICMP_DEST_UNREACH,ICMP_PORT_UNREACH,NULLICMP);
		}
		udpNoPorts++;
		free_p(bp);
		return;
	}
	/* Create space for the foreign socket info */
	packet = pushdown(bp,sizeof(struct socket));

	fsocket.address = ip->source;
	fsocket.port = udp.source;
	memcpy(&packet->data[0],(char *)&fsocket,sizeof(struct socket));

	/* Queue it */
	enqueue(&up->rcvq,packet);
	up->rcvcnt++;
	udpInDatagrams++;

	if(up->r_upcall) {
		(*up->r_upcall)(iface,up,up->rcvcnt);
	}
}

int
st_udp(struct udp_cb *udp)
{
	return tprintf("%lx%6u  %s\n",ptol(udp),udp->rcvcnt,pinet(&udp->socket));
}

/* Convert UDP header in internal format to an mbuf in external format */
static struct mbuf *
htonudp(struct udp *udp,struct mbuf *data,struct pseudo_header *ph)
{
	int16 checksum;

	/* Allocate UDP protocol header and fill it in */
	struct mbuf *bp = pushdown(data,UDPHDR);
	char *cp = bp->data;

	cp = put16(cp,udp->source);				/* Source port */
	cp = put16(cp,udp->dest);				/* Destination port */
	cp = put16(cp,udp->length);				/* Length */
	*cp++ = 0;								/* Clear checksum */
	*cp-- = 0;

	/* All zeros and all ones is equivalent in one's complement arithmetic;
	 * the spec requires us to change zeros into ones to distinguish an
 	 * all-zero checksum from no checksum at all
	 */
	if((checksum = cksum(ph,bp,ph->length)) == 0) {
		checksum = 0xffffffffL;
	}
	put16(cp,checksum);
	return bp;
}

/* Convert UDP header in mbuf to internal structure */
int
ntohudp(struct udp *udp,struct mbuf **bpp)
{
	char udpbuf[UDPHDR];

    if(pullup(bpp,udpbuf,UDPHDR) != UDPHDR) {
		return -1;
    }
	udp->source = get16(&udpbuf[0]);
	udp->dest = get16(&udpbuf[2]);
	udp->length = get16(&udpbuf[4]);
	udp->checksum = get16(&udpbuf[6]);
	return 0;
}

/* Extract UDP checksum value from a network-format header without
 * disturbing the header
 */
static int16
udpcksum(struct mbuf *bp)
{
	struct mbuf *dup;

    if(dup_p(&dup,bp,6,2) != 2) {
		return 0;
    }
	return pull16(&dup);
}


/* ------------------------------ UDP sub cmds -------------------------- */

/* Dump UDP statistics and control blocks */
static int
doudpstat(int argc,char **argv,void *p)
{
	struct udp_cb *udp;
	int i;

	for(i = 1; i <= NUMUDPMIB; i++) {
		tprintf("(%2u)udp%-17s%10lu",i,Udp_mib[i].name,Udp_mib[i].value.integer);
		tputs((i % 2) ? "     " : "\n");
	}
	if((i % 2) == 0) {
		tputs("\n");
	}
	tputs("&UCB Rcv-Q  Local socket\n");

	for(udp = Udps;udp != NULLUDP; udp = udp->next) {
		st_udp(udp);
	}
	return 0;
}

int
doudp(int argc,char **argv,void *p)
{
	struct cmds Udpcmds[] = {
		"status",	doudpstat,	0, 0,	NULLCHAR,
		NULLCHAR,
	};
	return subcmd(Udpcmds,argc,argv,p);
}

#endif /* UDP */
