/*
 * GT CLIPPER STANDARD HEADER
 *
 * File......: gt_alert.prg
 * Author....: Andy M Leighton
 * BBS.......: The Dark Knight Returns
 * Net/Node..: 050/069
 * User Name.: Andy Leighton
 * Date......: 21-05-93
 * Revision..: 1.1
 *
 * This is an original work by Andy Leighton and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

/*  $DOC$
 *  $FUNCNAME$
 *       GT_ALERT()
 *  $CATEGORY$
 *       Video
 *  $ONELINER$
 *       Alert() replacement
 *  $SYNTAX$
 *       GT_Alert(<aMss>, <aOpts>, <lSave>, <aColour>) --> nChoice
 *  $ARGUMENTS$
 *       <aMss>     - An array of messages
 *       <aOpts>    - An array of prompts / choices for the alert
 *       <lSave>    - .T. if we need the screen saving and restoring
 *       <aColour>  - Colour Spec Array
 *                      1st element, border & normal text & unselected prompt
 *                      2nd element, selected prompt
 *  $RETURNS$
 *       nPos       - The prompt/choice selected otherwise 0 if escape was
 *                    pressed
 *  $DESCRIPTION$
 *       The GT_Alert() function creates a slightly less simple dialog
 *       than the standard Alert() function.  Use it wherever you would
 *       Alert().
 *
 *       It has a number of differences / advantages over the standard
 *       alert().
 *
 *       These are
 *            Optional saving/restoring of the screen area.
 *            The message is passed as an array of character strings
 *            GT_Alert() can check the setkey handler.
 *
 *       KNOWN PROBLEMS
 *          Does not check setkey handler yet - read the code
 *          Spacing of prompts look poor in some cases.
 *          These problems are known and will be fixed RSN
 *
 *          These problems have now been fixed.
 *          If you find any more please report the,
 *
 *       CAVEATS
 *         Does not check that the elements in the arrays are character
 *          strings
 *         Does not check that the prompts will fit on the machine
 *
 *  $EXAMPLES$
 *          .
 *          .
 *          nChoice := GT_Alert( { "GT Example System",              ;
 *                                 "Do You Want To Quit" },          ;
 *                               { "Yes", "No", "Not Sure"}, TRUE)
 *
 *          do case
 *             case nChoice == 1
 *                ? "You have chosen to quit"
 *             case nChoice == 2
 *                ? "You have chosen not to quit"
 *             case nChoice == 3
 *                ? "Make up your mind"
 *             otherwise
 *                ? "Escape pressed"
 *          endcase
 *
 *  $SEEALSO$
 *  $END$
 */

#include "gt_lib.ch"

// translate for aMaxStrLen()
// works like the one in FuncKy

#translate aMaxStrLen(<a>)      =>     len(GT_AComp(<a>, AC_MAXLEN))


function GT_Alert(aMss, aOpts, lSave, aColour)

   local nLines
   local nWidth
   local nWideCh
   local nWidePr
	local i
   local nLeft
   local nTop
   local nRight
   local nBottom
   local nChoice
   local nPos
   local nLKey    := 0
   local aPos     := array(len(aOpts))
   local oldPos
   local lFinish  := FALSE
   local svScr

   default aColour to {'W+/R', 'W+/B'}
   default lSave   to TRUE

   nLines  := len(aMss) + 3
   nWidth  := aMaxStrLen(aMss) + 3
   nTop    := (maxrow() - nLines) / 2
   nBottom := (maxrow() + nLines) / 2
   nWideCh := 4

   for i := 1 to len(aOpts)
      nWideCh += len(aOpts[i]) + 2
   next
   nWidth  := max(nWidth, nWideCh)
   nLeft   := (maxcol() - nWidth) / 2
   nRight  := (maxcol() + nWidth) / 2

   if lSave
      svScr := GT_SaveScr(nTop, nLeft, nBottom, nRight)
   endif

   dispbegin()

   // draw the box

   @ nTop, nLeft, nBottom, nRight box B_SINGLE + " " color aColour[1]

   // draw the message strings

   for i := 1 to len(aMss)
      @ nTop + i + 1, nLeft + 2 say ;
                     strcenter(aMss[i], nWidth - 3) color aColour[1]
   next

   // work out spacing between prompts to make it look more
   // sexy than the original

   if nWideCh < aMaxStrLen(aMss) + 3
      nWidePr := -2
      for i := 1 to len(aOpts)
         nWidePr := nWidePr + len(aOpts[i]) + 2
      next
      nPos := 1 + int(max(nWidth - nWidePr, 2) / 2)
   else
      nPos := 3
   endif

   // draw the prompts

   for i := 1 to len(aOpts)
      @ nBottom - 1, nLeft + nPos say aOpts[i] color aColour[1]
      aPos[i] := nLeft + nPos
      nPos    := nPos + len(aOpts[i]) + 2
   next
   dispend()

   nChoice := 1
   oldPos  := 0

   do while !lFinish
      if oldPos != nChoice
         // we have moved so move the highlight bar

         dispbegin()
         if oldPos != 0
            @ nBottom - 1, aPos[oldPos] say aOpts[oldPos] color aColour[1]
         endif
         @ nBottom - 1, aPos[nChoice] say aOpts[nChoice] color aColour[2]
         oldPos := nChoice
         dispend()
      endif

      nLKey := GT_inkey(0)

      do case
         case nLKey == K_ESC              // ESC pressed, quit
            lFinish := TRUE
            nChoice := 0
         case nLKey == K_RETURN           // CR pressed, choose current
            lFinish := TRUE
         case nLKey == K_HOME             // HOME pressed, go to first prompt
            nChoice := 1
         case nLKey == K_END              // END pressed, go to last prompt
            nChoice := len(aPos)
         case nLKey == K_RIGHT            // RIGHT pressed, right a prompt
            nChoice++
            if nChoice > len(aPos)        // handle wraparound
               nChoice := len(aPos)
            endif
         case nLKey == K_LEFT             // LEFT pressed, left a prompt
            nChoice--
            if nChoice < 1
               nChoice := 1
            endif
      endcase
   enddo

   if lSave                               // Restore screen before leaving
      GT_RestScr(svScr)
   endif

return nChoice
