/*
    The source code contained within this file is protected under the
    laws of the United States of America and by International Treaty.
    Unless otherwise noted, the source contained herein is:

    Copyright (c)1990-1994 BecknerVision Inc - No Rights Reserved

    Written by John Wm Beckner
    BecknerVision Inc

        ** THIS FILE IS PUBLIC DOMAIN **

        ** Use of this code is at your own risk, absolutely no guarantees
        ** are made to its usefulness  - you must determine for yourself if
        ** this code is of any use to you.  If you do not agree with this, do
        ** not use this code. **
*/

#include "beckner.inc"
#include "set.ch"

////////////////
////////////////
//
// Purpose:
//    Fully interactive record editor
//
// Syntax:
//    fFullAdd([<aField>], [<aSayGet>], [<aPicture>], [<aValid>], [<lMove>],
//             [<lSeek>], [<lReuse>], [<lRecall>], [<lDelete>]) -> NIL
//
// Formal Arguments: (9)
//    Name        Description
//    ___________ ____________
//    aField      Field list [all fields]
//    aSayGet     "S"ay (display-only) or "G"et (modifiable) ["G"]
//    aPicture    Picture strings [none]
//    aValid      Validation strings [none]
//    lMove       .y. if record movement allowed [.y.]
//    lSeek       .y. if SEEK movement is preferred over GOTO movement [.y.]
//    lReuse      .y. to allow re-use of deleted records when adding [.y.]
//    lRecall     .y. if deleted records may be recalled [.y.]
//    lDelete     .y. if record can be deleted [.y.]
//
// Returns:
//    NIL
//
// Examples:
//    #include "beckner.inc"
//    FUNCTION TestIt()
//       CreateTest()
//       fFullAdd()           /* lets you add a record */
//       fDataEdit()          /* lets you edit record  */
//       CLOSE Test
//    ENDFUNCTION
//
//    #include "alias.ch"
//    STATIC FUNCTION CreateTest()
//       fCreateDbf("Test/Name/C/32/Balance/N/12/2/Date/D/Over21/L/Notes/M")
//       USE Test NEW EXCLUSIVE
//    ENDFUNCTION
//
// Files:
//    (current work area)
//
// Description:
//    Fully interactive record editor.  You can specify the field list in
//    <aField> or accept the default which is all fields.  Single element
//    arrays are used for S'87 compatibility.  Each subsequent array is on a
//    one-to-one correspondence with <aField>.  <aSayGet> contain
//    either "S" or "G" for SAY and GET respectively.  SAY fields are only
//    displayed, while GET fields are modifiable.  You can specify PICTURE
//    and VALID clause expressions in <aPicture> and <aValid>,
//    respectively.  If you want the record to be delete-able, then <lDelete>
//    must be .true. (the default).  If deleted records may be recalled, then
//    <lRecall> must be .true. (the default).  If deleted records are re-used
//    when adding new records, then <lReuse> must be .true. (the default).
//    If <lSeek> (the default), then you are prompted for a seek string based
//    on the current index expression, otherwise you are prompted for a record
//    number whenever you get a record directly.  This is only allowed if
//    <lMove> (the default), which allows record movement via a get option and
//    the plus and minus keys for single-record movement.
//
// See Also:
//    fAddRecord()
//    fFullAdd()
//    fBrowse()
//
// Category:
//    File Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

FUNCTION fDataEdit(aField, aSayGet, aPicture, aValidator,;
      lAllowMoves, lUseSeek, lReuseDeleteds, lAllowRecall, lAllowDeletion)
   LOCAL nFieldCount, lUsePicture := .n., lUseValid := .n., nCtr, nLastRow
   LOCAL nDataLen := 0, nPICTURELen, cType, uData, aCoordinate := {}
   LOCAL nWindowLen := 0, nRow, nCol, GETList := {}, nOption := 7, aData
   LOCAL nRecord, nGetRec, iData, cValid, lTemp, nTemp
   nLastRow := MaxRow()
   DEFAULT aField TO dbStruct()
   nFieldCount := Min(22, Len(aField))
   IF Len(aField)>22
      aSize(aField, 22)
   ENDIF
   DEFAULT lAllowDeletion TO .y.,;
         lAllowMoves TO .y.,;
         lUseSeek TO .y.,;
         lReuseDeleteds TO .y.,;
         lAllowRecall TO .y.,;
         lUseValid TO .y.,;
         lUsePicture TO .y.
   IF aSayGet=NIL
      aSize((aSayGet := {}), nFieldCount)
      aFill(aSayGet, "G")
   ENDIF
   FOR nCtr := 1 to nFieldCount
      nWindowLen := Max(nWindowLen, aField[nCtr, 3])
      nDataLen := Max(nDataLen, Max(iif(lUsePicture, aPicture[nCtr], 0),;
      aField[nCtr, 3]))
   NEXT
   nWindowLen := (nPICTURELen := nWindowLen += 4)+nDataLen + 1
   nWindowLen := min(MaxCol()-1, nWindowLen)
   eSave()
   aCoordinates := vWindow(nFieldCount, nWindowLen, .y. ,'EDITING')
   nRow := aCoordinates[1]-1
   nCol := aCoordinates[2]
   aEval(aField, {|uElement, nElNum| SetPos(nRow+nElNum, nCol),;
         qQout(Trim(uElement[1])+" "+;
         Replicate(".", 19-Len(Trim(uElement[1]))))})
   nCol := 24
   WHILE LOOPING
      @ nRow, aCoordinates[2] SAY "#"
      @ row(),Col() SAY RecNo();
            PICTURE replicate("9", len(ltrim(str(lastrec()))))
      @ nLastRow-1, 57 SAY iif(Deleted(), "*DELETED*", space(9))
      aSize((aData := {}), nFieldCount)
      aEval(aField, {|uElement, nElNum| aData[nElNum] :=;
            Eval(FieldBlock(uElement[1]))})
      FOR nCtr := 1 to nFieldCount
         IF lUsePicture
            IF aPicture[nCtr]=NIL
               IF ValType(aData[nCtr])="C" .and. Len(aData[nCtr])>40
                  @ nRow+nCtr, nCol SAY Left(aData[nCtr], 40)
               ELSE
                  @ nRow+nCtr, nCol SAY aData[nCtr]
               ENDIF
            ELSE
               @ nRow+nCtr, nCol SAY Transform(aData[nCtr], aPicture[nCtr])
            ENDIF
         ELSE
            IF ValType(aData[nCtr])="C" .and. Len(aData[nCtr])>40
               @ nRow+nCtr, nCol SAY Left(aData[nCtr], 40)
            ELSE
               @ nRow+nCtr, nCol SAY aData[nCtr]
            ENDIF
         ENDIF
      NEXT
      @ nLastRow,0
      @ nLastRow,0 SAY 'OPTIONS:'
      IF lAllowMoves
         @ nLastRow,Col()+2 PROMPT 'GET'
         @ nLastRow,Col()+2 PROMPT '+/next'
         @ nLastRow,Col()+2 PROMPT '-/prior'
      ENDIF
      @ nLastRow,Col()+2 PROMPT 'Edit'
      @ nLastRow,Col()+2 PROMPT 'Add'
      @ nLastRow,Col()+2 PROMPT 'Delete'
      @ nLastRow,Col()+2 PROMPT 'Quit'
      MENU TO nOption
      @ nLastRow,0
      DO CASE
      CASE nOption=1 .and. lAllowMoves
         nRecord := RecNo()
         IF !lUseSeek .or. IndexOrd()=0
            nGetRec=0
            @ nLastRow, 0 SAY 'Record #' GET nGetRec PICTURE ;
                  Replicate('9', Len(lTrim(Str(RecCount()))))
            READ
            @ nLastRow, 0
            GO nGetRec
         ELSE
            iData := &(IndexKey(IndexOrd()))
            @ nLastRow, 0 SAY IndexKey(IndexOrd())+':' GET iData
            READ
            @ nLastRow,0
            SEEK iData
            IF !Found()
               GO nRecord
               pBeep()
               LOOP
            ENDIF
         ENDIF
         IF Deleted()
            IF lAllowRecall
               iData := .y.
               @ nLastRow, 0 SAY 'This record is deleted.  Recall (Y/N)?' ;
                     GET iData PICTURE 'Y'
               READ
               IF iData
                  fLockRec()
                  RECALL
                  UNLOCK
               ELSE
                  GO nRecord
               ENDIF
            ELSE
               GO nRecord
               pBeep()
            ENDIF
         ENDIF
         LOOP
      CASE nOption=2 .and. lAllowMoves
         SKIP
         IF EOF()
            GO BOTTOM
            pBeep()
         ENDIF
         LOOP
      CASE nOption=3 .and. lAllowMoves
         SKIP -1
         IF BOF()
            GO BOTTOM
            pBeep()
         ENDIF
         LOOP
      CASE (nOption=4 .and. lAllowMoves) .or. nOption=1
         FOR nCtr := 1 to nFieldCount
            IF Upper(aSayGet[nCtr])='G'
               cValid := '.y.'
               IF lUseValid
                  cValid := aValidator[nCtr]
               ENDIF
               IF lUsePicture
                  IF aPicture[nCtr]=NIL
                     IF ValType(aData[nCtr])="C" .and. Len(aData[nCtr])>40
                        @ nRow+nCtr, nCol GET aData[nCtr];
                              PICTURE "@S40" VALID &(cValid)
                     ELSE
                        @ nRow+nCtr, nCol GET aData[nCtr] VALID &(cValid)
                     ENDIF
                  ELSE
                     @ nRow+nCtr, nCol GET aData[nCtr];
                           PICTURE aPicture[nCtr] VALID &(cValid)
                  ENDIF
               ELSE
                  IF ValType(aData[nCtr])="C" .and. Len(aData[nCtr])>40
                     @ nRow+nCtr, nCol GET aData[nCtr];
                           PICTURE "@S40" VALID &(cValid)
                  ELSE
                     @ nRow+nCtr, nCol GET aData[nCtr] VALID &(cValid)
                  ENDIF
               ENDIF
            ENDIF
         NEXT
         READ
         fLockRec()
         aEval(aField, {|uElement, nElNum| Eval(FieldBlock(uElement[1]),;
               aData[nElNum])})
         UNLOCK
         LOOP
      CASE nOption=5
         IF lReuseDeleteds
            lTemp := Set(_SET_DELETED, .n.)
            nTemp := IndexOrd()
            SET ORDER to 0
            LOCATE FOR Deleted()
            IF !Found()
               fAddRecord()
            ELSE
               fLockRec()
               RECALL
            ENDIF
            Set(_SET_DELETED, lTemp)
            UNLOCK
         ELSE
            fAddRecord()
            UNLOCK
         ENDIF
         aEval(aField, {|uElement, nElNum| aData[nElNum] :=;
               Eval(FieldBlock(uElement[1]))})
         @ nRow, aCoordinates[2]+1 SAY RecNo();
               PICTURE Replicate("9", Len(lTrim(Str(lastrec()))))
         FOR nCtr := 1 to nFieldCount
            IF Upper(aSayGet[nCtr])='G'
               cValid := '.y.'
               IF lUseValid
                  cValid := aValidator[nCtr]
               ENDIF
               IF lUsePicture
                  IF aPicture[nCtr]!=NIL
                     @ nRow+nCtr, nCol GET aData[nCtr];
                          PICTURE aPicture[nCtr] VALID &(cValid)
                  ELSE
                     IF ValType(aData[nCtr])="C" .and. Len(aData[nCtr])>40
                        @ nRow+nCtr, nCol GET aData[nCtr];
                              PICTURE "@S40" VALID &(cValid)
                     ELSE
                        @ nRow+nCtr, nCol GET aData[nCtr] VALID &(cValid)
                     ENDIF
                  ENDIF
               ELSE
                  IF ValType(aData[nCtr])="C" .and. Len(aData[nCtr])>40
                     @ nRow+nCtr, nCol GET aData[nCtr];
                           PICTURE "@S40" VALID &(cValid)
                  ELSE
                     @ nRow+nCtr, nCol GET aData[nCtr] VALID &(cValid)
                  ENDIF
               ENDIF
            ENDIF
         NEXT
         READ
         fLockRec()
         aEval(aField, {|uElement, nElNum| FieldPut(FieldPos(uElement[1]),;
               aData[nElNum])})
         UNLOCK
         LOOP
      CASE nOption=6
         IF lAllowDeletion
            IF vIsSure(nLastRow)
               fLockRec()
               DELETE
               UNLOCK
            ENDIF
         ELSE
            pBeep()
         ENDIF
         LOOP
      CASE nOption=7 .or. nOption=4
         EXIT
      ENDCASE
   ENDWHILE
   eRestore()
ENDFUNCTION
