/*
    The source code contained within this file is protected under the
    laws of the United States of America and by International Treaty.
    Unless otherwise noted, the source contained herein is:

    Copyright (c)1990-1994 BecknerVision Inc - No Rights Reserved

    Written by John Wm Beckner
    BecknerVision Inc

        ** THIS FILE IS PUBLIC DOMAIN **

        ** Use of this code is at your own risk, absolutely no guarantees
        ** are made to its usefulness  - you must determine for yourself if
        ** this code is of any use to you.  If you do not agree with this, do
        ** not use this code. **
*/

#include "beckner.inc"

////////////////
////////////////
//
// Purpose:
//    Returns an array containing a start and end date range
//
// Syntax:
//    dApprox(<cApprox>, [<aHelp>], [<cDelims>], [<aValid>]) -> aRange
//
// Formal Arguments: (4)
//    Name        Description
//    ___________ ____________
//    cApprox     Free format date string
//    aHelp       Array containing help information [see description]
//    cDelims     Free format string delimiters [", "]
//    aValid      Array containing codes [see description]
//
// Returns:
//    aRange      Array containg 2 elements, a start date and an end date
//
// Examples:
//    #include "beckner.inc"
//    FUNCTION TestIt()
//       LOCAL cTest1 := "07/01/95"
//       LOCAL cTest2 := "1st Quarter 1996"
//       LOCAL cTest3 := "Fall 1997"
//       ? dApprox(cTest1)    /* {07/01/95, 07/01/95} */
//       ? dApprox(cTest2)    /* {01/01/96, 03/31/96} */
//       ? dApprox(cTest3)    /* {09/21/97, 12/20/97} */
//    ENDFUNCTION
//
// Files:
//    none
//
// Description:
//    Returns a date range based on a free format date string.  <aHelp> is a
//    help screen, which defaults to:
//
//      +----------------------------+
//      |Please enter a valid date in|
//      |one of the following fields:|
//      |                            |
//      |mm/dd/yyyy   mm/yyyy        |
//      |                            |
//      |Or enter the year with your |
//      |choice below:               |
//      |                            |
//      |1st Quarter        January  |
//      |2nd Quarter        February |
//      |3rd Quarter        March    |
//      |4th Quarter        April    |
//      |Winter             May      |
//      |Spring             June     |
//      |Summer             July     |
//      |Fall               August   |
//      |                   September|
//      |                   October  |
//      |                   November |
//      |                   December |
//      +----------------------------+
//
//    <aValid> contains the determination codes, and defaults to the following:
//
//       {{3, "1ST", "Q1"}, {3, "2ND", "Q2"}, {3, "3RD", "Q3"},;
//       {3, "4TH", "Q4"}, {3, "WINTER", "S1"}, {3, "AUTUMN", "S2"},;
//       {4, "FALL", "S2"}, {3, "SPRING", "S3"}, {3, "SUMMER", "S4"},;
//       {3, "JANUARY", "M01"}, {3, "FEBRUARY", "M02"}, {3, "MARCH", "M03"},;
//       {3, "APRIL", "M04"}, {3, "MAY", "M05"}, {3, "JUNE", "M06"},;
//       {3, "JULY", "M07"}, {3, "AUGUST", "M08"}, {3, "SEPTEMBER", "M09"},;
//       {3, "OCTOBER", "M10"}, {3, "NOVEMBER", "M11"},;
//       {3, "DECEMBER", "M12"}}
//
//    In each primary array element, another array is contained, which contains
//    the number of characters which must match, the string to match, and the
//    internal code to return.  The following codes are valid:
//
//          Code  Description
//          ----  ----------------------------------
//          Q1    1st quarter
//          Q2    2nd quarter
//          Q3    3rd quarter
//          Q4    4th quarter
//          S1    1st season (Winter)
//          S2    2nd season (Spring)
//          S3    3rd season (Summer)
//          S4    4th season (Autumn) aka (Fall)
//          M01   January
//          M02   February
//          M03   March
//          M04   April
//          M05   May
//          M06   June
//          M07   July
//          M08   August
//          M09   September
//          M10   October
//          M11   November
//          M12   December
//
// Notes:
//    Thanks to Dr Charles Beattie for inspiring the creation of this function.
//
// Category:
//    Date Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

FUNCTION dApprox(cFreeFormat, aHelpScreen, cDelims, aValid)
   LOCAL aWord, dStart, dEnd, nMonth, nYear, cTemp, GetList := {}
   DEFAULT aValid to {{3, "1ST", "Q1"}, {3, "2ND", "Q2"}, {3, "3RD", "Q3"},;
         {3, "4TH", "Q4"}, {3, "WINTER", "S1"}, {3, "AUTUMN", "S2"},;
         {4, "FALL", "S2"}, {3, "SPRING", "S3"}, {3, "SUMMER", "S4"},;
         {3, "JANUARY", "M01"}, {3, "FEBRUARY", "M02"}, {3, "MARCH", "M03"},;
         {3, "APRIL", "M04"}, {3, "MAY", "M05"}, {3, "JUNE", "M06"},;
         {3, "JULY", "M07"}, {3, "AUGUST", "M08"}, {3, "SEPTEMBER", "M09"},;
         {3, "OCTOBER", "M10"}, {3, "NOVEMBER", "M11"},;
         {3, "DECEMBER", "M12"}}
   DEFAULT cDelims to ", "
   iif(aHelpScreen=NIL,;
         aHelpScreen := {"Please enter a valid date in",;
         "one of the following fields:",;
         "",;
         "mm/dd/yyyy   mm/yyyy",;
         "",;
         "Or enter the year with your",;
         "choice below:",;
         "",;
         "1st Quarter        January",;
         "2nd Quarter        February",;
         "3rd Quarter        March",;
         "4th Quarter        April",;
         "Winter             May",;
         "Spring             June",;
         "Summer             July",;
         "Fall               August",;
         "                   September",;
         "                   October",;
         "                   November",;
         "                   December"},)
   cFreeFormat := StrTran(StrTran(cFreeFormat, ","), ".")
   aWord       := aLine2Array(Upper(cFreeFormat))
   IF Empty(aWord)
      RETURN dApproxHelp(aHelpScreen, aValid)
   ENDIF
   IF "/"$aWord[1]
      IF !Empty(dStart := ctod(aWord[1]))
         RETURN {dStart, dStart}
      ENDIF
      IF sCount("/", aWord[1])>1
         RETURN dApproxHelp(aHelpScreen, aValid)
      ENDIF
      cTemp  := aWord[1]
      nMonth := Val(sParse(@cTemp))
      nYear  := Val(cTemp)
      IF nMonth>0 .and. nMonth<13 .and. nYear>0 .and.;
            nYear<iif(dSetCentury(), 2500, 100)
         RETURN {dStart := ctod(Str(nMonth)+"/1/"+Str(nYear)),;
               dEndMonth(dStart)}
      ENDIF
   ENDIF
   RETURN dApproxHelp(aHelpScreen, aValid)
ENDFUNCTION

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION dApproxHelp(aHelpScreen, aValid)
   MEMVAR GetList
   LOCAL aRC, cLine := Space(30), nCols
   PRIVATE GetList := {}
   aRC := vWindow(Len(aHelpScreen)+1, nCols := aMaxLength(aHelpScreen),;
         .y., "DATE HELP")
   @ aRC[1], aRC[2] GET cLine PICTURE "@S"+lTrim(Str(nCols))
   aEval(aHelpScreen, {|cLine| DevPos(++aRC[1], aRC[2]), qqout(cLine)})
   RETURN aConvert(cLine)
ENDFUNCTION

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION aConvert(cLine, aValid)
   LOCAL aWord, nFound := 0, dStart, dEnd, nLen, nYear
   aWord := aLine2Array(cLine)
   aEval(a1From2(aValid, 2), {|cWord, nElNum| iif(cWord==aWord[1],;
         nFound := nElNum, NIL)})
   IF !(nYear := Val(aWord(nLen := Len(aWord)))>0 .and.;
            Val(aWord(nLen))<Year(Date()))
      nYear := Val(nAsk("Year:", "N", "9999"))
   ENDIF
   aValid[3] := Upper(aValid[3])
   DO CASE
   CASE aValid[3]="S1"
      dStart := CtoD("12/21"+sMake(nYear))
      dEnd := CtoD("3/20/"+sMake(nYear+1))
   CASE aValid[3]="S2"
      dStart := CtoD("3/21"+sMake(nYear))
      dEnd := CtoD("6/20/"+sMake(nYear))
   CASE aValid[3]="S3"
      dStart := CtoD("6/21"+sMake(nYear))
      dEnd := CtoD("9/20/"+sMake(nYear))
   CASE aValid[3]="S4"
      dStart := CtoD("9/21"+sMake(nYear))
      dEnd := CtoD("12/20/"+sMake(nYear))
   CASE aValid[3]="M"
      dEnd := dEndMonth(dStart := CtoD(Right(aValid[3], 2)+"/1/"+sMake(nYear)))
   CASE aValid[3]="Q"
      dEnd := dEndQuarter(dStart := CtoD(SubStr(" 1 4 710", (Val(Right(aValid[3], 2))-1)*2, 2)+"/1/"+sMake(nYear)))
   ENDCASE
   RETURN {dStart, dEnd}
ENDFUNCTION
