

//----------------------------FLATARRY.CPP---------------------------------//
// author: Daniel W. Moore
//
// Support classes for disk-heap, EMS-heap, and debug heap VMemory objects
//

#include <dos.h>
#include <stdlib.h>
#include <stdio.h>
#include <alloc.h>

#include "flatarry.h"

// -- word-wise access to virtual arrays

size_t far & FlatArray::operator()(VHANDLE i) {
  return  *(size_t*)&((*this)[i*sizeof(size_t)]);
}

// -- debug-only virtual array object

HeapArray::HeapArray(long sz) : FlatArray( sz )
{
  p = (char far *) farcalloc(size=sz,sizeof(char));
}

char far & HeapArray::operator[] (VHANDLE x)
{
  if (x>=size) return (char far&) c;
  else {
    char huge *q = ((char huge*)p + x);
    char huge *q1 = q+1;        // make sure to normalize
    return *(char far*) (q1-1); // huge pointer
  }
}

HeapArray::~HeapArray() {
  farfree(p);
}

// -- array implemented as C++ stream on disk

DiskArray::DiskArray(VHANDLE sz, char *p, size_t nbits ) :
  file(p, ios::binary | ios::out | ios::in), FlatArray(sz)
{

  bufsize= (1 << nbits);
  mask=(1 << nbits)-1;
  page=-1;
  file.seekp(sz-1); file.put('\0');
  file.seekg(sz-1);
  long fs=file.tellg();
  if (fs >= sz-1)      // test size of file
    pbuf = new char [bufsize];
  else
    pbuf = NULL;  // if size fails, DiskArray (& thus also VMemory) object
}                 // is invalid!!

DiskArray::~DiskArray()
{
  access(-1);           // make sure last page used is swapped out
}

char far & DiskArray::operator[] (long index)
{
  long newpage;
  if ((newpage=(index & ~mask))!=page) {  // load required page, if not
    access(newpage); file.clear();        // already resident
  }
  return pbuf[index & mask];              // get byte
}

DiskArray::access(long newpage)           // change resident page...
{
  if (!file) return 0;
  if (!pbuf) return 0;
  if (page>=0)
  {
    file.seekp(page);                     // ... swapping out old
    if (file.fail()) return 0;
    file.write(pbuf,bufsize);
    if (file.fail()) return 0;
  }
  if (newpage>=0)
  {
    file.seekg(newpage);                  // ... swapping in new
    if (file.fail()) return 0;
    file.read(pbuf,bufsize);
    //if (file.fail()) return 0;
  }
  page=newpage;

  return 1;
}

// -- virtual array implemented as expanded memory pages

#if _EMS_

EMSArray::EMSArray(VHANDLE size)
  : FlatArray( EMS_PHYS_PAGESIZE * (((size-1)/EMS_PHYS_PAGESIZE)+1) )
{
  pbuf = 0;
  nbits=14;               // one quarter of a page frame
  bufsize= (1 << nbits);
  mask=(1 << nbits)-1;
  page=-1;                // start with no page loaded
  numEMSpages=get_size()/EMS_PHYS_PAGESIZE;

  EMShandle = -1;
  if (EMSinstall()) {                          // check EMS installed;
    EMShandle=EMSalloc(numEMSpages);           // allocate expanded mem.
    pbuf=(char far *) MK_FP(EMSframe(),0); // uses first quarter of
  }                                             // EMS page frame
}

EMSArray::~EMSArray()
{
  if (EMShandle > 0) EMSfree(EMShandle);      // avoid wasting the
}                                               // EMS resource!

EMSArray::access(long newpage)
{
    int j=EMSmap( EMShandle, newpage>>nbits, 0);  // map in new EMS page
    return j ? ( (page=newpage), 1 ) : 0;
}

char far & EMSArray::operator[] (long index)    // indexing operation for
{                                               //  expanded memory array
  long newpage;
  if ((newpage=(index & ~mask))!=page) {
    access(newpage);
  }
  return pbuf[index & mask];
}

#endif /*_EMS_*/
