/*      gvlib_l.h

        Definitions for assembler/C routines authored by George Vanous

        Designed for the Large memory model

        Produced by Vanous Enterprise 1990,1991,1992,1993
*/

#if     !defined(__GVLIBL_DEF_)
#define __GVLIBL_DEF_

#include <stdio.h>                /* necessary for "counter" and "counterr" */

/*****************************************************************************
 * TYPE DEFINITIONS
 *-----------------*/
typedef unsigned char byte;             /* (0-255) type as in Pascal        */
typedef signed char shortint;           /* (-128 to +127) type as in Pascal */

/*****************************************************************************
 * COLOR DEFINITIONS
 *------------------*/
#define BLACK        0x0                                              /* 00 */
#define BLUE         0x1                                              /* 01 */
#define GREEN        0x2                                              /* 02 */
#define CYAN         0x3                                              /* 03 */
#define RED          0x4                                              /* 04 */
#define MAGENTA      0x5                                              /* 05 */
#define BROWN        0x6                                              /* 06 */
#define LIGHTGRAY    0x7                                              /* 07 */
#define DARKGRAY     0x8                                              /* 08 */
#define LIGHTBLUE    0x9                                              /* 09 */
#define LIGHTGREEN   0xA                                              /* 10 */
#define LIGHTCYAN    0xB                                              /* 11 */
#define LIGHTRED     0xC                                              /* 12 */
#define LIGHTMAGENTA 0xD                                              /* 13 */
#define YELLOW       0xE                                              /* 14 */
#define WHITE        0xF                                              /* 15 */
#define BLINK        0x8   /* value to add to background color for blinking */

/*****************************************************************************
 * NUMBER DEFINITIONS
 *-------------------*/
#define ZERO      0x0                                                 /* 00 */
#define ONE       0x1                                                 /* 01 */
#define TWO       0x2                                                 /* 02 */
#define THREE     0x3                                                 /* 03 */
#define FOUR      0x4                                                 /* 04 */
#define FIVE      0x5                                                 /* 05 */
#define SIX       0x6                                                 /* 06 */
#define SEVEN     0x7                                                 /* 07 */
#define EIGHT     0x8                                                 /* 08 */
#define NINE      0x9                                                 /* 09 */
#define TEN       0xA                                                 /* 10 */
#define ELEVEN    0xB                                                 /* 11 */
#define TWELVE    0xC                                                 /* 12 */
#define THIRTEEN  0xD                                                 /* 13 */
#define FOURTEEN  0xE                                                 /* 14 */
#define FIFTEEN   0xF                                                 /* 15 */
#define SIXTEEN   0x10                                                /* 16 */
#define SEVENTEEN 0x11                                                /* 17 */
#define EIGHTEEN  0x12                                                /* 18 */
#define NINETEEN  0x13                                                /* 19 */
#define TWENTY    0x14                                                /* 20 */

/*****************************************************************************
 * GENERAL-PURPOSE DEFINITIONS
 *----------------------------*/
#define NO   -1            /* logical constant                              */
#define YES  +1            /* logical constant                              */
#define TRUE +1            /* logical constant                              */
#define FALSE 0            /* logical constant                              */
#define MAX -1             /* logical constant                              */
#define MIN -2             /* logical constant                              */
#define MAXX 640           /* total screen x-length                         */
#define MAXMAXX 640        /* total (screen)+(virtual screen) x-length      */
#define MAXY 480           /* total screen y-height                         */
#define MAXMAXY 816        /* total (screen)+(virtual screen) y-height      */
#define SIGNED -1          /* signed number                                 */
#define UNSIGNED 1         /* unsigned number                               */
#define TCUR 255           /* text cursor                                   */
#define MCUR 254           /* mouse cursor                                  */
#define MEM  253           /* memory cursor
                              (coordinates are stored in memory)            */
#define TEXTSAVE 0         /* store text video memory                       */
#define TEXTSHOW 1         /* restore text video memory                     */
#define TEXT25 0           /* 25-line text mode                             */
#define TEXT50 1           /* 50-line text mode                             */
#define TEXT25F 2          /* forced 25-line text mode                      */
#define TEXT50F 3          /* forced 50-line text mode                      */
#define GRAFIX 10          /* 640x480x16 graphics mode                      */
#define TEXTPAGEACTIVE -1  /* current active text video page                */
#define TEXTPAGEVISUAL -2  /* current visual text video page                */
#define TEXTPAGE1 1        /* text video page 1                             */
#define TEXTPAGE2 2        /* text video page 2                             */
#define TEXTPAGE3 3        /* text video page 3                             */
#define TEXTPAGE4 4        /* text video page 4                             */
#define TEXTPAGE5 5        /* text video page 5                             */
#define TEXTPAGE6 6        /* text video page 6                             */
#define TEXTPAGE7 7        /* text video page 7                             */
#define TEXTPAGE8 8        /* text video page 8                             */
#define LEFT  -1           /* leftward direction                            */
#define RIGHT +1           /* rightward direction                           */
#define UP    +2           /* upward direction                              */
#define DOWN  -2           /* downward direction                            */
#define ALIGN_NONE  -1	   /* perform no alignment when outputting text     */
#define ALIGN_HORZ   0	   /* horizontal center, shifting odd length left   */
#define ALIGN_VERT   1	   /* vertical center, shifting odd length up	    */
#define ALIGN_CENTER 2	   /* horizontal and vertical center                */
#define ALIGN_RIGHT  3     /* right justify text                            */
#define CURSORHIDE 11      /* "pcmnd" - hide cursor                         */
#define CURSORSHOW 1       /* "pcmnd" - show a hidden cursor                */
#define CURSORNOSAVE 0     /* "scmnd" - do not store cursor's position      */
#define CURSORSAVE 1       /* "scmnd" - store cursor's position             */
#define COPY_IMAGE 0x0003  /* hexadecimal code for performing a direct copy */
#define AND_IMAGE  0x0803  /* hexadecimal code for performing a logical AND */
#define OR_IMAGE   0x1003  /* hexadecimal code for performing a locical OR  */
#define XOR_IMAGE  0x1803  /* hexadecimal code for doing an eXclusive OR    */
#define COPY_IMAGE_SET 0x0013  /* perform a direct copy and set to scroll   */
#define AND_IMAGE_SET  0x0813  /* perform a logical AND and set to scroll   */
#define OR_IMAGE_SET   0x1013  /* perform a locical OR  and set to scroll   */
#define XOR_IMAGE_SET  0x1813  /* perform an eXclusive OR and set to scroll */

/*****************************************************************************
 * SCAN CODE DEFINITIONS
 *----------------------*/
#define SCAN_NONE          0x0                                        /* 00 */
#define SCAN_ESC           0x1                                        /* 01 */
#define SCAN_1             0x2                                        /* 02 */
#define SCAN_2             0x3                                        /* 03 */
#define SCAN_3             0x4                                        /* 04 */
#define SCAN_4             0x5                                        /* 05 */
#define SCAN_5             0x6                                        /* 06 */
#define SCAN_6             0x7                                        /* 07 */
#define SCAN_7             0x8                                        /* 08 */
#define SCAN_8             0x9                                        /* 09 */
#define SCAN_9             0xA                                        /* 10 */
#define SCAN_0             0xB                                        /* 11 */
#define SCAN_HYPHEN        0xC                                        /* 12 */
#define SCAN_EQUAL         0xD                                        /* 13 */
#define SCAN_BACKSPACE     0xE                                        /* 14 */
#define SCAN_TAB           0xF                                        /* 15 */
#define SCAN_Q             0x10                                       /* 16 */
#define SCAN_W             0x11                                       /* 17 */
#define SCAN_E             0x12                                       /* 18 */
#define SCAN_R             0x13                                       /* 19 */
#define SCAN_T             0x14                                       /* 20 */
#define SCAN_Y             0x15                                       /* 21 */
#define SCAN_U             0x16                                       /* 22 */
#define SCAN_I             0x17                                       /* 23 */
#define SCAN_O             0x18                                       /* 24 */
#define SCAN_P             0x19                                       /* 25 */
#define SCAN_BRACKET_LEFT  0x1A                                       /* 26 */
#define SCAN_BRACKET_RIGHT 0x1B                                       /* 27 */
#define SCAN_ENTER         0x1C                                       /* 28 */
#define SCAN_CTRL          0x1D                                       /* 29 */
#define SCAN_A             0x1E                                       /* 30 */
#define SCAN_S             0x1F                                       /* 31 */
#define SCAN_D             0x20                                       /* 32 */
#define SCAN_F             0x21                                       /* 33 */
#define SCAN_G             0x22                                       /* 34 */
#define SCAN_H             0x23                                       /* 35 */
#define SCAN_J             0x24                                       /* 36 */
#define SCAN_K             0x25                                       /* 37 */
#define SCAN_L             0x26                                       /* 38 */
#define SCAN_SEMICOLON     0x27                                       /* 39 */
#define SCAN_QUOTE         0x28                                       /* 40 */
#define SCAN_BACKQUOTE     0x29                                       /* 41 */
#define SCAN_SHIFT_LEFT    0x2A                                       /* 42 */
#define SCAN_BACKSLASH     0x2B                                       /* 43 */
#define SCAN_Z             0x2C                                       /* 44 */
#define SCAN_X             0x2D                                       /* 45 */
#define SCAN_C             0x2E                                       /* 46 */
#define SCAN_V             0x2F                                       /* 47 */
#define SCAN_B             0x30                                       /* 48 */
#define SCAN_N             0x31                                       /* 49 */
#define SCAN_M             0x32                                       /* 50 */
#define SCAN_COMMA         0x33                                       /* 51 */
#define SCAN_PERIOD        0x34                                       /* 52 */
#define SCAN_SLASH         0x35                                       /* 53 */
#define SCAN_SHIFT_RIGHT   0x36                                       /* 54 */
#define SCAN_GRAYSTAR      0x37                                       /* 55 */
#define SCAN_ALT           0x38                                       /* 56 */
#define SCAN_SPACE         0x39                                       /* 57 */
#define SCAN_CAPSLOCK      0x3A                                       /* 58 */
#define SCAN_F1            0x3B                                       /* 59 */
#define SCAN_F2            0x3C                                       /* 60 */
#define SCAN_F3            0x3D                                       /* 61 */
#define SCAN_F4            0x3E                                       /* 62 */
#define SCAN_F5            0x3F                                       /* 63 */
#define SCAN_F6            0x40                                       /* 64 */
#define SCAN_F7            0x41                                       /* 65 */
#define SCAN_F8            0x42                                       /* 66 */
#define SCAN_F9            0x43                                       /* 67 */
#define SCAN_F10           0x44                                       /* 68 */
#define SCAN_NUMLOCK       0x45                                       /* 69 */
#define SCAN_SCROLLOCK     0x46                                       /* 70 */
#define SCAN_HOME          0x47                                       /* 71 */
#define SCAN_UP            0x48                                       /* 72 */
#define SCAN_PGUP          0x49                                       /* 73 */
#define SCAN_GRAYMINUS     0x4A                                       /* 74 */
#define SCAN_LEFT          0x4B                                       /* 75 */
#define SCAN_KP5           0x4C                                       /* 76 */
#define SCAN_RIGHT         0x4D                                       /* 77 */
#define SCAN_GRAYPLUS      0x4E                                       /* 78 */
#define SCAN_END           0x4F                                       /* 79 */
#define SCAN_DOWN          0x50                                       /* 80 */
#define SCAN_PGDN          0x51                                       /* 81 */
#define SCAN_INS           0x52                                       /* 82 */
#define SCAN_DEL           0x53                                       /* 83 */
#define SCAN_SYSRQ         0x54                                       /* 84 */
#define SCAN_F11           0x57                                       /* 87 */
#define SCAN_F12           0x58                                       /* 88 */

/*****************************************************************************
 * GLOBAL DEFINITIONS
 *-------------------*/
typedef struct {
   byte ascii;         /* ASCII code of character pressed                   */
   byte scan;          /* SCAN code of character pressed                    */
} asciiscan;

typedef struct {
   shortint fgclr;     /* foreground color or NO if no color fill           */
   shortint bgclr;     /* background color or NO if no color fill           */
   char fillchar;      /* character to fill with or NO if no character fill */
   byte x;             /* abscissa (if TCUR, uses text cursor abscissa
                                    if MCUR, uses mouse cursor abscissa)    */
   byte y;             /* ordinate (if TCUR, uses text cursor ordinate
                                    if MCUR, uses mouse cursor ordinate)    */
   shortint length;    /* x-length of area                                  */
   shortint height;    /* y-height of area                                  */
   char *overwrite;    /* string to overwrite or NULL if overwrite all      */
   char *print;        /* pointer to string to print or NULL if no string   */
} filldata;

typedef struct {
   unsigned int size;  /* size of image in bytes                            */
   byte tclr;          /* transparency color of image
            * or number of 8-pixel groups per row if image is "16_i_save"ed */
   int x;              /* abscissa (if MCUR, uses mouse abscissa)           */
   int y;              /* abscissa (if MCUR, uses mouse ordinate)           */
   int length;         /* x-length of image                                 */
   int height;         /* y-height of image                                 */
   int how : 1;        /* relative to current scroll (TRUE) or not (FALSE)  */
   int for_future : 15;/* no current purpose (fills integer)		    */
} imagedata;

typedef struct {
   byte update;        /* TRUE "_ms_stat" has updated structure "mousedata" */
		       /* FALSE no structure variables have been updated    */
   int pos[2];         /* old mouse cursor x,y position                     */
   int buts[2];        /* old left,right button  0 not pressed  1 pressed   */
   int npos[2];        /* new mouse cursor x,y position                     */
   int nbuts[2];       /* new left,right button  0 not pressed  1 pressed   */
   int butlr[2];       /* x,y position of last left button release          */
   int butlp[2];       /* x,y position of last left button press            */
   int butrr[2];       /* x,y position of last right button release         */
   int butrp[2];       /* x,y position of last right button press           */
} mousedata;                                                                

typedef struct {
   shortint fgclr;     /* foreground color or NO if no color fill           */
   shortint bgclr;     /* background color or NO if no color fill           */
   byte command; /* ALIGN_NONE    - no command
                  * ALIGN_HORZ    - horizontal center; shifts odd lengths left
                  * ALIGN_VERT    - vertical center; shifts odd lengths up
                  * ALIGN_CENTER - horizontal and vertical center
                  * ALIGN_RIGHT   - right justify; flush to right margin    */
   byte x;             /* abscissa (if TCUR, uses text cursor abscissa
                                    if MCUR, uses mouse cursor abscissa)    */
   byte y;             /* ordinate (if TCUR, uses text cursor ordinate
                                    if MCUR, uses mouse cursor ordinate)    */
   char *string;       /* pointer to string to print                        */
} printdata;

typedef struct {
   unsigned int size;  /* size of image in bytes                            */
   shortint fgclr;     /* NO if no color fill                               */
   shortint bgclr;     /* NO if no color fill                               */
   byte x;             /* abscissa (if TCUR, uses text cursor abscissa
                                    if MCUR, uses mouse cursor abscissa)    */
   byte y;             /* ordinate (if TCUR, uses text cursor ordinate
                                    if MCUR, uses mouse cursor ordinate)    */
   byte length;        /* x-length                                          */
   byte height;        /* y-height                                          */
} textimagedata;

typedef struct {
   shortint fgclr;     /* foreground color or NO if no color fill           */
   shortint bgclr;     /* background color or NO if no color fill           */
   byte x;             /* abscissa (if TCUR, uses text cursor abscissa
                                    if MCUR, uses mouse cursor abscissa)    */
   byte y;             /* ordinate (if TCUR, uses text cursor ordinate
                                    if MCUR, uses mouse cursor ordinate)    */
} textstr;

/* ------------------------------------------------------------------------ */


/*****************************************************************************
 * LIGHT ROUTINES
 *---------------*/
void alloff(void);
   /* turns all three lights off (NUM LOCK, CAPS LOCK, and SCROLL LOCK)     */

void allon(void);
   /* turns all three lights on (NUM LOCK, CAPS LOCK, and SCROLL LOCK)      */

void capsoff(void);                      /* turns CAPS LOCK light off       */

void capson(void);                       /* turns CAPS LOCK light on        */

void numoff(void);                       /* turns NUM LOCK light off        */

void numon(void);                        /* turns NUM LOCK light on         */

void scrolloff(void);                    /* turns SCROLL LOCK light off     */

void scrollon(void);                     /* turns SCROLL LOCK light on      */

/* ------------------------------------------------------------------------ */


/*****************************************************************************
 * MISCELLANEOUS ROUTINES
 *-----------------------*/
void bestlib_init(void);                    /* initializes THE BEST LIBRARY */

void beep(void);                                           /* beeps speaker */

FILE *counter(char *text, char *title, char *authorname);
   /* opens data file "text" and records number of program executions
    * uses program name "title" and author name "authorname"
    * must be in a text mode or nothing will print
    * displays five unique shareware reminders every fifth execution
    * displays amount of memory available to program
    * NOTE #include <alloc.h> must be present or free memory will be large
    * returns file handle to data file                                      */

FILE *counterr(char *text, char *title, char *authorname, char *registree);
   /* counter registered
    * opens a data file and records number of times software has
      been run
    * must be in a text mode or nothing will print
    * displays five unique shareware reminders every fifth execution
    * displays amount of memory available to program
    * NOTE #include <alloc.h> must be present or free memory will be large
    * returns file handle to data file                                      */

char *dosshell(void);
   /* shells to DOS, returning on "EXIT"
    * returns NULL if successful, or a pointer to a string saying error     */

int getvideomode(void);                   /* returns current video mode     */

void msec(int delaytime);
   /* delays computer for "delaytime" milliseconds
    * first call initializes itself to computer                             */

void setvideomode(int mode);		    /* changes to video mode "mode" */

int stopw(int index, int delaytime);
   /* "index" refers to a stopwatch and must be between 0-24
    * "delaytime" is number of clock ticks to set stopwatch to
    * if called with "delaytime" 0 returns TRUE if finished, FALSE if not   */

/* ------------------------------------------------------------------------ */


/*****************************************************************************
 * MOUSE ROUTINES
 *---------------*/
void _ms_get(int *x, int *y);
   /* stores mouse abscissa in "x" and mouse ordinate in "y"
    * if mouse is in text mode, stores byte values (can be byte *x, byte *y)
    * if "y" = NULL, "x" assumed to be "int *xy" (or byte *xy if text mode) */

int _ms_getx(void);
   /* returns mouse abscissa
    * if mouse is in text mode, returns byte value                          */

int _ms_gety(void);
   /* returns mouse ordinate
    * if mouse is in text mode, returns byte value                          */

void _ms_hide(void);                                  /* hides mouse cursor */

int _ms_init(byte mode);
   /* initialize mouse for video mode "mode"
                                          TEXT25   25-line text mode
                                          TEXT50   50-line text mode
                                          GRAFIX   640x480x16 graphics mode
    * initializes structure "msdata" with default values
    * centers mouse cursor on screen and hides it (use "_ms_hide" to see it)
    * sets maximum abscissa and ordinate of mouse cursor to screen size
    * returns TRUE if successful or FALSE if no mouse driver present        */

int _ms_kill(void);
   /* completely disables mouse driver;
    * returns TRUE is successful or FALSE is failed                         */

void _ms_range(int minx, int miny, int maxx, int maxy);
   /* minimum abscissa "minx", ordinate "miny", and maximum abscissa "maxx",
      ordinate "maxy" is set as new mouse movement range
    * if "maxx" = MAX, sets maximum abscissa to screen maximum
    * if "maxy" = MAX, sets maximum ordinate to screen maximum              */

void _ms_set(int x, int y);
   /* position mouse cursor at "x","y"
    * if "x" = MEM, the "mousedata.npos[0]" is used
    * if "y" = MEM, the "mousedata.npos[1]" is used
    * if values are supplied for either "x" or "y", those are stored into
      "mousedata.npos[0]" or "mousedata.npos[1]", respectively              */

int _ms_setx(int x);
   /* set mouse cursor abscissa to "x"
    * if "x" = MEM, the "mousedata.npos[0]" is used
    * if a value is supplied for "x", it is stored into "mousedata.npos[0]"
    * returns mouse ordinate                                                */

int _ms_sety(int y);
   /* set mouse cursor ordinate to "y"
    * if "x" = MEM, the "mousedata.npos[1]" is used
    * if a value is supplied for "y", it is stored into "mousedata.npos[1]"
    * returns mouse abscissa                                                */

void _ms_shape(byte fgclr, byte bgclr, char ch);
   /* changes the text mouse cursor to ASCII character "ch"
    * "fgclr" = NO if no foreground, "bgclr" = NO if no background          */

void _ms_show(void);				   /* displays mouse cursor */

void _ms_stat(void);
   /* updates all "mousedata" information except pos[] and but[]            */

/* ------------------------------------------------------------------------ */


/*****************************************************************************
 * STRING ROUTINES
 *----------------*/
char casedn(char ch);                          /* returns "ch" in lowercase */

void casedn_ptr(char *ch);                    /* converts "ch" to lowercase */

char caseup(char ch);                          /* returns "ch" in uppercase */

void caseup_ptr(char *ch);                    /* converts "ch" to uppercase */

char getchr(void);
   /* gets next available keypress; if no keypress available, waits for
      one
    * ASCII code and scan code is returned in structure "asciiscan"
    * returns ASCII code                                                    */

char kbclear(void);
   /* clears keyboard buffer
    * returns ASCII code of last character that was in buffer               */

int keyhit(void);        /* returns 0 if no key ready, ASCII code otherwise */

int scanforbyte(char chrcompare, int elementno, char *startaddress);
   /* checks if character "chrcompare" is present in "startaddress"
    * checks "elementno" bytes (if 0, searches to end of string "startaddress"
    * returns offset + 1 or
      FALSE if no match found or if "startaddress" is not a string          */

int scanforstr(char *strcompare, int elementno, char *startaddress);
   /* checks if "strcompare" is present in "startaddress"
    * checks "elementno" bytes (if 0, searches to end of string "startaddress"
    * returns offset + 1 or FALSE if no match found or
      FALSE if either "strcompare" or "startaddress" are not strings        */

int scanforword(int intcompare, int elementno, int *startaddress);
   /* checks if integer "intcompare" is present in "startaddress"
    * checks "elementno" words (if 0, searches to end of array "startaddress";
    * returns offset + 1 or
              FALSE if no match found or if "elementno" = 0                 */

int sign(int number);                    /* returns sign of number "number" */

void strcasedown(char *str);
                           /* converts all characters in "str" to lowercase */
void strcasedown_pr(printdata *prdata);
        /* converts all characters in "printdata.string2print" to lowercase */

void strcaseup(char *str);
                    /* converts all characters in string "str" to uppercase */
void strcaseup_pr(printdata *prdata);
        /* converts all characters in "printdata.string2print" to uppercase */

int strcompare(char *str1, char *str2);
   /* compares "str1" to "str2"
    * returns +1 if "str1" is alphabetically greater than "str2"
    * returns  0 if "str1" and "str2" are same
    * returns -1 if "str1" is alphabetically less than "str2"               */

int strcopy(char *destination, char *source);
   /* copies string "source" into "destination"
    * copy stops when NULL character, '\x0', is found in "source"
    * returns TRUE if successful or FALSE if "source" is not a string       */

unsigned int strlength(char *str);               /* returns length of "str" */

/* ------------------------------------------------------------------------ */


/*****************************************************************************
 * TEXT MODE ROUTINES
 *-------------------*/
void _txt_attr_save(byte x, byte y, byte *fgclr, byte *bgclr);
   /* stores foreground color at "x","y" into "fgclr" and background color
      into "bgclr"
    * if "fgclr" = NULL, foreground is not stored
    * if "bgclr" = NULL, background is not stored                           */

void _txt_attr_show(byte x, byte y, byte fgclr, byte bgclr);
   /* prints foreground color "fgclr" and background color "bgclr" at "x","y"
    * if "fgclr" = NO, foreground is not modified
    * if "bgclr" = NO, background is not modified                           */

void _txt_attrs_fill(byte x, byte y,
                     byte length, byte height, byte fgclr, byte bgclr);
   /* the region with corner "x","y", x-length "length", and y-height
      "height" is filled in fgcolor "fglcr" and bgcolor "bgclr"
    * if "fgclr" = NO, foreground is not modified
    * if "bgclr" = NO, background is not modified                           */

int _txt_attrs_need(byte length, byte height);
   /* returns memory requirement for a "_txt_attrs_save"ed image of
      x-length "length" and y-height "height"                               */

void _txt_attrs_save(byte x, byte y,
                     byte length, byte height, textimagedata *attrs);
   /* stores the region of attributes from corner "x","y" with x-length
      "length" and y-height "height" into "attrs"
    * if "x" = MEM, "attrs.x" is used
    * if "y" = MEM, "attrs.y" is used
    * if "length" = MEM, "attrs.length" is used
    * if "height" = MEM, "attrs.height" is used                             */

void _txt_attrs_show(byte x, byte y,
                     byte fgclr, byte bgclr, textimagedata *attrs);
   /* prints the region of attributes "attrs" at top-left corner "x","y"
    * if "x" = MEM,  "attrs.x" is used
    * if "y" = MEM,  "attrs.y" is used
    * if "fgclr" = NO,  foreground color is not modified
                 = MEM, "attrs.fgclr" is used
                 = YES, modifies foreground with stored colors
    * if "bgclr" = NO,  background color is not modified
                 = MEM, "attrs.bgclr" is used
                 = YES, modifies background with stored colors              */

void _txt_blink(int what);
   /* if "what" = TRUE,  blinking characters
                = FALSE, 16 background colors                               */

char _txt_chr_attr_show(byte x, byte y, byte fgclr, byte bgclr, char ch);
   /* the character "ch" is printed in fgcolor "fgclr" and bgcolor "bgclr"
      at "x","y"
    * if "fgclr" = NO, foreground is not modified
    * if "bgclr" = NO, background is not modified
    * returns ASCII value of character that was overwritten                 */

char _txt_chr_erase(byte x, byte y);
   /* overwrites the character at "x","y" with a space
    * returns ASCII value of character that was overwritten                 */

char _txt_chr_save(byte x, byte y);
   /* returns ASCII value of character at "x","y"                           */

char _txt_chr_show(byte x, byte y, int count, char ch);
   /* prints "count" character "ch" at "x","y"
    * returns ASCII value of character that was overwritten                 */

char _txt_chrattr_save(byte x, byte y, char *chrattr);
   /* stores character with its attribute at "x","y" into "chrattr"
    * returns ASCII value of character that was saved                       */

char _txt_chrattr_show(byte x, byte y, byte fgclr, byte bgclr, char *chrattr);
   /* prints character with its attribute, stored in "chrattr", at "x","y"
    * if "fgclr" = NO, foreground is not modified
    * if "bgclr" = NO, background is not modified
    * returns ASCII value of character that was overwritten                 */

void _txt_chrattrs_erase(byte x, byte y, textimagedata *chrattrs);
   /* overwrites "chrattrs" with corner at "x","y" with spaces
    * if "x" = MEM, "chrattrs.x" is used
    * if "y" = MEM, "chrattrs.y" is used                                    */

int _txt_chrattrs_need(byte length, byte height);
   /* returns memory requirement for a "_txt_chrattrs_save"ed image         */

void _txt_chrattrs_save(byte x, byte y,
                        byte length, byte height, textimagedata *chrattrs);
   /* stores the region of characters and their attributes from corner
      "x","y" with x-length "length" and y-height "height" into "chrattrs"
    * if "x" = MEM, "chrattrs.x" is used
    * if "y" = MEM, "chrattrs.y" is used
    * if "length" = MEM, "chrattrs.length" is used
    * if "height" = MEM, "chrattrs.height" is used                          */

void _txt_chrattrs_show(byte x, byte y,
                        byte fgclr, byte bgclr, textimagedata *chrattrs);
   /* prints a region of character and their attributes "chrattrs" at
      top-left corner "x","y"
    * if "x" = MEM, "chrattrs.x" is used
    * if "y" = MEM, "chrattrs.y" is used
    * if "fgclr" = NO,  foreground color is not modified
                 = MEM, "chrattrs.fgclr" is used
    * if "bgclr" = NO,  background color is not modified
                 = MEM, "chrattrs.bgclr" is used                            */

void _txt_chrs_erase(byte x, byte y, textimagedata *chrs);
   /* overwrites "chrs" with corner at "x","y" with spaces
    * if "x" = MEM, "chrs.x" is used
    * if "y" = MEM, "chrs.y" is used                                        */

int _txt_chrs_need(byte length, byte height);
   /* returns memory requirement for a "_txt_chrs_save"ed image             */

void _txt_chrs_save(byte x, byte y,
                    byte length, byte height, textimagedata *chrs);
   /* stores the region of characters from corner "x","y" with x-length
      "length" and y-height "height" into "chrs"
    * if "length" = MEM, "chrs.length" is used
    * if "height" = MEM, "chrs.height" is used                              */

void _txt_chrs_show(byte x, byte y, textimagedata *chrs);
   /* prints the region of character "chrs" with top-left corner "x","y"
    * if "x" = MEM, "chrs.x" is used
    * if "y" = MEM, "chrs.y" is used                                        */

void _txt_erase(byte x, byte y, byte length, byte height);
   /* overwrites the region with corner at "x","y", x-length "length", and
      y-height "height" with spaces                                         */

void _txt_fillarea(byte x, byte y, filldata *fidata);
   /* fills screen according to structure "fidata"
    * the filling begins at corner "x","y"
    * if "x" = MEM, "fidata.x" is used
    * if "y" = MEM, "fidata.y" is used                                      */

int _txt_flood(byte fgclr, byte bgclr, char *str);
   /* floods screen with string "str" in "fgclr", "bgclr"
    * if "fgclr" = NO, foreground color is not modified
    * if "bgclr" = NO, background color is not modified
    * if "str" = NULL, no string is written
    * returns length of "str"                                               */

char _txt_getchre(byte x, byte y);
  /* gets next available keypress and echoes it to "x","y"; if no
     keypress is available, waits for one
    * ASCII code and scan code is returned in structure "asciiscan"
    * returns ASCII code                                                    */

byte _txt_getlet(byte x, byte y, char *msg, byte max,
                 byte fgclr, byte bgclr, char *def, char *str);
   /* gets alphabet input from keyboard
    * prints message "message" at "x","y" followed by default response
      "def" by a call to "_txt_str_show"
    * stores user input into "str"
    * accepts only alphabet characters as input
    * accepts a maximum of "max" characters; if "max" = -1, accepts a
      maximum of "strlength(str)" characters
    * will not allow input to exceed screen limit
    * hides text cursor and restores its position upon return
    * returns -1 if <ESC> is pressed or number of characters typed          */

byte _txt_getnum(byte x, byte y, char *msg, int min, int max,
                 byte fgclr, byte bgclr, int def, int *input);
   /* gets numeric input from keyboard
    * prints message "message" at "x","y" followed by default response
      "def" by a call to "_txt_str_show"
    * colors response space with "fgclr", "bgclr"
    * stores numerical value of user's input into "input"
    * accepts only numerical values, as well as '+' and '-'
    * accepts a maximum number of "max" or 32767 and a minimum number of
      "min" or -32768
    * will not allow input to exceed screen limit
    * hides text cursor and restores its position upon return
    * returns -1 if <ESC> is pressed else number of digits typed            */

byte _txt_getnumu(byte x, byte y, char *msg, unsigned int min,
  unsigned int max, byte fgclr, byte bgclr, int def, unsigned int *input);
   /* gets numeric input from keyboard
    * prints message "message" at "x","y" followed by default response
      "def" by a call to "_txt_str_show"
    * colors response space with "fgclr", "bgclr"
    * stores numerical value of user's input into "input"
    * accepts only numerical values
    * accepts a maximum number of "max" or 65535 and a minimum number of "min"
    * will not allow input to exceed screen limit
    * hides text cursor and restores its position upon return
    * returns -1 if <ESC> is pressed else number of digits typed            */

byte _txt_getstr(byte x, byte y, char *msg, byte max,
                 byte fgclr, byte bgclr, char *def, char *str);
   /* gets string input from keyboard
    * prints message "message" at "x","y" followed by default response
      "def" by a call to "_txt_str_show"
    * stores user input into "str"
    * accepts all ASCII characters and stores user input into "str"
    * accepts a maximum of "max" characters; if "max" = -1, accepts a
      maximum of "strlength(str)" characters
    * will not allow input to exceed screen limit
    * hides text cursor and restores its position upon return
    * returns -1 if <ESC> is pressed or number of characters typed          */

void _txt_mem(byte what, textimagedata *timage);
   /* if "what" = TEXTSHOW - restores text memory
                = TEXTSAVE - stores text memory
    * storage area "timage" is used
    * if "timage" = NULL, one built-in storage area is used                 */

void _txt_mode(byte mode);
   /* "mode" determines text mode:       TEXT25  - 25-line text mode
                                         TEXT50  - 50-line text mode
                                         TEXT25F - forced 25-line text mode
                                         TEXT50F - forced 50-line text mode */

byte _txt_pageget_a(void);           /* returns active text video page      */

byte _txt_pageget_v(void);           /* returns visual text video page      */

void _txt_pageset_a(byte page);
 /* sets text page "page" as active
  * if "page" = -1, the active page becomes the current visual text page    */

void _txt_pageset_v(byte page);
 /* sets a text page as visual, or seen 
  * if "page" = -1, the visual page becomes the current active text page    */

int _txt_print(byte x, byte y, printdata *prdata);
   /* prints according to structure "prdata"
    * the printing begins at "x","y"
    * if "x" = MEM, "prdata.x" is used
    * if "y" = MEM, "prdata.y" is used
    * returns length of "prdata.string"                                     */

void _txt_scroll(byte x, byte y, byte length, byte height, byte direction);
   /* smoothly scrolls screen region with corner at "x","y", x-length
      "length", and y-height "height" in direction "direction"
    * if "direction" = LEFT,  scrolls to the left
                     = RIGHT, scrolls to the right                          */

int _txt_str_erase(byte x, byte y, byte command, char *str);
   /* overwrites "str" at "x","y" with blanks (ASCII 32)
    * if "command" = ALIGN_NONE   no command
                   = ALIGN_HORZ   horizontal center; shifts odd lengths left
                   = ALIGN_VERT   vertical center; shifts odd lengths up
                   = ALIGN_CENTER horizontal and vertical center
                   = ALIGN_RIGHT  right justify; flush to right margin
    * returns length of "str"                                               */

int _txt_str_erase_pr(byte x, byte y, printdata *prdata);
   /* overwrites "prdata.string" at "x","y" with blanks (ASCII 32)
    * if "x" = MEM, "prdata.x" is used
    * if "y" = MEM, "prdata.y" is used
    * returns length of "prdata.string"                                     */

int _txt_str_save(byte x, byte y, byte length, char *str);
   /* reads "length" characters from "x","y" into "str"
    * if "length" = 0, uses length of "str" as number of characters to read
    * The NULL character is appended as last character in "str"
    * returns length of "str"                                               */

int _txt_str_show(byte x, byte y, byte command, char *str);
   /* prints string "str" at "x","y"
    * if "command" = ALIGN_NONE   no command
                   = ALIGN_HORZ   horizontal center; shifts odd lengths left
                   = ALIGN_VERT   vertical center; shifts odd lengths up
                   = ALIGN_CENTER horizontal and vertical center
                   = ALIGN_RIGHT  right justify; flush to right margin
    * returns length of "str"                                               */

int _txt_strattr_need(byte length);
   /* returns memory requirement for "_txt_strattr_save"ed image of
      x-length "length"                                                     */

int _txt_strattr_save(byte x, byte y, byte length, char *strattr);
   /* reads "length" characters their attributes from "x","y" into
      "strattr"; make sure you have allocated enough memory by a call to
      "_txt_strattr_need"
    * if "length" = 0, uses length of "str" / 2 as number of chars to read
    * The NULL character is appended as last character in "strattr"
    * returns character count of "strattr" (which is length / 2)            */

int _txt_strattr_show(byte x, byte y, byte command, char *strattr);
   /* prints string with attributes "strattr" at "x","y"
    * if "command" = ALIGN_NONE   no command
                   = ALIGN_HORZ   horizontal center; shifts odd lengths left
                   = ALIGN_VERT   vertical center; shifts odd lengths up
                   = ALIGN_CENTER horizontal and vertical center
                   = ALIGN_RIGHT  right justify; flush to right margin
    * returns character count of "strattr" (which is length / 2)            */

/* ========================================================================
   --------------------------- CURSOR ROUTINES ---------------------------- */

void _cur_get(byte *x, byte *y);
   /* stores cursor abscissa in "x" and cursor ordinate in "y"
    * if "y" = NULL, "x" is assumed to be "byte *xy"                        */

byte _cur_getx(void);                            /* returns cursor abscissa */

byte _cur_gety(void);                            /* returns cursor ordinate */

void _cur_off(void);                                    /* turns cursor off */

void _cur_on(void);                                     /* turns cursor on  */

byte _cur_set(byte x, byte y);
   /* sets cursor to "x","y"
    * if "x" = MCUR, cursor abscissa is set to mouse abscissa
    * if "y" = MCUR, cursor ordinate is set to mouse ordinate
    * returns cursor abscissa                                               */

byte _cur_setx(byte x);
   /* sets cursor abscissa to "x"
    * returns cursor ordinate                                               */

byte _cur_sety(byte y);
   /* sets cursor ordinate to "y"
    * returns cursor abscissa                                               */

void _cur_shapeget(byte *start, byte *end);
   /* gets cursor shape on active page
    * stores starting scan line into "start", ending scan line into "end"
    * if "end" = NULL, "start" is assumed to be "byte *start_end"           */

void _cur_shapeset(byte start, byte end);
   /* defines new cursor shape, starting at scan line "start", ending at
      scan line "end"
    * if "start" = NO, starting scan line of cursor is not modified
    * if "end" = NO, ending scan line of cursor is not modified
    * "start" and end" must be between 0 - 32                               */

/* ------------------------------------------------------------------------ */


/*****************************************************************************
 * VGA ROUTINES
 *-------------*/
void _16_boxfill(int x, int y, int length, int height, int color, int how);
   /* creates a solid box of color "color" with one corner at "x","y" and
      x-length "length" and y-height "height" (which can be negative)
    * returns 0 if successful or -1 if box dimensions are off-screen
    * "how" determines how image is drawn: COPY_IMAGE - draws image
                                           AND_IMAGE  - ANDs image
                                           OR_IMAGE   - ORs image
                                           XOR_IMAGE  - XORs image
            COPY_IMAGE_SET - draws image and sets to current scroll
            AND_IMAGE_SET  - ANDs  image and sets to current scroll
            OR_IMAGE_SET   - ORs   image and sets to current scroll
            XOR_IMAGE_SET  - XORs  image and sets to current scroll */

void _16_boxoutline(int x, int y, int length, int height, int color,int how);
   /* creates an outline of a box color "color" with one corner at "x","y"
      and x-length "length" and y-height "height" (which can be negative)
    * "length" and "height" must be at least 2
    * returns 0 if successful or -1 if box dimensions are off-screen
    * "how" determines how image is drawn: COPY_IMAGE - draws image
                                           AND_IMAGE  - ANDs image
                                           OR_IMAGE   - ORs image
                                           XOR_IMAGE  - XORs image
            COPY_IMAGE_SET - draws image and sets to current scroll
            AND_IMAGE_SET  - ANDs  image and sets to current scroll
            OR_IMAGE_SET   - ORs   image and sets to current scroll
            XOR_IMAGE_SET  - XORs  image and sets to current scroll         */

void _16_floodall(int color, int scroll);
   /* floods a screen-size with color "color
    * "scroll" determines relative to what flood will occur:
                                             TRUE  - relative to screen
                                             FALSE - relative to A000:0000  */

void _16_floodallall(int color);
   /* floods a 640x816 region of video RAM with color "color"               */

int _16_ispixel(int x, int y, int color, int scroll);
   /* returns TRUE if pixel at "x","y" is color "color" or FALSE if not
    * "scroll" determines relative to what pixel will be checked:
                                             TRUE  - relative to screen
                                             FALSE - relative to A000:0000  */

void _16_linebiglen(int x, int y, int length,
                    int height, int color, int how, int dimensions);
   /* draws line with one end at "x","y", x-length "length", and y-height
      "height" in color "color"
    * "how" determines how image is drawn: COPY_IMAGE - draws image
                                           AND_IMAGE  - ANDs image
                                           OR_IMAGE   - ORs image
                                           XOR_IMAGE  - XORs image
            COPY_IMAGE_SET - draws image and sets to current scroll
            AND_IMAGE_SET  - ANDs  image and sets to current scroll
            OR_IMAGE_SET   - ORs   image and sets to current scroll
            XOR_IMAGE_SET  - XORs  image and sets to current scroll
    * line is drawn using user-specified block size "dimensions"            */

void _16_linebigxy(int x1, int y1, int x2, int y2,
                   int color, int how, int dimensions);
   /* draws a line from "x1", "y1" to "x2", "y2" in color "color"
    * line is drawn using user-specified block size "dimensions"
    * "how" determines how image is drawn: COPY_IMAGE - draws image
                                           AND_IMAGE  - ANDs image
                                           OR_IMAGE   - ORs image
                                           XOR_IMAGE  - XORs image
            COPY_IMAGE_SET - draws image and sets to current scroll
            AND_IMAGE_SET  - ANDs  image and sets to current scroll
            OR_IMAGE_SET   - ORs   image and sets to current scroll
            XOR_IMAGE_SET  - XORs  image and sets to current scroll         */

void _16_linelen(int x, int y, int length, int height, int color, int how);
   /* draws line with one end at "x","y", x-length "length", and y-height
      "height" in color "color"
    * same as "linexy" but draws line using relative distances
    * returns 0 if successful or -1 if line dimensions are off-screen
    * "how" determines how image is drawn: COPY_IMAGE - draws image
                                           AND_IMAGE  - ANDs image
                                           OR_IMAGE   - ORs image
                                           XOR_IMAGE  - XORs image
            COPY_IMAGE_SET - draws image and sets to current scroll
            AND_IMAGE_SET  - ANDs  image and sets to current scroll
            OR_IMAGE_SET   - ORs   image and sets to current scroll
            XOR_IMAGE_SET  - XORs  image and sets to current scroll         */

void _16_linexy(int x1, int y1, int x2, int y2, int color, int how);
   /* draws a line from "x1", "y1" to "x2", "y2" in color "color"
    * same as "linecool" but draws line properly
    * returns 0 if successful or -1 if line dimensions are off-screen
    * "how" determines how image is drawn: COPY_IMAGE - draws image
                                           AND_IMAGE  - ANDs image
                                           OR_IMAGE   - ORs image
                                           XOR_IMAGE  - XORs image
            COPY_IMAGE_SET - draws image and sets to current scroll
            AND_IMAGE_SET  - ANDs  image and sets to current scroll
            OR_IMAGE_SET   - ORs   image and sets to current scroll
            XOR_IMAGE_SET  - XORs  image and sets to current scroll         */

void _16_pixel(int x, int y, int color, int scroll);
   /* changes pixel at "x","y" to "color"
    * "scroll" determines relative to what pixel will be set:
                                             TRUE  - relative to screen
                                             FALSE - relative to A000:0000  */

int _16_popupcolumn(int x, int y, int choice, int maxopts,
                    char *ptr[], int mousepresent);
   /* a column will pop up at "x","y" with maximum choices "maxopts"
    * "choice" will be first highlighted menu
    * "ptr" points to main menu options
    * "mousepresent" is TRUE if a mouse is present and has been initialized
      with "_ms_init()"                                                       */

int _16_popupmenu(int x, int y, int choice, int maxmenus,
                  char *ptr[], int data[][3], char *text[], int mousepresent);
   /* a menu will pop up at "x","y" with maximum choices "maxmenus"
    * "choice" will be first highlighted menu
    * "ptr" points to main menu options
    * "data" holds necessary data
      [number of submenus], [index into *text], [1 = active, 0 = inactive]
    * "text" points to all of submenus
    * "mousepresent" is TRUE if a mouse is present and has been initialized
      with "_ms_init()"                                                       */

int _16_whatpixel(int x, int y, int scroll);
   /* returns pixel color at "x","y"
    * "scroll" determines relative to what pixel will be read:
                                             TRUE  - relative to screen
                                             FALSE - relative to A000:0000  */

void highstandardvga(void);              /* changes to 640x480 16 color VGA */

int isitega(void);          /* returns TRUE if EGA compatible, FALSE if not */

int scrollevel(void);          /* returns number of rows currently scrolled */

void scrollrow(int rows);
   /* scrolls screen "rows" rows (+ve scrolls down, -ve scrolls up)
    * if "rows" = 0, restores video pointer to A000:0000                    */

/* ========================================================================
   ---------------------------- IMAGE ROUTINES ---------------------------- */

void _16_copy(imagedata *destination, imagedata *source);
   /* copies "source" into "destination"
    * both must be "16_c_save"ed, "16_i_save"ed, or "16_p_save"ed images
    * make sure "destination" has been been allocated sufficient memory     */

void _16_move(int oldx, int oldy, int newx, int newy,
              int id, imagedata *oldgraph, imagedata *graph[]);
   /* "oldgraph", at "oldx","oldy", is moved
    * the new image "image" becomes located at "newx","newy"
    * if "oldx" = MCUR, mouse cursor abscissa is used
                = MEM,  "oldgraph.x" is used
    * if "oldy" = MCUR, mouse cursor ordinate is used
                = MEM,  "oldgraph.y" is used
    * if "newx" = MCUR, mouse cursor abscissa is used
                = MEM,  "graph.x" is used
    * if "newy" = MCUR, mouse cursor ordinate is used
                = MEM,  "graph.y" is used
    * "segment" determines memory location of background image
    * "id" is a unique number indexing
    * "memoryidentity" that identifies graphic (it is a constant)
    * "segment" determines memory location of background image
    * two global arrays required: "memoryidentity" and "memoryaddress"
        unsigned int memoryidentity[MAXGRAPHICS];
        void *memoryaddress[MAXGRAPHICS];
      DO NOT MODIFY THESE VARIABLES -- THEY ARE USED BY THE ASSEMBLER ROUTINES
      "memoryidentity" is a unique graphic number for each separate image
      "memoryadddress" is an array of video memory offsets for each image
    * this procedure will animate "_16_c_save"ed images                     */

/* === === === === === === === === === === === === === === === === === ===
   ---------------------------- COMPRESS IMAGE ---------------------------- */

/* use all of these routines only with images saved by "_16_c_save"         */

void _16_c_move(int oldx, int oldy, int newx, int newy,
                imagedata *oldgraph, imagedata *graph, imagedata *segment[]);
   /* "oldgraph", at "oldx","oldy", is moved
    * the new image "image" becomes located at "newx","newy"
    * if "oldx" = MCUR, mouse cursor abscissa is used
                = MEM,  "oldgraph.x" is used
    * if "oldy" = MCUR, mouse cursor ordinate is used
                = MEM,  "oldgraph.y" is used
    * if "newx" = MCUR, mouse cursor abscissa is used
                = MEM,  "graph.x" is used
    * if "newy" = MCUR, mouse cursor ordinate is used
                = MEM,  "graph.y" is used
    * "segment" determines memory location of background image              */

unsigned int _16_c_need_scrn(int left, int top, int length, int height);
   /* returns memory requirement of image with top-left corner at
      "left","top" and x-length "length" and y-height "height"              */

unsigned int _16_c_need_wrst(int length, int height);
   /* returns worst-case memory requirement of theoretical image with
      x-length "length" and y-height "height"
    * use only with "_16_c_save"                                            */

int _16_c_percent(imagedata *image);
   /* returns how much percent smaller (or larger) a "_16_c_save"ed image is,
      compared to an uncompressed "_16_p_save"ed image (relative to 100)
    * a return value of 40 means 60% smaller whereas 125 means 25% larger   */

int _16_c_read_off(imagedata *image, int offset);
   /* returns color of pixel "offset" (starts at 0) of image "image"        */

int _16_c_read_xy(imagedata *image, int x, int y);
   /* returns color of pixel at "x","y" of image "image"                    */

void _16_c_save(int left, int top, int length, int height,
	int transparency_color, imagedata *image, unsigned int size, int how);
   /* stores screen location with top-left corner at "left","top",
      x-length "length", y-height "height", and transparency color
      "transparency_color" into image "image"
    * if byte size "size" is 0, it will be calculated automatically
    * "how" determines if image will be saved relative to current
      scroll or not:  TRUE  - image is saved relative to current scroll
                      FALSE - image is saved relative to A000:0000          */

void _16_c_show(int x, int y, imagedata *image);
   /* writes image "image" at "x","y"
    * if "x" = -1, "imagedata.x" is used for abscissa and "imagedata.y"
      is used for ordinate
    * bit "imagedata.how" determines if image is shown relative to
      screen (TRUE) or relative to A000:0000 (FALSE)                        */

void _16_i_to_c(imagedata *destination_c, imagedata *source_i);
   /* converts "_16_i_save"ed image "source_i"
      into "_16_c_save"ed image "destination_c"                             */

void _16_p_to_c(imagedata *destination_c, imagedata *source_p);
   /* converts "_16_p_save"ed image "source_p"
      into "_16_c_save"ed image "destination_c"                             */

/* === === === === === === === === === === === === === === === === === ===
   ------------------------------ PIXEL IMAGE ----------------------------- */

/* use all of these routines only with images saved by "_16_p_save"         */

void _16_c_to_p(imagedata *destination_p, imagedata *source_c);
   /* converts "_16_c_save"ed image "source_c"
      into "_16_p_save"ed image "destination_p"                             */

void _16_i_to_p(imagedata *destination_p, imagedata *source_i);
   /* converts "_16_i_save"ed image "source_i"
      into "_16_p_save"ed image "destination_p"                             */

void _16_p_move(int oldx, int oldy, int newx, int newy,
                imagedata *oldgraph, imagedata *graph, imagedata *segment[]);
   /* "oldgraph", at "oldx","oldy", is moved
    * the new image "image" becomes located at "newx","newy"
    * if "oldx" = MCUR, mouse cursor abscissa is used
                = MEM,  "oldgraph.x" is used
    * if "oldy" = MCUR, mouse cursor ordinate is used
                = MEM,  "oldgraph.y" is used
    * if "newx" = MCUR, mouse cursor abscissa is used
                = MEM,  "graph.x" is used
    * if "newy" = MCUR, mouse cursor ordinate is used
                = MEM,  "graph.y" is used
    * "segment" determines memory location of background image              */

unsigned int _16_p_need(int length, int height);
   /* returns memory requirement of a
      "_16_p_save"ed image of x-length "length and y-height "height"        */

int _16_p_read_off(imagedata *image, int offset);
   /* returns color of pixel "offset" (starts at 0) of image "image"        */

int _16_p_read_xy(imagedata *image, int x, int y);
   /* returns color of pixel at "x","y" of image "image"                    */

void _16_p_save(int left, int top, int length, int height,
	 int transparencycolor, imagedata *image, unsigned int size, int how);
   /* stores screen location with top-left corner at "left","top",
      x-length "length", y-height "height", and transparency color
      "transparency_color" into image "image"
    * if byte size "size" is 0, it will be calculated automatically
    * "how" determines if image will be saved relative to current
      scroll or not:  TRUE  - image is saved relative to current scroll
                      FALSE - image is saved relative to A000:0000          */

void _16_p_show(int x, int y, imagedata *image);
   /* writes image "image" at "x","y"
    * if "x" = -1, "imagedata.x" is used for abscissa and "imagedata.y"
      is used for ordinate
    * bit "imagedata.how" determines if image is shown relative to
      screen (TRUE) or relative to A000:0000 (FALSE)                        */

/* === === === === === === === === === === === === === === === === === ===
   ------------------------------ PLANE IMAGE ----------------------------- */

/* use all of these routines only with images saved by "_16_i_save"         */

void _16_c_to_i(imagedata *destination_i, imagedata *source_c);
   /* converts "_16_c_save"ed image "source_c"
      into "_16_i_save"ed image "destination_i"                             */

void _16_i_move(int oldx, int oldy, int newx, int newy,
             imagedata *oldgraph[], imagedata *graph[], imagedata *segment[]);
void _16_i_move_rect(int oldx, int oldy, int newx, int newy,
             imagedata *oldgraph[], imagedata *graph[], imagedata *segment[]);
   /* use "_16_i_move_rect" to move rectangular graphics only
    * "oldgraph", at "oldx","oldy", is moved
    * "oldgraph[0]" is the full-color graphic and "oldgraph[1]" is the full-
      black version of the graphic
    * the new image "image" becomes located at "newx","newy"
    * "graph[0]" is the full-color graphic and "graph[1]" is the
      full-black version of the graphic
    * if "oldx" = MCUR, mouse cursor abscissa is used
                = MEM,  "oldgraph.x" is used
    * if "oldy" = MCUR, mouse cursor ordinate is used
                = MEM,  "oldgraph.y" is used
    * if "newx" = MCUR, mouse cursor abscissa is used
                = MEM,  "graph.x" is used
    * if "newy" = MCUR, mouse cursor ordinate is used
                = MEM,  "graph.y" is used
    * "segment" determines memory location of background image              */

unsigned int _16_i_need(int length, int height);
   /* returns memory requirement of a
      "_16_i_save"ed image of x-length "length and y-height "height"        */

int _16_i_read_off(imagedata *image, int offset);
   /* returns color of pixel "offset" (starts at 0) of image "image"        */

int _16_i_read_xy(imagedata *image, int x, int y);
   /* returns color of pixel at "x","y" of image "image"                    */

void _16_i_save(int left, int top, int length, int height,
                imagedata *image, unsigned int size, int how);
   /* stores screen location with top-left corner at "left","top",
      x-length "length", and y-height "height" into image "image"
    * if byte size "size" is 0, it will be calculated automatically
    * "how" determines if image will be saved relative to current
      scroll or not:  TRUE  - image is saved relative to current scroll
                      FALSE - image is saved relative to A000:0000          */

void _16_i_show(int x, int y, imagedata *image, int how);
   /* writes image "image" at "x","y"
    * if "x" = -1, "imagedata.x" is used for abscissa and "imagedata.y"
      is used for ordinate
    * "how" determines how image is drawn: COPY_IMAGE - draws image
                                               AND_IMAGE  - ANDs image
                                               OR_IMAGE   - ORs image
                                               XOR_IMAGE  - XORs image
    * bit "imagedata.how" determines if image is shown relative to
      screen (TRUE) or relative to A000:0000 (FALSE)                        */

void _16_p_to_i(imagedata *destination_i, imagedata *source_p);
   /* converts "_16_p_save"ed image "source_p"
      into "_16_i_save"ed image "destination_i"                             */

void _16_restore_bg(int left, int top,
                    int length, int height, imagedata *segment[]);
   /* restores region of background with top-left
      corner at "left","top", x-length "length" and y-height "height"
    * background is stored in segment "segment"
    * bit "imagedata.how" determines if image is shown relative to
      screen (TRUE) or relative to A000:0000 (FALSE)                        */

/* ------------------------------------------------------------------------ */

#endif
