#[derive(Debug, PartialEq)]
pub struct Selector {
    pub menu_type: String,
    pub user_display: String,
    pub host: String,
    pub port: u16,
    pub path: String,
}

impl Selector {
    pub fn parse_from_menu(line: &str) -> Result<Selector, String> {
        let line = line.trim();

        if line == "." || line.is_empty() {
            return Err(String::from("Empty menu line"));
        }

        let menu_type = line.chars().next().unwrap();
        let after_type = &line[1..];
        let fields: Vec<_> = after_type.split('\t').collect();

        if fields.len() != 4 {
            return Err(String::from("Invalid number of tabs"));
        }

        if let Some(_url) = fields[1].strip_prefix("URL:") {
            return Err(String::from("URL links not supported"));
        }

        let user_display = fields[0];
        let mut path = fields[1].to_string();
        let host = fields[2];
        let port;

        if menu_type == 'i' || menu_type == '3' {
            return Ok(Selector {
                menu_type: menu_type.to_string(),
                user_display: user_display.to_string(),
                host: String::from(""),
                port: 0,
                path: String::from(""),
            });
        }

        if let Ok(num) = fields[3].parse::<u16>() {
            port = num;
        } else {
            return Err(String::from("Port is not a u16"));
        }

        // RFC 1436 says that the selector is *opaque*. This implies that they are always *absolute*.
        // We don't encounter something like "hello.txt" and then have to turn this into an absolute
        // path based on what we're currently fetching.
        if !path.starts_with('/') {
            path = format!("/{path}");
        }

        if path.contains(' ') {
            return Err(String::from("Spaces in paths not supported"));
        }

        Ok(Selector {
            menu_type: menu_type.to_string(),
            user_display: user_display.to_string(),
            host: host.to_string(),
            port,
            path,
        })
    }

    pub fn parse_from_url(url: &str) -> Result<Selector, String> {
        let parts: Vec<_> = url.split("://").collect();

        if parts.is_empty() || parts[0].to_lowercase() != "gopher" {
            return Err(String::from("Not a Gopher URL"));
        }

        let after_scheme = parts[1];
        if after_scheme.is_empty() {
            return Err(String::from("Nothing after scheme"));
        }

        let slashes: Vec<_> = after_scheme.split('/').collect();
        let host_port = slashes[0].to_string();

        let host;
        let port;
        let host_port_colon: Vec<_> = host_port.split(':').collect();
        if host_port_colon.len() == 1 {
            host = host_port_colon[0].to_string();
            port = 70;
        } else if host_port_colon.len() == 2 {
            if let Ok(num) = host_port_colon[1].parse::<u16>() {
                host = host_port_colon[0].to_string();
                port = num;
            } else {
                return Err(String::from("Port is not a u16"));
            }
        } else {
            return Err(String::from("Host/port has too many colons"));
        }

        if slashes.len() == 1 {
            // Empty selector means root path.
            return Ok(Selector {
                menu_type: String::from("1"),
                user_display: String::from(""),
                host,
                port,
                path: String::from("/"),
            });
        }

        let menu_type;
        let path_offset;
        if slashes[1].len() == 1 {
            menu_type = slashes[1].to_string();
            path_offset = 2;
        } else {
            // Menu type unknown/invalid, assume some binary file.
            menu_type = String::from("9");
            path_offset = 1;
        }

        let path = String::from("/") + &slashes[path_offset..].join("/");

        if path.contains(' ') {
            return Err(String::from("Spaces not allowed in URLs"));
        }

        // TODO Support percent decoding?

        Ok(Selector {
            menu_type,
            user_display: String::from(""),
            host,
            port,
            path,
        })
    }

    pub fn to_url(&self) -> Result<String, String> {
        if self.menu_type == "i" || self.menu_type == "3" {
            return Err(String::from("Selector not fetchable"));
        }

        Ok(format!(
            "gopher://{}:{}/{}{}",
            self.host, self.port, self.menu_type, self.path
        ))
    }
}

pub fn grab_gopher_selectors_from_menu(content: &str) -> Vec<Selector> {
    content
        .lines()
        .filter_map(|l| Selector::parse_from_menu(l).ok())
        .collect()
}

pub fn humanize_menu_item(line: &str) -> Result<String, String> {
    let selector = Selector::parse_from_menu(line)?;

    let prefix;
    if selector.menu_type == "i" {
        prefix = String::from("      ");
    } else if selector.menu_type == "0" {
        prefix = String::from("(FILE)");
    } else if selector.menu_type == "1" {
        prefix = String::from(" (DIR)");
    } else if selector.menu_type == "7" {
        prefix = String::from("   (?)");
    } else {
        prefix = format!("   ({})", selector.menu_type);
    }

    Ok(format!("{} {}", prefix, selector.user_display))
}

#[cfg(test)]
mod tests_parse_from_url {
    use super::*;

    #[test]
    fn parse_from_url_onlyhost() {
        assert_eq!(
            Selector::parse_from_url("gopher://foo.example"),
            Ok(Selector {
                menu_type: String::from("1"),
                user_display: String::from(""),
                host: String::from("foo.example"),
                port: 70,
                path: String::from("/"),
            }),
        );
    }

    #[test]
    fn parse_from_url_notype() {
        assert_eq!(
            Selector::parse_from_url("gopher://foo.example/bar.txt"),
            Ok(Selector {
                menu_type: String::from("9"),
                user_display: String::from(""),
                host: String::from("foo.example"),
                port: 70,
                path: String::from("/bar.txt"),
            }),
        );
    }

    #[test]
    fn parse_from_url_full() {
        assert_eq!(
            Selector::parse_from_url("gopher://foo.example:7070/I/my/subdir/test.img"),
            Ok(Selector {
                menu_type: String::from("I"),
                user_display: String::from(""),
                host: String::from("foo.example"),
                port: 7070,
                path: String::from("/my/subdir/test.img"),
            }),
        );
    }

    #[test]
    fn parse_from_url_fails() {
        assert!(Selector::parse_from_url("http://example.com").is_err());
        assert!(Selector::parse_from_url("gopher://").is_err());
        assert!(Selector::parse_from_url("gopher:/example.com/0/test.txt").is_err());
        assert!(Selector::parse_from_url("gopher://example.com:not_a_number/0/test.txt").is_err());
        assert!(Selector::parse_from_url("gopher://example.com:70:80/0/test.txt").is_err());
        assert!(Selector::parse_from_url("gopher://example.com:1000000/0/test.txt").is_err());
        assert!(Selector::parse_from_url("gopher://example.com/I/File with spaces.png").is_err());
    }
}

#[cfg(test)]
mod tests_parse_from_menu {
    use super::*;

    #[test]
    fn parse_from_menu_empty() {
        assert!(Selector::parse_from_menu("").is_err());
    }

    #[test]
    fn parse_from_menu_info() {
        assert_eq!(
            Selector::parse_from_menu("iFoobar\tselector\tserver\tport"),
            Ok(Selector {
                menu_type: String::from("i"),
                user_display: String::from("Foobar"),
                host: String::from(""),
                port: 0,
                path: String::from(""),
            }),
        );
    }

    #[test]
    fn parse_from_menu_normal() {
        assert_eq!(
            Selector::parse_from_menu("0A Text File\t/hello.txt\tserver\t70"),
            Ok(Selector {
                menu_type: String::from("0"),
                user_display: String::from("A Text File"),
                host: String::from("server"),
                port: 70,
                path: String::from("/hello.txt"),
            }),
        );
        assert_eq!(
            Selector::parse_from_menu("0A Text File\thello.txt\tserver\t70"),
            Ok(Selector {
                menu_type: String::from("0"),
                user_display: String::from("A Text File"),
                host: String::from("server"),
                port: 70,
                path: String::from("/hello.txt"),
            }),
        );
        assert_eq!(
            Selector::parse_from_menu("IImage\t/hello.png\tserver\t7071"),
            Ok(Selector {
                menu_type: String::from("I"),
                user_display: String::from("Image"),
                host: String::from("server"),
                port: 7071,
                path: String::from("/hello.png"),
            }),
        );
    }

    #[test]
    fn parse_from_menu_gargabe() {
        assert!(Selector::parse_from_menu("iSome short line").is_err());
        assert!(Selector::parse_from_menu("This is something that shouldn't be here.").is_err());
    }

    #[test]
    fn parse_from_menu_garbage_space_in_path() {
        assert!(
            Selector::parse_from_menu("0A Text File\t/hello my old friend.txt\tserver\t70")
                .is_err()
        );
    }

    #[test]
    fn parse_from_menu_dotline() {
        assert!(Selector::parse_from_menu(".").is_err());
    }
}

#[cfg(test)]
mod tests_humanize_menu_item {
    use super::*;

    #[test]
    fn humanize_menu_item_info() {
        assert_eq!(
            humanize_menu_item("iHello\t-\t-\t0"),
            Ok(String::from("       Hello"))
        );
        assert_eq!(
            humanize_menu_item("i\t-\t-\t0"),
            Ok(String::from("       "))
        );
    }

    #[test]
    fn humanize_menu_item_dir() {
        assert_eq!(
            humanize_menu_item("1Hello\t/some/path\tserver\t70"),
            Ok(String::from(" (DIR) Hello")),
        );
    }

    #[test]
    fn humanize_menu_item_file() {
        assert_eq!(
            humanize_menu_item("0Hello\t/some/path\tserver\t70"),
            Ok(String::from("(FILE) Hello")),
        );
    }

    #[test]
    fn humanize_menu_item_search() {
        assert_eq!(
            humanize_menu_item("7Hello\t/some/path\tserver\t70"),
            Ok(String::from("   (?) Hello")),
        );
    }

    #[test]
    fn humanize_menu_item_other_type() {
        assert_eq!(
            humanize_menu_item("gHello\t/some/path\tserver\t70"),
            Ok(String::from("   (g) Hello")),
        );
        assert_eq!(
            humanize_menu_item("IHello\t/some/path\tserver\t70"),
            Ok(String::from("   (I) Hello")),
        );
        assert_eq!(
            humanize_menu_item("9Hello\t/some/path\tserver\t70"),
            Ok(String::from("   (9) Hello")),
        );
    }

    #[test]
    fn humanize_menu_item_garbage() {
        assert!(humanize_menu_item("Random text").is_err());
    }
}

#[cfg(test)]
mod tests_to_url {
    use super::*;

    #[test]
    fn to_url_normal() {
        assert_eq!(
            Selector {
                user_display: String::from("<does not matter>"),
                menu_type: String::from("0"),
                host: String::from("example.com"),
                port: 7071,
                path: String::from("/this/is/some/file"),
            }
            .to_url(),
            Ok(String::from(
                "gopher://example.com:7071/0/this/is/some/file"
            )),
        );
        assert_eq!(
            Selector {
                user_display: String::from("<does not matter>"),
                menu_type: String::from("I"),
                host: String::from("example.com"),
                port: 70,
                path: String::from("/foo.png"),
            }
            .to_url(),
            Ok(String::from("gopher://example.com:70/I/foo.png")),
        );
    }

    #[test]
    fn to_url_not_fetchable() {
        assert!(
            Selector {
                user_display: String::from("This is an info line"),
                menu_type: String::from("i"),
                host: String::from("-"),
                port: 0,
                path: String::from("-"),
            }
            .to_url()
            .is_err()
        );
        assert!(
            Selector {
                user_display: String::from("This is an error line"),
                menu_type: String::from("3"),
                host: String::from("-"),
                port: 0,
                path: String::from("-"),
            }
            .to_url()
            .is_err()
        );
    }
}
