use std::collections::HashMap;
use std::env::args;
use std::time::{SystemTime, UNIX_EPOCH};

mod config;
mod db;
mod fetch;
mod selector;

fn timestamp_now() -> u64 {
    SystemTime::now()
        .duration_since(UNIX_EPOCH)
        .expect("Time did not go forward, hello NTP")
        .as_secs()
}

fn preview(output: &mut String, selector: &selector::Selector) {
    if !(selector.menu_type == "0" || selector.menu_type == "1") {
        return;
    }

    if let Ok(raw_content) = fetch::fetch(selector) {
        let mut i = 0;

        for line in raw_content.lines() {
            if i == 0 {
                output.push('\n');
            }

            let line_trimmed = line.trim_end().to_string();
            let to_display = if selector.menu_type == "0" {
                line_trimmed
            } else {
                match selector::humanize_menu_item(line) {
                    Ok(hline) => hline,
                    Err(_) => line_trimmed,
                }
            };
            output.push_str(&format!("> {to_display}\n"));

            i += 1;
            if i == 10 {
                break;
            }
        }
        if i > 0 {
            output.push_str(">\n");
            output.push_str("> ...\n");
            output.push_str(">\n");
            output.push('\n');
        }
    }
}

fn process_urls(
    urls: &[String],
    existing_db: &HashMap<String, u64>,
    do_preview: bool,
) -> HashMap<String, u64> {
    let mut db = existing_db.clone();
    let now = timestamp_now();

    let mut output = String::new();

    #[cfg(debug_assertions)]
    eprintln!("Current timestamp: {now}");

    for url in urls {
        let selector = selector::Selector::parse_from_url(url);
        if selector.is_err() {
            eprintln!("Invalid config, not a parsable URL: [{url}]");
            continue;
        }
        let selector = selector.unwrap();

        let content = match fetch::fetch(&selector) {
            Ok(content_loaded) => content_loaded,
            Err(msg) => {
                eprintln!("Could not fetch {url}: {msg}");
                continue;
            }
        };

        #[cfg(debug_assertions)]
        eprint!("{content}");

        let mut first_new = true;

        for selector_found in selector::grab_gopher_selectors_from_menu(&content) {
            let url_in_menu = match selector_found.to_url() {
                Ok(u) => u,
                Err(_) => continue, // Selector not fetchable.
            };

            let identifier = format!("{url} {url_in_menu}");

            #[cfg(debug_assertions)]
            eprintln!("[{url_in_menu}] -> [{identifier}]");

            if !db.contains_key(&identifier) {
                if first_new {
                    output.push_str(&format!("---- {url} ----\n"));
                    output.push('\n');
                    first_new = false;
                }
                output.push_str(&format!(": {url_in_menu}\n"));

                if do_preview {
                    preview(&mut output, &selector_found);
                }
            }

            db.insert(identifier, now);
        }

        if !first_new {
            output.push('\n');
        }
    }

    print!("{output}");

    db
}

fn prune_expired(existing_db: &HashMap<String, u64>) -> HashMap<String, u64> {
    let mut db = HashMap::new();
    let now = timestamp_now();

    #[cfg(debug_assertions)]
    let expire_secs = 10;
    #[cfg(not(debug_assertions))]
    let expire_secs = 86400 * 180;

    for (identifier, timestamp) in existing_db {
        if now - timestamp <= expire_secs {
            db.insert(identifier.to_owned(), timestamp.to_owned());
        }
    }

    db
}

fn main() {
    let mut do_preview = true;

    let args: Vec<String> = args().collect();

    let mut i = 1;
    while i < args.len() {
        if args[i] == "-P" {
            do_preview = false;
        }

        i += 1;
    }

    let urls = match config::load() {
        Ok(urls_loaded) => urls_loaded,
        Err(msg) => panic!("Could not load config: {msg}"),
    };

    let existing_db = db::load();
    let new_db = process_urls(&urls, &existing_db, do_preview);
    let pruned_db = prune_expired(&new_db);
    db::save(&pruned_db);
}
