use std::collections::HashMap;
use std::env::{home_dir, var};
use std::fs::{File, create_dir_all, read_to_string, rename};
use std::io::Write;
use std::path::PathBuf;

fn get_data_dir() -> PathBuf {
    let base_dir = match var("XDG_DATA_HOME") {
        Ok(val) => PathBuf::from(val),
        Err(_) => home_dir()
            .expect("No idea what your homedir is")
            .join(".local")
            .join("share"),
    };

    base_dir.join("rophcatch")
}

pub fn load() -> HashMap<String, u64> {
    let data_file = get_data_dir().join("seen");

    let mut db_map = HashMap::new();

    if let Ok(content) = read_to_string(&data_file) {
        for line in content.lines() {
            let fields: Vec<_> = line.split(' ').collect();
            if fields.len() != 3 {
                continue;
            }

            let seen_on_host = fields[0].to_string();
            let selector = fields[1].to_string();
            let timestamp = fields[2]
                .parse::<u64>()
                .unwrap_or_else(|_| panic!("Corrupt DB, field 2 not a u64: [{line}]"));

            db_map.insert(format!("{seen_on_host} {selector}"), timestamp);
        }
    }

    db_map
}

fn save_hashmap_to_file(db: &HashMap<String, u64>, data_file: &PathBuf) -> std::io::Result<()> {
    let mut fp = File::create(data_file)?;

    for (identifier, timestamp) in db {
        fp.write_all(format!("{identifier} {timestamp}\n").as_bytes())?;
    }

    Ok(())
}

pub fn save(db: &HashMap<String, u64>) {
    let data_dir = get_data_dir();

    // Ignore return value, because it is unclear to me how to distinguish between actual errors
    // and already existing paths.
    let _ = create_dir_all(&data_dir);

    let data_file = data_dir.join("seen.new");

    if let Err(msg) = save_hashmap_to_file(db, &data_file) {
        panic!("Could not save DB to temp file: {msg}");
    }
    if let Err(msg) = rename(data_file, data_dir.join("seen")) {
        panic!("Could not rename temp DB file: {msg}");
    }
}
