/* 
 * Apple // emulator for Linux: C portion of audio
 *
 * Copyright 2000 Robert Munafo
 *
 * This software package is subject to the GNU General Public License
 * version 2 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#include <fcntl.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>

#include "devaudio.h"
#include "cpu.h"
#include "glue.h"
#include "misc.h"
#include "prefs.h"

#ifdef COUNT_CYCLES

/* If /dev/audio method is being used, all the following variables are
 * used, otherwise they aren't */

/* This is the amount of time (in 10ths of a second) of the maximum desired
 * amount of silence. There is a tradeoff here: If you make it really short,
 * music programs sound wrong because the gaps (rests) get lost. But if
 * you make it really long, then the "maximum speed" setting (F9) doesn't
 * provide as much speed, because you can only run at maximum speed when
 * sound is not being played. I have found after a fair amount of fiddling
 * with the setting that the best setting seems to be about 1/2 second. */
#define MAX_SILENCE_TIME 5L

#define AUDIO_FREQ_ASK 48000
static long audio_freq;
#define CLOCK_SPEED_6502 1022727L
static long audio_incr;
static long audio_max_silence;
static long audio_flushrate;
#define AUDIO_PULSEBUF 512
static int audio_fd;
static long last_audio_cycles;

static char speaker_hi[AUDIO_PULSEBUF+2];
static char speaker_lo[AUDIO_PULSEBUF+2];
static int spkr_state;
static long audio_toflush;

/* This routine is CALLED FROM ASSEMBLY in response to read or write
 * accesses to $C030-$C03F, but only if /dev/audio sound output method was
 * set at startup. */
GLUE_C_READ(devaudio_read_speaker)
{
  long ctr;
  long first2;
  int len;
  char *buf;

  /* Find out how many clock cycles have gone by since the last time
   * the speaker was toggled */
  ctr = cpu65_current.cycles - last_audio_cycles;
  /* Convert to our sampling rate */
  ctr /= audio_incr;
  
  /* Range check, and update last_audio_cycles. Fractions of a sample
   * will automatically be counted towards the next sample; this
   * ensures the accuracy of pitch for sustained notes. */
  
  /* Don't output blocks of silence longer than audio_max_silence */
  if ((ctr > audio_max_silence) || (ctr < -1)) {
	/* There was a long delay. Just catch up. */
	ctr = audio_max_silence;
    last_audio_cycles = cpu65_current.cycles - (ctr * audio_incr);
	/* because there has been a long delay, the last speaker transition
     * hasn't been heard yet. That means we would get a "click"
     * at the beginning of the max_silence we're about to write out.
     * We avoid this click by toggling the state before doing the write. */
	spkr_state = spkr_state ? 0 : 1;
  } else if (ctr < 1) {
	/* This is to make sure the following actually generates a sound:
	 * loop1   lea $C030  ; toggle speaker
	 *         nop
     *         lea $C030  ; toggle speaker again
     *         dex
     *         bcc loop1
	 * However, we don't do this if the audio output is already more
     * than one full sample ahead of the 6502 -- otherwise, (particularly
     * when the /dev/audio sampling rate is very low, we'd get a
     * continuous high-pitched tone, which would almost certainly be of
     * the wrong frequency. */
	if (ctr == 0) {
	  ctr = 1;
	}
  }

  /* Okay, now write the audio bytes corresponding to the amount
   * of time that has elapsed */

  buf = spkr_state ? speaker_hi : speaker_lo;

#ifdef LAME_O
  /* The first two samples after a toggle are special */
  first2 = ctr;
  if (first2 > 2) {
	first2 = 2;
  }
  last_audio_cycles += (first2 * audio_incr);
  len = first2;
  write(audio_fd, buf+AUDIO_PULSEBUF, len);
  ctr -= first2;
#endif

  /* Output solid blocks of PULSEBUF size until we're down to less
   * then one PULSEBUF */
  while (ctr > AUDIO_PULSEBUF) {
	last_audio_cycles += (AUDIO_PULSEBUF * audio_incr);
	ctr -= AUDIO_PULSEBUF;
	len = AUDIO_PULSEBUF;
	write(audio_fd, buf, len);
  }
	
  /* Now output the last bit */
  if (ctr) {
	len = ctr;
	write(audio_fd, buf, len);
	last_audio_cycles += (ctr * audio_incr);
  }
  
  /* See if it's time to flush (synchronize) the audio output */
  audio_toflush += ctr;
  if (audio_toflush > audio_flushrate) {
	/* ioctl(audio_fd, I_FLUSH); */
	audio_toflush = 0;
  }
  
  /* Now actually toggle the speaker state */
  spkr_state = spkr_state ? 0 : 1;
}

void devaudio_init()
{
  int freq, result;
  int i;

	audio_fd = open("/dev/audio", O_WRONLY, 0);
	if (audio_fd == -1) {
	  /* It failed */
	  perror("/dev/audio");
	} else {
	  /* success */
	  sound_ready |= SOUND_DEVAUDIO;

	  /* Now set up all our variables, etc. */

	  /* First, try to set audio speed. If unsuccessful, we use the standard /dev/audio
	   * speed which is 8000 Hz (Sun workstations, Sparc, SunOS, Solaris) because almost
	   * all other /dev/audio implementations emulate the Sun */
	  freq = AUDIO_FREQ_ASK;
	  result = ioctl(audio_fd, SNDCTL_DSP_SPEED, &freq);
	  if (result == -1) {
		perror("SNDCTL_DSP_SPEED");
		audio_freq = 8000L;
	  } else {
		printf("Got speed %i\n", freq);
		audio_freq = freq;
	  }
	  audio_max_silence = (MAX_SILENCE_TIME * audio_freq) / 10L;
	  audio_flushrate = audio_freq / 50;
	  audio_incr = CLOCK_SPEED_6502 / audio_freq;

          {
             long fragspec;
             fragspec = 0x00020008L;
             result = ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &fragspec);
          }

	  /* Set up our buffer used to write to the speaker. */
	  for(i=0; i<AUDIO_PULSEBUF; i++) {
		speaker_hi[i] = 140; /* volume */
		speaker_lo[i] = 128;
	  }
	  /* The first two samples are in-between. This is to simulate the speaker's inertia:
	   * If you toggle the speaker, you don't actually get a perfect square wave pulse
	   * out but instead you get a sort of exponential curve tending towards an asymtote */
	  speaker_hi[AUDIO_PULSEBUF] = 136; speaker_hi[AUDIO_PULSEBUF+1] = 138;
	  speaker_lo[AUDIO_PULSEBUF] = 136; speaker_lo[AUDIO_PULSEBUF+1] = 132;

	  spkr_state = 0;
	  audio_toflush = 0;
	  last_audio_cycles = 0;
	}
}

#else /* !COUNT_CYCLES */
/* If COUNT_CYCLES not active, this driver can't function. */

/* Non-working stub just to keep linker satisfied.  Might even cause a crash
 * (calling conventions, etc.) if actually used. 
 * 
 * This won't actually be called -- since init is also a no-op, 
 * the devaudio will never be enabled.
 */
void devaudio_read_speaker(void){}

void devaudio_init(void){}

#endif /* !COUNT_CYCLES */
