/* 
 * Apple // emulator for Linux: C support for 6502 on i386
 *
 * Copyright 1994 Alexander Jean-Claude Bottema
 * Copyright 1995 Stephen Lee
 * Copyright 1997, 1998 Aaron Culliney
 * Copyright 1998, 1999, 2000, 2001 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 2 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#include <string.h>

#include "cpu.h"

/* different than in defs.h! */
#define C_Flag_6502	0x1
#define X_Flag_6502	0x20
#define I_Flag_6502	0x4
#define V_Flag_6502	0x40
#define B_Flag_6502	0x10
#define D_Flag_6502	0x8
#define Z_Flag_6502	0x2
#define N_Flag_6502	0x80

#ifdef COUNT_CYCLES
/* These are "best-case" timing
 * Certain instruction have "penalty" cycles under some conditions:
 *  +1 if a branch is taken
 *  +1 (making +2 total) if branch taken to a different page.
 *  +1 on an absolute-index, indirect-index (but not index-indirect)
 *     read where the index pushes the EA to a new page.
 * 
 * I've arbitrarily assigned a cycle count of 2 to the hang instructions, 
 * consistent with the working instructions in their column.
 *
 * Based on John West's "64doc,v" file.
 */
static const unsigned int timings_nmos[256] = 
{
  7, 6, 2, 8, 3, 3, 5, 5,   3, 2, 2, 2, 4, 4, 6, 6, 	/* 0? */
  2, 5, 2, 8, 4, 4, 6, 6,   2, 4, 2, 7, 4, 4, 7, 7, 	/* 1? */
  6, 6, 2, 8, 3, 3, 5, 5,   4, 2, 2, 2, 4, 4, 6, 6, 	/* 2? */
  2, 5, 2, 8, 4, 4, 6, 6,   2, 4, 2, 7, 4, 4, 7, 7, 	/* 3? */
  6, 6, 2, 8, 3, 3, 5, 5,   3, 2, 2, 2, 3, 4, 6, 6, 	/* 4? */
  2, 5, 2, 8, 4, 4, 6, 6,   2, 4, 2, 7, 4, 4, 7, 7, 	/* 5? */
  6, 6, 2, 8, 3, 3, 5, 5,   4, 2, 2, 2, 5, 4, 6, 6, 	/* 6? */
  2, 5, 2, 8, 4, 4, 6, 6,   2, 4, 2, 7, 4, 4, 7, 7, 	/* 7? */
 
  2, 6, 2, 6, 3, 3, 3, 3,   2, 2, 2, 2, 4, 4, 4, 4, 	/* 8? */
  2, 6, 2, 6, 4, 4, 4, 4,   2, 5, 2, 5, 5, 5, 5, 5, 	/* 9? */
  2, 6, 2, 6, 3, 3, 3, 3,   2, 2, 2, 2, 4, 4, 4, 4, 	/* A? */
  2, 5, 2, 5, 4, 4, 4, 4,   2, 4, 2, 4, 4, 4, 5, 5, 	/* B? */
  2, 6, 2, 8, 3, 3, 5, 5,   2, 2, 2, 2, 4, 4, 6, 6, 	/* C? */
  2, 5, 2, 8, 4, 4, 6, 6,   2, 4, 2, 7, 4, 4, 7, 7, 	/* D? */
  2, 6, 2, 8, 3, 3, 5, 5,   2, 2, 2, 2, 4, 4, 6, 6, 	/* E? */
  2, 5, 2, 8, 4, 4, 6, 6,   2, 4, 2, 7, 4, 4, 7, 7, 	/* F? */
};

/* 65C02 has no undocumented instructions -- all unused codes are NOPs.
 * I'm assuming they take the same time as a NOP.  26 new instructions
 * are added.
 * 
 * It executes 6C one slower, and normally executes absolute-indexed RMWs
 * one faster.
 *
 * Penalties:
 *   +1 penalty on decimal operations
 *   +1 on absolute-indexed read-mod-write
 *	(so same speed as 6502 in this case) 
 *   Plus the 6502 issues.
 */ 
static const unsigned int timings_cmos[256] = 
{
  7, 6, 2, 2, 5, 3, 5, 2,   3, 2, 2, 2, 5, 4, 6, 2, 	/* 0? */
  2, 5, 5, 2, 5, 4, 6, 2,   2, 4, 2, 2, 5, 4, 7, 2, 	/* 1? */
  6, 6, 2, 2, 3, 3, 5, 2,   4, 2, 2, 2, 4, 4, 6, 2, 	/* 2? */
  2, 5, 5, 2, 4, 4, 6, 2,   2, 4, 2, 2, 4, 4, 7, 2, 	/* 3? */
  6, 6, 2, 2, 2, 3, 5, 2,   3, 2, 2, 2, 3, 4, 6, 2, 	/* 4? */
  2, 5, 5, 2, 2, 4, 6, 2,   2, 4, 3, 2, 2, 4, 7, 2, 	/* 5? */
  6, 6, 2, 2, 3, 3, 5, 2,   4, 2, 2, 2, 6, 4, 6, 2, 	/* 6? */
  2, 5, 5, 2, 4, 4, 6, 2,   2, 4, 4, 2, 6, 4, 7, 2, 	/* 7? */
 
  3, 6, 2, 2, 3, 3, 3, 2,   2, 2, 2, 2, 4, 4, 4, 2, 	/* 8? */
  2, 6, 5, 2, 4, 4, 4, 2,   2, 5, 2, 2, 4, 5, 5, 2, 	/* 9? */
  2, 6, 2, 2, 3, 3, 3, 2,   2, 2, 2, 2, 4, 4, 4, 2, 	/* A? */
  2, 5, 5, 2, 4, 4, 4, 2,   2, 4, 2, 2, 4, 4, 5, 2, 	/* B? */
  2, 6, 2, 2, 3, 3, 5, 2,   2, 2, 2, 2, 4, 4, 6, 2, 	/* C? */
  2, 5, 5, 2, 2, 4, 6, 2,   2, 4, 3, 2, 2, 4, 7, 2, 	/* D? */
  2, 6, 2, 2, 3, 3, 5, 2,   2, 2, 2, 2, 4, 4, 6, 2, 	/* E? */
  2, 5, 5, 2, 2, 4, 6, 2,   2, 4, 4, 2, 2, 4, 7, 2, 	/* F? */
}; 
#endif /* COUNT_CYCLES */

static void initialize_code_tables(void)
{
    int		i;

    for (i = 0; i < 256; i++)
    {
	unsigned char	val = 0;

	if (i & C_Flag)
	    val |= C_Flag_6502;
	if (i & X_Flag)
	    val |= X_Flag_6502;
	if (i & I_Flag)
	    val |= I_Flag_6502;
	if (i & V_Flag)
	    val |= V_Flag_6502;
	if (i & B_Flag)
	    val |= B_Flag_6502;
	if (i & D_Flag)
	    val |= D_Flag_6502;
	if (i & Z_Flag)
	    val |= Z_Flag_6502;
	if (i & N_Flag)
	    val |= N_Flag_6502;

	cpu65_flags_encode[ i ] = val | 0x20;
	cpu65_flags_decode[ val ] = i;
    }
}


void cpu65_set(int flags)
{
     initialize_code_tables();

     switch (flags & 0xf)
     {
     case CPU65_NMOS:
	if (flags & CPU65_FAULT)
            memcpy(cpu65__opcodes,cpu65__nmosbrk,1024);	
        else
	    memcpy(cpu65__opcodes,cpu65__nmos,1024);
	break;
#ifdef COUNT_CYCLES 
	memcpy(cpu65__timings,timings_nmos,1024);
#endif /* COUNT_CYCLES */
     case CPU65_C02:
        memcpy(cpu65__opcodes,cpu65__cmos,1024);
#ifdef COUNT_CYCLES
	memcpy(cpu65__timings,timings_cmos,1024);
#endif /* COUNT_CYCLES */
        break;
     default:
        abort();
     }
	
     cpu65__signal = 0;
}

void cpu65_interrupt(int reason)
{
    cpu65__signal = reason;
}

