/* 
 * Apple // emulator for Linux:	svgalib graphics support
 *
 * Copyright 1994 Alexander Jean-Claude Bottema
 * Copyright 1995 Stephen Lee
 * Copyright 1997, 1998 Aaron Culliney
 * Copyright 1998, 1999, 2000, 2001 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 2 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#include <vga.h>
#include <vgakeyboard.h>
#include <string.h>
#include <stdio.h>
#include <sys/kd.h>
#include <termios.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <stdlib.h>

#include "video.h"
#include "vidpriv.h"
#include "misc.h"
#include "keys.h"

/* I need at least 560x192 for 80col.  it seems that direct linear
 * framebuffer access to anything in the realm of 640x480x256 not
 * standard with vga/svga?
 * Memory sizes are hardcode here.  ugly ugly ugly.
 */

static unsigned char	*svga_GM;			/* SVGA base address of graphic area */
static unsigned char	vga_mem_page_0[64000];		/* page0 framebuffer */
static unsigned char	vga_mem_page_1[64000];		/* page1 framebuffer */


/* -------------------------------------------------------------------------
    video_setpage(p):    Switch to screen page p
   ------------------------------------------------------------------------- */
void video_setpage(int p)
{
    if (p == video__current_page) return;

    if (p)
    {
	memcpy(vga_mem_page_0,svga_GM,64000);
	memcpy(svga_GM,vga_mem_page_1,64000);
	video__current_page = 1;
	video__fb1 = vga_mem_page_0;
	video__fb2 = svga_GM;
    }
    else
    {
	memcpy(vga_mem_page_1,svga_GM,64000);
	memcpy(svga_GM,vga_mem_page_0,64000);
	video__current_page = 0;
	video__fb1 = svga_GM;
	video__fb2 = vga_mem_page_1;
    }
}

/* -------------------------------------------------------------------------
    c_initialize_colors():    Initialize color palette
   ------------------------------------------------------------------------- */

static void c_initialize_colors()
{
    /* Apple graphics colors */
    vga_setpalette( COLOR_BLACK,	 0,  0,  0 );
    vga_setpalette( COLOR_DARK_BLUE,	 0,  0, 32 );
    vga_setpalette( COLOR_DARK_GREEN,	 0, 36,  0 );
    vga_setpalette( COLOR_BLUE,		28,  6, 63 );
    vga_setpalette( COLOR_BROWN,	37, 21, 10 );
    vga_setpalette( COLOR_LIGHT_GREY,	37, 42, 42 );
    vga_setpalette( COLOR_GREEN,	 0, 63,  0 );
    vga_setpalette( COLOR_AQUA,		32, 63, 32 );
    vga_setpalette( COLOR_MAGENTA,	48,  0, 12 );
    vga_setpalette( COLOR_PURPLE,	41, 13, 42 );
    vga_setpalette( COLOR_DARK_GREY,	26, 26, 26 );
    vga_setpalette( COLOR_SKY_BLUE,	 3, 47, 58 );
    vga_setpalette( COLOR_ORANGE,	63,  6, 11 );
    vga_setpalette( COLOR_PINK,		63, 39, 37 );
    vga_setpalette( COLOR_YELLOW,	63, 63,  0 );
    vga_setpalette( COLOR_WHITE,	63, 63, 63 );

    /* Flashing colors - alternated periodically */
    vga_setpalette( COLOR_FLASHING_BLACK,  0,  0,  0 );
    vga_setpalette( COLOR_FLASHING_WHITE, 63, 63, 63 );

    /* Colors for the configuration menus */
    vga_setpalette( COLOR_IF_RED,	  62,  0,  0 );
    vga_setpalette( COLOR_IF_GREEN,	   0, 62,  0 );
    vga_setpalette( COLOR_IF_BLUE,	   0,  0, 40 );

}

/* -------------------------------------------------------------------------
    c_initialize_keyboard()
   ------------------------------------------------------------------------- */

static void c_initialize_keyboard()
{
    if (keyboard_init()) {
    	printf("Error: Could not switch to RAW keyboard mode.\n");
	exit(-1);
    }

    keyboard_translatekeys(DONT_CATCH_CTRLC);
    keyboard_seteventhandler( c_read_raw_key );
}

static void c_flash_cursor(int on) {
    if (!on) {
	vga_setpalette( COLOR_FLASHING_BLACK, 0, 0, 0 );
	vga_setpalette( COLOR_FLASHING_WHITE, 63, 63, 63 );
    } else {
	vga_setpalette( COLOR_FLASHING_BLACK, 63, 63, 63 );
	vga_setpalette( COLOR_FLASHING_WHITE, 0, 0, 0 );	
    }
}

/* -------------------------------------------------------------------------
    video_init()
   ------------------------------------------------------------------------- */
void video_init(void) {
    /* I'm forcing VGA mode since it seems to be supported by
     * the lowest common denominator cards.
     */
    printf("Using standard VGA 320x200 mode.\n");
    printf("Press RETURN to continue...");
    getchar();

    vga_setchipset( VGA );
    if (vga_init()) {
	printf("Cannot initialize svgalib!\n");
	exit(1);
    }

    vga_setmode( G320x200x256 );

/*    vga_claimvideomemory( 131072 );*/
    c_initialize_keyboard();
    svga_GM = video__fb1 = vga_getgraphmem();
    video__fb2 = vga_mem_page_1;

    memset(video__fb1,0,64000); /* start as black */
    memset(video__fb2,0,64000);

    c_initialize_colors();






}

void video_shutdown(void) {
    vga_setmode(TEXT);
    keyboard_close();
}

void video_sync(int block)
{
    static int flash_count;

    if (block) 
    {
	/* keyboard_waitforupdate();    --- seems to cause bad crashes... */
	usleep(TIMER_DELAY);
        keyboard_update();
    }
    else
	keyboard_update();

    switch (++flash_count)
    {
    case 6: 
        c_flash_cursor(1);
        break;
    case 12: 
        c_flash_cursor(0);
        flash_count = 0; 
        break;
    default:
        break;
    }
}

struct assoc
{
  const char *name;
  int value;
};

static struct assoc scancode_map[] = 
{ 
{"0", 11},
{"1", 2},
{"2", 3}, 
{"3", 4},
{"4", 5},
{"5", 6},
{"6", 7},
{"7", 8},
{"8", 9},
{"9", 10},
{"A", 30},
{"Alt_L", 56},
{"Alt_R", 100},
{"B", 48},
{"BackSpace", 14},
{"Break", 101},
{"C", 46},
{"Caps_Lock", 58},
{"Control_L", 29},
{"Control_R", 97},
{"D", 32},
{"Delete", 111},
{"Down", 108},
{"E", 18},
{"End", 107},
{"Escape", 1},
{"F", 33},
{"F1", 59},
{"F10", 68},
{"F11", 87},
{"F12", 88},
{"F2", 60},
{"F3", 61},
{"F4", 62},
{"F5", 63},
{"F6", 64},
{"F7", 65},
{"F8", 66},
{"F9", 67},
{"G", 34},
{"H", 35},
{"Home", 102},
{"I", 23},
{"Insert", 110},
{"J", 36},
{"K", 37},
{"KP_0", 82},
{"KP_1", 79},
{"KP_2", 80},
{"KP_3", 81},
{"KP_4", 75},
{"KP_5", 76},
{"KP_6", 77},
{"KP_7", 71},
{"KP_8", 72},
{"KP_9", 73},
{"KP_Divide", 98},
{"KP_Enter", 96},
{"KP_MULTIPLY", 55},
{"KP_Period", 83},
{"KP_Plus", 78},
{"KP_Subtract", 74},
{"L", 38},
{"Left", 105},
{"M", 50},
{"Menu", 127},
{"Meta_L", 125},
{"Meta_R", 126},
{"N", 49},
{"Num_Lock", 69},
{"O", 24},
{"P", 25},
{"Page_Down", 109},
{"Page_Up", 104},
{"Pause", 119},
{"Print", 99},
{"Q", 16},
{"R", 19},
{"Return", 28},
{"Right", 106},
{"S", 31},
{"Scroll_Lock", 70},
{"Shift_L", 42},
{"Shift_R", 54},
{"T", 20},
{"Tab", 15},
{"U", 22},
{"Up", 103},
{"V", 47},
{"W", 17},
{"X", 45},
{"Y", 21},
{"Z", 44},
{"apostrophe", 40},
{"backslash", 43},
{"bracketleft", 26},
{"bracketright", 27},
{"comma", 51},
{"equal", 13},
{"grave", 41},
{"less", 86},
{"minus", 12},
{"period", 52},
{"semicolon", 39},
{"slash", 53},
{"space", 57}
};

static int comparer(const void *a,const void *b)
{
    static const struct assoc *aa,*bb;

    aa = a; bb = b;

    return strcmp(aa->name,bb->name);
}

int video_keycode(const char *keyname)
{
    struct assoc key, *ret;

    key.name = keyname;

    ret = bsearch(&key,scancode_map,
                  sizeof scancode_map / sizeof(struct assoc), 
                  sizeof (struct assoc),&comparer);

    if (!ret) return -1;

    return ret->value;
}
