/* 
 * Apple // emulator for Linux: Combined Timer/Clock/Harddrive
 *
 * Copyright 1994 Alexander Jean-Claude Bottema
 * Copyright 1995 Stephen Lee
 * Copyright 1997, 1998 Aaron Culliney
 * Copyright 1998, 1999, 2000, 2001 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 2 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include <math.h>
#include <time.h>
#include <sys/time.h>

#include "glue.h"
#include "misc.h"
#include "cpu.h"

struct timeval time1, time2;

FILE *hard_file[2];
const char *hard_filename[2] = { "part1.hdv", "part2.hdv" };

void clockhd_setmark ();
void clockhd_readwatch ();
void clockhd_prodosclk ();
void clockhd_prodosdisk ();
void clockhd_boot ();

GLUE_C_READ (clockhd_setmark)
{
  time1 = time2;
  gettimeofday (&time2, 0);
}

/* This function is used as an Applesoft USR() vector.
 *
 * It returns the seconds between the last two reads at C0X0, formatted
 * as an Applesoft floating point number.  
 *
 */
GLUE_C_GATE (clockhd_readwatch, base_cxrom, apple_ii_64k)
{
  double timediff;
  int exponent;

#if 0
  printf ("Time vectors: %10d.%06d , %10d.%06d\n",
	  time1.tv_sec, time1.tv_usec, time2.tv_sec, time2.tv_usec);
#endif /* 0 */

  timediff = (time2.tv_usec - time1.tv_usec);
  timediff /= 1000000.0;
  timediff += (time2.tv_sec - time1.tv_sec);

#if 0
  printf ("Time difference: %e (%a)\n", timediff, timediff);
#endif /* 0 */

  /* Now, we convert to Applesoft floating point format */

  exponent = ilogb (timediff) + 1;

  if (exponent <= -0x80)
    {
      /* Actually, this could only happen if the time difference reported
       * by the OS was zero. 
       *
       * I'm not even going to handle the case of an overflow -- which 
       * would represent an interval of over 10^30 years.      
       */
      memset (base_stackzp + 0x9d, 0, 6);
    }
  else
    {
      base_stackzp[0x9d] = exponent + 0x80;	/* exponent */
      *((unsigned int *) (base_stackzp + 0x9e)) = ntohl ((unsigned int) scalb (timediff, 32 - exponent));	/* mantissa */
      base_stackzp[0xa2] = 0;	/* sign */

#if 0
      printf ("Reg: %02X %02X %02X %02X %02X %02x\n",
	      base_stackzp[0x9d], base_stackzp[0x9e],
	      base_stackzp[0x9f], base_stackzp[0xa0],
	      base_stackzp[0xa1], base_stackzp[0xa2]);
#endif /* 0 */
    }
}

/* This function updates the time stored in the ProDOS global page
 * (BF90-BF93)
 *
 */
GLUE_C_GATE (clockhd_prodosclk, base_cxrom, apple_ii_64k)
{
  time_t now;
  struct tm *lt;

  time (&now);

  lt = localtime (&now);

  base_ramwrt[0xBF90] = (((lt->tm_mon + 1) % 8) << 5) + lt->tm_mday;
  base_ramwrt[0xBF91] = ((lt->tm_year % 100) << 1) + (lt->tm_mon >= 7);
  base_ramwrt[0xBF92] = lt->tm_min;
  base_ramwrt[0xBF93] = lt->tm_hour;
}

GLUE_C_GATE (clockhd_prodosdisk, base_cxrom, apple_ii_64k)
{
  int cmd, unit, x, address;
  unsigned char *start;
  union emu_dma_arg arg;

  cmd = base_stackzp[0x42];
  unit = base_stackzp[0x43] >> 7;

#if 0
  printf ("Cmd: %02X %02X %02X %02X %02X %02x\n",
	  base_stackzp[0x42], base_stackzp[0x43],
	  base_stackzp[0x44], base_stackzp[0x45],
	  base_stackzp[0x46], base_stackzp[0x47]);
#endif /* 0 */

  if (!hard_filename[unit])
    {
      /*
       * If no filename is defined, return Not Connected ($28).
       */

      cpu65_current.a = 0x28;
      cpu65_current.f |= C_Flag;
      return;

    }

  /* Only allow FORMAT to nonexistent files */
  if (cmd != 3 && !hard_file[unit])
    {
      /* If a filename is defined, but no file exists return 
       * $27 (I/O Error).  This suggests that the media may
       * become usable with a FORMAT.
       */

      cpu65_current.a = 0x27;
      cpu65_current.f |= C_Flag;
      return;
    }

  switch (cmd)
    {
    case 0:			/* STATUS */
      break;
    case 1:			/* READ */
    case 2:			/* WRITE */
      fseek (hard_file[unit], (base_stackzp[0x46] +
			       (base_stackzp[0x47] << 8)) * 512, SEEK_SET);

      address = base_stackzp[0x44] + (base_stackzp[0x45] << 8);
      arg.file = hard_file[unit];

      if (cmd == 1)
	emu_dma (EMUDMA_IN | EMUDMA_FILE, address, arg, 512);
      else
	emu_dma (EMUDMA_OUT | EMUDMA_FILE, address, arg, 512);

      if (ferror (hard_file[unit]))
	{
	  cpu65_current.a = 0x27;
	  cpu65_current.f |= C_Flag;
	  return;
	}
      break;
    case 3:			/* FORMAT */
      if (hard_file[unit])
	fclose (hard_file[unit]);
      hard_file[unit] = fopen (hard_filename[unit], "w+");
      if (!hard_file[unit])
	{
	  cpu65_current.a = 0x27;
	  cpu65_current.f |= C_Flag;
	  return;
	}
      break;
    }

  cpu65_current.a = 0x0;
  cpu65_current.f &= ~C_Flag;


}


GLUE_C_GATE (clockhd_boot, base_cxrom, apple_ii_64k)
{
  int err;
  union emu_dma_arg arg;

  err = 1;

  if (hard_file[0])
    {
      fseek (hard_file[0], 0, SEEK_SET);
      arg.file = hard_file[0];
      emu_dma (EMUDMA_IN | EMUDMA_FILE, 0x800, arg, 512);
      err = ferror (hard_file[0]) || base_ramrd[0x800] != 0x01;
    }

  if (err)
    {
      /* No usable bootblock.
       * 
       * We put the emulator into an infinite loop -- Sloppy, but not 
       * much worse than what the disk ][ does.
       * 
       * Reportedly, some cards could redirect back into the ROM
       * and make it move on to lower-priority cards.  However,
       * I'm not 100% sure how to pull this off.
       */
      base_ramrd[0x801] = 0x80;  /* BRA */
      base_ramrd[0x802] = 0xFE;  /*  -2 */
    }

  /* Set up return to 0x0801.  A bit hacky -- this is admittedly a 
   * weakness in my callgate design.
   *
   * Note that 6502 return addresses on the stack are biased. We want to
   * "return" to 0x801, but 0x800 must be stored.
   */
  base_stackzp[0x1FF] = 0x08;
  base_stackzp[0x1FE] = 0x00;
  cpu65_current.sp = 0xfd;

  /* ProDOS seems to need our slot number << 4 in X to boot, and also has 
   * code that will treat us as a Disk ][ if A is less than 3.  I haven't
   * seen any mention of this in the notes I have available.
   */

  cpu65_current.x = (cpu65_current.pc >> 4) & 0x70; 
  cpu65_current.a = 0xff;


}

void
clockhd_install (int slot)
{
  int base;

  if (hard_filename[0])
    hard_file[0] = fopen (hard_filename[0], "r+");
  if (hard_filename[1])
    hard_file[1] = fopen (hard_filename[1], "r+");

  /* null rom */
  memset (apple_ii_64k[0] + 0xC000 + 0x100 * slot, 0, 0x100);

  base = 0xC000 + (slot << 8);

  /* ProDOS disk driver stigmata */
  apple_ii_64k[0][base + 0x01] = 0x20;	/* meaningless ID bytes */
  apple_ii_64k[0][base + 0x05] = 0x03;
  apple_ii_64k[0][base + 0x07] = 0x3C;
  apple_ii_64k[0][base + 0xFC] = 0xFF;	/* Capacity of drive (maximum) */
  apple_ii_64k[0][base + 0xFD] = 0xFF;
  apple_ii_64k[0][base + 0xFE] = 0x5F;	/* 2 drives, interruptable, 
					 * all commands recognized 
					 */
  apple_ii_64k[0][base + 0xFF] = 0xC0;	/* Entry point */

  cpu65_vmem[base + 0x00].r = clockhd_boot;
  cpu65_vmem[base + 0x40].r = clockhd_readwatch;
  cpu65_vmem[base + 0x80].r = clockhd_prodosclk;
  cpu65_vmem[base + 0xC0].r = clockhd_prodosdisk;

  /* Softswitches */
  base = 0xC080 + (slot << 4);

  cpu65_vmem[base + 0x0].r = clockhd_setmark;
}

