/* 
 * Apple // emulator for Linux: Keyboard handler
 *
 * Copyright 1994 Alexander Jean-Claude Bottema
 * Copyright 1995 Stephen Lee
 * Copyright 1997, 1998 Aaron Culliney
 * Copyright 1998, 1999, 2000, 2001 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 2 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <linux/keyboard.h>

#include "keys.h"
#include "misc.h"
#include "video.h"
#include "interface.h"
#include "cpu.h"
#include "prefs.h"

struct assoc
{
  const char *name;
  int value;
};

struct keymap
{
    unsigned char plain[MAX_KEYCODES];
    unsigned char ctrl[MAX_KEYCODES];
    unsigned char shift[MAX_KEYCODES];
    unsigned char both[MAX_KEYCODES];
    unsigned char lock[MAX_KEYCODES];
};

/* parameters for generic and keyboard-simulated joysticks */
short		joy_x = 127;
short		joy_y = 127;
unsigned char	key_register = 0;

static int		next_key = -1;
static int		in_mygetch = 0;

int			keys_down = 0;

static unsigned short		max_speed = 0;

static struct keymap	keymap2plus, keymap2e;

/* ----------------------------------------------------
    functions
   ---------------------------------------------------- */

/* -------------------------------------------------------------------------
 * This routine is called periodically to update the state of the emulator.
 * 	-handles switching to menus
 * 	-polls PC Joystick
 * 	-update palette for flashing text
 * ------------------------------------------------------------------------- */
void c_periodic_update(int dummysig) {
    int current_key;

    video_sync(0);

    if (next_key >= 0)
    {	
	current_key = next_key;
	next_key = -1;

	if (current_key < 128) {
	    key_register = current_key | 0x80;
	} else switch (current_key) {
	case RST:
	    cpu65_interrupt(ResetSig);
	    break;
	case BOT:
	    cpu65_interrupt(RebootSig);
	    break;
	case J_C:
	    if (joy_mode != JOY_KYBD) break;
	    joy_x = joy_center_x;
	    joy_y = joy_center_y;
	    break;
	case kS6D1:
	    c_interface_select_diskette( 0 );
	    break;
	case kS6D2:
	    c_interface_select_diskette( 1 );
	    break;
	case kPause:
	    while (c_mygetch(1) == -1) { }/*busy loop*/
	    break;
	case kKBHelp:
	    c_interface_keyboard_layout();
	    break;
#ifdef DEBUGGER
	case kDebug:
	    cpu65_interrupt(EnterDebugSig);
	    break;
#endif
	case kFast:
	    if (max_speed != 0)
	        cpu65_delay = max_speed, max_speed = 0;
	    else
	        max_speed = cpu65_delay, cpu65_delay = 1;
	    break;
	case kMenu:
	    if (max_speed != 0)
		cpu65_delay = max_speed, max_speed = 0;
	    c_interface_parameters();
	    break;
	case kCaps:	/* set/reset caps lock */
	    keys_down ^= BLock;
            break;
        }
    }

    /* simulated joystick */
    if (joy_mode == JOY_KYBD) {
	if (keys_down & (BJUL | BJ_U | BJUR))
	{
		if (joy_y > joy_step)
		    joy_y -= joy_step;
		else
		    joy_y = 0;
	}

	if (keys_down & (BJDL | BJ_D | BJDR))
	{
		if (joy_y < joy_range - joy_step)
		    joy_y += joy_step;
		else
		    joy_y = joy_range-1;
	}

	if (keys_down & (BJUL | BJ_L | BJDL))
	{
		if (joy_x > joy_step)
		    joy_x -= joy_step;
		else
		    joy_x = 0;
	}

	if (keys_down & (BJUR | BJ_R | BJDR))
	{
		if (joy_x < joy_range - joy_step)
		    joy_x += joy_step;
		else
		    joy_x = joy_range-1;
	}
    }
    else if (joy_mode == JOY_DIGITAL)
    {
	if (keys_down & (BJUL | BJ_U | BJUR))
	    joy_y = 0;
        else if (keys_down & (BJDL | BJ_D | BJDR))
	    joy_y = joy_range - 1;
	else
	    joy_y = joy_center_y;

	if (keys_down & (BJUL | BJ_L | BJDL))
	    joy_x = 0;
 	else if (keys_down & (BJUR | BJ_R | BJDR))
	    joy_x = joy_range - 1;
	else
	    joy_x = joy_center_x;
    }
#ifdef PC_JOYSTICK
    else if (joy_mode == JOY_PCJOY)
        poll_pcjoystick();
#endif
    else if (joy_mode == JOY_OFF)
        joy_x = joy_y = 256;
}

/* -------------------------------------------------------------------------
    void c_read_raw_key() : handle a scancode
   ------------------------------------------------------------------------- */
void c_read_raw_key(int scancode, int pressed) {
    const unsigned char *keyrow = NULL;
    struct keymap *keymap;
    int action;
  
    /* determine which key mapping to use */

    if (apple_mode == IIE_MODE || in_mygetch) 
        keymap = &keymap2e;
    else
        keymap = &keymap2plus;

    if ((keys_down & (BkLShift|BkRShift)) &&
	(keys_down & (BkLCtrl|BkRCtrl)))
	keyrow = keymap->both;
    else if (keys_down & (BkLCtrl|BkRCtrl))
	keyrow = keymap->ctrl;
    else if (keys_down & (BkLShift|BkRShift))
	keyrow = keymap->shift;
    else if (keys_down & BLock)
	keyrow = keymap->lock;
    else
	keyrow = keymap->plain;

    action = keyrow[scancode];

    /* key is pressed */
    if (pressed) {

	if (action >= 224) 
            keys_down |= 1 << (action - 224);
        else next_key = action;
    }

    /* key is released */
    else {

	if (action >= 224) 
            keys_down &= ~(1 << (action - 224));
    }
}

int c_mygetch(int block)
{
    int retval;

    in_mygetch = 1;

    if (block) while (next_key == -1)
        video_sync(1);
    else
        video_sync(0);

    in_mygetch = 0;

    retval = next_key;
    next_key = -1;

    return retval;
}

static struct assoc action_names[] = {
  {"Jcenter", J_C},
  {"Jdown", J_D},
  {"Jdownleft", JDL},
  {"Jdownright", JDR},
  {"Jleft", J_L},
  {"Jright", J_R},
  {"Jup", J_U},
  {"Jupleft", JUL},
  {"Jupright", JUR},
  {"button0", JB0},
  {"button1", JB1},
  {"button2", JB2},
  {"caps", kCaps},
  {"ctrl1", kLCtrl},
  {"ctrl2", kRCtrl},
  {"debug", kDebug},
  {"end", kEND},
  {"fast", kFast},
  {"home", kHOME},
  {"keyhelp", kKBHelp},
  {"menu", kMenu},
  {"null", kNONE},
  {"pagedown", kPGDN},
  {"pageup", kPGUP},
  {"pause", kPause},
  {"reboot", BOT},
  {"reset", RST},
  {"s6d1", kS6D1},
  {"s6d2", kS6D2},
  {"shift1", kLShift},
  {"shift2", kRShift},
  {"space", 32}
};

static int comparer(const void *a,const void *b)
{
    static const struct assoc *aa,*bb;

    aa = a; bb = b;

    return strcmp(aa->name,bb->name);
}

static int action_num(const char *actname)
{
    struct assoc key, *ret;

    if (actname[0] && !actname[1]) /* one letter */
    {
        return actname[0];
    }

    if (actname[0] == '^' && actname[1] && !actname[2]) /* ctrl char */
    { 
        if (actname[1] == '?') return 127;

	if (actname[1] >= '@' && actname[1] <= '_') 
            return actname[1] - 64;
    }

    key.name = actname;

    ret = bsearch(&key,action_names,
                  sizeof action_names / sizeof(struct assoc), 
                  sizeof (struct assoc),
                  &comparer);

    if (!ret) 
    {
        fprintf(stderr,"unknown key action `%s'\n",actname);
        return kNONE;
    }

    return ret->value;
}

static void load_bindings(FILE *file,struct keymap *target)
{
    char *buffer;
    size_t size;
    char *word;
    int code,plain,shift,ctrl,both,lock;

    memset(target->plain,kNONE,MAX_KEYCODES);
    memset(target->shift,kNONE,MAX_KEYCODES);
    memset(target->ctrl,kNONE,MAX_KEYCODES);
    memset(target->both,kNONE,MAX_KEYCODES);
    memset(target->lock,kNONE,MAX_KEYCODES);

    buffer = 0; size = 0;

    while (getline(&buffer,&size,file) != EOF)
    {
        word = strtok(buffer,"\n\t ");

	code = video_keycode(word);

	if (code < 0)
	{
	    fprintf(stderr,"unknown key `%s'\n",word);
	}

        word = strtok(0,"\n\t ");

	plain = action_num(word);

	word = strtok(0,"\n\t ");
	if (word)
	{
 	    shift = action_num(word);

	    word = strtok(0,"\n\t ");

 	    ctrl = action_num(word);

	    word = strtok(0,"\n\t ");

 	    both = action_num(word);

	    word = strtok(0,"\n\t ");

 	    lock = action_num(word);
        }
        else 
        {
            lock = both = ctrl = shift = plain;
        }

	target->plain[code] = plain;
	target->shift[code] = shift;
	target->ctrl[code] = ctrl;
	target->both[code] = both;
	target->lock[code] = lock;    
    }
} 

void initialize_keymaps(void)
{
    FILE *f;

    f = fopen_cat_exit(system_path,"plus.key","r");
    load_bindings(f,&keymap2plus);
    fclose(f);

    f = fopen_cat_exit(system_path,"keymap.key","r");
    load_bindings(f,&keymap2e);
    fclose(f);
}

