
/*
 *   FID.C -- a utility to browse Apple II .DSK image files
 *
 *   By Paul Schlyter  (pausch@saaf.se)
 *
 *   2000 April-May
 *
 *   Ver 1.0:  2000-05-16
 *
 *   2001-03-17:  Bug fix in dumpBufferAsIntBasicFile(): bug in
 *                detokenization of IntBasic source
 *
 *   2001-10-10:  Bug fix in dumpfile(): didn't chain to next TS-list
 *                for files longer than 122 sectors
 *
 *   2001-12-26:  Response to an email - fixed some bugs related to
 *                dump output files:
 *                1: Now the output file is explicitly closed
 *                2: Now also the last sector is dumped. This bug
 *                   appeared when I fixed the 2001-10-10 bug above.
 *                Added a feature: disk image files can now be dumped
 *                in raw form.
 *
 *   2002-01-07:  Fixed a bug in F_findnext() for UNIX so it also
 *                works in explicit subdirectories
 *
 *   2002-01-09:  Fixed a bug in dumpBufferAsApplesoftFile() which
 *                earlier aborted a dump prematurely.  Also dumpfile()
 *                no longer exits with error if fewer sectors are
 *                present in TS-list than indicated by file length
 *                in directory.
 *
 *   2002-01-15:  General overhaul in file dump routine: now handles
 *                multiple TS-lists and sparse files as well.
 *                Allocation "holes" are filled in with sectors filled
 *                with 00h bytes.
 *
 *   2002-03-21:  Typo in Applesoft token table: "DEL" ==> "DEF"
 */

/*
 *   #define one of these as 1:
 *
 *     MSVC_WIN32   Ms VC++ ver 4 or later, for Win-32
 *
 *     MSC_DOS      Ms VC++ ver 1.52 or earlier,
 *                    or Ms C ver 6 or earlier, for DOS/Win-16
 *
 *     TURBO_C      Turbo C ver 2 (probably works on TC++ too)
 *
 *     UNIX         gcc under Free-BSD (probably works under other unices too)
 *
 *     GENERIC      Uses only Standard C.  Should work on any ANSI C compiler.
 *                  Cannot do wildcard searches though.
 *
 */

#define UNIX  1

#if MSVC_WIN32
#pragma message("Compiling for Win-32")
#endif

#if MSC_DOS
#pragma message("Compiling for DOS and Microsoft C")
#endif

#if TURBO_C
#pragma message("Compiling for DOS and Borland/Turbo C/C++")
#endif

#if UNIX
#pragma message("Compiling for UNIX")
#endif

#if GENERIC
#pragma message("Compiling for Generic Standard C")
#endif

#if MSVC_WIN32+MSC_DOS+TURBO_C+UNIX+GENERIC != 1
#error Must choose one single environment!
#endif



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

typedef unsigned char  U8;
typedef signed   char  S8;
typedef unsigned short U16;
typedef unsigned long  U32;

#ifdef uint
#undef uint
#endif
#define uint  unsigned int


/* Commandline parameter definitions, with default values */

struct params
{
    int dumpfile;
} params = { 0 };


/* Utility functions */

char *gnb( char *s )
{
    while( *s && !isspace(*s) )
        s++;
    return s;
}  /* gnb */

char *gnnb( char *s )
{
    while( isspace(*s) )
        s++;
    return s;
}  /* gnb */

char *trimline( char *s )
{
    if ( s )
    {
        char *t = gnnb(s);
        char *u;
        for( u=gnb(t); *gnnb(u)!='\0'; u=gnb(gnnb(u)) )
            ;
        *u='\0';
        strcpy( s, t );
    }
    return s;
}  /* trimline */

char *getline( char *prompt )
{
#define BUF_SIZ  64
#define N_BUF    4
    static char buffer[BUF_SIZ*N_BUF];
    static char *bf = buffer;
    char *b = bf;
    if ( prompt && *prompt )
        printf( "%s: ", prompt ), fflush(stdout);
    fgets( b, BUF_SIZ, stdin );
    bf += BUF_SIZ;
    if ( bf-buffer >= BUF_SIZ*N_BUF )
        bf = buffer;
    return b;
#undef N_BUF
#undef BUF_SIZ
}  /* getline */

int yes( char *prompt )
{
    char *ans = trimline(getline(prompt));
    return  toupper(*ans) == 'Y' ;
}  /* yes */

int choice( char *prompt, char *choices )
{
    char *s, cho;
    if ( choices == NULL  ||  *choices == '\0' )
        return -1;
    for(;;)
    {
        if ( prompt && *prompt )
            printf( "%s: ", prompt );
        printf( "(%c", *(s=choices) );
        while( *++s )
            printf( "/%c", *s );
        printf( ") " );
        fflush(stdout);
        cho = *trimline(getline("")), cho = (char) toupper(cho);
        if ( cho == '\0' )
            return -1;
        for( s=choices; *s; s++ )
        {
            if ( toupper(*s) == cho )
                return  (int) (s-choices);
        }
        printf( "Bad choice - retry!\n" );
    }
}  /* choice */

int pause( int count, FILE *f )
{
    static int n, n0;
    if ( f != stdout )
        return 0;       /* Ignore for file output */
    if ( count > 0 )
    {
        n = n0 = count;
        return 0;       /* Init counter */
    }
    if ( n0 == 0 )
        return 0;
    if ( n-- == 0 )
    {                   /* Pause after n0 lines */
        char *pr = "<Enter> to continue, Stop<Enter> to stop, All<Enter> to list all";
retry:  switch( *trimline(getline(pr)) )
        {
            case 'S': case 's':
                return -1;
            case 'A': case 'a':
                n0 = 0;
                break;
            case 0:
                break;
            default:
                goto retry;
        }
        n = n0;
    }
    return 0;
}  /* pause */

FILE *openfile( char *prompt, int *binary, char *outname )
{
    if ( outname )
        *outname = '\0';
    for(;;)
    {
        char *openmode = "w";
        FILE *f;
        char *fname = trimline(getline(prompt));
        *binary = 0;
        if ( *fname == '\0' )
            return  stdout;
        switch( choice( "Dump as ASCII or Binary?  A/B", "AB" ) )
        {
            case -1:
                printf( "Cancelled\n" );
                return  NULL;
            case 1:
                *binary = 1;
                openmode = "wb";
                break;
            case 0:
            default:
                *binary = 0;
                openmode = "w";
                break;
        }
        if ( (f = fopen( fname, "r" )) != NULL )
        {
            fclose(f);
            if ( ! yes("File exists - overwrite? (Y/N) ") )
                continue;
        }
        f = fopen( fname, openmode );
        if ( f == NULL )
            printf( "Cannot open %s\n", fname );
        if ( outname )
            strcpy( outname, fname );
        return f;
    }
}  /* openfile */

#define isStdout(f)  ( f == stdout )

long filesize( FILE *f )
{
    struct stat stat;
    if ( fstat( fileno(f), &stat ) < 0 )
        return -1;
    return stat.st_size;
}  /* filesize */

U16 get16( U8 *where )
{   /* Fetch 16 bits in little-endian format */
    return  (U16)( where[0] + where[1]*256 );
}  /* get16 */

void put16( U8 *where, U16 value )
{   /* Store 16 bits in little-endian format */
    where[0] = (U8) value;
    where[1] = (U8) (value >> 8);
}  /* put16 */


/* Apple II DOS 3.3 disk data structures */

#pragma pack(1)

struct TS
{
    U8 track, sector;
};

struct VTOC
{
      U8 dummy1[1];
    struct TS cat;         /* Track/sector of catalog */
    U8 DOSver;
      U8 dummy2[2];
    U8 diskVolNo;          /* Disk volume number: default $FE */
      U8 dummy3[0x20];
    U8 maxTSpairs;         /* Max TSpairs within one TSlist (122 for 256 byte sectors) */
      U8 dummy4[8];
    U8 lastTrackAlloc;     /* Last track were sectors were allocated */
    S8 allocDir;           /* Allocation direction: +1 or -1 */
      U8 dummy5[2];
    U8 tracksOnDisk;       /* Tracks per diskette (default 35) */
    U8 sectorsPerTrack;    /* Sectors per diskette (default 13 or 16) */
    U8 bytesPerSector[2];  /* Bytes per sector (default 256), little endian */
    U8 bitmapTrack[50][4]; /* Bitmap of free/allocated sectors */
    /*
     *  Sectors numbered 0..(maxsect-1)
     *  Tracks numbered 0..(maxtrack-1)
     *  Volume label: default=FE, match-any-volume=00
     *  32 bits/track where each bit represents one sector: 1=free, 0=used
     *  First bytes: 8 last sectors on track - highest sector maps to highest bit
     *  Second byte: 8 next-to-last sectors on tracn
     *  'th byte:    8 first sectors on track
     *  Ev. remaining bytes: unused, set to 0
     */
};

struct TSlist
{
      U8 dummy1[1];
    struct TS nextTS;  /* Track/sector of next TSlist (track=0 if no more) */
      U8 dummy2[2];
    U8 sectOffset[2];  /* Sector offset (U16, little endian) in file
                          of first sector descr by this list */
      U8 dummy3[5];
    struct TS ts[122];
};

struct DirEntry
{
    struct TS ts;       /* Track/sector of 1st TS list sector.
                         * Track is usually $11.
                         *  $FF => deleted file (orig track copied to
                         *                       last char of filename)
                         *  $00 => entry never used
                         */
    U8 fileType;        /* File type and flags:
                         * Bit 7: set if file locked, clear if file unlocked
                         * $00 - T  Text file
                         * $01 - I  Integer Basic file
                         * $02 - A  Applesoft Basic file
                         * $04 - B  Binary file
                         * $08 - S  S type file
                         * $10 - R  Relocatable object module file
                         * $20 - A  A type file
                         * $40 - B  B type file
                         */
    char filename[30];  /* File name */
    U8 fileLength[2];   /* File length in sectors (little endian) */
};

struct CatSect
{
      U8 dummy1[1];
    struct TS next;        /* Track/sector of next catalog sector */
      U8 dummy2[8];
    struct DirEntry de[7];
};

#pragma pack()

/* End of Apple II DOS 3.3 disk data structures */


struct A2disk
{
    FILE *f;
    int tracks, sectors;
    U32 size;          /* Size in sectors */
    U16 bytesPerSector;
    struct VTOC VTOC;
};

struct A_fblk
{
    struct A2disk *A2;
    struct CatSect cat;
    struct DirEntry currde;
    struct TS ts;
    int n;
    char afn[40];
};

#define VTOC_TRACK   0x11
#define VTOC_SECT    0
#define BYTES_PER_SECTOR  0x100

struct TS VTOCts = { VTOC_TRACK, VTOC_SECT };

int readSector( struct A2disk *A2, void *buffer, struct TS ts )
{
    long linsect = ts.track * (long) A2->sectors + ts.sector;
    if ( fseek( A2->f, linsect * BYTES_PER_SECTOR, SEEK_SET ) != 0 )
        return -2;  /* I/O error */
    if ( fread( buffer, A2->bytesPerSector, 1, A2->f ) != 1 )
        return -2;  /* I/O error */
    return 0;
}  /* readSector */

int tryGeometry( struct A2disk *A2, uint trySectors )
{
    A2->sectors = trySectors;
    if ( (A2->size%trySectors) != 0 )
        goto invalidGeometry;
    if ( A2->size < trySectors*VTOC_TRACK )
        goto invalidGeometry;
    if ( readSector( A2, &(A2->VTOC), VTOCts ) < 0 )
        goto IO_error;
    if ( A2->VTOC.DOSver > 3 )
        goto invalidGeometry;
    if ( A2->VTOC.allocDir != +1  &&  A2->VTOC.allocDir != -1 )
        goto invalidGeometry;
    if ( (uint) A2->VTOC.sectorsPerTrack != trySectors )
        goto invalidGeometry;
    if ( (U32) A2->VTOC.tracksOnDisk != A2->size/trySectors )
        goto invalidGeometry;
    if ( get16(A2->VTOC.bytesPerSector) != A2->bytesPerSector )
        goto invalidGeometry;
    A2->tracks = (int) (A2->size / A2->sectors);
    return 0;
invalidGeometry:
    A2->sectors = 0;
    return -1;
IO_error:
    A2->sectors = 0;
    return -2;
}  /* tryGeometry */

int findDiskgeometry( struct A2disk *A2 )
{
    long fsize = filesize(A2->f);
    int sectry[] =
    {
        16,32,13,
        3,4,5,6,7,8,9,10,11,12,14,15,
        17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,-1
    };
    int n;

    A2->bytesPerSector = BYTES_PER_SECTOR;
    if ( fsize < (long) A2->bytesPerSector  ||
         (fsize%A2->bytesPerSector) != 0L )
        goto not_an_Apple2_disk_image;
    A2->size = fsize/A2->bytesPerSector;
    for( n=0; sectry[n]!=-1; n++ )
    {
        switch( tryGeometry( A2, sectry[n]) )
        {
            case 0:
                return 0;
            case -2:
                return -2;   /* I/O error */
            default: break;
        }
    }
not_an_Apple2_disk_image:
    return -1;
}  /* findDiskGeometry */

char TYP( U8 ft )
{
    switch( ft & 0x7F )
    {
        case 0x00:  return 'T';
        case 0x01:  return 'I';
        case 0x02:  return 'A';
        case 0x04:  return 'B';
        case 0x08:  return 'S';
        case 0x10:  return 'R';
        case 0x20:  return 'a';
        case 0x40:  return 'b';
        default:    return '-';
    }
}  /* TYP */

/*
 *  Apple II screen codes
 *
 *  $00-$1F   ASCII $40-$5F, inverse
 *  $20-$3F   ASCII $20-$3F, inverse
 *  $40-$5F   ASCII $40-$5F, blinking
 *  $60-$7F   ASCII $20-$3F, blinking
 *  $80-$9F   ASCII $40-$5F, normal
 *  $A0-$DF   ASCII $20-$5F, normal
 *  $E0-$FF   ASCII $20.$3F, normal ($60-$7F on IIe/IIc)
 */

/* Convert filename in Apple II charcodes to ASCII */
char *NAME( char *fn, int maxlen )
{
    static char namebuf[64];
    int n;
    if ( maxlen <= 0 )
        return NULL;
    if ( maxlen > sizeof(namebuf)-1 )
        maxlen = sizeof(namebuf)-1;
    memset( namebuf, 0, sizeof(namebuf) );
    memcpy( namebuf, fn, maxlen );
    for( n=0; n<maxlen; n++ )
    {
        if ( namebuf[n] & 0x80 )
            namebuf[n] &= 0x7F;
        else
        {
            namebuf[n] &= 0x3F;
            namebuf[n] ^= 0x20;
            namebuf[n] += 0x20;
        }
    }
    return trimline(namebuf);
}  /* NAME */

int catalog( struct A2disk *A2 )
{
    struct TS ts = A2->VTOC.cat;
    struct CatSect cat;
    for(;;)
    {
        int n;
        if ( ts.track == 0 )
            goto exit;
        if ( readSector( A2, &cat, ts ) < 0 )
            goto IO_error;
        for( n=0; n<7; n++ )
        {
            struct DirEntry *de = &(cat.de[n]);
            U8 ft = TYP(de->fileType);
            char locked = (char)((de->fileType & 0x80) ? '*' : ' ');
            char *name = NAME(de->filename,sizeof(de->filename));
            if ( de->ts.track == 0x00 )
                continue;  /* End of directory */
            if ( de->ts.track == 0xFF )
                continue;  /* File deleted */
            printf( "%c%c %03d %s\n", locked, ft,
                    get16(de->fileLength)&0xFF, name );
        }
        ts = cat.next;
    }
exit:
    return 0;
IO_error:
    return -1;
}  /* catalog */

#if UNIX
int stricmp( char *a, char *b )
{
    for( ; *a && *b; a++,b++ )
    {
        int ca = toupper(*a);
        int cb = toupper(*b);
        if ( ca != cb )
            return ca - cb;
    }
    return *a - *b;
}
#endif

/******************* Wildcard match stuff *******************/

int nowc( char *s )
{
    for( ; *s; s++ )
    {
        if ( *s == '*'  ||  *s == '?' )
            return 0;
    }
    return 1;
}  /* nowc */

char *nexts( char *s )
{
    if ( s && *s )
        return s+1;
    else
        return s;
}  /* nexts */

int cmatch( char *a, char *b )
{
    return  tolower(*a) == tolower(*b);
}  /* cmatch */

#define WILDS  '?'
#define WILDM  '*'

int fname_match( char *ufn, char *afn )
{
    for(;;)
    {
        if ( nowc(afn) && stricmp(afn,ufn)==0 )
            return 1;   /* Remaining strings match */
        if ( *afn == WILDS )
        {
            ufn = nexts(ufn), afn++;
            continue;
        }
        if ( *afn == WILDM )
        {
            char *u = ufn;
            if ( *++afn == '\0' )
                return 1;   /* "*" matches anything */
            for( ; *afn==WILDM || *afn==WILDS; afn++ )
            {
                if ( *afn == WILDS )
                    u++;
            }
            if ( *afn == '\0' )
                return 1;  /* Match */
            for( ; *u; u++ )
            {
                if ( cmatch(u,afn) )
                    if ( fname_match(u,afn) )
                        return 1;   /* Match */
            }
            return 0;   /* Mismatch */
        }
        if ( ! cmatch(afn++,ufn++) )
            return 0;  /* Mismatch */
    }
}  /* fname_match */

/************************************************************/

int A_findnext( struct A_fblk *A_fblk )
{
    for(;;)
    {
        if ( A_fblk->ts.track == 0 )
            goto exit;
        if ( readSector( A_fblk->A2, &(A_fblk->cat), A_fblk->ts ) < 0 )
            goto IO_error;
        for( ; A_fblk->n<7; A_fblk->n++ )
        {
            struct DirEntry *de = &(A_fblk->cat.de[A_fblk->n]);
        /* U8 ft = TYP(de->fileType); */
        /* char locked = (de->fileType & 0x80) ? '*' : ' '; */
            char *name = NAME(de->filename,sizeof(de->filename));
            if ( de->ts.track == 0x00 )
                continue;  /* End of directory */
            if ( de->ts.track == 0xFF )
                continue;  /* File deleted */
            if ( fname_match( name, A_fblk->afn ) )
            {
                A_fblk->currde = *de;
                A_fblk->n++;
                return 1;   /* Found ! */
            }
        }
        A_fblk->n = 0;
        A_fblk->ts = A_fblk->cat.next;
    }
exit:
    A_fblk->n++;
    return 0;
IO_error:
    A_fblk->n++;
    return -1;
}  /* A_findnext */

int A_findfirst( struct A2disk *A2, char *afn, struct A_fblk *A_fblk )
{
    A_fblk->A2 = A2;
    A_fblk->n = 0;
    A_fblk->ts = A2->VTOC.cat;
    strcpy( A_fblk->afn, afn );
    return  A_findnext( A_fblk );
}  /* A_findfirst */

int dumpBufferRaw( U8 *data, uint len, FILE *f )
{
    return  fwrite( data, len, 1, f ) == 1 ? 0 : -1;
}  /* dumpBufferRaw */

int dumpBufferAsBinFile( U8 *data, char *fname, uint len, FILE *f )
{
    uint start, length, n;
    start  = *data++, start  += 256 * *data++;
    length = *data++, length += 256 * *data++;
    len -= 4;
    pause(22,f);
    for( n=0; n<length; n++, data++ )
    {
        if ( (n%16) == 0 )
        {
            fprintf( f, "\n" );
            if ( pause(0,f) < 0 )
                goto exit;
            fprintf( f, "%04X:", start+n );
        }
        fprintf( f, " %02X", *data );
    }
    fprintf( f, "\n" );
    if ( f != stdout )
        fprintf( f, "\nBSAVE %s,A$%04X,L$%04X\n", fname, start, length );
exit:
    return 0;
}  /* dumpBufferAsBinFile */

int dumpBufferAsApplesoftFile( U8 *data, char *fname, uint len, FILE *f )
/*
 *   Applesoft file format:
 *
 *   <Length_of_file> (16-bit little endian)
 *   <Line>
 *   ......
 *   <Line>
 *
 *   where <Line> is:
 *   <Next addr>  (16-bit little endian)
 *   <Line no>    (16-bit little endian: 0-65535)
 *   <Tokens and/or characters>
 *   <End-of-line marker: $00 $00 bytes>
 *
 */
{
    static char *atoken[128] =
    {
        "END","FOR","NEXT","DATA","INPUT","DEL","DIM","READ",
        "GR","TEXT","PR#","IN#","CALL","PLOT","HLIN","VLIN",
        "HGR2","HGR","HCOLOR=","HPLOT","DRAW","XDRAW","HTAB",
        "HOME","ROT=","SCALE=","SHLOAD","TRACE","NOTRACE",
        "NORMAL","INVERSE","FLASH","COLOR=","POP","VTAB",
        "HIMEM=","LOMEM=","ONERR","RESUME","RECALL","STORE",
        "SPEED=","LET","GOTO","RUN","IF","RESTORE","&","GOSUB",
        "RETURN","REM","STOP","ON","WAIT","LOAD","SAVE","DEF",
        "POKE","PRINT","CONT","LIST","CLEAR","GET","NEW",
        "TAB(","TO","FN","SPC(","THEN","AT","NOT","STEP","+",
        "-","*","/","^","AND","OR",">","=","<","SGN","INT",
        "ABS","USR","FRE","SCRN(","PDL","POS","SQR","RND",
        "LOG","EXP","COS","SIN","TAN","ATN","PEEK","LEN",
        "STR$","VAL","ASC","CHR$", "LEFT$","RIGHT$","MID$","",
        "","","","","","","","","","","","","","","","","","","",""
    };
    U8 *data0 = data;
    int alen = get16(data);
    pause(22,f);
    for( data+=2; get16(data) && (data-data0 < alen); )
    {
        unsigned int lineno;
        data += 2;
        lineno = get16(data), data += 2;
        fprintf( f, "%u ", lineno );
        for( ; *data; data++ )
        {
            if ( *data & 0x80 )
                fprintf( f, " %s ", atoken[*data-0x80] );
            else
                fprintf( f, "%c", *data );
        }
        fprintf( f, "\n" ), data++;
        if ( pause(0,f) < 0 )
            goto exit;
    }
    len = len, data = data;
    if ( f != stdout )
        fprintf( f, "\nSAVE %s\n", fname );
exit:
    return 0;
}  /* dumpBufferAsApplesoftFile */


int canBeIntBasicFile( U8 *data, uint len )
{
    U8 *data0 = data;
    int alen = get16(data);
    for( data+=2; *data && (data-data0 <= alen); )
    {
        /*unsigned int lineno;*/
        unsigned int linelen = *data++;
        /*lineno = get16(data),*/ data += 2;
        /* Integer Basic lines always ends with a 0x01 */
        if ( data[linelen-4] != 0x01 )
            return 0;
        /* Integer Basic lines never start with control chars */
        if ( *data >= 0x81 && *data <= 0x98 )
            return 0;
        data += linelen-3;
    }
    len = len;
    return 1;
}  /* canBeIntBasicFile */



int dumpBufferAsIntBasicFile( U8 *data, char *fname, uint len, FILE *f )
/*
 *   Integer Basic file format:
 *
 *   <Length_of_file> (16-bit little endian)
 *   <Line>
 *   ......
 *   <Line>
 *
 *   where <Line> is:
 *      1 byte:   Line length
 *      2 bytes:  Line number, binary little endian
 *      <token>
 *      <token>
 *      <token>
 *      .......
 *      <end-of-line token>
 *
 *   <token> is one of:
 *      $12 - $7F:   Tokens as listed below: 1 byte/token
 *      $80 - $FF:   ASCII characters with high bit set
 *      $B0 - $B9:   Integer constant, 3 bytes:  $B0-$B9,
 *                     followed by the integer value in
 *                     2-byte binary little-endian format
 *                     (Note: a $B0-$B9 byte preceded by an
 *                      alphanumeric ASCII(hi_bit_set) byte
 *                      is not the start of an integer
 *                      constant, but instead part of a
 *                      variable name)
 *
 *   <end-of-line token> is:
 *      $01:         One byte having the value $01
 *                   (Note: a $01 byte may also appear
 *                    inside an integer constant)
 *
 *  Note that the tokens $02 to $11 represent commands which
 *  can be executed as direct commands only -- any attempt to
 *  enter then into an Integer Basic program will be rejected
 *  as a syntax error.  Therefore, no Integer Basic program
 *  which was entered through the Integer Basic interpreter
 *  will contain any of the tokens $02 to $11.  The token $00
 *  appears to be unused and won't appear in Integer Basic
 *  programs either.  However, $00 is used as an end-of-line
 *  marker in S-C Assembler source files, which also are of
 *  DOS file type "I".
 *
 *  (note here a difference from Applesoft Basic, where there
 *  are no "direct mode only" commands - any Applesoft commands
 *  can be entered into an Applesoft program as well).
 *
 */
{
#define REM_TOKEN   0x5D
#define UNARY_PLUS  0x35
#define UNARY_MINUS 0x36
#define QUOTE_START 0x28
#define QUOTE_END   0x29
    static char *itoken[128] =
    {
        /* $00-$0F */
        "HIMEM:","<$01>", "_",     " : ",
        "LOAD",  "SAVE",  "CON",   "RUN",   /* Direct commands */
        "RUN",   "DEL",   ",",     "NEW",
        "CLR",   "AUTO",  ",",     "MAN",

        /* $10-$1F */
        "HIMEM:","LOMEM:","+",     "-",     /* Binary ops */
        "*",     "/",     "=",     "#",
        ">=",    ">",     "<=",    "<>",
        "<",     "AND",   "OR",    "MOD",

        /* $20-$2F */
        "^",     "+",     "(",     ",",
        "THEN",  "THEN",  ",",     ",",
        "\"",    "\"",    "(",     "!",
        "!",     "(",     "PEEK",  "RND",

        /* $30-$3F */
        "SGN",   "ABS",   "PDL",   "RNDX",
        "(",     "+",     "-",     "NOT",   /* Unary ops */
        "(",     "=",     "#",     "LEN(",
        "ASC(",  "SCRN(", ",",     "(",

        /* $40-$4F */
        "$",     "$",     "(",     ",",
        ",",     ";",     ";",     ";",
        ",",     ",",     ",",     "TEXT",  /* Statements */
        "GR",    "CALL",  "DIM",   "DIM",

        /* $50-$5F */
        "TAB",   "END",   "INPUT", "INPUT",
        "INPUT", "FOR",   "=",     "TO",
        "STEP",  "NEXT",  ",",     "RETURN",
        "GOSUB", "REM",   "LET",   "GOTO",

        /* $60-$6F */
        "IF",    "PRINT", "PRINT", "PRINT",
        "POKE",  ",",     "COLOR=","PLOT",
        ",",     "HLIN",  ",",     "AT",
        "VLIN",  ",",     "AT",    "VTAB",

        /* $70-$7F */
        "=",     "=",     ")",     ")",
        "LIST",  ",",     "LIST",  "POP",
        "NODSP", "DSP",  "NOTRACE","DSP",
        "DSP",   "TRACE", "PR#",   "IN#",
    };

    U8 *data0 = data;
    int alen = get16(data);
    pause(22,f);
    for( data+=2; *data && (data-data0 <= alen); )
    {
        int inREM = 0, inQUOTE = 0;
        int lastAN = 0, leadSP = 0, lastTOK = 0;
        unsigned int lineno;
        unsigned int linelen = *data++;
        lineno = get16(data), data += 2;
        linelen = linelen;
        fprintf( f, "%u ", lineno );
        for( ; *data!=0x01; data++ )
        {
            leadSP = leadSP || lastAN;
            if ( *data & 0x80 )
            {
                if ( !inREM && !inQUOTE && !lastAN && (*data >= 0xB0 && *data <= 0xB9) )
                {
                    signed short integer = get16(data+1);
                    int leadspace = lastTOK && leadSP;
                    fprintf( f, leadspace ? " %d" : "%d", (int) integer );
                    data += 2;
                    leadSP = 1;
                }
                else
                {
                    char c = (char) (*data & 0x7F);
                    int leadspace = !inREM && !inQUOTE &&
                                    lastTOK && leadSP && isalnum(c);
                    if ( leadspace )
                        fprintf( f, " " );
                    if ( c >= 0x20 )
                        fprintf( f, "%c", c );
                    else
                        fprintf( f, "^%c", c+0x40 );
                    lastAN = isalnum(c);
                }
                lastTOK = 0;
            }
            else
            {
                char *tok = itoken[*data];
                char lastchar = tok[strlen(tok)-1];
                int leadspace = leadSP &&
                          ( isalnum(tok[0]) ||
                            *data == UNARY_PLUS ||
                            *data == UNARY_MINUS ||
                            *data == QUOTE_START  );
                switch( *data )
                {
                    case REM_TOKEN:   inREM = 1;    break;
                    case QUOTE_START: inQUOTE = 1;  break;
                    case QUOTE_END:   inQUOTE = 0;  break;
                    default:  break;
                }
                fprintf( f, leadspace ? " %s" : "%s", tok );
                lastAN  = 0;
                leadSP = isalnum(lastchar) || lastchar == ')' || lastchar == '\"';
                lastTOK = 1;
            }
        }
        fprintf( f, "\n" ), data++;
        if ( pause(0,f) < 0 )
            goto exit;
    }
    len = len;
    if ( f != stdout )
        fprintf( f, "\nSAVE %s\n", fname );
exit:
    return 0;
}  /* dumpBufferAsIntBasicFile */


int canBeSCasmFile( U8 *data, uint len )
{
    U8 *data0 = data;
    int alen = get16(data);
    for( data+=2; *data && (data-data0 <= alen); )
    {
        U8 *p;
        /*unsigned int lineno;*/
        unsigned int linelen = *data++;
        /*lineno = get16(data),*/ data += 2;
        /* S-C Assembler lines always ends with a 0x00 */
        if ( data[linelen-4] != 0x00 )
            return 0;
        /* S-C Assembler lines never contains bytes above 0xC0 */
        for( p=data+linelen-4; p>=data; p-- )
        {
            if ( *p > 0xC0 )
                return 0;
        }
        data += linelen-3;
    }
    len = len;
    return 1;
}  /* canBeSCasmFile */



int dumpBufferAsSCasmFile( U8 *data, char *fname, uint len, FILE *f )
/*
 *   S-C Assembler file format:
 *
 *   <Length_of_file> (16-bit little endian)
 *   <Line>
 *   ......
 *   <Line>
 *
 *   where <Line> is:
 *   <Length>     (8-bit: including length byte, line no, contents, zero>
 *   <Line no>    (16-bit little endian: 0-65535)
 *   <Characters>
 *   <Zero byte>  ($00, to mark the end of the S-C Asm source line)
 *
 *   where <Characters> are an arbitrary sequence of:
 *   <Literal character>:      $20 to $7E - literal characters
 *   <Compressed spaces>:      $80 to $BF - represents 1 to 63 spaces
 *   <Compressed repetition>:  $C0 <n> <ch> - represents <ch> n times
 *
 */
{
    U8 *data0 = data;
    int alen = get16(data);
    pause(22,f);
    for( data+=2; *data && (data-data0 <= alen); )
    {
        unsigned int ilen, lineno;
        ilen = *data++;
        ilen = ilen;
        lineno = get16(data), data += 2;
        fprintf( f, "%04u ", lineno );
        for( ; *data; data++ )
        {
            if ( *data == 0xC0 )
            {
                int rep = *++data;
                int chr = *++data;
                while( rep-- > 0 )
                    fprintf( f, "%c", chr );
            }
            else if ( *data & 0x80 )
            {
                int n = *data - 0x80;
                while( n-- > 0 )
                    fprintf( f, " " );
            }
            else
                fprintf( f, "%c", *data );
        }
        fprintf( f, "\n" ), data++;
        if ( pause(0,f) < 0 )
            goto exit;
    }
    len = len;
    if ( f != stdout )
        fprintf( f, "\nSAVE %s\n", fname );
exit:
    return 0;
}  /* dumpBufferAsSCasmFile */


int dumpBufferAsTextFile( U8 *data, char *fname, uint len, FILE *f )
{
    uint n;
    int last = -1;
    pause(22,f);
    for( n=0; n<len; n++ )
    {
        switch( data[n] )
        {
            case 0:
                if ( last != 0 )
                {
                    fprintf( f, "----------------------------------\n" );
                    if ( pause(0,f) < 0 )
                        goto exit;
                }
                break;
            case 0x8D:
                fprintf( f, "\n" );
                if ( pause(0,f) < 0 )
                    goto exit;
                break;
            default:
                fprintf( f, "%c", data[n] & 0x7F );
                break;
        }
        last = data[n];
    }
    fprintf( f, "\nTextfile %s\n", fname );
exit:
    return 0;
}  /* dumpBufferAsTextFile */


int dumpBuffer( U8 *data, char *fname, char type, uint len )
{
    int rc = 0, bin = 0;
    char outfname[64];
    FILE *f = openfile( "Dump file - output file name (<Enter>=screen)",
                        &bin, outfname );
    if ( f == NULL )
        return -1;

    if ( bin )
        type = 0;

    switch( type )
    {
        case 0:
            printf( "Dumping %s in raw form (%u bytes) to %s\n",
                    fname, len, outfname );
            rc = dumpBufferRaw( data, len, f );
            break;
        case 'B':
            printf( "Dumping %s as monitor hex dump to %s\n",
                    fname, outfname );
            rc = dumpBufferAsBinFile( data, fname, len, f );
            break;
        case 'T':
            printf( "Dumping %s as text file to %s\n",
                    fname, outfname );
            rc = dumpBufferAsTextFile( data, fname, len, f );
            break;
        case 'A':
            printf( "Dumping %s as Applesoft source to %s\n",
                    fname, outfname );
            rc = dumpBufferAsApplesoftFile( data, fname, len, f );
            break;
        case 'I':
            if ( canBeSCasmFile( data, len ) )
            {
                printf( "Dumping %s as S-C Assembler source to %s\n",
                        fname, outfname );
                rc = dumpBufferAsSCasmFile( data, fname, len, f );
            }
            else if ( canBeIntBasicFile( data, len ) )
            {
                printf( "Dumping %s as Integer Basic source to %s\n",
                        fname, outfname );
                rc = dumpBufferAsIntBasicFile( data, fname, len, f );
            }
            else switch( choice( "Cannot determine file type - dump as IntBasic or SC-Asm file?", "IS" ) )
            {
                case -1:
                    printf( "Dump cancelled\n" );
                    break;
                case 0:
                    printf( "Dumping %s as Integer Basic source to %s\n",
                            fname, outfname );
                    rc = dumpBufferAsIntBasicFile( data, fname, len, f );
                case 1:
                    printf( "Dumping %s as S-C Assembler source to %s\n",
                            fname, outfname );
                    rc = dumpBufferAsSCasmFile( data, fname, len, f );
            }
            break;
        default:
            printf( "\nDump of file %s (type %c) not implemented\n",
                    fname, type );
            printf( "First four bytes: %02X %02X %02X %02X\n",
                    data[0], data[1], data[2], data[3] );
            return -1;
    }
    if ( fclose(f) == EOF )
        return -1;
    else
        return rc;
}  /* dumpBuffer */

int TSzero( struct TS ts )
{
    return  ts.track == 0  &&  ts.sector == 0;
}  /* TSzero */

int TSok( struct TS ts, struct A2disk *A2 )
{
    return  ts.track  < A2->VTOC.tracksOnDisk    &&
            ts.sector < A2->VTOC.sectorsPerTrack;
}  /* TSok */

struct TS *getTSlist( struct A2disk *A2, struct TS TSlistpos, uint *TS_sectors )
{
    static struct TSlist TSlist;
    int listsize = A2->VTOC.tracksOnDisk * A2->VTOC.sectorsPerTrack * sizeof(struct TS);
    int partlistsize = A2->VTOC.maxTSpairs * sizeof(struct TS);
    struct TS *list = malloc(listsize);
    if ( list == NULL )
        return NULL;   /* Out of memory */
    memset( list, 0, listsize );
    *TS_sectors = 0;
    if ( TSzero(TSlistpos) )
        return list;     /* No TS-list, empty file */
    /* Read first TS-list */
    if ( readSector( A2, &(TSlist), TSlistpos ) < 0 )
        goto error;
    /* Count it, copy disk allocation info */
    memcpy( list+(*TS_sectors)*A2->VTOC.maxTSpairs, TSlist.ts, partlistsize );
    (*TS_sectors)++;
    /* Check for additional TS-lists */
    while( ! TSzero(TSlist.nextTS) )
    {
        if ( !TSok(TSlist.nextTS, A2) )
            goto error;   /* Bad link - exit with error */
        if ( readSector( A2, &(TSlist), TSlist.nextTS ) < 0 )
            goto error;
        /* Count additional TS-list, copy allocation info */
        memcpy( list+(*TS_sectors)*A2->VTOC.maxTSpairs, TSlist.ts, partlistsize );
        (*TS_sectors)++;
    }
    /* Done! Return disk allocation vector */
    return list;
error:
    free(list);
    return NULL;
}  /* getTSlist */

int TSlastItem( struct TS *list, struct A2disk *A2 )
{
    int i, n = -1;
    int listitems = A2->VTOC.tracksOnDisk * A2->VTOC.sectorsPerTrack;
    for( i=0; i<listitems; i++ )
    {
        if ( ! TSzero(list[i]) )
             n = i;
    }
    return n;
}  /* TSlastItem */

int TSitems( struct TS *list, struct A2disk *A2 )
{
    int i, n=0;
    int listitems = A2->VTOC.tracksOnDisk * A2->VTOC.sectorsPerTrack;
    for( i=0; i<listitems; i++ )
    {
        if ( ! TSzero(list[i]) )
             n++;
    }
    return n;
}  /* TSlastItem */

int dumpfile( struct A2disk *A2, char *fname )
{
    static struct A_fblk A_fblk;
    static struct TS *list = NULL;
    uint TS_sectors, sectors_in_dir, sectors_on_disk, sectors_seq;
    uint n, filebufsize;
    U8 *data = NULL;
    char *fname_found;
    int rc = A_findfirst( A2, fname, &A_fblk );
    if ( rc != 1 )
    {
        printf( "%s not found\n", fname );
        return 0;
    }
/* ======================================================= */
    do
    {
        fname_found = NAME(A_fblk.currde.filename,sizeof(A_fblk.currde.filename));
        sectors_in_dir = get16(A_fblk.currde.fileLength);
        list = getTSlist( A2, A_fblk.currde.ts, &TS_sectors );
        if ( list == NULL )
            goto IO_error;
        sectors_on_disk = TSitems( list, A2 );
        sectors_seq  = TSlastItem( list, A2 ) + 1;
        if ( sectors_in_dir != sectors_on_disk + TS_sectors )
            printf( "Size error: directory says file is %d sectors, disk has %d sectors\n",
                    sectors_in_dir, sectors_on_disk + TS_sectors );
        if ( sizeof(uint)==2 && sectors_seq > 255 )
        {
            printf( "%d sectors too much to dump in a 16-bit executable\n" );
            return -1;
        }
        filebufsize = sectors_seq*BYTES_PER_SECTOR;
        data = (U8 *) malloc( filebufsize );
        if ( data == NULL )
            goto oom;
        memset( data, 0, filebufsize );
        for( n=0; n<sectors_seq; n++ )
        {
            if ( TSzero(list[n]) )
                continue;
            if ( readSector( A2, data+n*BYTES_PER_SECTOR, list[n] ) < 0 )
                goto IO_error;
        }
        dumpBuffer( data, fname_found, TYP(A_fblk.currde.fileType), filebufsize );
        free(list);
        free(data);
    }  while( A_findnext( &A_fblk ) == 1 );
    return 1;
IO_error:
    if ( list )
        free( list );
    if ( data )
        free( data );
    return -1;
oom:
    return -1;
}  /* dumpfile */

int fid( char *fname )
{
    static struct A2disk A2;

    if ( (A2.f = fopen( fname, "rb" )) == NULL )
    {
        printf( "Cannot open Apple II disk image file %s\n", fname );
        return -1;
    }
    printf( "\n%s:\n", fname );

    switch( findDiskgeometry( &A2 ) )
    {
        case -1:  goto not_an_Apple2_disk_image;
        case -2:  goto IO_error;
        default:  break;
    }
    catalog(&A2);
    if ( params.dumpfile )  for(;;)
    {
        char *fn = trimline(getline("Apple ][ file to dump"));
        if ( *fn == '\0' )
            break;
        dumpfile(&A2,fn);
        printf( "=======================================\n" );
        printf( "\n%s:\n", fname );
        catalog(&A2);
    }
    fclose(A2.f);
    return 0;

not_an_Apple2_disk_image:
    fclose(A2.f);
    printf( "Not an Apple ][ disk image file\n" );
    return -1;

IO_error:
    fclose(A2.f);
    printf( "I/O error\n" );
    return -1;
}  /* fid */

char *lastslash( char *path )
{
    char *p;
    if ( path == NULL  ||  *path == '\0' )
        return NULL;
    for( p=path+strlen(path)-1; p>=path; p-- )
    {
        if ( *p == '\\' || *p == '/' )
            return p;
        if ( p == path )
            return NULL;
    }
    return NULL;
}  /* lastslash */

char *fnam( char *path )
{
    char *ls;
    if ( path[0] && path[1] == ':' )
        path += 2;
    ls = lastslash(path);
    return  ls ? ls+1 : path;
}  /* fnam */

char *inpath( char *path, char *fname )
{
    static char newpath[80];
    if ( path == NULL || fname == NULL )
        return NULL;
    strcpy( newpath, path );
    strcpy( fnam(newpath), fname );
    return newpath;
}  /* inpath */

/**** findfirst/findnext struff ****/
#if TURBO_C
#include <dir.h>

struct ffblk FindData;
char *amb_fn = NULL;

char *F_findfirst( char *afn )
{
    amb_fn = afn;
    if ( findfirst(afn,&FindData,0) == 0 )
        return inpath( amb_fn, FindData.ff_name );
    else
    {
        amb_fn = NULL;
        return NULL;
    }
}  /* F_findfirst */

char *F_findnext(void)
{
    if ( findnext(&FindData) == 0 )
        return inpath( amb_fn, FindData.ff_name );
    else
    {
        amb_fn = NULL;
        return NULL;
    }
}  /* F_findnext */


#elif MSVC_WIN32
#include <io.h>

struct _finddata_t FindData;
long hFile;
char *amb_fn = NULL;

void findclose_if_needed(void)
{
    if ( amb_fn )
    {
        _findclose(hFile);
        hFile = -1L, amb_fn = NULL;
    }
}  /* findclose_if_needed */

void findclose_at_exit(void)
{
    static int initialized = 0;
    if ( ! initialized )
    {
        atexit(findclose_if_needed);
        initialized = 1;
    }
}  /* findclose_at_exit */

char *F_findfirst( char *afn )
{
    findclose_at_exit();
    findclose_if_needed();
    amb_fn = afn;
    if ( (hFile = _findfirst(afn,&FindData)) != -1L )
        return inpath( amb_fn, FindData.name );
    else
    {
        findclose_if_needed();
        return NULL;
    }
}  /* F_findfirst */

char *F_findnext(void)
{
    if ( amb_fn == NULL )
        return NULL;
    else if ( _findnext(hFile,&FindData) == 0 )
        return inpath( amb_fn, FindData.name );
    else
    {
        findclose_if_needed();
        return NULL;
    }
}  /* F_findnext */


#elif MSC_DOS

#include <dos.h>

struct find_t FindData;
char *amb_fn = NULL;

char *F_findfirst( char *afn )
{
    amb_fn = afn;
    if ( _dos_findfirst(afn,0,&FindData) == 0 )
        return inpath( amb_fn, FindData.name );
    else
    {
        amb_fn = NULL;
        return NULL;
    }
}  /* F_findfirst */

char *F_findnext(void)
{
    if ( _dos_findnext(&FindData) == 0 )
        return inpath( amb_fn, FindData.name );
    else
    {
        amb_fn = NULL;
        return NULL;
    }
}  /* F_findnext */

#elif UNIX
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>

int isfile( char *path )
{
    struct stat statbuf;
    if ( stat( path, &statbuf ) != 0 )
        return 0;
    return  (statbuf.st_mode & S_IFREG) != 0;
}  /* isfile */

typedef struct dirent DIR_ENT;
struct U_fblk
{
    DIR *dp;
    DIR_ENT *de;
    char amb_fn[128];
} U_fblk = { NULL, NULL, {0} };

char *dn( char *afn )
{
    char *ls;
    static char dirname[80];
    strcpy( dirname, afn );
    ls = lastslash(dirname);
    if ( ls == NULL )
        return ".";
    *ls = '\0';
    return  dirname;
}  /* dn */

void findclose_if_needed(void)
{
    if (U_fblk.dp)
        closedir(U_fblk.dp);
    U_fblk.dp = NULL, U_fblk.de = NULL;
    memset( U_fblk.amb_fn, 0, sizeof(U_fblk.amb_fn) );
}  /* findclose_if_needed */

void findclose_at_exit(void)
{
    static int initialized = 0;
    if ( ! initialized )
    {
        atexit(findclose_if_needed);
        initialized = 1;
    }
}  /* findclose_at_exit */

char *F_findnext(void)
{
    if ( U_fblk.amb_fn[0] == '\0'  ||  U_fblk.dp == NULL )
        return NULL;
    while( (U_fblk.de = readdir(U_fblk.dp)) != NULL )
    {
        char *upath = inpath( U_fblk.amb_fn, U_fblk.de->d_name );
        if (isfile(upath) && fname_match(U_fblk.de->d_name,fnam(U_fblk.amb_fn)))
            return  upath;
    }
    return NULL;
}  /* F_findnext */

char *F_findfirst( char *afn )
{
    char *dirname = dn(afn);
    findclose_at_exit();
    findclose_if_needed();
    strcpy( U_fblk.amb_fn, afn );
    if ( (U_fblk.dp = opendir(dirname)) == NULL )
        return NULL;
    return F_findnext();
}  /* F_findfirst */

#elif defined(GENERIC)
/* No wildcard searches possible      */
/* Only checks if afn exists as a ufn */

char *F_findfirst( char *afn )
{
    FILE *f = fopen(afn,"r");
    if ( f == NULL )
        return NULL;
    fclose(f);
    return afn;
}  /* F_findfirst */

char *F_findnext(void)
{
    return NULL;
}  /* F_findnext */


#else
#error  Must define compiler for findfirst/findnext
#endif
/**************************************/

void usage(void)
{
    printf( "\n"
            "Usage:   fid  [-d]  <afn1> [<anf2> ... <afnn>]\n"
            "       -d:  dump files in disk images\n" ); 
    exit(0);
}  /* usage */

int main( int argc, char *argv[])
{
    int n = 0;
#if 0
    if ( sizeof(struct VTOC) != BYTES_PER_SECTOR )
        n = printf( "struct VTOC has bad size: %d, should be %d\n",
                    sizeof(struct VTOC), BYTES_PER_SECTOR );
    if ( sizeof(struct TSlist) != BYTES_PER_SECTOR )
        n = printf( "struct TSlist has bad size: %d, should be %d\n",
                    sizeof(struct TSlist), BYTES_PER_SECTOR );
    if ( sizeof(struct CatSect) != BYTES_PER_SECTOR )
        n = printf( "struct CatSect has bad size: %d, should be %d\n",
                    sizeof(struct CatSect), BYTES_PER_SECTOR );
    if ( n )
        return 1;
#endif

    if ( argc <= 1 )
        usage();
    for( n=1; n<argc; n++ )
    {
        char *fn, *arg;
        arg = argv[n];
        if ( arg[0] != '-' )
        {
            for( fn=F_findfirst(arg); fn; fn=F_findnext() )
                fid(fn);
        }
        else switch( arg[1] )
        {
            case 'd': case 'D':
                params.dumpfile = 1;
                break;
            default:
                usage();
        }
    }

    return 0;
}  /* main */
