/***********************************************************************
 *
 *	umslave.c	Slave command (receiving) mode
 *			Part of the UMDSS Project
 *			(C)opyright 1992 Morgan Davis Group
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>

#define	SCOPE	extern
#include "umdss.h"

	/* Forward references to external functions */
	
extern void cleanup();
extern void logentry();

/*
 *	slave()		Take orders from the master site
 */

int slave()
{
	int bogus, timeouts, pid, c;
	char tempfile1[L_tmpnam], tempfile2[L_tmpnam];

	get_job_info();		/* wait for Bytes: and Jobs: reports */

	bogus = timeouts = 0;

	for (;;) {
		/* Send ENQ to tell master it is ready */
		write_char(ENQ_PROMPT);
		c = get_timed_char(ENQ_CMD_WAIT);
		switch (c) {
		case TIMEOUT:
			if (++timeouts >= MAX_TIMEOUTS) {
				logentry("*", "<ERROR: NOT RESPONDING>");
				cleanup(ERROR);
			}
			break;

		case HANGUP_CMD:
			/* Master is commanding us to hangup */
			signal(SIGHUP, SIG_IGN);
			return(HANGUP_CMD);

		case SEND_CMD:
			/* Master is done sending to us and wants us */
			/* to switch roles (become master) to send to it. */
			return(SEND_CMD);

		case BINARY_CMD:
		case RECEIVE_CMD:
			/* We are commanded to receive a file */
#ifdef DEBUG
			logentry("?","<DEBUG: GOT R>");
#endif
			timeouts = bogus = 0;
			tmpnam(tempfile1);
#ifdef DEBUG
			sprintf(buf, "<DEBUG: TMPNAM = '%s'>", tempfile1);
			logentry("?", buf);
#endif
			if (xfer(rcvcmd, tempfile1)) {
				logentry("*", "<ALERT: RECEIVE FAILED>");
				break;
			}

			/* Munge it */
			if (process_mail(tempfile1, tempfile2))
				break; /* error processing file */

			/* Now mail the processed file through rmail */
			if (!(pid = fork())) {	 /* Child process */
				freopen("/dev/null", "a", stderr);
				freopen(tempfile2, "r", stdin);
				execlp("rmail", "rmail", towho, (char *)NULL);
				logentry("*", "<ERROR: RMAIL EXECLP() FAILED>");
				cleanup(errno);
			} else if (pid == -1) {
				logentry("*", "<ERROR: RMAIL FORK() FAILED>");
				cleanup(errno);
			}

			/* Now log the transfer */
			(void) stat(tempfile2, &stbuf);
			sprintf(buf, "%lu %s", (unsigned long)stbuf.st_size, towho);
			logentry("R", buf);

			jobsRcvd++;
			bytesRcvd += stbuf.st_size;
			logstatus("");

			/* Wait for rmail to finish */
			fwait(pid);
			unlink(tempfile2);
			break;

		case CANCEL_CMD:		/* eat CANs */
			break;

		default:
#ifdef DEBUG
			sprintf(buf, "<DEBUG: GOT $%02hx, WANTED H|S|R|B>", c);
			logentry("?", buf);
#endif
			if (++bogus >= MAX_BOGUS) {
				logentry("*", "<ERROR: TOO MUCH LINE NOISE>");
				cleanup(ERROR);
			}
			write_char(CANCEL_CMD);
			write_char(CANCEL_CMD);
			while(get_timed_char(2) != TIMEOUT);
		}
	}
}


/* process_mail():
 *
 *	1. Modify 'From ' line from the form
 *		 "From site!user (date)" to
 *		 "From user (date) remote from site"
 *	2. Get target address from 'Ppath: ' line and discard it.
 *	3. Copy body of message
 */

int process_mail(tempfile1, tempfile2)
	char *tempfile1, *tempfile2;
{
	register char c, *cp;
	register FILE *file1, *file2;
	int n;

	/* Open 2 temporary files for processing this mail */

	tmpnam(tempfile2);
	if ((file2 = fopen(tempfile2, "w")) == NULL ||
			(file1 = fopen(tempfile1, "r")) == NULL ) {
		logentry("*", "<ERROR: PROCESSING RECEIVED FILE>");
		unlink(tempfile1);
		unlink(tempfile2); 
		cleanup(ERROR);
	}

	*towho = '\0';

	while ((fgets(buf, BUFLEN, file1) != NULL) && (*buf != '\n')) {
		/* Process the 'From site!user (date)' line */
		if (!strncmp("From ", buf, 5)) {
			if (!sindex(buf, " remote from ")) {
				if ((cp = strchr(buf, '\n')))
					*cp = '\0';
				cp = strrchr(buf, '!');
				*cp++ = '\0';
				fprintf(file2, "From %s remote from %s\n", cp, buf+5);
			}
		} else if (!strncmp("Ppath: ", buf, 7)) {
			/* Scan 'Ppath: ' line to find  target address */
			cp = buf + 7;
			n = strlen(hostname);
			if (!strncmp(cp, hostname, n)) {
				cp += n;
				if (*cp == '!')
					cp++;
				else
					cp = buf + 7;
			}
			strcpy(towho, cp);
			if ((cp = strchr(towho, '\n')))
				*cp = '\0';
		} else
			fputs(buf, file2);
	}
	fputc('\n', file2);

	while((n = fread(buf, sizeof(char), BUFLEN, file1))) {
		while (!buf[n-1] && n > 0)		/* strip nulls */
			n--;
		fwrite(buf, sizeof(char), n, file2);	/* Message body */
	}

	fclose(file1);
	unlink(tempfile1);		 /* Finished with 1st file */

	fclose(file2);
	if (!*towho) {
		logentry("*", "<ERROR: RECEIVED W/O PPATH>");
		unlink(tempfile2);
		return(ERROR);
	}
	return(NO_ERROR);
}

/*
 *	get_job_info()
 *	Accepts job info from remote site and just
 *	tosses it.  We only do this just to keep I/O
 *	streams synchronized.
 */
 
int get_job_info()
{
	unsigned long bytes, jobs;

	fflush(stdout);

	tty_alarmed = 0;
	alarm(20);

	if (scanf("%*s%lu", &bytes) != 1 || tty_alarmed) {
		alarm(0);
		logentry("*", "<ALERT: GET BYTES FAILED>");
		return(ERROR);
	}

	if (scanf("%*s%lu", &jobs) != 1 || tty_alarmed) {
		alarm(0);
		logentry("*", "<ALERT: GET JOBS FAILED>");
		return(ERROR);
	}

	alarm(0);

#ifdef DEBUG
	{
	sprintf(buf, "<DEBUG: EXPECT %lu JOBS FOR %lu BYTES>", jobs, bytes);
	logentry("?", buf);
	}
#endif
	while (get_timed_char(2) != TIMEOUT);
	return (NO_ERROR);
}
