#include <sys/types.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>

#include "serial.h"

/* USE_TERMIOS by default if neither USE_SGTTY nor USE_TERMIO defined */

#ifdef USE_SGTTY
#include <sgtty.h>
#define	TERMSTRUCT	struct sgttyb
#endif
#ifdef USE_TERMIO
#include <termio.h>
#define	TERMSTRUCT	struct termio
#else
#undef USE_TERMIOS
#define USE_TERMIOS
#include <termios.h>
#define	TERMSTRUCT	struct termios
#endif

#ifndef USE_TERMIOS
#ifdef USE_SGTTY
#define	SPEEDLIST	300, 600, 1200, 1800, 2400, \
			4800, 9600, 19200, 38400
#define	SPEEDCNT	9
#else
#define	SPEEDLIST	0, 50, 75, 110, 134.5, 150, 200, \
			300, 600, 1200, 1800, 2400, 4800, \
			9600, 19200, 38400
#define	SPEEDCNT	16
unsigned speeds[SPEEDCNT] = { SPEEDLIST };
#endif
#endif

TERMSTRUCT save_stty, cur_stty;
int tty_alarmed = 0;

int tty_save() {
#ifdef USE_SGTTY
	return(ioctl(0, TIOCGETP, &save_stty));
#endif
#ifdef USE_TERMIO
	return(ioctl(0, TCGETA, &save_stty));
#else
	return(tcgetattr(0, &save_stty));
#endif
}

int tty_restore() {
#ifdef USE_SGTTY
	return(ioctl(0, TIOCSETP, &save_stty));
#endif
#ifdef USE_TERMIO
	return(ioctl(0, TCSETAW, &save_stty));
#else
	return(tcsetattr(0, TCSAFLUSH, &save_stty));
#endif
}

int tty_rawmode() {
	int status;
	cur_stty = save_stty;
#ifdef USE_SGTTY
	cur_stty.sg_flags &= ~ECHO;
	cur_stty.sg_flags |= HUPCL;
	cur_stty.sg_flags |= RAW;
	status = ioctl(fileno(stdin), TIOCSETP, &cur_stty);
#endif
#ifdef USE_TERMIO
	cur_stty.c_oflag = cur_stty.c_iflag = cur_stty.c_lflag = 0;
	cur_stty.c_iflag |= IGNBRK | IGNPAR;
	cur_stty.c_cflag |= (CS8 | CREAD | HUPCL);
	cur_stty.c_cc[VMIN] = 1;
	cur_stty.c_cc[VTIME] = 0;
	status = ioctl(0, TCSETAW, &cur_stty);
#else
	cfmakeraw(&cur_stty);
	status = tcsetattr(0, TCSANOW, &cur_stty);
#endif

	setvbuf(stdout, NULL, _IONBF, 0);

	signal(SIGALRM, tty_sigalrm);
	alarm(0);
	tty_alarmed = 0;

	return(status);
}

unsigned long tty_speed() {
#ifdef USE_SGTTY
	return(cur_stty.sg_ispeed);
#endif
#ifdef USE_TERMIO
	return(cur_stty.c_cflag & CBAUD);
#else
	return(cfgetispeed(&cur_stty));
#endif
}

int tty_hangup() {
#ifdef USE_SGTTY
	while(get_timed_char(2) != TIMEOUT);	/* pause before hangup */
	cur_stty.sg_ispeed = 0;
	return(ioctl(fileno(stdin), TIOCSETP, &cur_stty));
#endif
#ifdef USE_TERMIO
	while(get_timed_char(2) != TIMEOUT);	/* pause before hangup */
	cur_stty.c_cflag &= ~CBAUD;
	return(ioctl(fileno(stdin), TCSETAW, &save_stty));
#else
	cfsetspeed(&cur_stty, 0);
	return(tcsetattr(fileno(stdin), TCSAFLUSH, &save_stty));
#endif
}


void tty_sigalrm(signo)
	int signo;
{
	int flags;

	alarm(1);
	tty_alarmed = 1;		/* Reset alarm to avoid race window */
	signal(SIGALRM, tty_sigalrm);	/* that can cause hanging in read() */

	flags = fcntl(0, F_GETFL, 0);
	fcntl(0, F_SETFL, flags | FNDELAY);
}

/*
 *	get_timed_char()
 *
 *	Reads characters with time-out processing.  The character read
 *	is returned, unless a time-out occurs, which returns -1.
 */

int get_timed_char(seconds)
	int seconds;
{
	int status;
	char c;

	tty_alarmed = 0;
	alarm(seconds);
	status = read(0, &c, 1);
	alarm(0);

	switch (status) {
	case 1:
		return ((int)c & 0xFF);

	case -1:
		status = fcntl(0, F_GETFL, 0);
		fcntl(0, F_SETFL, status & ~FNDELAY);
	}
	return (-1);
}


/*
 *	write_clear_char()	Flushes stdin and then writes a character
 *	write_char()		Writes a character
 */

void write_clear_char(c)
	char c;
{
	while (get_timed_char(2) != -1);
	write(1, &c, 1);
}

void write_char(c)
	char c;
{
	write(1, &c, 1);
}
