/**********************************************************************
 *
 *	mdxfer.c	Morgan Davis XMODEM (transfer routines)
 *			(C)opyright 1992 Morgan Davis Group
 *
 */

#include <stdio.h>
#include <unistd.h>		/* constants for fseek */

#include "mdx.h"
#include "serial.h"

#define	THE_ERROR(r)	(((r) == TIMEOUT) ? TIMEOUT : FAILED)


/***********************************************************************
 *
 * cancel_check() returns TRUE if character received within one
 * second is a cancel.  Otherwise, returns FALSE.
 *
 */

int cancel_check()
{
	return ((get_timed_char(1) == CAN) ? 1 : 0);
}


/***********************************************************************
 *
 * getsize() takes a start of header (or start of transfer) character
 * and determines the proper data block size to expect.  The size is
 * returned as an int.
 */
 
unsigned int getsize(response)
	int response;
{
	switch(response) {
	case 'L':
	case SSTX:	return(4096);

	case 'K':
	case STX:	return(1024);

	case 'C':
	case SOH:
	default:	return(128);
	}
}	


/***********************************************************************
 *
 * checkit() does either a CRC-16 or a checksum (depending on the state
 * of the crc16 flag that is passed) on the buffer pointed to by ptr
 * for count bytes.  The result is returned as an int.
 *
 * 7/1/93 mwd -- uses updcrc to do faster table driven calc (updcrc.c)
 */

unsigned int checkit(ptr, count, crc16)
	char *ptr;
	unsigned int count;
	int crc16;
{
	register unsigned int crc = 0, i;

	if (crc16) {
#if 0
		while (count--) {
			crc = crc ^ (int)*ptr++ << 8;
			for (i = 0; i < 8; ++i) {
				if (crc & 0x8000)
					crc = crc << 1 ^ 0x1021;
				else
					crc = crc << 1;
			}
		}
#else
		crc = updcrc(crc, ptr, count);
#endif
	} else {
		while (count--)
			crc = (crc + *ptr++) & 0xFF;
	}
	return(crc);
}


/**********************************************************************
 *
 * send_file() is the main entry point for sending a file.  The file
 * should already be opened by the caller, and the file's reference
 * pointer it passed.  It returns one of the transfer result codes as
 * listed at the top of this file.
 */

int xsend(fname, blocksize)
	char *fname;
	int blocksize;
{
	char *xferbuf;
	int result;
	FILE *sfile;

	if ((xferbuf = (char *)malloc(MAXBUF + FRAMING)) == NULL)
		return (MEMFAIL);

	if ((sfile = fopen(fname, "r")) == NULL) {
		free(xferbuf);
		return(OSERROR);
	}

	result = sx(sfile, xferbuf, blocksize);

	free(xferbuf);
	return(result);
}	


/**********************************************************************
 *
 * sx() is called by send_file() to do all the work.  It returns
 * a transfer result back to send_file().
 */

int sx(sfile, xferbuf, blocksize)
	FILE *sfile;
	char *xferbuf;
	int blocksize;
{
	int i, j, bytesread, response;
	unsigned int sectnum = 1, crc, reqsize, done, crcflag;
	unsigned int attempts = 0;
	unsigned long bytesleft;

	done = 0;

	/* wait for initiation from receiving side */

	do {
		response = get_timed_char(1);
		switch (response) {
		case 'C':
			crcflag = 1;
			response = -1;
			if (get_timed_char(1) == 'K') {
				response = 'K';
				if (get_timed_char(1) == 'L')
					response = 'L';
			}
			break;
		case NAK:
			crcflag = 0;
			break;
		case CAN:
			if (cancel_check())
				return (CANCELLED);
		default:
			if (++attempts >= MAX_NAK)
				return (TIMEOUT);
			response = 0;
		}
	} while (!response);

	reqsize = (response == -1) ? blocksize : getsize(response);	

	/* Determine file size */

	fseek(sfile, (long)0, SEEK_END);
	bytesleft = ftell(sfile);
	fseek(sfile, (long)0, SEEK_SET);

	/* Main packet sending loop */

	do {
		if (bytesleft >= reqsize)
			blocksize = reqsize;
		else
			blocksize = (bytesleft >= 1024) ? 1024 : 128;

		if((bytesread = fread(xferbuf+3, 1, blocksize, sfile)) == -1)
			return(OSERROR);

		if (!bytesread)
			break;		/* eof */

		bytesleft -= bytesread;

		if (bytesread < blocksize) {
			for ( i = bytesread + 3; i < (blocksize + 3); i++ )
				xferbuf[i] = '\0';
			++done;
		}

		xferbuf[0] = SOH + ((blocksize >= 1024) ? 1 : 0) |
				((blocksize == 4096) ? 0x80 : 0x00);
		xferbuf[1] = sectnum;
		xferbuf[2] = ~sectnum;

		crc = checkit(xferbuf + 3, blocksize, crcflag);
		if (crcflag) {
			xferbuf[blocksize + 3] = crc >> 8;
			xferbuf[blocksize + 4] = crc & 0x00FF;
		} else
			xferbuf[blocksize + 3] = crc;

		attempts = 0;
		do {
/*			write(stdout, xferbuf, blocksize + 4 + crcflag); */
			fwrite(xferbuf, sizeof(char), blocksize + 4 + crcflag, stdout);
			i = WRITE_DELAY + (blocksize > 128) ?
				(blocksize / 256) : 0;
			response = get_timed_char(i);
			if (response == CAN && cancel_check())
				return(CANCELLED);
			if (response != ACK && ++attempts == MAX_ERROR) {
				for (i = 0; i < 4; i++)
					write_char(CAN);
				return (THE_ERROR(response));
			}
		} while (response != ACK);
		++sectnum;
	} while (!done);

	attempts = 0;
	do {
		write_char(EOT);
	} while ((get_timed_char(5) != ACK) && (attempts++ < MAX_EOT));

	return (GOOD);
}


/***********************************************************************
 *
 * receive_file() receives a file to the name passed.  Unlike send_file()
 * which takes the pointer to a previously opened file, receive_file()
 * takes the intended file's name and opens the file itself.  It calls
 * rx() to do the work, and then return the appropriate transfer result.
 */

int xrecv(fname, crcflag, blocksize)
	char *fname;
	int crcflag, blocksize;
{
	char *xferbuf;
	int result;
	FILE *rfile;

	if ((xferbuf = (char *)malloc(MAXBUF + FRAMING)) == NULL)
		return (MEMFAIL);

	if ((rfile = fopen(fname, "w")) == NULL) {
		free(xferbuf);
		return(OSERROR);
	}

	result = rx(rfile, xferbuf, crcflag, blocksize);
	free(xferbuf);
	fclose(rfile);
	
	if (result != GOOD)
		unlink(fname);

	return(result);
}	


/***********************************************************************
 *
 * rx() is the guts of the receive.  This long, messy routine does all
 * the work and returns a transfer result to receive_file().
 */

int rx(rfile, xferbuf, crcflag, blocksize)
	FILE *rfile;
	char *xferbuf;
	int crcflag, blocksize;
{
	int i, c, response;
	unsigned int crc, packcrc, reqsize, starting, waitnext = 5;
	unsigned int error_flag, error_count, errmax = MAX_RCVHI;
	unsigned int crchi, crclo;
	char our_packnum, packnum, packnumchk;

	error_count = our_packnum = 0;
	starting = 1;

	fflush(stdin);

	for (;;) {
		do {
			if (starting) {
				if (crcflag && (error_count < NAK_THRESH)) {
					write_char('C');
					if (blocksize > 128) {
						write_char('K');
						if (blocksize > 1024)
							write_char('L');
					}
				} else {
					crcflag = 0;
					write_char(NAK);
				}
			} else if (error_count)
				write_clear_char(NAK);

			response = get_timed_char(waitnext);
			switch (response) {
			case EOT:
				write_char(ACK);
				return(GOOD);
			case SOH:
			case STX:
			case SSTX:
				break;
			case CAN:
				if (cancel_check())
					return(CANCELLED);
			default:
				if (++error_count >= errmax)
					return(THE_ERROR(response));
				response = 0;
			}
		} while (!response);

		if (starting) {
			starting = error_count = 0;
			waitnext = 10;
			errmax = MAX_ERROR;
		}

		reqsize = getsize(response);

		packnum = get_timed_char(3);
		packnumchk = get_timed_char(3);

		if (packnum == ~packnumchk) {
			if (packnum == (our_packnum + 1)) {
				for (error_flag = i = 0; i < reqsize; i++) {
					if ((xferbuf[i] = get_timed_char(3)) == TIMEOUT) {
						++error_flag;
						break;
					}
				}
				if (!error_flag) {
					crchi = ((crcflag) ? get_timed_char(3) : 0);
					crclo = get_timed_char(3);
					crc = (crchi << 8) | (crclo & 0xFF);
					packcrc = checkit(xferbuf, reqsize, crcflag) & 0xFFFF;
					if (crc == packcrc) {
/* no newline conversion
						for (i = 0; i < reqsize; i++)
							if (xferbuf[i] == 13)
								xferbuf[i] = '\n';
*/
						fwrite(xferbuf, 1, reqsize, rfile);
						write_char(ACK);
						error_count = 0;
						our_packnum = packnum;
					} else {
						++error_count;
					}
				} else {
					++error_count;
				}
			} else {
				if (packnum == our_packnum) {
					write_clear_char(ACK);
					error_count = 0;
				} else
					return(FAILED);	/* this is fatal */
			}
		} else {
			++error_count;
		}
	}
}
