	Page	58,132
	Title	EXPRESS.ASM	Expression Handler
;******************************************************************************
;
;   Name:	EXPRESS.ASM	Expression Handler
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the code for the expression
;	handler.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	Get_Expression		; Get expression value routine
;
;  External Declarations
;
	Extrn	Get_Parameter:Near	; Get parameter routine       (SUPPORT)
	Extrn	Upper_Case:Near 	; Convert to upper case       (SUPPORT)
	Extrn	Match_Parameter:Near	; Match parameter routine     (SUPPORT)
	Extrn	ASCII_Binary:Near	; ASCII to binary conversion  (SUPPORT)
	Extrn	Stack_Init:Near 	; Initialize stack routine	(STACK)
	Extrn	Push_Byte:Near		; Push byte onto stack routine	(STACK)
	Extrn	Push_Word:Near		; Push word onto stack routine	(STACK)
	Extrn	Push_Dword:Near 	; Push dword onto stack routine (STACK)
	Extrn	Pop_Byte:Near		; Pop byte from stack routine	(STACK)
	Extrn	Pop_Word:Near		; Pop word from stack routine	(STACK)
	Extrn	Pop_Dword:Near		; Pop dword from stack routine	(STACK)
	Extrn	Top_Byte:Near		; Get top byte from stack	(STACK)
	Extrn	Top_Word:Near		; Get top word from stack	(STACK)
	Extrn	Top_Dword:Near		; Get top dword from stack	(STACK)
	Extrn	Input_Buffer:Byte	; Input buffer storage area	 (DATA)
	Extrn	Parm_Buffer:Byte	; Parameter buffer storage area  (DATA)
	Extrn	Delimit_Input:Byte	; Input delimiter table 	 (DATA)
	Extrn	Delimit_Express:Byte	; Expression delimiter table	 (DATA)
	Extrn	Flag_Encode:Byte	; CPU flag encoding table	 (DATA)
	Extrn	Flag_Decode:Byte	; CPU flag decoding table	 (DATA)
	Extrn	System_Flag:Byte	; Apple emulator system flag byte(DATA)
	Extrn	Build_Stack:Byte	; Build private stack area	 (DATA)
	Extrn	Result_Stack:Byte	; Result private stack area	 (DATA)
	Extrn	Last:Byte		; Last parameter type value	 (DATA)
	Extrn	Rank:Byte		; Expression rank value 	 (DATA)
	Extrn	Radix:Word		; Current numeric radix value	 (DATA)
;
;  LOCAL Equates
;
HEX		Equ	16		; Base 16 - Hexadecimal
DECIMAL 	Equ	10		; Base 10 - Decimal
OCTAL		Equ	8		; Base 8  - Octal
BINARY		Equ	2		; Base 2  - Binary
BASE_SHIFT	Equ	03h		; Base shift value
CONST_INPUT	Equ	0Bh		; Constant input precedence value
CONST_STACK	Equ	0Ch		; Constant stack precedence value
CONST_RANK	Equ	01h		; Constant rank value
TYPE_OPER	Equ	000h		; Operator type value
TYPE_CONST	Equ	0FFh		; Constant type value
MIN_PRECEDENCE	Equ	000h		; Minimum precedence value
MAX_PRECEDENCE	Equ	0FFh		; Maximum precedence value
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	Include 	Strucs.inc	; Include the structure definitions
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	Get_Expression	Get Expression Value Procedure
	Page	+
;******************************************************************************
;
;	Get_Expression(Parameter, Offset, Stack_Frame)
;
;		Save the required registers
;		Call routine to initialize the build stack
;		Call routine to initialize the result stack
;		Initialize last token type and expression rank value
;		While there are more tokens
;			Call routine to get the next token
;			Call routine to get token input precedence
;			While input precedence <= stack top stack precedence
;				Call routine to pop off stack top
;				Call routine to evaluate popped value
;			EndWhile
;			Call routine to push token on stack
;		EndWhile
;		If there is an expression present (At least one token)
;			While there are entries on the stack
;				Call routine to pop off stack top
;				Call routine to evaluate popped value
;			EndWhile
;			If this is a valid expression (Rank is 1)
;				Clear carry to indicate valid expression
;			Else
;				Set carry to indicate expression error
;			Endif
;		Else
;			Set zero flag indicating no expression
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to expression parameter
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX    - Expression value
;		BX    - Delimiter type/Parameter length
;		DX    - Parse offset updated
;		FL    - Carry flag set if invalid expression
;			Zero flag set if no expression (Valid expression only)
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Expression	Proc	Near		; Get expression procedure
	Save	cx,di,bp,es		; Save the required registers
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	lea	di,ds:[Build_Stack]	; Get pointer to the build stack
	call	Stack_Init		; Call routine to initialize stack
	lea	di,ds:[Result_Stack]	; Get pointer to the results stack
	call	Stack_Init		; Call routine to initialize stack
	and	ds:[System_Flag],Not EXPRESSION
	mov	ds:[Last],TYPE_OPER	; Initialize last type to operator
	xor	al,al			; Zero the expression
	mov	ds:[Rank],al		;		      rank value
	lea	di,ds:[Build_Stack]	; Get pointer to private build stack
Expression_Loop:
	call	Get_Token		; Call routine to get next input token
	jc	Check_Expression	; Jump if no more tokens available
	or	ds:[System_Flag],EXPRESSION
	Save	bx,dx,si		; Save the required register values
	mov	bl,al			; Save the current
	mov	si,cx			;		   token values
	call	Input_Value		; Call routine to get input precedence
	mov	bh,ah			; Save the input precedence value
Check_Loop:
	mov	ch,MIN_PRECEDENCE	; Assume minimum precedence value
	call	Top_Dword		; Get the top of the build stack
	jc	Check_Precedence	; Jump if nothing on the stack
	mov	cx,dx			; Setup to get the stack precedence
	call	Stack_Value		; Call routine to get stack precedence
	mov	ch,ah			; Save the stack precedence value
Check_Precedence:
	cmp	bh,ch			; Check input against stack precedence
	ja	Expression_Stack	; Jump if time to put value on stack
	call	Pop_Dword		; Pop the top off the build stack
	jc	Expression_Bad		; Jump if stack underflow
	cmp	bh,ch			; Check input against stack precedence
	je	Expression_Next 	; Jump if time to get next token
	mov	cx,dx			; Setup the actual stack value
	call	Evaluate		; Call routine to handle evaluation
	jc	Expression_Bad		; Jump if this is a bad expression
	jmp	Short Check_Loop	; Keep checking the stack top
Expression_Stack:
	mov	al,bl			; Restore the original
	mov	dx,si			;		       next token values
	call	Push_Dword		; Call routine to push token on stack
	jc	Expression_Bad		; Jump if stack overflow
Expression_Next:
	Restore bx,dx,si		; Restore the required registers values
	test	bx,END_OF_EXPRESS	; Check for end of expression delimiter
	jnz	Check_Expression	; Jump if this is end of expression
	jmp	Short Expression_Loop	; Loop till expression parsing complete
Expression_Bad:
	Restore bx,dx,si		; Restore the required registers values
	jmp	Expression_Error	; Go return expression error
Check_Expression:
	jnz	Expression_Error	; Jump if this is really a token error
	test	ds:[System_Flag],EXPRESSION
	jz	Expression_Exit 	; Jump if there was no expression
Clear_Loop:
	mov	bp,dx			; Save the current parse offset
	call	Pop_Dword		; Pop the top off the build stack
	jc	Check_Rank		; Jump if stack is empty
	mov	cx,dx			; Setup the actual stack value
	call	Evaluate		; Call routine to handle evaluation
	mov	dx,bp			; Restore the current parse offset
	jc	Expression_Error	; Jump if this is a bad expression
	jmp	Short Clear_Loop	; Loop till the build stack is empty
Check_Rank:
	mov	dx,bp			; Restore the current parse offset
	mov	al,ds:[Rank]		; Get the expression rank value
	dec	al			; Check for a valid expression
	jnz	Expression_Error	; Jump if the expression is invalid
Get_Result:
	lea	di,ds:[Result_Stack]	; Get pointer to the result stack
	call	Pop_Word		; Call routine to pop off actual result
	jc	Expression_Error	; Jump if there is no result
	test	ds:[System_Flag],EXPRESSION
	jmp	Short Expression_Exit	; Go return control to the caller
Expression_Error:
	stc				; Set carry indicating special/error
Expression_Exit:
	Restore cx,di,bp,es		; Restore the required registers
	ret				; Return to the caller
Get_Expression	Endp			; End of the Get_Expression procedure
	Subttl	Get_Token	Get Next Token Procedure
	Page	+
;******************************************************************************
;
;	Get_Token(Parameter, Offset, Stack_Frame)
;
;		Save the required registers
;		Call routine to get the next parameter
;		If a parameter is available
;			If a null parameter is present
;				If delimiter is an end of expresssion delimiter
;					Set carry indicating no token
;				Else not end of expression
;					If unknown delimiter type (Non-operator)
;						Decrement back to delimiter
;						Set carry indicating no token
;					Else operator type delimiter
;						If subtraction delimiter
;							If last type operator
;								Set uniary minus
;							Endif
;						Endif
;						If not pop operation
;							Set last type to oper.
;						Endif
;						Clear carry indicating token
;					Endif
;				Endif
;			Else non-null parameter
;				If last type value is constant
;					Reset the parse offset value
;					Set carry indicating no token
;				Endif
;				Call routine to convert to uppercase
;				Get pointer to register table
;				Call routine to check for a match
;				If there is a register match
;					Call routine to get register value
;					Set last type value to constant
;					Clear carry indicating token
;				Else not a register
;					If delimiter was a base delimiter type
;						Get the requested base value
;					Else
;						Use the default base value
;					Endif
;					Call routine to convert to binary
;					If a valid constant value
;						Set last type value to constant
;						Clear carry indicating token
;					Else invalid parameter
;						Set carry indicating no token
;					Endif
;				Endif
;		Else no parameter
;			Set carry indicating no token available
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to expression parameter
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;		ES    - Data segment
;
;	Registers on Exit:
;
;		AL    - Token type value (Operator vs. Constant)
;		BX    - Delimiter type/Parameter length
;		CX    - Token value (Operator type/Actual value)
;		DX    - Parse offset updated
;		FL    - Carry flag set if token error
;			Zero flag set if no token (Token error only)
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Token	Proc	Near		; Get next token procedure
	Save	di			; Save the required registers
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,CR			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Express] ; Get expression delimiter table
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	push	dx			; Save the current parse offset value
	call	Get_Parameter		; Call routine to get next parameter
	mov	bx,ax			; Save parameter length/delimiter type
	pop	cx			; Restore original parse offset value
	jc	No_Token		; Jump if no parameter available
	jnz	Check_Type		; Jump if a non-null parameter given
Check_Delimiter:
	test	bh,END_OF_EXPRESS	; Check for end of expression delimiter
	jnz	No_Token		; Jump if this is end of expression
	test	bh,OP_MASK		; Check for operator delimiter
	jz	Invalid_Delimiter	; Jump if not an operator delimiter
	mov	cl,bh			; Save the actual
	xor	ch,ch			;		  operator type value
Check_Operator:
	cmp	cl,OP_SUB		; Check for subtraction operation
	jne	Set_Operator		; Jump if not subtraction operation
	cmp	ds:[Last],TYPE_OPER	; Check the last parameter type value
	jne	Set_Operator		; Jump if last parameter was a constant
	mov	cl,OP_NEG		; Change operation to negation
Set_Operator:
	mov	al,TYPE_OPER		; Set token type to operator
	cmp	cl,OP_POP		; Check for close parenthesis operator
	je	Token_Done		; Jump if this is a close operator
	mov	ds:[Last],al		; Set last parameter type value
Token_Done:
	clc				; Clear carry indicating token available
	jmp	Short Token_Exit	; Go return control to the caller
Check_Type:
	cmp	ds:[Last],TYPE_CONST	; Check for back to back constants
	jne	Check_Register		; Jump if not back to back constants
	mov	dx,cx			; Reset the current parse offset
	jmp	Short No_Token		; Go return no more tokens available
Check_Register:
	call	Upper_Case		; Call routine to convert to uppercase
	push	si			; Save the parameter pointer
	lea	si,ds:[Register_Table]	; Get pointer to start of register table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the match table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	pop	si			; Restore the parameter pointer
	jc	Check_Value		; Jump if this is not a register
Register_Token:
	or	bh,bh			; Check for end of input buffer
	jz	Register_Skip		; Jump if end of input buffer
	dec	dx			; Decrement to parse delimiter next
Register_Skip:
	xor	ah,ah			; Convert match number to full word
	shl	ax,1			; Convert match number to table index
	xchg	ax,bx			; Setup to call correct register routine
	call	ds:[bx + Register_Jump] ; Call correct routine to get register
	xchg	ax,bx			; Restore the register order
	mov	al,TYPE_CONST		; Set token type to constant
	mov	ds:[Last],al		; Set last parameter type value
	jmp	Short Token_Exit	; Go return control to the caller
Invalid_Delimiter:
	dec	dx			; Decrement back to unknown delimiter
No_Token:
	xor	cx,cx			; Set zero indicating no more tokens
	stc				; Set carry indicating token error
	jmp	Short Token_Exit	; Go return control to the caller
Check_Value:
	mov	ax,ds:[Radix]		; Get the current numeric radix value
	test	bh,BASE_TYPE		; Check for a base type delimiter
	jz	Convert_Value		; Jump if not a base type delimiter
	mov	al,bh			; Get the base delimiter value
	and	ax,BASE_MASK		; Mask off all but the base value
	shr	ax,BASE_SHIFT		; Convert base value to table index
	xchg	ax,bx			; Setup to access the base table
	mov	bx,ds:[bx + Radix_Table]; Get the actual radix value
	xchg	ax,bx			; Restore the register order
Convert_Value:
	call	ASCII_Binary		; Call routine to convert to binary
	jnc	Value_Token		; Jump if this is a valid number
	xor	cx,cx			; Clear zero flag
	dec	cx			;		  indicating invalid
	stc				; Set carry indicating token error
	jmp	Short Token_Exit	; Go return control to the caller
Value_Token:
	or	bh,bh			; Check for end of input buffer
	je	Value_Skip		; Jump if end of the input buffer
Radix_Check:
	test	bh,BASE_TYPE		; Check for a base type delimiter
	jnz	Value_Skip		; Jump if a base type delimiter
	dec	dx			; Decrement to parse delimiter next
Value_Skip:
	mov	cx,ax			; Save the actual numeric value
	mov	al,TYPE_CONST		; Set token type to constant
	mov	ds:[Last],al		; Set last parameter type value
Token_Exit:
	Restore di			; Restore the required registers
	ret				; Return to the caller
Get_Token	Endp			; End of the Get_Token procedure
	Subttl	Get_SP		Get Stack Pointer Register Procedure
	Page	+
;******************************************************************************
;
;	Get_SP(Stack_Frame)
;
;		Get the 65C02 stack pointer register
;		Clear carry indicating no errors
;		Return to the caller
;
;	Registers on Entry:
;
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		CX    - Stack pointer register value
;		FL    - Carry flag cleared indicating no error
;
;******************************************************************************
		Even			; Force procedure to even address
Get_SP		Proc	Near		; Get stack pointer register procedure
	mov	cx,ss:[bp.BX_Register]	; Get the 65C02 stack pointer register
	clc				; Clear carry indicating no errors
	ret				; Return to the caller
Get_SP		Endp			; End of the Get_SP procedure
	Subttl	Get_PC		Get Program Counter Register Procedure
	Page	+
;******************************************************************************
;
;	Get_PC(Stack_Frame)
;
;		Get the 65C02 program counter register
;		Clear carry indicating no errors
;		Return to the caller
;
;	Registers on Entry:
;
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		CX    - Program counter register value
;		FL    - Carry flag cleared indicating no error
;
;******************************************************************************
		Even			; Force procedure to even address
Get_PC		Proc	Near		; Get program counter reg. procedure
	mov	cx,ss:[bp.SI_Register]	; Get the 65C02
	dec	cx			;		program counter register
	clc				; Clear carry indicating no errors
	ret				; Return to the caller
Get_PC		Endp			; End of the Get_PC procedure
	Subttl	Get_ACC 	Get Accumulator Register Procedure
	Page	+
;******************************************************************************
;
;	Get_ACC(Stack_Frame)
;
;		Save the required registers
;		Get the 65C02 accumulator register value
;		Sign extend the accumulator value
;		Restore the required registers
;		Clear carry indicating no errors
;		Return to the caller
;
;	Registers on Entry:
;
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		CX    - Accumulator register value (Sign extended)
;		FL    - Carry flag cleared indicating no error
;
;******************************************************************************
		Even			; Force procedure to even address
Get_ACC 	Proc	Near		; Get accumulator register procedure
	Save	ax			; Save the required registers
	mov	al,ss:[bp.DL_Register]	; Get the 65C02 accumulator register
	cbw				; Sign extend the accumulator value
	mov	cx,ax			; Save the accumulator value
	Restore ax			; Restore the required registers
	clc				; Clear carry indicating no errors
	ret				; Return to the caller
Get_ACC 	Endp			; End of the Get_ACC procedure
	Subttl	Get_X		Get X Index Register Procedure
	Page	+
;******************************************************************************
;
;	Get_X(Stack_Frame)
;
;		Save the required registers
;		Get the 65C02 X index register value
;		Sign extend the X index value
;		Restore the required registers
;		Clear carry indicating no errors
;		Return to the caller
;
;	Registers on Entry:
;
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		CX    - X index register value (Sign extended)
;		FL    - Carry flag cleared indicating no error
;
;******************************************************************************
		Even			; Force procedure to even address
Get_X		Proc	Near		; Get X index register procedure
	Save	ax			; Save the required registers
	mov	al,ss:[bp.CH_Register]	; Get the 65C02 X index register value
	cbw				; Sign extend the X index value
	mov	cx,ax			; Save the X index value
	Restore ax			; Restore the required registers
	clc				; Clear carry indicating no errors
	ret				; Return to the caller
Get_X		Endp			; End of the Get_X procedure
	Subttl	Get_Y		Get Y Index Register Procedure
	Page	+
;******************************************************************************
;
;	Get_Y(Stack_Frame)
;
;		Save the required registers
;		Get the 65C02 Y index register value
;		Sign extend the Y index value
;		Restore the required registers
;		Clear carry indicating no errors
;		Return to the caller
;
;	Registers on Entry:
;
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		CX    - Y index register value (Sign extended)
;		FL    - Carry flag cleared indicating no error
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Y		Proc	Near		; Get Y index register procedure
	Save	ax			; Save the required registers
	mov	al,ss:[bp.CL_Register]	; Get the 65C02 Y index register value
	cbw				; Sign extend the Y index value
	mov	cx,ax			; Save the Y index value
	Restore ax			; Restore the required registers
	clc				; Clear carry indicating no errors
	ret				; Return to the caller
Get_Y		Endp			; End of the Get_Y procedure
	Subttl	Get_FL		Get Flags Register Procedure
	Page	+
;******************************************************************************
;
;	Get_FL(Stack_Frame)
;
;		Save the required registers
;		Get the 65C02 flags register value
;		Set the r (Reserved) flag bit
;		Encode flag bits from 80x86 to 65C02
;		Zero extend the flags value
;		Restore the required registers
;		Clear carry indicating no errors
;		Return to the caller
;
;	Registers on Entry:
;
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		CX    - Flags register value (Zero extended)
;		FL    - Carry flag cleared indicating no error
;
;******************************************************************************
		Even			; Force procedure to even address
Get_FL		Proc	Near		; Get flags register procedure
	Save	bx			; Save the required registers
	mov	bl,ss:[bp.DH_Register]	; Get the 65C02 flags register value
	or	bl,CPU_R		; Make sure reserved bit is set
	xor	bh,bh			; Convert flag value to full word
	mov	cl,cs:[bx + Flag_Encode]; Get the encoded flag value
	xor	ch,ch			; Zero extend the flags value
	Restore bx			; Restore the required registers
	clc				; Clear carry indicating no errors
	ret				; Return to the caller
Get_FL		Endp			; End of the Get_FL procedure
	Subttl	Input_Value	Get Input Precedence Procedure
	Page	+
;******************************************************************************
;
;	Input_Value(Type, Value)
;
;		Save the required registers
;		If type value is a constant
;			Get the constant input precedence value
;		Else
;			Lookup the correct input precedence value
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Parameter type value (Operator/Constant)
;		CX    - Actual parameter value
;
;	Registers on Exit:
;
;		AH    - Input precedence value
;
;******************************************************************************
		Even			; Force procedure to even address
Input_Value	Proc	Near		; Get input precedence procedure
	Save	bx			; Save the required registers
	mov	ah,CONST_INPUT		; Default to a constant parameter type
	cmp	al,TYPE_CONST		; Check for a constant type parameter
	je	Input_Exit		; Jump if this is a constant type
	mov	bx,cx			; Get the actual operator type value
	and	bx,OP_MASK		; Mask off all but the operator type
	dec	bx			; Convert operator type to table index
	mov	ah,ds:[bx + Input_Table]; Get the input precedence value
Input_Exit:
	Restore bx			; Restore the required registers
	ret				; Return to the caller
Input_Value	Endp			; End of the Input_Value procedure
	Subttl	Stack_Value	Get Stack Precedence Procedure
	Page	+
;******************************************************************************
;
;	Stack_Value(Type, Value)
;
;		Save the required registers
;		If type value is a constant
;			Get the constant stack precedence value
;		Else
;			Lookup the correct stack precedence value
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Parameter type value (Operator/Constant)
;		CX    - Actual parameter value
;
;	Registers on Exit:
;
;		AH    - Stack precedence value
;
;******************************************************************************
		Even			; Force procedure to even address
Stack_Value	Proc	Near		; Get stack precedence procedure
	Save	bx			; Save the required registers
	mov	ah,CONST_STACK		; Default to a constant parameter type
	cmp	al,TYPE_CONST		; Check for a constant type parameter
	je	Stack_Exit		; Jump if this is a constant type
	mov	bx,cx			; Get the actual operator type value
	and	bx,OP_MASK		; Mask off all but the operator type
	dec	bx			; Convert operator type to table index
	mov	ah,ds:[bx + Stack_Table]; Get the stack precedence value
Stack_Exit:
	Restore bx			; Restore the required registers
	ret				; Return to the caller
Stack_Value	Endp			; End of the Stack_Value procedure
	Subttl	Rank_Value	Get Rank Value Procedure
	Page	+
;******************************************************************************
;
;	Rank_Value(Type, Value)
;
;		Save the required registers
;		If type value is a constant
;			Get the constant rank value
;		Else
;			Lookup the correct rank value
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Parameter type value (Operator/Constant)
;		CX    - Actual parameter value
;
;	Registers on Exit:
;
;		AH    - Rank value
;
;******************************************************************************
		Even			; Force procedure to even address
Rank_Value	Proc	Near		; Get rank value procedure
	Save	bx			; Save the required registers
	mov	ah,CONST_RANK		; Default to a constant parameter type
	cmp	al,TYPE_CONST		; Check for a constant type parameter
	je	Rank_Exit		; Jump if this is a constant type
	mov	bx,cx			; Get the actual operator type value
	and	bx,OP_MASK		; Mask off all but the operator type
	dec	bx			; Convert operator type to table index
	mov	ah,ds:[bx + Rank_Table] ; Get the actual rank value
Rank_Exit:
	Restore bx			; Restore the required registers
	ret				; Return to the caller
Rank_Value	Endp			; End of the Rank_Value procedure
	Subttl	Evaluate	Evaluate Result Procedure
	Page	+
;******************************************************************************
;
;	Evaluate(Type, Value)
;
;		Save the required registers
;		Call routine to get the rank value
;		Update the current expression rank value
;		If current rank value is valid (Non-zero)
;			If type value is a constant
;				Call routine to push constant on result stack
;			Else type value is operator
;				Call correct routine to evaluate operator
;			Endif
;		Else invalid expression
;			Set carry indicating invalid expression
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Parameter type value (Operator/Constant)
;		CX    - Actual parameter value
;
;	Registers on Exit:
;
;		FL    - Carry set if invalid expression
;
;******************************************************************************
		Even			; Force procedure to even address
Evaluate	Proc	Near		; Evaluate result procedure
	Save	ax,bx,di		; Save the required registers
	lea	di,ds:[Result_Stack]	; Get pointer to private result stack
	call	Rank_Value		; Call routine to get the rank value
	add	ds:[Rank],ah		; Update the expression rank value
	jnz	Evaluate_Continue	; Jump if still a valid expression
Eval_Error:
	stc				; Set carry indicating expression error
	jmp	Short Evaluate_Exit	; Go return control to the caller
Evaluate_Continue:
	cmp	al,TYPE_CONST		; Check for a constant parameter
	jne	Eval_Operator		; Jump if this is not a constant
Eval_Constant:
	mov	ax,cx			; Get the actual constant value
	call	Push_Word		; Push the value onto the stack
	jmp	Short Evaluate_Exit	; Go return control to the caller
Eval_Operator:
	mov	bx,cx			; Get the actual operator type
	and	bx,OP_MASK		; Mask off all but the operator type
	dec	bx			; Convert operator type
	shl	bx,1			;			to table index
	call	ds:[bx + Op_Table]	; Call routine to handle operation
Evaluate_Exit:
	Restore ax,bx,di		; Restore the required registers
	ret				; Return to the caller
Evaluate	Endp			; End of the Evaluate procedure
	Subttl	AND_Operation	Logical AND Operation Procedure
	Page	+
;******************************************************************************
;
;	AND_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the logical AND operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
AND_Operation	Proc	Near		; Logical AND operation procedure
	Save	ax,cx			; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	AND_Exit		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	AND_Exit		; Jump if there is a stack underflow
	and	ax,cx			; Perform the logical AND operation
	call	Push_Word		; Put result value back onto the stack
AND_Exit:
	Restore ax,cx			; Restore the required registers
	ret				; Return to the caller
AND_Operation	Endp			; End of the AND_Operation procedure
	Subttl	OR_Operation	Logical OR Operation Procedure
	Page	+
;******************************************************************************
;
;	OR_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the logical OR operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
OR_Operation	Proc	Near		; Logical OR operation procedure
	Save	ax,cx			; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	OR_Exit 		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	OR_Exit 		; Jump if there is a stack underflow
	or	ax,cx			; Perform the logical OR operation
	call	Push_Word		; Put result value back onto the stack
OR_Exit:
	Restore ax,cx			; Restore the required registers
	ret				; Return to the caller
OR_Operation	Endp			; End of the OR_Operation procedure
	Subttl	XOR_Operation	Logical XOR Operation Procedure
	Page	+
;******************************************************************************
;
;	XOR_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the logical XOR operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
XOR_Operation	Proc	Near		; Logical XOR operation procedure
	Save	ax,cx			; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	XOR_Exit		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	XOR_Exit		; Jump if there is a stack underflow
	xor	ax,cx			; Perform the logical XOR operation
	call	Push_Word		; Put result value back onto the stack
XOR_Exit:
	Restore ax,cx			; Restore the required registers
	ret				; Return to the caller
XOR_Operation	Endp			; End of the XOR_Operation procedure
	Subttl	NOT_Operation	Logical NOT Operation Procedure
	Page	+
;******************************************************************************
;
;	NOT_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Perform the logical NOT operation
;			Call routine to push result onto result stack
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
NOT_Operation	Proc	Near		; Logical NOT operation procedure
	Save	ax			; Save the required registers
	call	Pop_Word		; Pop the operand from stack
	jc	NOT_Exit		; Jump if there is a stack underflow
	not	ax			; Perform the logical NOT operation
	call	Push_Word		; Put result value back onto the stack
NOT_Exit:
	Restore ax			; Restore the required registers
	ret				; Return to the caller
NOT_Operation	Endp			; End of the NOT_Operation procedure
	Subttl	Right_Operation Shift Right Operation Procedure
	Page	+
;******************************************************************************
;
;	Right_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the shift right operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Right_Operation Proc	Near		; Shift right operation procedure
	Save	ax,cx			; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	Right_Exit		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	Right_Exit		; Jump if there is a stack underflow
	shr	ax,cl			; Perform the shift right operation
	call	Push_Word		; Put result value back onto the stack
Right_Exit:
	Restore ax,cx			; Restore the required registers
	ret				; Return to the caller
Right_Operation Endp			; End of the Right_Operation procedure
	Subttl	Left_Operation	Shift Left Operation Procedure
	Page	+
;******************************************************************************
;
;	Left_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the shift left operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Left_Operation	Proc	Near		; Shift left operation procedure
	Save	ax,cx			; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	Left_Exit		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	Left_Exit		; Jump if there is a stack underflow
	shl	ax,cl			; Perform the shift left operation
	call	Push_Word		; Put result value back onto the stack
Left_Exit:
	Restore ax,cx			; Restore the required registers
	ret				; Return to the caller
Left_Operation	Endp			; End of the Left_Operation procedure
	Subttl	Mod_Operation	Modulus Operation Procedure
	Page	+
;******************************************************************************
;
;	Mod_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the modulus operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Mod_Operation	Proc	Near		; Modulus operation procedure
	Save	ax,cx,dx		; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	Mod_Exit		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	Mod_Exit		; Jump if there is a stack underflow
	xor	dx,dx			; Setup to perform modulus operation
	div	cx			; Perform the modulus
	mov	ax,dx			;		      operation
	call	Push_Word		; Put result value back onto the stack
Mod_Exit:
	Restore ax,cx,dx		; Restore the required registers
	ret				; Return to the caller
Mod_Operation	Endp			; End of the Mod_Operation procedure
	Subttl	Add_Operation	Addition Operation Procedure
	Page	+
;******************************************************************************
;
;	Add_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the add operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Add_Operation	Proc	Near		; Addition operation procedure
	Save	ax,cx			; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	Add_Exit		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	Add_Exit		; Jump if there is a stack underflow
	add	ax,cx			; Perform the addition operation
	call	Push_Word		; Put result value back onto the stack
Add_Exit:
	Restore ax,cx			; Restore the required registers
	ret				; Return to the caller
Add_Operation	Endp			; End of the Add_Operation procedure
	Subttl	Sub_Operation	Subtraction Operation Procedure
	Page	+
;******************************************************************************
;
;	Sub_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the subtraction operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Sub_Operation	Proc	Near		; Subtraction operation procedure
	Save	ax,cx			; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	Sub_Exit		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	Sub_Exit		; Jump if there is a stack underflow
	sub	ax,cx			; Perform the subtraction operation
	call	Push_Word		; Put result value back onto the stack
Sub_Exit:
	Restore ax,cx			; Restore the required registers
	ret				; Return to the caller
Sub_Operation	Endp			; End of the Sub_Operation procedure
	Subttl	Mul_Operation	Multiplication Operation Procedure
	Page	+
;******************************************************************************
;
;	Mul_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the multiplication operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Mul_Operation	Proc	Near		; Multiplication operation procedure
	Save	ax,cx,dx		; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	Mul_Exit		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	Mul_Exit		; Jump if there is a stack underflow
	mul	cx			; Perform the multiplcation operation
	call	Push_Word		; Put result value back onto the stack
Mul_Exit:
	Restore ax,cx,dx		; Restore the required registers
	ret				; Return to the caller
Mul_Operation	Endp			; End of the Mul_Operation procedure
	Subttl	Div_Operation	Division Operation Procedure
	Page	+
;******************************************************************************
;
;	Div_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Get second value from result stack
;			If no stack underflow
;				Perform the division operation
;				Call routine to push result onto result stack
;			Else stack underflow
;				Set carry indicating expression error
;			Endif
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Div_Operation	Proc	Near		; Division operation procedure
	Save	ax,cx,dx		; Save the required registers
	call	Pop_Word		; Pop first operand from stack
	jc	Div_Exit		; Jump if there is a stack underflow
	mov	cx,ax			; Save the first operand value
	call	Pop_Word		; Pop second operand from stack
	jc	Div_Exit		; Jump if there is a stack underflow
	xor	dx,dx			; Setup to perform division operation
	div	cx			; Perform the division operation
	call	Push_Word		; Put result value back onto the stack
Div_Exit:
	Restore ax,cx,dx		; Restore the required registers
	ret				; Return to the caller
Div_Operation	Endp			; End of the Div_Operation procedure
	Subttl	Neg_Operation	Negation Operation Procedure
	Page	+
;******************************************************************************
;
;	Neg_Operation(Result_Stack)
;
;		Save the required registers
;		Get first value from result stack
;		If no stack underflow
;			Perform the negation operation
;		Else stack underflow
;			Set carry indicating expression error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Neg_Operation	Proc	Near		; Negation operation procedure
	Save	ax			; Save the required registers
	call	Pop_Word		; Pop the operand from stack
	jc	Neg_Exit		; Jump if there is a stack underflow
	neg	ax			; Perform the negation operation
	call	Push_Word		; Put result value back onto the stack
Neg_Exit:
	Restore ax			; Restore the required registers
	ret				; Return to the caller
Neg_Operation	Endp			; End of the Neg_Operation procedure
	Subttl	Push_Operation	Push Operation Procedure (Dummy)
	Page	+
;******************************************************************************
;
;	Push_Operation(Result_Stack)
;
;		Set carry flag indicating expression error
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Push_Operation	Proc	Near		; Push operation procedure
	stc				; Set carry indicating expression error
	ret				; Return to the caller
Push_Operation	Endp			; End of the Push_Operation procedure
	Subttl	Pop_Operation	Pop Operation Procedure (Dummy)
	Page	+
;******************************************************************************
;
;	Pop_Operation(Result_Stack)
;
;		Set carry flag indicating expression error
;		Return to the caller
;
;	Registers on Entry:
;
;		DI    - Pointer to private result stack
;
;	Registers on Exit:
;
;		FL    - Carry set if stack error (Expression error)
;
;******************************************************************************
		Even			; Force procedure to even address
Pop_Operation	Proc	Near		; Pop operation procedure
	stc				; Set carry indicating expression error
	ret				; Return to the caller
Pop_Operation	Endp			; End of the Pop_Operation procedure
;******************************************************************************
;
;	Define the register match and jump tables
;
;******************************************************************************
		Even			; Force byte to even address
Register_Table	Equ	This Byte	; Start of the register match table
		Db	4		; Size of match table entries
		Db	2,"SP "         ; Stack pointer register name
		Db	2,"PC "         ; Program counter register name
		Db	1,"ACC"         ; Accumulator register name
		Db	1,"X  "         ; X index register name
		Db	1,"Y  "         ; Y index register name
		Db	1,"FL "         ; Flags register name
		Db	0		; End of match table
Register_Jump	Equ	This Word	; Start of the register jump table
		Dw	Offset Get_SP	; Get stack pointer register value
		Dw	Offset Get_PC	; Get program counter register value
		Dw	Offset Get_ACC	; Get accumulator register value
		Dw	Offset Get_X	; Get X index register value
		Dw	Offset Get_Y	; Get Y index register value
		Dw	Offset Get_FL	; Get flags register value
;******************************************************************************
;
;	Define the numeric radix table
;
;******************************************************************************
		Even			; Force byte to even address
Radix_Table	Equ	This Word	; Start of the numeric radix table
		Dw	BINARY		; Binary base value (2)
		Dw	OCTAL		; Octal base value (8)
		Dw	DECIMAL 	; Decimal base value (10)
		Dw	HEX		; Hexadecimal base value (16)
;******************************************************************************
;
;	Define the input and stack precedence tables
;
;******************************************************************************
Input_Table	Equ	This Byte	; Start of input precedence table
		Db	01h		; Logical AND operation input precedence
		Db	01h		; Logical OR operation input precedence
		Db	01h		; Logical XOR operation input precedence
		Db	03h		; Shift right operation input precedence
		Db	03h		; Shift left operation input precedence
		Db	03h		; Modulus operation input precedence
		Db	05h		; Addition operation input precedence
		Db	05h		; Subtraction operation input precedence
		Db	07h		; Multiplication input precedence
		Db	07h		; Division operation input precedence
		Db	09h		; Logical NOT operation input precedence
		Db	09h		; Negation operation input precedence
		Db	0Dh		; Push operation input precedence
		Db	00h		; Pop operation input precedence
Stack_Table	Equ	This Byte	; Start of stack precedence table
		Db	02h		; Logical AND operation stack precedence
		Db	02h		; Logical OR operation stack precedence
		Db	02h		; Logical XOR operation stack precedence
		Db	04h		; Shift right operation stack precedence
		Db	04h		; Shift left operation stack precedence
		Db	04h		; Modulus operation stack precedence
		Db	06h		; Addition operation stack precedence
		Db	06h		; Subtraction operation stack precedence
		Db	08h		; Multiplication stack precedence
		Db	08h		; Division operation stack precedence
		Db	0Ah		; Logical NOT operation stack precedence
		Db	0Ah		; Negation operation stack precedence
		Db	00h		; Push operation stack precedence
		Db	00h		; Pop operation stack precedence
Rank_Table	Equ	This Byte	; Start of rank table
		Db	-1		; Logical AND operation rank value
		Db	-1		; Logical OR operation rank value
		Db	-1		; Logical XOR operation rank value
		Db	-1		; Shift right operation rank value
		Db	-1		; Shift left operation rank value
		Db	-1		; Modulus operation rank value
		Db	-1		; Addition operation rank value
		Db	-1		; Subtraction operation rank value
		Db	-1		; Multiplication operation rank value
		Db	-1		; Division operation rank value
		Db	0		; Logical NOT operation rank value
		Db	0		; Negation operation rank value
		Db	0		; Push operation rank value
		Db	0		; Pop operation rank value
;******************************************************************************
;
;	Define the operation table
;
;******************************************************************************
Op_Table	Equ	This Word	; Start of the operation table
		Dw	Offset AND_Operation
		Dw	Offset OR_Operation
		Dw	Offset XOR_Operation
		Dw	Offset Right_Operation
		Dw	Offset Left_Operation
		Dw	Offset Mod_Operation
		Dw	Offset Add_Operation
		Dw	Offset Sub_Operation
		Dw	Offset Mul_Operation
		Dw	Offset Div_Operation
		Dw	Offset NOT_Operation
		Dw	Offset Neg_Operation
		Dw	Offset Push_Operation
		Dw	Offset Pop_Operation
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the Express module
