	Page	58,132
	Title	APPLE.ASM	Apple Emulator (65C02 Processor)
;******************************************************************************
;
;   Name:	APPLE.ASM	Apple Emulator (65C02 Processor)
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the code for the Apple
;	specific part of the Apple emulator.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	Initialize		; Apple emulator initialization routine
	Public	Read_Memory		; Read 65C02 Apple memory routine
	Public	Write_Memory		; Write 65C02 Apple memory routine
	Public	System_Request		; System request routine       (Req.)
	Public	System_Reset		; Apple system reset routine   (Req.)
	Public	System_Debug		; System debug routine	       (Req.)
	Public	Break_System		; Apple break system routine   (Int.)
	Public	Trace_System		; Apple trace system routine   (Int.)
	Public	Debug_System		; Apple debug system routine   (Int.)
	Public	Reset_System		; Apple reset system routine   (Int.)
	Public	Request_System		; Apple request system routine (Int.)
	Public	Error			; Apple emulator error routine
	Public	Exit			; Apple emulator exit routine
;
;  External Declarations
;
	Extrn	Clear_Memory:Near	; Clear memory routine	       (MEMORY)
	Extrn	Configuration:Near	; Apple configuration routine  (CONFIG)
	Extrn	Video_Init:Near 	; Video initialization routine	(VIDEO)
	Extrn	Video_Reset:Near	; Video reset routine		(VIDEO)
	Extrn	CPU_Reset:Near		; CPU reset routine		(RESET)
	Extrn	BUS_Reset:Near		; Bus reset routine		(RESET)
	Extrn	Debug:Near		; Apple ][ system debug routine (DEBUG)
	Extrn	Dump_Code:Near		; Dump code listing routine	(DEBUG)
	Extrn	Dump_Data:Near		; Dump RAM data routine 	(DEBUG)
	Extrn	Dump_Stack:Near 	; Dump stack data routine	(DEBUG)
	Extrn	Print_Status:Near	; Print status routine		(DEBUG)
	Extrn	Print_Register:Near	; Print registers routine	(DEBUG)
	Extrn	Int_Init:Near		; Interrupt initialization	  (INT)
	Extrn	Int_Reset:Near		; Interrupt reset routine	  (INT)
	Extrn	Set_LED:Near		; Set keyboard LED routine   (KEYBOARD)
	Extrn	Key_Reset:Near		; Keyboard reset routine     (KEYBOARD)
	Extrn	Flush_Keyboard:Near	; Flush keyboard routine     (KEYBOARD)
	Extrn	Check_Key:Near		; Check for input routine    (KEYBOARD)
	Extrn	Get_Key:Near		; Get keyboard code routine  (KEYBOARD)
	Extrn	Last_Key:Byte		; Last keyboard scan code    (KEYBOARD)
	Extrn	Error_Table:Word	; Error string table		 (DATA)
	Extrn	System_ROM:Word 	; Apple system ROM file name	 (DATA)
	Extrn	Read_Table:Word 	; Read memory page table	 (DATA)
	Extrn	Write_Table:Word	; Write memory page table	 (DATA)
	Extrn	System_Flag:Byte	; Apple emulator system flag byte(DATA)
	Extrn	Emulate_Flag:Byte	; Emulator system flag byte	 (DATA)
	Extrn	RAM_Space:Word		; RAM space segment value	 (DATA)
	Extrn	ALT_Space:Word		; Alternate RAM space segment	 (DATA)
	Extrn	ERR_NO_MEMORY:Abs	; Not enough memory error code	 (DATA)
	Extrn	ERR_NO_SYSTEM_FILE:Abs	; No system ROM file error code  (DATA)
	Extrn	ERR_BAD_SYSTEM_FILE:Abs ; Bad system ROM file error code (DATA)
	Extrn	ERR_BAD_SYSTEM_IMAGE:abs; Bad system ROM image error code(DATA)
	Extrn	Code_Address:Word	; Current code dump address	(DEBUG)
	Extrn	Last_Code:Word		; Last code dump address	(DEBUG)
	Extrn	Code_Count:Word 	; Code breakpoints count value	(DEBUG)
	Extrn	Data_Count:Word 	; Data breakpoints count value	(DEBUG)
	Extrn	Opcode_Table:Byte	; Debugger opcode table 	(DEBUG)
	Extrn	Code_Table:Word 	; Code breakpoint address table (DEBUG)
	Extrn	Data_Table:Word 	; Data breakpoint address table (DEBUG)
	Extrn	Type_Table:Word 	; Debugger type table		(DEBUG)
	Extrn	Trace_Count:Word	; Trace count value storage area(DEBUG)
	Extrn	Value:Word		; Current value storage area	(DEBUG)
;
;  LOCAL Equates
;
BREAK_OFFSET	Equ	02h		; Breakpoint match offset value
SYSTEM_ADDRESS	Equ	0D000h		; Apple system ROM load address
SYSTEM_SIZE	Equ	03000h		; Apple system ROM size (Bytes)
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	Include 	Strucs.inc	; Include the structure definitions
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	Initialize	Initialize the Apple Emulator
	Page	+
;******************************************************************************
;
;	Initialize()
;
;		Clear direction flag (All moves are forward)
;		Try to allocate the 65C02 RAM space (64k)
;		If able to allocate the RAM space
;			Clear the Apple ][ hardware and ROM memory areas
;			Call routine to load the Apple system ROM
;			Call routine to handle Apple configuration
;			Clear the Apple ][ system RAM area
;			Call routine to initialize interrupts
;			Call routine to initialize video system
;			Call routine to initilize keyboard LED's
;			Try to allocate the alternate 65C02 RAM space (64k)
;			If able to allocate the RAM space
;				Save the alternate RAM space segment
;				Call routine to initialize memory
;				Call routine to setup 65C02 registers (Reset)
;			Else unable to allocate the needed RAM space
;				Set error code to not enough memory
;				Call the error routine
;				Call routine to exit the emulator
;			Endif
;		Else unable to allocate the needed RAM space
;			Set error code to not enough memory
;			Call the error routine
;			Call routine to exit the emulator
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		DL    - Accumulator initialized        (0h)
;		DH    - 65C02 flags initialized        (0h)
;		CL    - Y index register initialized   (0h)
;		CH    - X index register initialized   (0h)
;		BX    - Stack pointer initialized    (1FFh)
;		SI    - Program counter initialized (FFFCh)
;		DS    - 65C02 RAM space
;		ES    - Video RAM segment
;
;******************************************************************************
		Even			; Force procedure to even address
Initialize	Proc	Near		; Emulator initialization procedure
	cld				; Clear direction flag (All forward)
	mov	ah,ALLOCATE_MEMORY	; Get the allocate memory function code
	mov	bx,RAM_SIZE		; Get number of paragraphs to allocate
	int	DOS			; Try to allocate the 65C02 RAM space
	jnc	Continue		; Jump if no errors allocating space
Memory_Error:
	mov	al,ERR_NO_MEMORY	; Get not enough memory error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit the emulator
Continue:
	mov	ds,ax			; Setup RAM space segment address
	mov	cs:[RAM_Space],ax	; Save the RAM space segment address
	mov	ah,ALLOCATE_MEMORY	; Get the allocate memory function code
	mov	bx,RAM_SIZE		; Get number of paragraphs to allocate
	int	DOS			; Try to allocate the 65C02 RAM space
	jc	Memory_Error		; Jump if errors allocating space
	mov	cs:[ALT_Space],ax	; Save alternate RAM space segment
	mov	ds,ax			; Setup alternate RAM space segment
	mov	si,Hardware_Area	; Get pointer to start of hardware area
	mov	cx,(Size Hardware_Area) + (Size ROM_Area)
	call	Clear_Memory		; Call routine to clear memory
	mov	ds,cs:[RAM_Space]	; Setup the 65C02 RAM space segment
	mov	si,Hardware_Area	; Get pointer to start of hardware area
	mov	cx,(Size Hardware_Area) + (Size ROM_Area)
	call	Clear_Memory		; Call routine to clear the memory
	call	Load_System		; Call routine to load system ROM
	call	Configuration		; Call routine to handle configuration
	mov	si,RAM_Area		; Get pointer to start of RAM area
	mov	cx,Size RAM_Area	; Get the number of bytes in RAM area
	call	Clear_Memory		; Call routine to clear the memory
	mov	ds,cs:[ALT_Space]	; Setup alternate 65C02 RAM segment
	mov	si,RAM_Area		; Get pointer to start of RAM area
	mov	cx,Size RAM_Area	; Get the number of bytes in RAM area
	call	Clear_Memory		; Call routine to clear the memory
	mov	ds,cs:[RAM_Space]	; Setup the 65C02 RAM space segment
	call	Int_Init		; Call routine to initialize interrupts
	call	Video_Init		; Call routine to initialize video
	call	Set_LED 		; Call routine to initialize LED's
	call	CPU_Reset		; Call routine to reset the 65C02
	ret				; Return to the caller
Initialize	Endp			; End of the Initialize procedure
	Subttl	Load_System	Load the Apple System ROM
	Page	+
;******************************************************************************
;
;	Load_System(RAM_Space)
;
;		Save the required registers
;		Try to open the Apple System ROM data file
;		If no errors opening the System ROM file
;			Try to read in the System ROM image
;			If errors reading the System ROM
;				Set error code to bad System ROM file
;				Call the error routine
;				Call routine to exit the emulator
;			Endif
;			Close the System ROM file
;		Else errors opening the System ROM file
;			Set error code to no System ROM file
;			Call the error routine
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Copy ROM image to alternate RAM segment
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Load_System	Proc	Near		; Load System ROM procedure
	Save	es			; Save the required registers
	mov	ax,cs			; Get the current code segment value
	mov	bx,ds			; Save 65C02 RAM space segment value
	mov	ds,ax			; Setup to open the system ROM file
	mov	ah,OPEN_FILE		; Get the open file function code
	mov	al,READ_ONLY		; Get the read file access code
	lea	dx,cs:[System_ROM]	; Get pointer to system ROM file name
	int	DOS			; Try to open the system ROM file
	jnc	Read_System_ROM 	; Jump if no errors opening file
	mov	al,ERR_NO_SYSTEM_FILE	; Get no system ROM file error code
System_ROM_Error:
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Read_System_ROM:
	mov	ds,bx			; Restore 65C02 RAM space segment value
	mov	bx,ax			; Move file handle to BX register
	mov	ah,READ_FILE		; Get read file function code
	mov	dx,SYSTEM_ADDRESS	; Setup the data buffer address
	mov	cx,SYSTEM_SIZE		; Get the system ROM image size (12k)
	int	DOS			; Try to read the system ROM image
	mov	cx,ax			; Save ROM image size in CX register
	mov	al,ERR_BAD_SYSTEM_FILE	; Get bad system ROM file error code
	jc	System_ROM_Error	; Jump if error trying to read the file
	cmp	cx,SYSTEM_SIZE		; Check for all of image read in
	mov	al,ERR_BAD_SYSTEM_IMAGE ; Get bad system ROM image error code
	jne	System_ROM_Error	; Jump if part of image missing
	mov	ah,CLOSE_FILE		; Get close file function code
	int	DOS			; Close the system ROM file
	mov	es,cs:[ALT_Space]	; Get alternate RAM space segment
	mov	si,SYSTEM_ADDRESS	; Get pointer to start of system ROM
	mov	di,si			; Setup to copy the system ROM
	mov	cx,SYSTEM_SIZE Shr 1	; Get the size of the system ROM
	rep	movsw			; Copy system ROM image to alternate RAM
	Restore es			; Restore the required registers
	ret				; Return to the caller
Load_System	Endp			; End of the Load_System procedure
	Subttl	Error		Emulator Error Routine
	Page	+
;******************************************************************************
;
;	Error(Error_Code)
;
;		Save the required registers
;		Print the correct error message
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Error code
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Error		Proc	Near		; Emulator error procedure
	Save	ds			; Save the required registers
	xor	ah,ah			; Convert error code to full word
	shl	ax,1			; Convert error code to table index
	mov	bx,ax			; Move table index into BX register
	mov	si,cs:[bx+Error_Table]	; Get the error string address
	mov	ax,cs			; Get current CS register value
	mov	ds,ax			; Setup to access the error string
	mov	ah,WRITE_FILE		; Get write file function code
	mov	bx,STD_ERROR		; Get the standard error handle
	lodsb				; Get the length of the error string
	mov	cl,al			; Move string length into CL register
	xor	ch,ch			; Convert string length into full word
	mov	dx,si			; Setup error string address
	int	DOS			; Print the error string to handle
	Restore ds			; Restore the required registers
	ret				; Return to the caller
Error		Endp			; End of the Error procedure
	Subttl	Exit		Emulator Exit Routine
	Page	+
;******************************************************************************
;
;	Exit()
;
;		Call routine to reset interrupt vectors
;		Call routine to reset video
;		Call routine to reset keyboard
;		Zero the return code
;		Exit to DOS
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Exit		Proc	Near		; Emulator exit procedure
	call	Int_Reset		; Call the interrupt reset routine
	call	Video_Reset		; Call the video reset routine
	call	Key_Reset		; Call the keyboard reset routine
	mov	ah,TERMINATE		; Get the terminate function code
	xor	al,al			; Zero the return code
	int	DOS			; Return control to MS-DOS
	ret				; Return to the caller (Never executed)
Exit		Endp			; End of the Exit procedure
	Subttl	Read_Memory	Read 65C02 Apple Memory Routine
	Page	+
;******************************************************************************
;
;	Read_Memory(Effective_Address)
;
;		Get the memory page from effective address
;		Jump to correct routine to process read
;
;	Registers on Entry:
;
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		AL    - 65C02 Memory value
;		BP    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Read_Memory	Proc	Near		; Read 65C02 Apple memory procedure
	mov	bp,di			; Get the effective address
	shr	bp,8			; Get page number into BP register
	shl	bp,1			; Convert page number to table index
	jmp	cs:[bp + Read_Table]	; Jump to correct routine to read
	ret				; Return to the caller (Never executed)
Read_Memory	Endp			; End of the Read_Memory procedure
	Subttl	Write_Memory	Write 65C02 Apple Memory Routine
	Page	+
;******************************************************************************
;
;	Write_Memory(Effective_Address, Data_Value)
;
;		Get the memory page from effective address
;		Jump to correct routine to process write
;
;	Registers on Entry:
;
;		DS:DI - 65C02 Effective address
;		AL    - 65C02 Memory value
;
;	Registers on Exit:
;
;		AX    - Destroyed
;		DI    - Destroyed
;		BP    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Write_Memory	Proc	Near		; Write 65C02 Apple memory procedure
	mov	bp,ax			; Save memory value in BP register
	mov	ax,di			; Get the effective address
	shr	ax,8			; Get page number into AX register
	shl	ax,1			; Convert page number to table index
	xchg	bp,ax			; Setup to jump to correct routine
	jmp	cs:[bp + Write_Table]	; Jump to correct routine to write
	ret				; Return to the caller (Never executed)
Write_Memory	 Endp			; End of the Write_Memory procedure
	Subttl	System_Request	Emulator System Request Routine (Request)
	Page	+
;******************************************************************************
;
;	System_Request(Scan_Code)
;
;		If this is a make code
;			If not already in the system
;				Set system request interrupt type
;			Endif
;		Endif this is a break code
;		Return control to the caller
;
;	Registers on Entry:
;
;		AH    - Scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
System_Request	Proc	Near		; Emulator system request procedure
	or	ah,ah			; Check for a make or break code
	js	Request_Done		; Jump if this is a break code
	test	cs:[Emulate_Flag],SYS_REQUEST
	jnz	Request_Done		; Jump if already in system procedure
	or	cs:[Emulate_Flag],SYS_REQUEST
Request_Done:
	ret				; Return to the caller
System_Request	 Endp			; End of the System_Request procedure
	Subttl	System_Reset	Apple System Reset Routine (Request)
	Page	+
;******************************************************************************
;
;	System_Reset(Scan_Code)
;
;		If the last scan code does NOT match (NOT a repeat)
;			If this is a make code
;				Set system reset interrupt type
;			Endif this is a break code
;		Endif this key is repeating
;		Return control to the caller
;
;	Registers on Entry:
;
;		AH    - Scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
System_Reset	Proc	Near		; Apple system reset procedure
	cmp	ah,cs:[Last_Key]	; Check for a repeat scan code
	je	Reset_Done		; Jump if this key is repeating
	or	ah,ah			; Check for a make or break code
	js	Reset_Done		; Jump if this is a break code
	or	cs:[Emulate_Flag],SYS_RESET
Reset_Done:
	ret				; Return to the caller
System_Reset	 Endp			; End of the System_Reset procedure
	Subttl	System_Debug	Emulator System Debug Routine (Request)
	Page	+
;******************************************************************************
;
;	System_Debug(Scan_Code)
;
;		If this is a make code
;			Set system debug interrupt type
;			If system is in trace mode
;				Turn trace mode off
;			Endif
;		Endif this is a break code
;		Return control to the caller
;
;	Registers on Entry:
;
;		AH    - Scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
System_Debug	Proc	Near		; Emulator system debug procedure
	or	ah,ah			; Check for a make or break code
	js	Debug_Done		; Jump if this is a break code
	or	cs:[Emulate_Flag],SYS_DEBUG
	test	cs:[Emulate_Flag],SYS_TRACE
	jz	Debug_Done		; Jump if not in trace mode
	and	cs:[Emulate_Flag],Not SYS_TRACE
Debug_Done:
	ret				; Return to the caller
System_Debug	 Endp			; End of the System_Debug procedure
	Subttl	Break_System	Break System Routine (Interrupt)
	Page	+
;******************************************************************************
;
;	Break_System(Interrupt_Number, Pending, RAM_Space, Registers, Address)
;
;		Save the required registers
;
;
;		Restore the required registers
;		Return control to the caller
;
;	Registers on Entry:
;
;		AH    - Interrupt number
;		AL    - Pending interrupt flags
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DI    - Current opcode address
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Break_System	Proc	Near		; Break system procedure
	Save	ax,cx,es		; Save the required registers (Minimum)
	mov	ax,cs			; Setup access to
	mov	es,ax			;		  the data segment
	mov	cs:[Value],cx		; Save the original X/Y register values
Code_Check:
	mov	cx,cs:[Code_Count]	; Get the number of code breakpoints
	jcxz	Data_Check		; Jump if no code breakpoints to check
	push	di			; Save the current opcode address
	lea	di,ds:[Code_Table]	; Get pointer to the code address table
	mov	ax,si			; Get the actual
	dec	ax			;		 65C02 program counter
	repne	scasw			; Check for a breakpoint match
	pop	di			; Restore the current opcode address
	jne	Data_Check		; Jump if no breakpoint match
	Restore ax,cx,es		; Clear the break stack frame (Minimum)
	call	Debug			; Break out to the system debugger
	Save	ax,cx,es		; Restore break stack frame (Minimum)
Data_Check:
	mov	cx,cs:[Data_Count]	; Get the number of data breakpoints
	jcxz	Break_Done		; Jump if no data breakpoints to check
	mov	cx,cs:[Value]		; Restore original X/Y register values
	push	di			; Save the current opcode address
	mov	al,Byte Ptr ds:[si - 1] ; Get the current opcode value
	mov	ah,Size Opcode_Debug	; Get size of opcode debug structure
	mul	ah			; Compute the offset to opcode entry
	mov	di,ax			; Setup to access the opcode table
	mov	al,cs:[Opcode_Table + di.Op_Address]
	mov	cs:[Value],ax		; Save the addressing type value
	and	ax,ADDRESS_MASK 	; Mask off all but the addressing type
	mov	di,ax			; Setup to access the effect table
	call	cs:[Effect_Table + di]	; Call routine to get effective address
	jc	No_Match		; Jump if no effective address
	mov	cx,cs:[Data_Count]	; Get the number of data breakpoints
	lea	di,ds:[Data_Table]	; Get pointer to the data address table
	repne	scasw			; Check for a breakpoint match
	jne	No_Match		; Jump if no breakpoint match
	lea	ax,ds:[Data_Table]	; Get pointer to the data address table
	sub	di,ax			; Compute breakpoint
	sub	di,BREAK_OFFSET 	;		     match offset
	mov	ax,cs:[Value]		; Restore the addressing type value
	and	ax,Not ADDRESS_MASK	; Mask off the addressing mode
	test	ax,cs:[Type_Table + di] ; Check for correct type value
	jz	No_Match		; Jump if incorrect type value
	pop	di			; Restore the current opcode address
	Restore ax,cx,es		; Clear the break stack frame (Minimum)
	call	Debug			; Break out to the system debugger
	Save	ax,cx,es		; Restore break stack frame (Minimum)
	jmp	Short Break_Done	; Go finish the breakpoint interrupt
No_Match:
	pop	di			; Restore the current opcode address
Break_Done:
	test	cs:[Emulate_Flag],SYS_BREAK
	jz	Break_Exit		; Jump if no more break requests
	or	dh,CPU_R		; Set emulator interrupt flag bit
Break_Exit:
	Restore ax,cx,es		; Restore the required registers
	ret				; Return control to the caller
Break_System	Endp			; End of the Break_System procedure
	Subttl	Trace_System	Trace System Routine (Interrupt)
	Page	+
;******************************************************************************
;
;	Trace_System(Interrupt_Number, Pending, RAM_Space, Registers, Address)
;
;		If there is keyboard input
;			Set trace count to one (Cause break)
;		Endif
;		Decrement the trace counter
;		If more traces to perform (Count non-zero)
;			Setup the Apple ][ system debugger stack frame
;			Call routine to dump the data values
;			Call routine to print the status line
;			Setup the current 65C02 code address
;			Call routine to dump the code at current address
;			Call routine to print the 65C02 registers
;			Call routine to dump the 65C02 stack
;			Remove the Apple ][ system debugger stack frame
;			Set the emulator interrupt flag bit
;		Else trace break time (Count zero)
;			Call the Apple ][ system debugger routine
;		Endif
;		Return control to the caller
;
;	Registers on Entry:
;
;		AH    - Interrupt number
;		AL    - Pending interrupt flags
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DI    - Current opcode address
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Trace_System	Proc	Near		; Trace system procedure
	call	Check_Key		; Call routine to check for input
	jz	Trace_Continue		; Jump if no keyboard input
	mov	cs:[Trace_Count],1	; Setup to exit the trace
Trace_Continue:
	dec	cs:[Trace_Count]	; Decrement the trace count value
	jz	Trace_Break		; Jump if no more traces to perform
	Save	ax,bx,cx,dx,si,di,bp	; Setup the debugger stack frame
	mov	bp,sp			; Setup access to the stack frame
	Save	ds,es			; Save the emulator segment values
	mov	ax,cs			; Setup to access
	mov	ds,ax			;		  the data segment
	mov	ax,ss:[bp.SI_Register]	; Get the 65C02
	dec	ax			;		program counter (SI-1)
	cmp	ax,ds:[Last_Code]	; Check against last code address
	jb	Code_Update		; Jump if outside last code dump
	cmp	ax,ds:[Code_Address]	; Check against last code address
	jae	Code_Update		; Jump if outside last code dump
	mov	ax,ds:[Last_Code]	; Use last code dump address value
Code_Update:
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Dump_Data		; Call routine to dump the data
	call	Print_Status		; Call routine to print status line
	call	Dump_Code		; Call routine to dump the code
	call	Print_Register		; Call routine to print register line
	call	Dump_Stack		; Call routine to dump the stack
	Restore ds,es			; Restore the emulator segment values
	Restore ax,bx,cx,dx,si,di,bp	; Remove the debugger stack frame
	or	dh,CPU_R		; Set emulator interrupt flag bit
	jmp	Short Trace_Exit	; Go return control to the caller
Trace_Break:
	and	cs:[Emulate_Flag],Not SYS_TRACE
	call	Debug			; Break out to the system debugger
Trace_Exit:
	ret				; Return control to the caller
Trace_System	Endp			; End of the Trace_System procedure
	Subttl	Debug_System	Debug System Routine (Interrupt)
	Page	+
;******************************************************************************
;
;	Debug_System(Interrupt_Number, Pending, RAM_Space, Registers, Address)
;
;		Clear the system debug interrupt bit
;		Call the Apple ][ system debugger routine
;		Return control to the caller
;
;	Registers on Entry:
;
;		AH    - Interrupt number
;		AL    - Pending interrupt flags
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DI    - Current opcode address
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Debug_System	Proc	Near		; Debug system procedure
	and	cs:[Emulate_Flag],Not SYS_DEBUG
	call	Debug			; Call the Apple ][ debug system
Debug_Exit:
	ret				; Return control to the caller
Debug_System	Endp			; End of the Debug_System procedure
	Subttl	Request_System	Request System Routine (Interrupt)
	Page	+
;******************************************************************************
;
;	Request_System(Interrupt_Number, Pending, RAM_Space, Registers, Address)
;
;		Clear the system request interrupt bit
;
;		Return control to the caller
;
;	Registers on Entry:
;
;		AH    - Interrupt number
;		AL    - Pending interrupt flags
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DI    - Current opcode address
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Request_System	Proc	Near		; Request system procedure
	and	cs:[Emulate_Flag],Not SYS_REQUEST

	call	Exit			; Call routine to exit the emulator

	ret				; Return control to the caller
Request_System	Endp			; End of the Request_System procedure
	Subttl	Reset_System	Reset System Routine (Interrupt)
	Page	+
;******************************************************************************
;
;	Reset_System(Interrupt_Number, Pending, RAM_Space, Registers, Address)
;
;		Save the required registers
;		Clear the system reset interrupt bit
;		Call routine to reset the processor registers
;		Setup to execute the next opcode
;		Restore the required registers
;		Return control to the caller
;
;	Registers on Entry:
;
;		AH    - Interrupt number
;		AL    - Pending interrupt flags
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DI    - Current opcode address
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		DL    - Accumulator initialized        (0h)
;		DH    - 65C02 flags initialized        (0h)
;		CL    - Y index register initialized   (0h)
;		CH    - X index register initialized   (0h)
;		BX    - Stack pointer initialized    (1FFh)
;		SI    - Program counter initialized (FFFCh)
;		DI    - Set to new opcode address
;
;******************************************************************************
		Even			; Force procedure to even address
Reset_System	Proc	Near		; Reset system procedure
	Save	ax			; Save the required registers
	and	cs:[Emulate_Flag],Not SYS_RESET
	call	BUS_Reset		; Call routine to reset Apple ][ bus
	call	CPU_Reset		; Call routine to reset 65C02 processor
	Setup				; Setup the new opcode address
	Restore ax			; Restore the required registers
	ret				; Return control to the caller
Reset_System	Endp			; End of the Reset_System procedure
	Subttl	Compute_BAD	Bad Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_BAD(Registers)
;
;		Set carry indicating no effective address
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_BAD	Proc	Near		; Bad addressing mode procedure
	stc				; Set carry indicating no address
	ret				; Return to the caller
Compute_BAD	Endp			; End of the Compute_BAD procedure
	Subttl	Compute_NONE	No Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_NONE(Registers)
;
;		Set carry indicating no effective address
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_NONE	Proc	Near		; No addressing mode procedure
	stc				; Set carry indicating no address
	ret				; Return to the caller
Compute_NONE	Endp			; End of the Compute_NONE procedure
	Subttl	Compute_IMM	Immediate Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_IMM(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_IMM	Proc	Near		; Immediate addressing mode procedure
	Save	si,di			; Save the required registers
	DoImm				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_IMM	Endp			; End of the Compute_IMM procedure
	Subttl	Compute_ABS	Absolute Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_ABS(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_ABS	Proc	Near		; Absolute addressing mode procedure
	Save	si,di			; Save the required registers
	DoAbs				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_ABS	Endp			; End of the Compute_ABS procedure
	Subttl	Compute_DP	Direct Page Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_DP(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_DP	Proc	Near		; Direct page addressing mode procedure
	Save	si,di			; Save the required registers
	DoDP				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_DP	Endp			; End of the Compute_DP procedure
	Subttl	Compute_DIX	Direct Page Indexed X Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_DIX(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_DIX	Proc	Near		; Direct page indexed X procedure
	Save	si,di			; Save the required registers
	DoDIX				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_DIX	Endp			; End of the Compute_DIX procedure
	Subttl	Compute_DIY	Direct Page Indexed Y Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_DIY(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_DIY	Proc	Near		; Direct page indexed Y procedure
	Save	si,di			; Save the required registers
	DoDIX				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_DIY	Endp			; End of the Compute_DIY procedure
	Subttl	Compute_AIX	Absolute Indexed X Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_AIX(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_AIX	Proc	Near		; Absolute indexed X procedure
	Save	si,di			; Save the required registers
	DoAIX				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_AIX	Endp			; End of the Compute_AIX procedure
	Subttl	Compute_AIY	Absolute Indexed Y Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_AIY(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_AIY	Proc	Near		; Absolute indexed Y procedure
	Save	si,di			; Save the required registers
	DoAIY				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_AIY	Endp			; End of the Compute_AIY procedure
	Subttl	Compute_AI	Absolute Indirect Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_AI(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_AI	Proc	Near		; Absolute indirect procedure
	Save	si,di			; Save the required registers
	DoAI				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_AI	Endp			; End of the Compute_AI procedure
	Subttl	Compute_DI	Direct Page Indirect Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_DI(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_DI	Proc	Near		; Direct page indirect procedure
	Save	si,di			; Save the required registers
	DoDI				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_DI	Endp			; End of the Compute_DI procedure
	Subttl	Compute_AIIX	Absolute Indexed Indirect X Addressing Routine
	Page	+
;******************************************************************************
;
;	Compute_AIIX(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_AIIX	Proc	Near		; Absolute indexed indirect X procedure
	Save	si,di			; Save the required registers
	DoAIIX				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_AIIX	Endp			; End of the Compute_AIIX procedure
	Subttl	Compute_DIIX	Direct Page Indexed Indirect X Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_DIIX(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_DIIX	Proc	Near		; Direct indexed indirect X procedure
	Save	si,di			; Save the required registers
	DoDIIX				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_DIIX	Endp			; End of the Compute_DIIX procedure
	Subttl	Compute_DIIY	Direct Page Indexed Indirect Y Mode Routine
	Page	+
;******************************************************************************
;
;	Compute_DIIY(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_DIIY	Proc	Near		; Direct indexed indirect Y procedure
	Save	si,di			; Save the required registers
	DoDIIY				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_DIIY	Endp			; End of the Compute_DIIY procedure
	Subttl	Compute_PCR	Program Counter Relative Addressing Routine
	Page	+
;******************************************************************************
;
;	Compute_PCR(Registers)
;
;		Save the required registers
;		Compute the effective address value
;		Clear carry indicating effective address
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX    - Effective address
;
;******************************************************************************
		Even			; Force procedure to even address
Compute_PCR	Proc	Near		; PC relative addressing procedure
	Save	si,di			; Save the required registers
	DoPCR				; Compute the effective address
	mov	ax,di			; Save the effective address value
	clc				; Clear carry indicating address
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Compute_PCR	Endp			; End of the Compute_PCR procedure
;******************************************************************************
;
;	Define the effective address table
;
;******************************************************************************
Effect_Table	Equ	This Word	; Start of effective address jump table
	Dw	Compute_BAD		; Illegal opcode addressing type
	Dw	Compute_NONE		; No addressing type
	Dw	Compute_IMM		; Immediate addressing type
	Dw	Compute_ABS		; Absolute addressing type
	Dw	Compute_DP		; Direct page addressing type
	Dw	Compute_DIX		; Direct page indexed with X type
	Dw	Compute_DIY		; Direct page indexed with Y type
	Dw	Compute_AIX		; Absolute indexed with X type
	Dw	Compute_AIY		; Absolute indexed with Y type
	Dw	Compute_AI		; Absolute indirect addressing type
	Dw	Compute_DI		; Direct page indirect addressing type
	Dw	Compute_AIIX		; Absolute indexed indirect X type
	Dw	Compute_DIIX		; Direct page indexed indirect X type
	Dw	Compute_DIIY		; Direct page indexed indirect Y type
	Dw	Compute_PCR		; Program counter relative type
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the Apple module
