/*********************************************************************
 *                                                                   *
 *                     XGS : Apple IIGS Emulator                     *
 *                                                                   *
 *        Written and Copyright (C)1996 by Joshua M. Thompson        *
 *                                                                   *
 *  You are free to distribute this code for non-commercial purposes *
 * I ask only that you notify me of any changes you make to the code *
 *     Commercial use is prohibited without my written permission    *
 *                                                                   *
 *********************************************************************/
/*********************************************************************
 *                                                                   *
 *                     HPPA Sound Emulation                          *
 *                                                                   *
 *  Written and Copyright (C)1997 by Cornelius Cook, c-cook@uiuc.edu *
 *          Thanks to Michael Hipp for HPUX code examples            *
 *                     Copy Policy: GPL V2.0                         *
 *                                                                   *
 *********************************************************************/

/*
 * File: arch/hppa/snd-drv.c
 *
 * The HPUX raw /dev/audio sound output driver.
 */

#include "xgs.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include <sys/audio.h>
#include "sound.h"
#include "snd-drv.h"

/* Where should the sound go to?  "AUDIO_TO_*" are #define'd in snd-drv.h */
#ifdef AUDIO_TO_LINEOUT
#define AUDIO_TO AUDIO_OUT_LINEOUT
#elif AUDIO_TO_SPEAKER
#define AUDIO_TO AUDIO_OUT_SPEAKER
#elif AUDIO_TO_HEADPHONE
#define AUDIO_TO AUDIO_OUT_HEADPHONE
#else
#define AUDIO_TO -1
#endif

/* Hard code what stereo is to HPUX */
#define AUDIO_STEREO 2

/* for the /dev/audio file */
static int audiofd;
/* for holding the converted audio streams */
static unsigned char snd_out_buffer[OUTPUT_BUFFER_SIZE*4];

/* any errors during audio init should do these four things ... */
int SND_ERROR(char * msg) {
	perror(msg);
	if (audiofd) close(audiofd);
	audiofd = -1;
	return 0;
}

int SND_outputInit(int rate)
{
  struct audio_describe ades;
  struct audio_gain again;
  int gain,i,audio;
  long closest=0;

  audiofd = open("/dev/audio",O_RDWR);

  if(audiofd < 0)
    return 0;

  /* get info from the device */
  if (ioctl(audiofd,AUDIO_DESCRIBE,&ades)<0)
	return SND_ERROR("hppa sound AUDIO_DESCRIBE");

  /* get the current gains */
  again.channel_mask = AUDIO_CHANNEL_0 | AUDIO_CHANNEL_1;
  if (ioctl(audiofd,AUDIO_GET_GAINS,&again)<0)
	return SND_ERROR("hppa sound AUDIO_GET_GAINS");

  /* set to half maximum gain */
  /* leave volume unchanged */
  again.cgain[0].transmit_gain = ades.max_transmit_gain / 2;
  again.cgain[1].transmit_gain = ades.max_transmit_gain / 2;
  again.channel_mask = AUDIO_CHANNEL_0 | AUDIO_CHANNEL_1;
  if (ioctl(audiofd,AUDIO_SET_GAINS,&again)<0) 
	return SND_ERROR("hppa sound AUDIO_SET_GAINS");

  /* if there is a desired destination, set it */
  if(AUDIO_TO != -1)
  {
	if (ioctl(audiofd,AUDIO_SET_OUTPUT,AUDIO_TO)<0)
		return SND_ERROR("hppa sound AUDIO_SET_OUTPUT");
  }

  /* find the closest sampling rate available */
  for(i=0;i<ades.nrates;i++)
  {
    if (labs(closest - rate) >= labs(ades.sample_rate[i] - rate))
	closest = ades.sample_rate[i];
    if(rate == ades.sample_rate[i])
      break;
  }
  if(i == ades.nrates)
  {
    fprintf(stderr,"hppa sound: sample-rate %ld not available.\nusing closest available rate: %ld.\n",rate,closest);
    /*rate=closest;*/
  }

  /* set the format to signed 16 bit */
  if (ioctl(audiofd,AUDIO_SET_DATA_FORMAT,AUDIO_FORMAT_LINEAR16BIT)<0)
	return SND_ERROR("hppa sound AUDIO_SET_DATA_FORMAT: AUDIO_FORMAT_LINEAR16BIT");

  /* might need to set the channels twice ?? */
  for (i=0;i<2;i++)
  if (ioctl(audiofd,AUDIO_SET_CHANNELS,AUDIO_STEREO)<0)
	return SND_ERROR("hppa sound AUDIO_SET_CHANNELS: AUDIO_STEREO");

  /* set the sampling rate */
  if (ioctl(audiofd,AUDIO_SET_SAMPLE_RATE,closest)<0)
	return SND_ERROR("hppa sound AUDIO_SET_SAMPLE_RATE");
  
  /* inform API what the actual rate is now */
  return closest;
}


size_t SND_outputWrite(snd_sample_struct *buffer, size_t len)
{
  int i;
  unsigned char * ptr=snd_out_buffer;

  /* did we initialize?  if not, we can't play sound */
  if (audiofd==-1) return 0;

  for (i = 0 ; i < len ; i++) {
	/* convert the ENDIAN-ness of the sound for the HP */
		*(ptr++) = (unsigned char)((buffer[i].left >> 10) & 0xff);
		*(ptr++) = (unsigned char)((buffer[i].left >> 2) & 0x3f);
		*(ptr++) = (unsigned char)((buffer[i].right >> 10) & 0xff);
		*(ptr++) = (unsigned char)((buffer[i].right >> 2) & 0x3f);
  }
  write(audiofd,(unsigned char *)snd_out_buffer, len*2);
  return len;
}

void SND_outputShutdown(void)
{
  if (audiofd) close (audiofd);
}
