/*********************************************************************
 *                                                                   *
 *                     XGS : Apple IIGS Emulator                     *
 *                                                                   *
 *        Written and Copyright (C)1996 by Joshua M. Thompson        *
 *                                                                   *
 *  You are free to distribute this code for non-commercial purposes *
 * I ask only that you notify me of any changes you make to the code *
 *     Commercial use is prohibited without my written permission    *
 *                                                                   *
 *********************************************************************/

/*
 * File: memory.c
 *
 * Memory management routines.
 */

#include "xgs.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef EMX_HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif 	/* end of #ifdef EMX_HAVE_SYS_TYPES_H */
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include "adb.h"
#include "clock.h"
#include "disks.h"
#include "emul.h"
#include "iwm.h"
#include "smtport.h"
#include "sound.h"
#include "video.h"

int	mem_ramsize;

int	mem_shadow_text;
int	mem_shadow_hires1;
int	mem_shadow_hires2;
int	mem_shadow_super;
int	mem_shadow_aux;
int	mem_shadow_lc;

int	mem_slot_reg[8];

int	mem_lcbank2;
int	mem_lcread;
int	mem_lcwrite;
int	mem_lcsecond;

int	mem_auxrd;
int	mem_auxwrt;
int	mem_altzp;
int	mem_80store;

int	mem_intcxrom;
int	mem_slotc3rom;
int	mem_rombank;

int	mem_diagtype;

int	mem_rom03;

byte	*slow_memory;
byte	*fast_memory;
byte	*rom_memory;
byte	*save_memory;
byte	*c07x_memory;

byte	junk_memory[256];

mem_pagestruct	mem_pages[65536];

word32	mem_slowram_changed[512];

const word32	mem_change_masks[256] = {
	0x00000001, 0x00000001, 0x00000001, 0x00000001, 0x00000001, 0x00000001, 0x00000001, 0x00000001,
	0x00000002, 0x00000002, 0x00000002, 0x00000002, 0x00000002, 0x00000002, 0x00000002, 0x00000002, 
	0x00000004, 0x00000004, 0x00000004, 0x00000004, 0x00000004, 0x00000004, 0x00000004, 0x00000004,
	0x00000008, 0x00000008, 0x00000008, 0x00000008, 0x00000008, 0x00000008, 0x00000008, 0x00000008,
	0x00000010, 0x00000010, 0x00000010, 0x00000010, 0x00000010, 0x00000010, 0x00000010, 0x00000010,
	0x00000020, 0x00000020, 0x00000020, 0x00000020, 0x00000020, 0x00000020, 0x00000020, 0x00000020,
	0x00000040, 0x00000040, 0x00000040, 0x00000040, 0x00000040, 0x00000040, 0x00000040, 0x00000040,
	0x00000080, 0x00000080, 0x00000080, 0x00000080, 0x00000080, 0x00000080, 0x00000080, 0x00000080,
	0x00000100, 0x00000100, 0x00000100, 0x00000100, 0x00000100, 0x00000100, 0x00000100, 0x00000100,
	0x00000200, 0x00000200, 0x00000200, 0x00000200, 0x00000200, 0x00000200, 0x00000200, 0x00000200,
	0x00000400, 0x00000400, 0x00000400, 0x00000400, 0x00000400, 0x00000400, 0x00000400, 0x00000400,
	0x00000800, 0x00000800, 0x00000800, 0x00000800, 0x00000800, 0x00000800, 0x00000800, 0x00000800,
	0x00001000, 0x00001000, 0x00001000, 0x00001000, 0x00001000, 0x00001000, 0x00001000, 0x00001000,
	0x00002000, 0x00002000, 0x00002000, 0x00002000, 0x00002000, 0x00002000, 0x00002000, 0x00002000,
	0x00004000, 0x00004000, 0x00004000, 0x00004000, 0x00004000, 0x00004000, 0x00004000, 0x00004000,
	0x00008000, 0x00008000, 0x00008000, 0x00008000, 0x00008000, 0x00008000, 0x00008000, 0x00008000,
	0x00010000, 0x00010000, 0x00010000, 0x00010000, 0x00010000, 0x00010000, 0x00010000, 0x00010000,
	0x00020000, 0x00020000, 0x00020000, 0x00020000, 0x00020000, 0x00020000, 0x00020000, 0x00020000,
	0x00040000, 0x00040000, 0x00040000, 0x00040000, 0x00040000, 0x00040000, 0x00040000, 0x00040000,
	0x00080000, 0x00080000, 0x00080000, 0x00080000, 0x00080000, 0x00080000, 0x00080000, 0x00080000,
	0x00100000, 0x00100000, 0x00100000, 0x00100000, 0x00100000, 0x00100000, 0x00100000, 0x00100000,
	0x00200000, 0x00200000, 0x00200000, 0x00200000, 0x00200000, 0x00200000, 0x00200000, 0x00200000,
	0x00400000, 0x00400000, 0x00400000, 0x00400000, 0x00400000, 0x00400000, 0x00400000, 0x00400000,
	0x00800000, 0x00800000, 0x00800000, 0x00800000, 0x00800000, 0x00800000, 0x00800000, 0x00800000,
	0x01000000, 0x01000000, 0x01000000, 0x01000000, 0x01000000, 0x01000000, 0x01000000, 0x01000000,
	0x02000000, 0x02000000, 0x02000000, 0x02000000, 0x02000000, 0x02000000, 0x02000000, 0x02000000,
	0x04000000, 0x04000000, 0x04000000, 0x04000000, 0x04000000, 0x04000000, 0x04000000, 0x04000000,
	0x08000000, 0x08000000, 0x08000000, 0x08000000, 0x08000000, 0x08000000, 0x08000000, 0x08000000,
	0x10000000, 0x10000000, 0x10000000, 0x10000000, 0x10000000, 0x10000000, 0x10000000, 0x10000000,
	0x20000000, 0x20000000, 0x20000000, 0x20000000, 0x20000000, 0x20000000, 0x20000000, 0x20000000,
	0x40000000, 0x40000000, 0x40000000, 0x40000000, 0x40000000, 0x40000000, 0x40000000, 0x40000000,
	0x80000000, 0x80000000, 0x80000000, 0x80000000, 0x80000000, 0x80000000, 0x80000000, 0x80000000,
};

/* Dispatch table for reads to page $C0xx I/O space	*/

byte (*mem_io_read_dispatch[256])(byte) = {
	ADB_readKey,		/* $C000 */
	MEM_set80store,		/* $C001 */
	MEM_clearAuxRd,		/* $C002 */
	MEM_setAuxRd,		/* $C003 */
	MEM_clearAuxWrt,	/* $C004 */
	MEM_setAuxWrt,		/* $C005 */
	MEM_clearIntCXROM,	/* $C006 */
	MEM_setIntCXROM,	/* $C007 */
	MEM_clearAltZP,		/* $C008 */
	MEM_setAltZP,		/* $C009 */
	MEM_clearSlotC3ROM,	/* $C00A */
	MEM_setSlotC3ROM,	/* $C00B */
	VID_clear80col,		/* $C00C */
	VID_set80col,		/* $C00D */
	VID_clearAltCh,		/* $C00E */
	VID_setAltCh,		/* $C00F */
	ADB_clearKey,		/* $C010 */
	MEM_getLCbank,		/* $C011 */
	MEM_getLCread,		/* $C012 */
	MEM_getAuxRd,		/* $C013 */
	MEM_getAuxWrt,		/* $C014 */
	MEM_getIntCXROM,	/* $C015 */
	MEM_getAltZP,		/* $C016 */
	MEM_getSlotC3ROM,	/* $C017 */
	MEM_get80store,		/* $C018 */
	EMUL_getVBL,		/* $C019 */
	VID_getText,		/* $C01A */
	VID_getMixed,		/* $C01B */
	VID_getPage2,		/* $C01C */
	VID_getHires,		/* $C01D */
	VID_getAltCh,		/* $C01E */
	VID_get80col,		/* $C01F */
	MEM_ioUnimp,		/* $C020 */
	MEM_ioUnimp,		/* $C021 */
	VID_getColorReg,	/* $C022 */
	VID_getVGCIntReg,	/* $C023 */
	ADB_readMouse,		/* $C024 */
	ADB_readModifiers,	/* $C025 */
	ADB_readCommand,	/* $C026 */
	ADB_readStatus,		/* $C027 */
	MEM_ioUnimp,		/* $C028 */
	VID_getNewVideo,	/* $C029 */
	MEM_ioUnimp,		/* $C02A */
	MEM_ioUnimp,		/* $C02B */
	MEM_ioUnimp,		/* $C02C */
	MEM_getSlotReg,		/* $C02D */
	VID_getVertCnt,		/* $C02E */
	VID_getHorzCnt,		/* $C02F */
	SND_clickSpeaker,	/* $C030 */
	IWM_getDiskReg,		/* $C031 */
	MEM_ioUnimp,		/* $C032 */
	CLK_getData,		/* $C033 */
	CLK_getControl,		/* $C034 */
	MEM_getShadowReg,	/* $C035 */
	MEM_getCYAReg,		/* $C036 */
	MEM_ioUnimp,		/* $C037 */
	MEM_ioUnimp,		/* $C038 */
	MEM_ioUnimp,		/* $C039 */
	MEM_ioUnimp,		/* $C03A */
	MEM_ioUnimp,		/* $C03B */
	SND_readSoundCtl,	/* $C03C */
	SND_readSoundData,	/* $C03D */
	SND_readSoundAddrL,	/* $C03E */
	SND_readSoundAddrH,	/* $C03F */
	MEM_ioUnimp,		/* $C040 */
	MEM_getIntEnReg,	/* $C041 */
	MEM_ioUnimp,		/* $C042 */
	MEM_ioUnimp,		/* $C043 */
	ADB_readM2MouseX,	/* $C044 */
	ADB_readM2MouseY,	/* $C045 */
	MEM_getDiagType,	/* $C046 */
	MEM_ioUnimp,		/* $C047 */
	MEM_ioUnimp,		/* $C048 */
	MEM_ioUnimp,		/* $C049 */
	MEM_ioUnimp,		/* $C04A */
	MEM_ioUnimp,		/* $C04B */
	MEM_ioUnimp,		/* $C04C */
	MEM_ioUnimp,		/* $C04D */
	MEM_ioUnimp,		/* $C04E */
	MEM_ioUnimp,		/* $C04F */
	VID_clearText,		/* $C050 */
	VID_setText,		/* $C051 */
	VID_clearMixed,		/* $C052 */
	VID_setMixed,		/* $C053 */
	VID_clearPage2,		/* $C054 */
	VID_setPage2,		/* $C055 */
	VID_clearHires,		/* $C056 */
	VID_setHires,		/* $C057 */
	MEM_ioUnimp,		/* $C058 */
	MEM_ioUnimp,		/* $C059 */
	MEM_ioUnimp,		/* $C05A */
	MEM_ioUnimp,		/* $C05B */
	MEM_ioUnimp,		/* $C05C */
	MEM_ioUnimp,		/* $C05D */
	VID_setDblRes,		/* $C05E */
	VID_clearDblRes,	/* $C05F */
	MEM_ioUnimp,		/* $C060 */
	ADB_readCommandKey,	/* $C061 */
	ADB_readOptionKey,	/* $C062 */
	MEM_ioUnimp,		/* $C063 */
	ADB_readPaddle0,	/* $C064 */
	ADB_readPaddle1,	/* $C065 */
	ADB_readPaddle2,	/* $C066 */
	ADB_readPaddle3,	/* $C067 */
	MEM_getStateReg,	/* $C068 */
	MEM_ioUnimp,		/* $C069 */
	MEM_ioUnimp,		/* $C06A */
	MEM_ioUnimp,		/* $C06B */
	MEM_ioUnimp,		/* $C06C */
	MEM_ioUnimp,		/* $C06D */
	MEM_ioUnimp,		/* $C06E */
	MEM_ioUnimp,		/* $C06F */
	ADB_triggerPaddles,	/* $C070 */
	MEM_getC071,		/* $C071 */
	MEM_getC072,		/* $C072 */
	MEM_getC073,		/* $C073 */
	MEM_getC074,		/* $C074 */
	MEM_getC075,		/* $C075 */
	MEM_getC076,		/* $C076 */
	MEM_getC077,		/* $C077 */
	MEM_getC078,		/* $C078 */
	MEM_getC079,		/* $C079 */
	MEM_getC07A,		/* $C07A */
	MEM_getC07B,		/* $C07B */
	MEM_getC07C,		/* $C07C */
	MEM_getC07D,		/* $C07D */
	MEM_getC07E,		/* $C07E */
	MEM_getC07F,		/* $C07F */
	MEM_setLCx80,		/* $C080 */
	MEM_setLCx81,		/* $C081 */
	MEM_setLCx82,		/* $C082 */
	MEM_setLCx83,		/* $C083 */
	MEM_ioUnimp,		/* $C084 */
	MEM_ioUnimp,		/* $C085 */
	MEM_ioUnimp,		/* $C086 */
	MEM_ioUnimp,		/* $C087 */
	MEM_setLCx88,		/* $C088 */
	MEM_setLCx89,		/* $C089 */
	MEM_setLCx8A,		/* $C08A */
	MEM_setLCx8B,		/* $C08B */
	MEM_ioUnimp,		/* $C08C */
	MEM_ioUnimp,		/* $C08D */
	MEM_ioUnimp,		/* $C08E */
	MEM_ioUnimp,		/* $C08F */
	MEM_ioUnimp,		/* $C090 */
	MEM_ioUnimp,		/* $C091 */
	MEM_ioUnimp,		/* $C092 */
	MEM_ioUnimp,		/* $C093 */
	MEM_ioUnimp,		/* $C094 */
	MEM_ioUnimp,		/* $C095 */
	MEM_ioUnimp,		/* $C096 */
	MEM_ioUnimp,		/* $C097 */
	MEM_ioUnimp,		/* $C098 */
	MEM_ioUnimp,		/* $C099 */
	MEM_ioUnimp,		/* $C09A */
	MEM_ioUnimp,		/* $C09B */
	MEM_ioUnimp,		/* $C09C */
	MEM_ioUnimp,		/* $C09D */
	MEM_ioUnimp,		/* $C09E */
	MEM_ioUnimp,		/* $C09F */
	MEM_ioUnimp,		/* $C0A0 */
	MEM_ioUnimp,		/* $C0A1 */
	MEM_ioUnimp,		/* $C0A2 */
	MEM_ioUnimp,		/* $C0A3 */
	MEM_ioUnimp,		/* $C0A4 */
	MEM_ioUnimp,		/* $C0A5 */
	MEM_ioUnimp,		/* $C0A6 */
	MEM_ioUnimp,		/* $C0A7 */
	MEM_ioUnimp,		/* $C0A8 */
	MEM_ioUnimp,		/* $C0A9 */
	MEM_ioUnimp,		/* $C0AA */
	MEM_ioUnimp,		/* $C0AB */
	MEM_ioUnimp,		/* $C0AC */
	MEM_ioUnimp,		/* $C0AD */
	MEM_ioUnimp,		/* $C0AE */
	MEM_ioUnimp,		/* $C0AF */
	MEM_ioUnimp,		/* $C0B0 */
	MEM_ioUnimp,		/* $C0B1 */
	MEM_ioUnimp,		/* $C0B2 */
	MEM_ioUnimp,		/* $C0B3 */
	MEM_ioUnimp,		/* $C0B4 */
	MEM_ioUnimp,		/* $C0B5 */
	MEM_ioUnimp,		/* $C0B6 */
	MEM_ioUnimp,		/* $C0B7 */
	MEM_ioUnimp,		/* $C0B8 */
	MEM_ioUnimp,		/* $C0B9 */
	MEM_ioUnimp,		/* $C0BA */
	MEM_ioUnimp,		/* $C0BB */
	MEM_ioUnimp,		/* $C0BC */
	MEM_ioUnimp,		/* $C0BD */
	MEM_ioUnimp,		/* $C0BE */
	MEM_ioUnimp,		/* $C0BF */
	MEM_ioUnimp,		/* $C0C0 */
	MEM_ioUnimp,		/* $C0C1 */
	MEM_ioUnimp,		/* $C0C2 */
	MEM_ioUnimp,		/* $C0C3 */
	MEM_ioUnimp,		/* $C0C4 */
	MEM_ioUnimp,		/* $C0C5 */
	MEM_ioUnimp,		/* $C0C6 */
	MEM_ioUnimp,		/* $C0C7 */
	MEM_ioUnimp,		/* $C0C8 */
	MEM_ioUnimp,		/* $C0C9 */
	MEM_ioUnimp,		/* $C0CA */
	MEM_ioUnimp,		/* $C0CB */
	MEM_ioUnimp,		/* $C0CC */
	MEM_ioUnimp,		/* $C0CD */
	MEM_ioUnimp,		/* $C0CE */
	MEM_ioUnimp,		/* $C0CF */
	MEM_ioUnimp,		/* $C0D0 */
	MEM_ioUnimp,		/* $C0D1 */
	MEM_ioUnimp,		/* $C0D2 */
	MEM_ioUnimp,		/* $C0D3 */
	MEM_ioUnimp,		/* $C0D4 */
	MEM_ioUnimp,		/* $C0D5 */
	MEM_ioUnimp,		/* $C0D6 */
	MEM_ioUnimp,		/* $C0D7 */
	MEM_ioUnimp,		/* $C0D8 */
	MEM_ioUnimp,		/* $C0D9 */
	MEM_ioUnimp,		/* $C0DA */
	MEM_ioUnimp,		/* $C0DB */
	MEM_ioUnimp,		/* $C0DC */
	MEM_ioUnimp,		/* $C0DD */
	MEM_ioUnimp,		/* $C0DE */
	MEM_ioUnimp,		/* $C0DF */
	IWM_readC0E0,		/* $C0E0 */
	IWM_readC0E1,		/* $C0E1 */
	IWM_readC0E2,		/* $C0E2 */
	IWM_readC0E3,		/* $C0E3 */
	IWM_readC0E4,		/* $C0E4 */
	IWM_readC0E5,		/* $C0E5 */
	IWM_readC0E6,		/* $C0E6 */
	IWM_readC0E7,		/* $C0E7 */
	IWM_readC0E8,		/* $C0E8 */
	IWM_readC0E9,		/* $C0E9 */
	IWM_readC0EA,		/* $C0EA */
	IWM_readC0EB,		/* $C0EB */
	IWM_readC0EC,		/* $C0EC */
	IWM_readC0ED,		/* $C0ED */
	IWM_readC0EE,		/* $C0EE */
	IWM_readC0EF,		/* $C0EF */
	MEM_ioUnimp,		/* $C0F0 */
	MEM_ioUnimp,		/* $C0F1 */
	MEM_ioUnimp,		/* $C0F2 */
	MEM_ioUnimp,		/* $C0F3 */
	MEM_ioUnimp,		/* $C0F4 */
	MEM_ioUnimp,		/* $C0F5 */
	MEM_ioUnimp,		/* $C0F6 */
	MEM_ioUnimp,		/* $C0F7 */
	MEM_ioUnimp,		/* $C0F8 */
	MEM_ioUnimp,		/* $C0F9 */
	MEM_ioUnimp,		/* $C0FA */
	MEM_ioUnimp,		/* $C0FB */
	MEM_ioUnimp,		/* $C0FC */
	MEM_ioUnimp,		/* $C0FD */
	MEM_ioUnimp,		/* $C0FE */
	MEM_ioUnimp		/* $C0FF */
};

/* Dispatch table for writes to page $C0xx I/O space	*/

byte (*mem_io_write_dispatch[256])(byte) = {
	MEM_clear80store,	/* $C000 */
	MEM_set80store,		/* $C001 */
	MEM_clearAuxRd,		/* $C002 */
	MEM_setAuxRd,		/* $C003 */
	MEM_clearAuxWrt,	/* $C004 */
	MEM_setAuxWrt,		/* $C005 */
	MEM_clearIntCXROM,	/* $C006 */
	MEM_setIntCXROM,	/* $C007 */
	MEM_clearAltZP,		/* $C008 */
	MEM_setAltZP,		/* $C009 */
	MEM_clearSlotC3ROM,	/* $C00A */
	MEM_setSlotC3ROM,	/* $C00B */
	VID_clear80col,		/* $C00C */
	VID_set80col,		/* $C00D */
	VID_clearAltCh,		/* $C00E */
	VID_setAltCh,		/* $C00F */
	ADB_clearKey,		/* $C010 */
	MEM_ioUnimp,		/* $C011 */
	MEM_ioUnimp,		/* $C012 */
	MEM_ioUnimp,		/* $C013 */
	MEM_ioUnimp,		/* $C014 */
	MEM_ioUnimp,		/* $C015 */
	MEM_ioUnimp,		/* $C016 */
	MEM_ioUnimp,		/* $C017 */
	MEM_ioUnimp,		/* $C018 */
	MEM_ioUnimp,		/* $C019 */
	MEM_ioUnimp,		/* $C01A */
	MEM_ioUnimp,		/* $C01B */
	MEM_ioUnimp,		/* $C01C */
	MEM_ioUnimp,		/* $C01D */
	MEM_ioUnimp,		/* $C01E */
	MEM_ioUnimp,		/* $C01F */
	MEM_ioUnimp,		/* $C020 */
	MEM_ioUnimp,		/* $C021 */
	VID_setColorReg,	/* $C022 */
	VID_setVGCIntReg,	/* $C023 */
	MEM_ioUnimp,		/* $C024 */
	MEM_ioUnimp,		/* $C025 */
	ADB_setCommand,		/* $C026 */
	ADB_setStatus,		/* $C027 */
	MEM_ioUnimp,		/* $C028 */
	VID_setNewVideo,	/* $C029 */
	MEM_ioUnimp,		/* $C02A */
	MEM_ioUnimp,		/* $C02B */
	MEM_ioUnimp,		/* $C02C */
	MEM_setSlotReg,		/* $C02D */
	MEM_ioUnimp,		/* $C02E */
	MEM_ioUnimp,		/* $C02F */
	SND_clickSpeaker,	/* $C030 */
	IWM_setDiskReg,		/* $C031 */
	VID_clearVGCInt,	/* $C032 */
	CLK_setData,		/* $C033 */
	CLK_setControl,		/* $C034 */
	MEM_setShadowReg,	/* $C035 */
	MEM_setCYAReg,		/* $C036 */
	MEM_ioUnimp,		/* $C037 */
	MEM_ioUnimp,		/* $C038 */
	MEM_ioUnimp,		/* $C039 */
	MEM_ioUnimp,		/* $C03A */
	MEM_ioUnimp,		/* $C03B */
	SND_writeSoundCtl,	/* $C03C */
	SND_writeSoundData,	/* $C03D */
	SND_writeSoundAddrL,	/* $C03E */
	SND_writeSoundAddrH,	/* $C03F */
	MEM_ioUnimp,		/* $C040 */
	MEM_setIntEnReg,	/* $C041 */
	MEM_ioUnimp,		/* $C042 */
	MEM_ioUnimp,		/* $C043 */
	MEM_ioUnimp,		/* $C044 */
	MEM_ioUnimp,		/* $C045 */
	MEM_ioUnimp,		/* $C046 */
	MEM_setVBLClear,	/* $C047 */
	MEM_ioUnimp,		/* $C048 */
	MEM_ioUnimp,		/* $C049 */
	MEM_ioUnimp,		/* $C04A */
	MEM_ioUnimp,		/* $C04B */
	MEM_ioUnimp,		/* $C04C */
	MEM_ioUnimp,		/* $C04D */
	MEM_ioUnimp,		/* $C04E */
	MEM_ioUnimp,		/* $C04F */
	VID_clearText,		/* $C050 */
	VID_setText,		/* $C051 */
	VID_clearMixed,		/* $C052 */
	VID_setMixed,		/* $C053 */
	VID_clearPage2,		/* $C054 */
	VID_setPage2,		/* $C055 */
	VID_clearHires,		/* $C056 */
	VID_setHires,		/* $C057 */
	MEM_ioUnimp,		/* $C058 */
	MEM_ioUnimp,		/* $C059 */
	MEM_ioUnimp,		/* $C05A */
	MEM_ioUnimp,		/* $C05B */
	MEM_ioUnimp,		/* $C05C */
	MEM_ioUnimp,		/* $C05D */
	VID_setDblRes,		/* $C05E */
	VID_clearDblRes,	/* $C05F */
	MEM_ioUnimp,		/* $C060 */
	MEM_ioUnimp,		/* $C061 */
	MEM_ioUnimp,		/* $C062 */
	MEM_ioUnimp,		/* $C063 */
	MEM_ioUnimp,		/* $C064 */
	MEM_ioUnimp,		/* $C065 */
	MEM_ioUnimp,		/* $C066 */
	MEM_ioUnimp,		/* $C067 */
	MEM_setStateReg,	/* $C068 */
	MEM_ioUnimp,		/* $C069 */
	MEM_ioUnimp,		/* $C06A */
	MEM_ioUnimp,		/* $C06B */
	MEM_ioUnimp,		/* $C06C */
	MEM_ioUnimp,		/* $C06D */
	MEM_ioUnimp,		/* $C06E */
	MEM_ioUnimp,		/* $C06F */
	ADB_triggerPaddles,	/* $C070 */
	MEM_ioUnimp,		/* $C071 */
	MEM_ioUnimp,		/* $C072 */
	MEM_ioUnimp,		/* $C073 */
	MEM_ioUnimp,		/* $C074 */
	MEM_ioUnimp,		/* $C075 */
	MEM_ioUnimp,		/* $C076 */
	MEM_ioUnimp,		/* $C077 */
	MEM_ioUnimp,		/* $C078 */
	MEM_ioUnimp,		/* $C079 */
	MEM_ioUnimp,		/* $C07A */
	MEM_ioUnimp,		/* $C07B */
	MEM_ioUnimp,		/* $C07C */
	MEM_ioUnimp,		/* $C07D */
	MEM_ioUnimp,		/* $C07E */
	MEM_ioUnimp,		/* $C07F */
	MEM_setLCx80,		/* $C080 */
	MEM_setLCx81,		/* $C081 */
	MEM_setLCx82,		/* $C082 */
	MEM_setLCx83,		/* $C083 */
	MEM_ioUnimp,		/* $C084 */
	MEM_ioUnimp,		/* $C085 */
	MEM_ioUnimp,		/* $C086 */
	MEM_ioUnimp,		/* $C087 */
	MEM_setLCx88,		/* $C088 */
	MEM_setLCx89,		/* $C089 */
	MEM_setLCx8A,		/* $C08A */
	MEM_setLCx8B,		/* $C08B */
	MEM_ioUnimp,		/* $C08C */
	MEM_ioUnimp,		/* $C08D */
	MEM_ioUnimp,		/* $C08E */
	MEM_ioUnimp,		/* $C08F */
	MEM_ioUnimp,		/* $C090 */
	MEM_ioUnimp,		/* $C091 */
	MEM_ioUnimp,		/* $C092 */
	MEM_ioUnimp,		/* $C093 */
	MEM_ioUnimp,		/* $C094 */
	MEM_ioUnimp,		/* $C095 */
	MEM_ioUnimp,		/* $C096 */
	MEM_ioUnimp,		/* $C097 */
	MEM_ioUnimp,		/* $C098 */
	MEM_ioUnimp,		/* $C099 */
	MEM_ioUnimp,		/* $C09A */
	MEM_ioUnimp,		/* $C09B */
	MEM_ioUnimp,		/* $C09C */
	MEM_ioUnimp,		/* $C09D */
	MEM_ioUnimp,		/* $C09E */
	MEM_ioUnimp,		/* $C09F */
	MEM_ioUnimp,		/* $C0A0 */
	MEM_ioUnimp,		/* $C0A1 */
	MEM_ioUnimp,		/* $C0A2 */
	MEM_ioUnimp,		/* $C0A3 */
	MEM_ioUnimp,		/* $C0A4 */
	MEM_ioUnimp,		/* $C0A5 */
	MEM_ioUnimp,		/* $C0A6 */
	MEM_ioUnimp,		/* $C0A7 */
	MEM_ioUnimp,		/* $C0A8 */
	MEM_ioUnimp,		/* $C0A9 */
	MEM_ioUnimp,		/* $C0AA */
	MEM_ioUnimp,		/* $C0AB */
	MEM_ioUnimp,		/* $C0AC */
	MEM_ioUnimp,		/* $C0AD */
	MEM_ioUnimp,		/* $C0AE */
	MEM_ioUnimp,		/* $C0AF */
	MEM_ioUnimp,		/* $C0B0 */
	MEM_ioUnimp,		/* $C0B1 */
	MEM_ioUnimp,		/* $C0B2 */
	MEM_ioUnimp,		/* $C0B3 */
	MEM_ioUnimp,		/* $C0B4 */
	MEM_ioUnimp,		/* $C0B5 */
	MEM_ioUnimp,		/* $C0B6 */
	MEM_ioUnimp,		/* $C0B7 */
	MEM_ioUnimp,		/* $C0B8 */
	MEM_ioUnimp,		/* $C0B9 */
	MEM_ioUnimp,		/* $C0BA */
	MEM_ioUnimp,		/* $C0BB */
	MEM_ioUnimp,		/* $C0BC */
	MEM_ioUnimp,		/* $C0BD */
	MEM_ioUnimp,		/* $C0BE */
	MEM_ioUnimp,		/* $C0BF */
	MEM_ioUnimp,		/* $C0C0 */
	MEM_ioUnimp,		/* $C0C1 */
	MEM_ioUnimp,		/* $C0C2 */
	MEM_ioUnimp,		/* $C0C3 */
	MEM_ioUnimp,		/* $C0C4 */
	MEM_ioUnimp,		/* $C0C5 */
	MEM_ioUnimp,		/* $C0C6 */
	MEM_ioUnimp,		/* $C0C7 */
	MEM_ioUnimp,		/* $C0C8 */
	MEM_ioUnimp,		/* $C0C9 */
	MEM_ioUnimp,		/* $C0CA */
	MEM_ioUnimp,		/* $C0CB */
	MEM_ioUnimp,		/* $C0CC */
	MEM_ioUnimp,		/* $C0CD */
	MEM_ioUnimp,		/* $C0CE */
	MEM_ioUnimp,		/* $C0CF */
	MEM_ioUnimp,		/* $C0D0 */
	MEM_ioUnimp,		/* $C0D1 */
	MEM_ioUnimp,		/* $C0D2 */
	MEM_ioUnimp,		/* $C0D3 */
	MEM_ioUnimp,		/* $C0D4 */
	MEM_ioUnimp,		/* $C0D5 */
	MEM_ioUnimp,		/* $C0D6 */
	MEM_ioUnimp,		/* $C0D7 */
	MEM_ioUnimp,		/* $C0D8 */
	MEM_ioUnimp,		/* $C0D9 */
	MEM_ioUnimp,		/* $C0DA */
	MEM_ioUnimp,		/* $C0DB */
	MEM_ioUnimp,		/* $C0DC */
	MEM_ioUnimp,		/* $C0DD */
	MEM_ioUnimp,		/* $C0DE */
	MEM_ioUnimp,		/* $C0DF */
	IWM_writeC0E0,		/* $C0E0 */
	IWM_writeC0E1,		/* $C0E1 */
	IWM_writeC0E2,		/* $C0E2 */
	IWM_writeC0E3,		/* $C0E3 */
	IWM_writeC0E4,		/* $C0E4 */
	IWM_writeC0E5,		/* $C0E5 */
	IWM_writeC0E6,		/* $C0E6 */
	IWM_writeC0E7,		/* $C0E7 */
	IWM_writeC0E8,		/* $C0E8 */
	IWM_writeC0E9,		/* $C0E9 */
	IWM_writeC0EA,		/* $C0EA */
	IWM_writeC0EB,		/* $C0EB */
	IWM_writeC0EC,		/* $C0EC */
	IWM_writeC0ED,		/* $C0ED */
	IWM_writeC0EE,		/* $C0EE */
	IWM_writeC0EF,		/* $C0EF */
	MEM_ioUnimp,		/* $C0F0 */
	MEM_ioUnimp,		/* $C0F1 */
	MEM_ioUnimp,		/* $C0F2 */
	MEM_ioUnimp,		/* $C0F3 */
	MEM_ioUnimp,		/* $C0F4 */
	MEM_ioUnimp,		/* $C0F5 */
	MEM_ioUnimp,		/* $C0F6 */
	MEM_ioUnimp,		/* $C0F7 */
	MEM_ioUnimp,		/* $C0F8 */
	MEM_ioUnimp,		/* $C0F9 */
	MEM_ioUnimp,		/* $C0FA */
	MEM_ioUnimp,		/* $C0FB */
	MEM_ioUnimp,		/* $C0FC */
	MEM_ioUnimp,		/* $C0FD */
	MEM_ioUnimp,		/* $C0FE */
	MEM_ioUnimp		/* $C0FF */
};

int MEM_init(void)
{
	int		i,rom_base;
	char		*path;
	FILE		*fp;
	struct stat	stats;

	if (mem_ramsize < 1) mem_ramsize = 1;
	if (mem_ramsize > 8) mem_ramsize = 8;

	printf("    - Allocating 128 KB for slow RAM: ");
	slow_memory = malloc(2*65536);
	if (slow_memory == NULL) {
		printf("Failed\n");
		return 1;
	}
	memset(slow_memory, 0, 2*65536);
	printf("Done\n");

	printf("    - Allocating %d MB for fast RAM: ",mem_ramsize);
	fast_memory = malloc(mem_ramsize * 1048576);
	if (fast_memory == NULL) {
		printf("Failed\n");
		return 1;
	}
	memset(fast_memory, 0, mem_ramsize * 1048576);
	printf("Done\n");

	printf("    - Allocating 128 KB save RAM: ");
	save_memory = malloc(2*65536);
	if (save_memory == NULL) {
		printf("Failed\n");
		return 1;
	}
	memset(save_memory, 0, 2*65536);
	printf("Done\n");

	printf("    - Loading ROM image: ");
	path = EMUL_expandPath(ROM_FILE);
	if (stat(path, &stats)) {
		printf("Failed\n");
		return 2;
	}
	if (stats.st_size == 131072) {
		mem_rom03 = 0;
	} else if (stats.st_size == 262144) {
		mem_rom03 = 1;
	} else {
		printf("Failed\n");
		return 2;
	}
	rom_memory = malloc(stats.st_size);
	if (rom_memory == NULL) {
		printf("Failed\n");
		return 2;
	}
	
	if ((fp = fopen(path,"rb")) == NULL) {
		printf("Failed\n");
		return 2;
	}
	if (fread(rom_memory,1,stats.st_size,fp) != stats.st_size) {
		printf("Failed\n");
		return 2;
	}
	fclose(fp);
	printf("Done\n");

	if (mem_rom03) {
		printf("    - ROM 03 emulation mode\n");
		rom_base = 0xFC00;
		c07x_memory = rom_memory + 0x03C070;
	} else {
		printf("    - ROM 01 emulation mode\n");
		rom_base = 0xFE00;
		c07x_memory = rom_memory + 0x01C070;
	}

	/* Initialize all pages */

	printf("\nInitializing emulator memory\n");
	for (i = 0x0000 ; i < mem_ramsize * 4096 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = fast_memory + (i * 256);
		mem_pages[i].readFlags = mem_pages[i].writeFlags = 0;
	}
	for (i = mem_ramsize * 4096 ; i < 0xE000 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = junk_memory;
		mem_pages[i].readFlags = mem_pages[i].writeFlags = MEM_FLAG_INVALID;
	}
	for (i = 0xE000 ; i < 0xE200 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = slow_memory + ((i-0xE000)*256);
		mem_pages[i].readFlags = mem_pages[i].writeFlags = MEM_FLAG_SPECIAL;
	}

	for (i = 0xE200 ; i < rom_base ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = junk_memory;
		mem_pages[i].readFlags = mem_pages[i].writeFlags = MEM_FLAG_INVALID;
	}
	for (i = rom_base ; i < 0x10000 ; i++) {
		mem_pages[i].readPtr = rom_memory + ((i - rom_base) * 256);
		mem_pages[i].writePtr = junk_memory;
		mem_pages[i].readFlags = 0;
		mem_pages[i].writeFlags = MEM_FLAG_INVALID;
	}

	for (i = 0xE800 ; i < 0xEA00 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = save_memory + ((i-0xE800) * 256);
		mem_pages[i].readFlags = mem_pages[i].writeFlags = 0;
	}

	return 0;
}

void MEM_update()
{
}

/* Memory manager reset routine. Here we reset the memory soft swithches to a	*/
/* reasonable power-up/reset state, and then rebuild the mem_pages[] array.	*/

void MEM_reset()
{
	int	i;

	mem_80store = 0;
	mem_auxrd = 0;
	mem_auxwrt = 0;
	mem_altzp = 0;

	mem_lcbank2 = 0;
	mem_lcread = 0;
	mem_lcwrite = 0;
	mem_lcsecond = 0;

	mem_shadow_text = 1;
	mem_shadow_hires1 = 1;
	mem_shadow_hires2 = 1;
	mem_shadow_super = 0;
	mem_shadow_aux = 1;
	mem_shadow_lc = 1;

	for (i = 0 ; i < 5 ; i++) mem_slot_reg[i] = 0;
	mem_slot_reg[7] = 1;

	MEM_rebuildMainMem();
	MEM_rebuildAltZpMem();
}

void MEM_shutdown()
{
	printf("\nShutting down emulator memory\n");
	if (slow_memory) free(save_memory);
	if (slow_memory) free(fast_memory);
	if (slow_memory) free(slow_memory);
	if (slow_memory) free(rom_memory);
}

void MEM_rebuildMainMem()
{
	int	i,offset1,offset2;

	offset1 = mem_auxrd? 0x10000 : 0;
	offset2 = mem_auxwrt? 0x10000 : 0;

	for (i = 0x0002 ; i < 0x0004 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}

	if (mem_80store) {
		offset1 = offset2 = vid_page2? 0x10000 : 0;
	}

	for (i = 0x0004 ; i < 0x0008 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}

	offset1 = mem_auxrd? 0x10000 : 0;
	offset2 = mem_auxwrt? 0x10000 : 0;

	for (i = 0x0008 ; i < 0x0020 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}

	if (mem_80store && vid_hires) {
		offset1 = offset2 = vid_page2? 0x10000 : 0;
	}

	for (i = 0x0020 ; i < 0x0040 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}

	offset1 = mem_auxrd? 0x10000 : 0;
	offset2 = mem_auxwrt? 0x10000 : 0;

	for (i = 0x0040 ; i < 0x00C0 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}
	MEM_rebuildShadowMem();
}

void MEM_rebuildAltZpMem()
{
	int	i,offset;

	offset = mem_altzp? 0x10000 : 0;

	for (i = 0x0000 ; i < 0x0002 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = fast_memory + (i * 256) + offset;
	}
	MEM_rebuildLangCardMem();
}

void MEM_rebuildLangCardMem()
{
	int	i,offset1,offset2;

	MEM_buildLanguageCard(0xE000,0,slow_memory);
	MEM_buildLanguageCard(0xE100,0,slow_memory+0x10000);
	offset1 = mem_altzp? 0x10000 : 0;
	if (mem_shadow_lc) {
		MEM_buildLanguageCard(0x0000,offset1,fast_memory);
		MEM_buildLanguageCard(0x0100,0x10000,fast_memory);
	} else {
		offset1 = mem_auxrd? 0x10000 : 0;
		offset2 = mem_auxwrt? 0x10000 : 0;
		for (i = 0x00C0 ; i < 0x0100 ; i++) {
			mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
			mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
		}
		for (i = 0x01C0 ; i < 0x0200 ; i++) {
			mem_pages[i].readPtr = fast_memory + (i*256);
			mem_pages[i].writePtr = fast_memory + (i*256);
		}
	}
}

void MEM_rebuildShadowMem()
{
	int	i,flag,flag2;

	if (mem_80store) {
		flag = vid_page2? MEM_FLAG_SHADOW_E1 : MEM_FLAG_SHADOW_E0;
	} else {
		flag = mem_auxwrt? MEM_FLAG_SHADOW_E1 : MEM_FLAG_SHADOW_E0;
	}

	for (i = 0x0004 ; i < 0x0008 ; i++) {
		mem_pages[i].writeFlags = mem_shadow_text? flag : 0;
	}

	if (mem_80store && vid_hires) {
		flag = vid_page2? MEM_FLAG_SHADOW_E1 : MEM_FLAG_SHADOW_E0;
	} else {
		flag = mem_auxwrt? MEM_FLAG_SHADOW_E1 : MEM_FLAG_SHADOW_E0;
	}

	if (flag == MEM_FLAG_SHADOW_E1) {
		flag2 = (mem_shadow_hires1 && mem_shadow_aux) || mem_shadow_super;
	} else {
		flag2 = mem_shadow_hires1;
	}

	for (i = 0x0020 ; i < 0x0040 ; i++) {
		mem_pages[i].writeFlags = flag2? flag : 0;
	}

	flag = mem_auxwrt? MEM_FLAG_SHADOW_E1 : MEM_FLAG_SHADOW_E0;

	if (flag == MEM_FLAG_SHADOW_E1) {
		flag2 = (mem_shadow_hires2 && mem_shadow_aux) || mem_shadow_super;
	} else {
		flag2 = mem_shadow_hires2;
	}

	for (i = 0x0040 ; i < 0x0060 ; i++) {
		mem_pages[i].writeFlags = flag2? flag : 0;
	}

	if (mem_shadow_super && mem_auxwrt) {
		flag = MEM_FLAG_SHADOW_E1;
	} else {
		flag = 0;
	}

	for (i = 0x0060 ; i < 0x00A0 ; i++) {
		mem_pages[i].writeFlags = flag;
	}

	for (i = 0x0104 ; i < 0x0108 ; i++) {
		mem_pages[i].writeFlags = mem_shadow_text? MEM_FLAG_SHADOW_E1 : 0;
	}

	flag = (mem_shadow_hires1 && mem_shadow_aux) || mem_shadow_super;

	for (i = 0x0120 ; i < 0x0140 ; i++) {
		mem_pages[i].writeFlags = flag? MEM_FLAG_SHADOW_E1 : 0;
	}

	flag = (mem_shadow_hires2 && mem_shadow_aux) || mem_shadow_super;

	for (i = 0x0140 ; i < 0x0160 ; i++) {
		mem_pages[i].writeFlags = flag? MEM_FLAG_SHADOW_E1 : 0;
	}

	for (i = 0x0160 ; i < 0x01A0 ; i++) {
		mem_pages[i].writeFlags = mem_shadow_super? MEM_FLAG_SHADOW_E1 : 0;
	}
}

/* This routine builds a language card in the given bank.  This	*/
/* means building all mapping info from $C000 up to $FFFF.	*/

void MEM_buildLanguageCard(int bank,int offset,byte *ram)
{
	int	i,lcbank,rom_lastbank;

	rom_lastbank = mem_rom03? 0x030000 : 0x010000;

	mem_pages[bank+0xC0].readFlags = MEM_FLAG_IO;
	mem_pages[bank+0xC0].writeFlags = MEM_FLAG_IO;

	for (i = 0xC1 ; i < 0xC7 ; i++) {
		if (mem_slot_reg[i - 0xC0]) {
			mem_pages[bank+i].readPtr = junk_memory;
			mem_pages[bank+i].readFlags = MEM_FLAG_INVALID;
		} else {
			mem_pages[bank+i].readPtr = rom_memory + rom_lastbank + (i * 256);
			mem_pages[bank+i].readFlags = 0;
		}
		mem_pages[bank+i].writePtr = junk_memory;
		mem_pages[bank+i].writeFlags = MEM_FLAG_INVALID;
	}

	mem_pages[bank+0xC7].readPtr = smpt_rom;
	mem_pages[bank+0xC7].readFlags = 0;

	for (i = 0xC8 ; i < 0xD0 ; i++) {
		mem_pages[bank+i].readPtr = rom_memory + rom_lastbank + (i * 256);
		mem_pages[bank+i].readFlags = 0;
		mem_pages[bank+i].writePtr = junk_memory;
		mem_pages[bank+i].writeFlags = MEM_FLAG_INVALID;
	}

	lcbank = mem_lcbank2? 0x1000 : 0;

	for (i = 0xD0 ; i < 0xE0 ; i++) {
		if (mem_lcread) {
			mem_pages[bank+i].readPtr = ram + (i*256) + offset - lcbank;
		} else {
			mem_pages[bank+i].readPtr = rom_memory + rom_lastbank + (i * 256);
		}
		if (mem_lcwrite) {
			mem_pages[bank+i].writePtr = ram + (i*256) + offset - lcbank;
			mem_pages[bank+i].writeFlags = 0;
		} else {
			mem_pages[bank+i].writePtr = junk_memory;
			mem_pages[bank+i].writeFlags = MEM_FLAG_INVALID;
		}
	}
	for (i = 0xE0 ; i < 0x100 ; i++) {
		if (mem_lcread) {
			mem_pages[bank+i].readPtr = ram + (i*256) + offset;
		} else {
			mem_pages[bank+i].readPtr = rom_memory + rom_lastbank + (i * 256);
		}
		if (mem_lcwrite) {
			mem_pages[bank+i].writePtr = ram + (i*256) + offset;
			mem_pages[bank+i].writeFlags = 0;
		} else {
			mem_pages[bank+i].writePtr = junk_memory;
			mem_pages[bank+i].writeFlags = MEM_FLAG_INVALID;
		}
	}
}

byte MEM_ioUnimp(byte val)
{
	return 0;
}

byte MEM_clear80store(byte val)
{
	mem_80store = 0;
	MEM_rebuildMainMem();
	return 0;
}

byte MEM_set80store(byte val)
{
	mem_80store = 1;
	MEM_rebuildMainMem();
	return 0;
}

byte MEM_get80store(byte val)
{
	return mem_80store? 0x80 : 0x00;
}

byte MEM_clearAuxRd(byte val)
{
	mem_auxrd = 0;
	MEM_rebuildMainMem();
	return 0;
}

byte MEM_setAuxRd(byte val)
{
	mem_auxrd = 1;
	MEM_rebuildMainMem();
	return 0;
}

byte MEM_getAuxRd(byte val)
{
	return mem_auxrd? 0x80 : 0x00;
}

byte MEM_clearAuxWrt(byte val)
{
	mem_auxwrt = 0;
	MEM_rebuildMainMem();
	return 0;
}

byte MEM_setAuxWrt(byte val)
{
	mem_auxwrt = 1;
	MEM_rebuildMainMem();
	return 0;
}

byte MEM_getAuxWrt(byte val)
{
	return mem_auxwrt? 0x80 : 0x00;
}

byte MEM_clearIntCXROM(byte val)
{
	mem_intcxrom = 0;
	return 0;
}

byte MEM_setIntCXROM(byte val)
{
	mem_intcxrom = 1;
	return 0;
}

byte MEM_getIntCXROM(byte val)
{
	return mem_intcxrom? 0x80 : 0x00;
}

byte MEM_clearAltZP(byte val)
{
	mem_altzp = 0;
	MEM_rebuildAltZpMem();
	return 0;
}

byte MEM_setAltZP(byte val)
{
	mem_altzp = 1;
	MEM_rebuildAltZpMem();
	return 0;
}

byte MEM_getAltZP(byte val)
{
	return mem_altzp? 0x80 : 0x00;
}

byte MEM_clearSlotC3ROM(byte val)
{
	mem_slotc3rom = 0;
	return 0;
}

byte MEM_setSlotC3ROM(byte val)
{
	mem_slotc3rom = 1;
	return 0;
}

byte MEM_getSlotC3ROM(byte val)
{
	return mem_slotc3rom? 0x80 : 0x00;
}

byte MEM_getLCbank(byte val)
{
	return mem_lcbank2? 0x80 : 0x00;
}

byte MEM_getLCread(byte val)
{
	return mem_lcread? 0x80 : 0x00;
}

byte MEM_getSlotReg(byte val)
{
	register int	i;

	val = 0;
	for (i = 7 ; i >= 0 ; i--) {
		val = val << 1;
		val |= mem_slot_reg[i];
	}
	return val;
}

byte MEM_setSlotReg(byte val)
{
	register int	i;

	for (i = 0 ; i < 8 ; i++) {
		mem_slot_reg[i] = val & 0x01;
		val = val >> 1;
	}
	MEM_rebuildLangCardMem();
	return 0;
}

byte MEM_getShadowReg(byte val)
{
	val = 0;
	if (!mem_shadow_text) val |= 0x01;
	if (!mem_shadow_hires1) val |= 0x02;
	if (!mem_shadow_hires2) val |= 0x04;
	if (!mem_shadow_super) val |= 0x08;
	if (!mem_shadow_aux) val |= 0x10;
	if (!mem_shadow_lc) val |= 0x40;
	return val;
}

byte MEM_setShadowReg(byte val)
{
	mem_shadow_text = (val & 0x01)? 0 : 1;
	mem_shadow_hires1 = (val & 0x02)? 0 : 1;
	mem_shadow_hires2 = (val & 0x04)? 0 : 1;
	mem_shadow_super = (val & 0x08)? 0 : 1;
	mem_shadow_aux = (val & 0x10)? 0 : 1;
	mem_shadow_lc = (val & 0x40)? 0 : 1;
	MEM_rebuildLangCardMem();
	MEM_rebuildShadowMem();
	return 0;
}

byte MEM_getCYAReg(byte val)
{
	val = 0;
	if (emul_speed) val |= 0x80;
	if (iwm_slot7_motor) val |= 0x08;
	if (iwm_slot6_motor) val |= 0x04;
	if (iwm_slot5_motor) val |= 0x02;
	if (iwm_slot4_motor) val |= 0x01;
	return val;
}

byte MEM_setCYAReg(byte val)
{
	emul_speed = (val & 0x80)? 1 : 0;
	iwm_slot7_motor = (val & 0x08)? 1 : 0;
	iwm_slot6_motor = (val & 0x04)? 1 : 0;
	iwm_slot5_motor = (val & 0x02)? 1 : 0;
	iwm_slot4_motor = (val & 0x01)? 1 : 0;
	if (emul_speed) {
		if (emul_speed2) {
			emul_target_cycles = 0;
			emul_target_speed = 0.0;
		} else {
			emul_target_cycles = 2500000;
			emul_target_speed = 2.5;
		}
	} else {
		emul_target_cycles = 1000000;
		emul_target_speed = 1.0;
	}
	return 0;
}

byte MEM_getIntEnReg(byte val)
{
	val = 0;
	if (emul_qtrsecirq) val |= 0x10;
	if (emul_vblirq) val |= 0x08;
	if (adb_m2mouseswirq) val |= 0x04;
	if (adb_m2mousemvirq) val |= 0x02;
	if (adb_m2mouseenable) val |= 0x01;
	return val;
}

byte MEM_setIntEnReg(byte val)
{
	adb_m2mouseenable = (val & 0x01)? 1 : 0;
	adb_m2mousemvirq = (val & 0x02)? 1 : 0;
	adb_m2mouseswirq = (val & 0x04)? 1 : 0;
	emul_vblirq = (val & 0x08)? 1 : 0;
	emul_qtrsecirq = (val & 0x10)? 1 : 0;
	return 0;
}

byte MEM_getDiagType(byte val)
{
	return mem_diagtype;
}

byte MEM_setVBLClear(byte val)
{
	if (mem_diagtype & 0x10) CPU_clearIRQ();
	if (mem_diagtype & 0x08) CPU_clearIRQ();
	mem_diagtype &= ~0x18;
	return 0;
}

byte MEM_getStateReg(byte val)
{
	val = 0;
	if (mem_intcxrom) val |= 0x01;
	if (mem_rombank) val |= 0x02;
	if (mem_lcbank2) val |= 0x04;
	if (!mem_lcread) val |= 0x08;
	if (mem_auxwrt) val |= 0x10;
	if (mem_auxrd) val |= 0x20;
	if (vid_page2) val |= 0x40;
	if (mem_altzp) val |= 0x80;
	return val;
}

byte MEM_setStateReg(byte val)
{
	mem_intcxrom = (val & 0x01)? 1:0;
	mem_rombank = (val & 0x02)? 1:0;
	mem_lcbank2 = (val & 0x04)? 1:0;
	mem_lcread = (val & 0x08)? 0:1;
	mem_auxwrt = (val & 0x10)? 1:0;
	mem_auxrd = (val & 0x20)? 1:0;
	(val & 0x40)? VID_setPage2(0) : VID_clearPage2(0);
	mem_altzp = (val & 0x80)? 1:0;
	MEM_rebuildMainMem();
	MEM_rebuildAltZpMem();
	return 0;
}

byte MEM_setLCx80(byte val)
{
	mem_lcbank2 = 1;
	mem_lcread = 1;
	mem_lcwrite = 0;
	mem_lcsecond = 0x80;
	MEM_rebuildLangCardMem();
	return 0;
}

byte MEM_setLCx81(byte val)
{
	mem_lcbank2 = 1;
	mem_lcread = 0;
	if (mem_lcsecond == 0x81) mem_lcwrite = 1;
	mem_lcsecond = 0x81;
	MEM_rebuildLangCardMem();
	return 0;
}

byte MEM_setLCx82(byte val)
{
	mem_lcbank2 = 1;
	mem_lcread = 0;
	mem_lcwrite = 0;
	mem_lcsecond = 0x82;
	MEM_rebuildLangCardMem();
	return 0;
}

byte MEM_setLCx83(byte val)
{
	mem_lcbank2 = 1;
	mem_lcread = 1;
	if (mem_lcsecond == 0x83) mem_lcwrite = 1;
	mem_lcsecond = 0x83;
	MEM_rebuildLangCardMem();
	return 0;
}

byte MEM_setLCx88(byte val)
{
	mem_lcbank2 = 0;
	mem_lcread = 1;
	mem_lcwrite = 0;
	mem_lcsecond = 0x88;
	MEM_rebuildLangCardMem();
	return 0;
}

byte MEM_setLCx89(byte val)
{
	mem_lcbank2 = 0;
	mem_lcread = 0;
	if (mem_lcsecond == 0x89) mem_lcwrite = 1;
	mem_lcsecond = 0x89;
	MEM_rebuildLangCardMem();
	return 0;
}

byte MEM_setLCx8A(byte val)
{
	mem_lcbank2 = 0;
	mem_lcread = 0;
	mem_lcwrite = 0;
	mem_lcsecond = 0x8A;
	MEM_rebuildLangCardMem();
	return 0;
}

byte MEM_setLCx8B(byte val)
{
	mem_lcbank2 = 0;
	mem_lcread = 1;
	if (mem_lcsecond == 0x8B) mem_lcwrite = 1;
	mem_lcsecond = 0x8B;
	MEM_rebuildLangCardMem();
	return 0;
}

byte MEM_getC071(byte val)
{
	return c07x_memory[0x01];
}

byte MEM_getC072(byte val)
{
	return c07x_memory[0x02];
}

byte MEM_getC073(byte val)
{
	return c07x_memory[0x03];
}

byte MEM_getC074(byte val)
{
	return c07x_memory[0x04];
}

byte MEM_getC075(byte val)
{
	return c07x_memory[0x05];
}

byte MEM_getC076(byte val)
{
	return c07x_memory[0x06];
}

byte MEM_getC077(byte val)
{
	return c07x_memory[0x07];
}

byte MEM_getC078(byte val)
{
	return c07x_memory[0x08];
}

byte MEM_getC079(byte val)
{
	return c07x_memory[0x09];
}

byte MEM_getC07A(byte val)
{
	return c07x_memory[0x0A];
}

byte MEM_getC07B(byte val)
{
	return c07x_memory[0x0B];
}

byte MEM_getC07C(byte val)
{
	return c07x_memory[0x0C];
}

byte MEM_getC07D(byte val)
{
	return c07x_memory[0x0D];
}

byte MEM_getC07E(byte val)
{
	return c07x_memory[0x0E];
}

byte MEM_getC07F(byte val)
{
	return c07x_memory[0x0F];
}
