/*********************************************************************
 *                                                                   *
 *                     XGS : Apple IIGS Emulator                     *
 *                                                                   *
 *        Written and Copyright (C)1996 by Joshua M. Thompson        *
 *                                                                   *
 *  You are free to distribute this code for non-commercial purposes *
 * I ask only that you notify me of any changes you make to the code *
 *     Commercial use is prohibited without my written permission    *
 *                                                                   *
 *********************************************************************/

/*
 * File: vid-main.c
 *
 * Video subysstem initialization and and core routines.
 */

#include "xgs.h"
#include "video.h"

Screen		*vid_screen;
Display		*vid_display;
Visual		*vid_visual;
XImage		*vid_image;
PIXEL		*vid_buffer;
Window		vid_root,vid_window;
PIXEL		vid_black,vid_white;
GC		vid_gc;
Colormap	vid_colormap;

XFontStruct	*vid_fontinfo;

PIXEL		*vid_lines[VID_HEIGHT];

byte		*vid_font40[2];
byte		*vid_font80[2];

void		(*VID_updateRoutine)(void);

#ifdef MITSHM
XShmSegmentInfo	vid_shminfo;
#endif

int VID_init(void)
{
	XSizeHints	size_hints;
	XWMHints	wm_hints;
	XEvent		event;

	XVisualInfo	vTemplate;
	XVisualInfo	*visualList;
	int		visualsMatched;
	int		visual_chosen;
	XSetWindowAttributes win_attr;
	XGCValues	gcvals;

	FILE		*fp;
	int		i;

	printf("\nInitializing emulator console\n");
	printf("    - Opening display: ");
	vid_display = XOpenDisplay(NULL);
	if (!vid_display) {
		printf("Failed\n");
		return 1;
	}
	vid_screen = DefaultScreenOfDisplay(vid_display);
	vid_black = BlackPixelOfScreen(vid_screen);
	vid_white = WhitePixelOfScreen(vid_screen);
	printf("Done\n");

	printf("    - Picking a visual: ");
	vTemplate.screen = DefaultScreen(vid_display);
	vTemplate.depth = VID_DEPTH;
	visualList = XGetVisualInfo(vid_display,
		VisualScreenMask | VisualDepthMask,
		&vTemplate, &visualsMatched);
	if(visualsMatched == 0) {
		fprintf(stderr, "no visuals!\n");
		return 1;
	}
	visual_chosen = -1;
	for(i = 0; i < visualsMatched; i++) {
		if (visualList[i].class == VID_CLASS) {
			visual_chosen = i;
			break;
		}
	}
	if (visual_chosen < 0) {
		printf("Failed\n");
		return 1;
	} else {
		printf("id %08x, screen %d, depth %d, class %d\n", (int) visualList[visual_chosen].visualid, visualList[visual_chosen].screen, visualList[visual_chosen].depth, visualList[visual_chosen].class);
	}
	vid_visual = visualList[visual_chosen].visual;

	printf("    - Creating window: ");
	vid_root = RootWindowOfScreen(vid_screen);
#ifdef PRIVATE_CMAP
	vid_colormap = XCreateColormap(vid_display, vid_root, vid_visual, AllocNone);
#else
	vid_colormap = DefaultColormapOfScreen(vid_screen);
#endif
	win_attr.border_pixel = vid_white;
	win_attr.background_pixel = vid_black;
	win_attr.colormap = vid_colormap;
	vid_window = XCreateWindow(vid_display,vid_root,0,0,
		VID_WIN_WIDTH,VID_WIN_HEIGHT, 0, vTemplate.depth, InputOutput,
		vid_visual, CWColormap | CWBackPixel | CWBorderPixel,
		&win_attr);
	if (!vid_window) {
		printf("Failed\n");
		return 1;
	}
	VID_allocTextColors();

	printf("    - Setting status font: ");
	vid_gc = XCreateGC(vid_display, vid_window, 0, NULL);
	vid_fontinfo = XLoadQueryFont(vid_display,VID_STATUS_FONT);
	gcvals.font = vid_fontinfo->fid;
	gcvals.fill_style = FillSolid;
	XChangeGC(vid_display, vid_gc, GCFillStyle | GCFont, &gcvals);
	printf("Done");

	XStoreName(vid_display, vid_window, VERSION);
	size_hints.flags = PSize | PMinSize | PMaxSize;
	size_hints.base_width = VID_WIN_WIDTH;
	size_hints.min_width = VID_WIN_WIDTH;
	size_hints.max_width = VID_WIN_WIDTH;
	size_hints.base_height = VID_WIN_HEIGHT;
	size_hints.min_height = VID_WIN_HEIGHT;
	size_hints.max_height = VID_WIN_HEIGHT;
	wm_hints.input = True;
	wm_hints.flags = InputHint;
	XSetWMNormalHints(vid_display, vid_window, &size_hints);
	XSetWMHints(vid_display, vid_window, &wm_hints);
	XSelectInput(vid_display,vid_window,FocusChangeMask|ExposureMask|KeyPressMask|KeyReleaseMask);
	XMapRaised(vid_display, vid_window);
	XWindowEvent(vid_display, vid_window, ExposureMask, &event);
	printf("Done\n");

	printf("    - Creating Image: ");
#ifdef MITSHM

	vid_image = XShmCreateImage(vid_display, vid_visual, VID_DEPTH, ZPixmap, NULL, &vid_shminfo,
   				    VID_WIDTH, VID_HEIGHT);
	if (!vid_image) {
		printf("Failed\n");
		return 1;
	}
	vid_shminfo.shmid = shmget(IPC_PRIVATE, vid_image->bytes_per_line*vid_image->height, IPC_CREAT|0777);
	if (vid_shminfo.shmid < 0) {
		printf("Failed\n");
		return 1;
	}
	vid_shminfo.shmaddr = shmat(vid_shminfo.shmid,NULL,0);
	if (!vid_shminfo.shmaddr) {
		printf("Failed\n");
		return 1;
	}
	vid_shminfo.readOnly = False;
	vid_image->data = vid_shminfo.shmaddr;
	vid_buffer = (PIXEL *) vid_shminfo.shmaddr;
	if(!XShmAttach(vid_display, &vid_shminfo)) {
		printf("Failed\n");
		return 1;
	}
#else
	vid_buffer = (PIXEL *) malloc(sizeof(PIXEL) * VID_WIDTH * VID_HEIGHT);
	if (!vid_buffer) {
		printf("Failed\n");
		return 1;
	}
	vid_image = XCreateImage(vid_display, vid_visual, VID_DEPTH, ZPixmap, 0, (void *) vid_buffer,
				 VID_WIDTH, VID_HEIGHT, sizeof(PIXEL)*8 ,0);
	if (!vid_image) {
		printf("Failed\n");
		return 1;
	}
#endif

	vid_lines[0] = vid_buffer;
	for (i = 1 ; i < VID_HEIGHT ; i++) {
		vid_lines[i] = vid_lines[i-1] + VID_WIDTH;
	}

	printf("Done\n");

	printf("    - Loading 40-col font from file \"%s\": ",FONT40_FILE);
	if (!(vid_font40[0] = malloc(57344))) {
		printf("Failed\n");
		return 1;
	}
	if (!(vid_font40[1] = malloc(57344))) {
		printf("Failed\n");
		return 1;
	}
	if ((fp = fopen(FONT40_FILE,"r")) == NULL) {
		printf("Failed\n");
		return 1;
	}
	if (fread(vid_font40[0],1,57344,fp) != 57344) {
		printf("Failed\n");
		fclose(fp);
		return 1;
	}
	if (fread(vid_font40[1],1,57344,fp) != 57344) {
		printf("Failed\n");
		fclose(fp);
		return 1;
	}
	fclose(fp);
	printf("Done\n");

	printf("    - Loading 80-col font from file \"%s\": ",FONT80_FILE);
	if (!(vid_font80[0] = malloc(28672))) {
		printf("Failed\n");
		return 1;
	}
	if (!(vid_font80[1] = malloc(28672))) {
		printf("Failed\n");
		return 1;
	}
	if ((fp = fopen(FONT80_FILE,"r")) == NULL) {
		printf("Failed\n");
		return 1;
	}
	if (fread(vid_font80[0],1,28672,fp) != 28672) {
		printf("Failed\n");
		fclose(fp);
		return 1;
	}
	if (fread(vid_font80[1],1,28672,fp) != 28672) {
		printf("Failed\n");
		fclose(fp);
		return 1;
	}
	fclose(fp);
	printf("Done\n");

	return 0;
}

/* Update the X display by sending the current image. This routine is	*/
/* called automatically by the EMUL_update() procedure.			*/

void VID_update()
{
	(*VID_updateRoutine)();
#ifdef MITSHM
	XShmPutImage(vid_display, vid_window, vid_gc, vid_image, 0, 0, 0, 0, VID_WIDTH, VID_HEIGHT, False);
#else
	XPutImage(vid_display, vid_window, vid_gc, vid_image, 0, 0, 0, 0, VID_WIDTH, VID_HEIGHT);
#endif
	XFlush(vid_display);
}

void VID_reset()
{
	vid_80col = 0;
	vid_altcharset = 0;
	vid_text = 1;
	vid_mixed = 0;
	vid_page2 = 0;
	vid_hires = 0;
	vid_dblres = 0;
	vid_currmode = 0;

	vid_a2mono = 0;
	vid_linear = 0;
	vid_super = 0;

	vid_bordercolor = 0;
	vid_textcolor[0] = 0;
	vid_textcolor[1] = 15;

	VID_newMode();
}

void VID_shutdown()
{
	printf("\nShutting down emulator console\n");
	printf("    - Destroying image\n");
#ifdef MITSHM
	XShmDetach(vid_display,&vid_shminfo);
	shmdt(vid_shminfo.shmaddr);
	shmctl(vid_shminfo.shmid,IPC_RMID,0);
#else
	XDestroyImage(vid_image);
#endif
	printf("    - Closing display\n");
	VID_deallocTextColors();
	XFreeColormap(vid_display, vid_colormap);
	XCloseDisplay(vid_display);
}

void VID_drawStatus1(char *msg)
{
	int	height,margin;

	height = vid_fontinfo->ascent + vid_fontinfo->descent;
	margin = vid_fontinfo->ascent;

	XSetForeground(vid_display, vid_gc, vid_white);
	XSetBackground(vid_display, vid_gc, vid_black);

	XDrawImageString(vid_display, vid_window, vid_gc, 0, VID_HEIGHT+height+margin, msg, strlen(msg));
}

void VID_drawStatus2(char *msg)
{
	int	height,margin;

	height = vid_fontinfo->ascent + vid_fontinfo->descent;
	margin = vid_fontinfo->ascent;

	XSetForeground(vid_display, vid_gc, vid_white);
	XSetBackground(vid_display, vid_gc, vid_black);

	XDrawImageString(vid_display, vid_window, vid_gc, 0, VID_HEIGHT+height*2+margin, msg, strlen(msg));
}

/* Handle the gruntwork of switching video modes. We update the vid_currmode	*/
/* variable, add/remove the necessary fault handlers, and then redraw the 	*/
/* screen in the new mode.							*/

void VID_newMode()
{
	int	oldmode,x,y;

	oldmode = vid_currmode;
	if (vid_super) {
		vid_currmode = VID_MODE_SUPER;
		for (x = 0 ; x < 15 ; x++) {
			for (y = 0 ; y < 15 ; y++) {
				vid_old_super_palettes[x][y] = 0xFFFF;
			}
		}
	} else {
		if (vid_text) {
			if (vid_80col) {
				vid_currmode = (!mem_80store && vid_page2)? VID_MODE_80TEXT2 : VID_MODE_80TEXT1;
			} else {
				vid_currmode = (!mem_80store && vid_page2)? VID_MODE_40TEXT2 : VID_MODE_40TEXT1;
			}
		} else {
			if (vid_hires) {
				if (vid_dblres && vid_80col) {
					vid_currmode = (!mem_80store && vid_page2)? VID_MODE_DHIRES2 : VID_MODE_DHIRES1;
				} else {
					vid_currmode = (!mem_80store && vid_page2)? VID_MODE_HIRES2 : VID_MODE_HIRES1;
				}
			} else {
				if (vid_dblres && vid_80col) {
					vid_currmode = (!mem_80store && vid_page2)? VID_MODE_DLORES2 : VID_MODE_DLORES1;
				} else {
					vid_currmode = (!mem_80store && vid_page2)? VID_MODE_LORES2 : VID_MODE_LORES1;
				}
			}
		}
	}
	if (oldmode != vid_currmode) {
		for (x = 0 ; x < 512 ; x++) mem_slowram_changed[x] = 0xFFFFFFFF;
		VID_drawBorder();
	}
	switch(vid_currmode) {
		case VID_MODE_40TEXT1:	VID_updateRoutine = VID_refreshText40Page1;
					break;
		case VID_MODE_40TEXT2:	VID_updateRoutine = VID_refreshText40Page2;
					break;
		case VID_MODE_80TEXT1:	VID_updateRoutine = VID_refreshText80Page1;
					break;
		case VID_MODE_80TEXT2:	VID_updateRoutine = VID_refreshText80Page2;
					break;
		case VID_MODE_LORES1:	VID_updateRoutine = VID_refreshLoresPage1;
					break;
		case VID_MODE_LORES2:	VID_updateRoutine = VID_refreshLoresPage2;
					break;
		case VID_MODE_DLORES1:	VID_updateRoutine = VID_refreshDLoresPage1;
					break;
		case VID_MODE_DLORES2:	VID_updateRoutine = VID_refreshDLoresPage2;
					break;
		case VID_MODE_HIRES1:	VID_updateRoutine = VID_refreshHiresPage1;
					break;
		case VID_MODE_HIRES2:	VID_updateRoutine = VID_refreshHiresPage2;
					break;
		case VID_MODE_DHIRES1:	VID_updateRoutine = VID_refreshDHiresPage1;
					break;
		case VID_MODE_DHIRES2:	VID_updateRoutine = VID_refreshDHiresPage2;
					break;
		case VID_MODE_SUPER:	VID_updateRoutine = VID_refreshSuperHires;
					break;
	}
}

void VID_drawBorder()
{
	register int	hwidth,vwidth,x,y;
	register PIXEL	color;
	
	if (vid_currmode == VID_MODE_SUPER) {
		hwidth = 0;
		vwidth = 40;
	} else {
		hwidth = 40;
		vwidth = 48;
	}
	color = vid_textpalette[vid_bordercolor];
	for (y = 0 ; y < vwidth ; y++) {
		for (x = 0 ; x < VID_WIDTH ; x++) {
			*(vid_lines[y]+x) = color;
		}
	}
	if (hwidth) {
		for (y = vwidth ; y < VID_HEIGHT - vwidth ; y++) {
			for (x = 0; x < hwidth ; x++) {
				*(vid_lines[y]+x) = color;
			}
			for (x = VID_WIDTH - vwidth - 1 ; x < VID_WIDTH ; x++) {
				*(vid_lines[y]+x) = color;
			}
		}
	}
	for (y = VID_HEIGHT - vwidth ; y < VID_HEIGHT; y++) {
		for (x = 0 ; x < VID_WIDTH ; x++) {
			*(vid_lines[y]+x) = color;
		}
	}
}
