/*********************************************************************
 *                                                                   *
 *                     XGS : Apple IIGS Emulator                     *
 *                                                                   *
 *        Written and Copyright (C)1996 by Joshua M. Thompson        *
 *                                                                   *
 *  You are free to distribute this code for non-commercial purposes *
 * I ask only that you notify me of any changes you make to the code *
 *     Commercial use is prohibited without my written permission    *
 *                                                                   *
 *********************************************************************/

/*
 * File: dc2img.c
 *
 * This is a short but simple program to convert a Macintosh DiskCopy
 * image into an XGS ".img" file.
 *
 * Usage: dc2img sourcefile destfile
 */

#include <xgs.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "disks.h"
#include "diskcopy.h"

byte	block_buffer[512];

diskcopy_header	dcimage;
xgs_image	image;

void swapWord(unsigned short *val)
{
	wswap	temp;
	byte	i;

	temp.W = *val;
	i = temp.B.B1;
	temp.B.B1 = temp.B.B2;
	temp.B.B2 = i;
	*val = temp.W;
}

void swapLong(unsigned long *val)
{
	lswap	temp;
	byte	i;

	temp.L = *val;
	i = temp.B.B1;
	temp.B.B1 = temp.B.B4;
	temp.B.B4 = i;
	i = temp.B.B2;
	temp.B.B2 = temp.B.B3;
	temp.B.B3 = i;
	*val = temp.L;
}

void main(int argc, char *argv[])
{
	int		i,j;
	char		buffer[64],*cp;
	unsigned long	checksum;
	unsigned short	blocks;
	word		ctmp;
	FILE		*fp1,*fp2;

	if ((fp1 = fopen(argv[1],"r")) == NULL) {
		perror("Can't open source file");
		exit(1);
	}

	if ((fp2 = fopen(argv[2],"w")) == NULL) {
		perror("Can't open source file");
		exit(1);
	}

	fread((void *) &dcimage,1,sizeof(diskcopy_header),fp1);

#ifdef LSB_FIRST
	swapLong(&dcimage.dataSize);
	swapLong(&dcimage.tagSize);
	swapLong(&dcimage.dataChecksum);
	swapLong(&dcimage.tagChecksum);
	swapWord(&dcimage.private);
#endif

	if (dcimage.private != 0x0100) {
		printf("Unknown DiskCopy format. Cannot continue.\n");
		fclose(fp1);
		fclose(fp2);
		exit(1);
	}

	strncpy(buffer,dcimage.diskName+1,dcimage.diskName[0]);
	buffer[(int) dcimage.diskName[0]] = 0;

	printf("\nDiskCopy image statistics:\n\n");

	printf("    Disk name : %s\n\n",buffer);

	printf("    Data size : %ld bytes\n",dcimage.dataSize);
	printf("Data checksum : $%08lX\n\n",dcimage.dataChecksum);

	printf("     Tag size : %ld bytes\n",dcimage.tagSize);
	printf(" Tag checksum : $%08lX\n\n",dcimage.tagChecksum);

	switch (dcimage.diskFormat) {
		case 0x00 :	cp = "400K";
				break;
		case 0x01 :	cp = "800K";
				break;
		case 0x02 :	cp = "720K";
				break;
		case 0x03 :	cp = "1440K";
				break;
		default :	cp = "Unknown";
				break;
	}
	printf("    Disk type : %s\n",cp);
	switch(dcimage.formatByte) {
		case 0x12 :	cp = "400K Macintosh";
				break;
		case 0x22 :	cp = ">400K Macintosh";
				break;
		case 0x24 :	cp = "800K Apple II";
				break;
		default :	cp = "Unknown";
				break;
	}
	printf("  Disk format : %s\n\n",cp);

	bzero((void *) &image,sizeof(image));

	blocks = dcimage.dataSize / 512;
	printf("** Creating an XGS image file of %hd blocks.\n",blocks);

	image.magic[0] = 'X';
	image.magic[1] = 'G';
	image.magic[2] = 'S';
	image.magic[3] = '!';
	image.version = 0x00;
	image.protected = 1;
	image.num_blocks = blocks;

#ifndef LSB_FIRST
	swapWord(&image.num_blocks);
#endif

	cp = strdup(buffer);	/* save this in case we need it */

	printf("\nEnter a name for this image (1-32 chars) or RETURN to use DiskCopy name:\n");
	gets(buffer);
	if (strlen(buffer)) {
		strncpy(image.name,buffer,31);
		free(cp);
	} else {
		strncpy(image.name,cp,31);
	}

	fwrite(&image,1,sizeof(image),fp2);

	printf("\nCopying data: ");
	checksum = 0;
	for (i = 0 ; i < blocks ; i++) {
		fread(block_buffer,1,512,fp1);
		fwrite(block_buffer,1,512,fp2);
		for (j = 0 ; j < 256 ; j++) {
			ctmp = *((word *) block_buffer+j);
#ifdef LSB_FIRST
			swapWord(&ctmp);
#endif
			checksum += ctmp;
			ctmp = checksum & 0x01;
			checksum = (checksum >> 1) | (ctmp << 31);
		}
		printf(".");
		fflush(stdout);
	}

	if (checksum == dcimage.dataChecksum) {
		printf("\n\nChecksums match. Image is OK.\n");
	} else {
		printf("\n\nWARNING: data checksum mismatch (wanted $%08lX, got $%08lX). Image may be corrupt.\n",dcimage.dataChecksum,checksum);
	}

	fclose(fp1);
	fclose(fp2);
}
