/*********************************************************************
 *                                                                   *
 *                     XGS : Apple IIGS Emulator                     *
 *                                                                   *
 *        Written and Copyright (C)1996 by Joshua M. Thompson        *
 *                                                                   *
 *  You are free to distribute this code for non-commercial purposes *
 * I ask only that you notify me of any changes you make to the code *
 *     Commercial use is prohibited without my written permission    *
 *                                                                   *
 *********************************************************************/

/*
 * File: mem-main.c
 *
 * Memory manager initialization and core routines.
 */

#include "xgs.h"
#include "disks.h"
#include "video.h"

int MEM_init(void)
{
	int	i;
	FILE	*fp;

	if (mem_ramsize < 1) mem_ramsize = 1;
	if (mem_ramsize > 8) mem_ramsize = 8;

	printf("    - Allocating %d KB for ROM: ",ROMSIZE*64);
	rom_memory = malloc(ROMSIZE*65536);
	if (rom_memory == NULL) {
		printf("Failed\n");
		return 1;
	}
	printf("Done\n");

	printf("    - Allocating 128 KB for slow RAM: ");
	slow_memory = malloc(2*65536);
	if (slow_memory == NULL) {
		printf("Failed\n");
		return 1;
	}
	printf("Done\n");

	printf("    - Allocating %d MB for fast RAM: ",mem_ramsize);
	fast_memory = malloc(mem_ramsize * 1048576);
	if (fast_memory == NULL) {
		printf("Failed\n");
		return 1;
	}
	printf("Done\n");

	printf("    - Loading ROM from file \"%s\": ",ROM_FILE);
	if ((fp = fopen(ROM_FILE,"r")) == NULL) {
		printf("Failed\n");
		return 2;
	}
	if (fread(rom_memory,1,ROMSIZE*65536,fp) != ROMSIZE*65536) {
		printf("Failed\n");
		return 2;
	}
	fclose(fp);
	printf("Done\n");

	/* Initialize all pages */

	printf("\nInitializing emulator memory\n");
	for (i = 0x0000 ; i < mem_ramsize * 4096 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = fast_memory + (i * 256);
		mem_pages[i].readFlags = mem_pages[i].writeFlags = 0;
	}
	for (i = mem_ramsize * 4096 ; i < 0xE000 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = junk_memory;
		mem_pages[i].readFlags = mem_pages[i].writeFlags = MEM_FLAG_INVALID;
	}
	for (i = 0xE000 ; i < 0xE200 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = slow_memory + ((i-0xE000)*256);
		mem_pages[i].readFlags = mem_pages[i].writeFlags = 0;
	}
#ifdef ROM_03
	for (i = 0xE200 ; i < 0xFC00 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = junk_memory;
		mem_pages[i].readFlags = mem_pages[i].writeFlags = MEM_FLAG_INVALID;
	}
	for (i = 0xFC00 ; i < 0x10000 ; i++) {
		mem_pages[i].readPtr = rom_memory + ((i-0xFC00)*256);
		mem_pages[i].writePtr = junk_memory;
		mem_pages[i].readFlags = 0;
		mem_pages[i].writeFlags = MEM_FLAG_INVALID;
	}
#else
	for (i = 0xE200 ; i < 0xFE00 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = NULL;
		mem_pages[i].readFlags = mem_pages[i].writeFlags = MEM_FLAG_INVALID;
	}
	for (i = 0xFE00 ; i < 0x10000 ; i++) {
		mem_pages[i].readPtr = rom_memory + ((i-0xFE00)*256);
		mem_pages[i].writePtr = junk_memory;
		mem_pages[i].readFlags = 0;
		mem_pages[i].writeFlags = MEM_FLAG_INVALID;
	}
#endif

	return 0;
}

void MEM_update()
{
}

/* Memory manager reset routine. Here we reset the memory soft swithches to a	*/
/* reasonable power-up/reset state, and then rebuild the mem_pages[] array.	*/

void MEM_reset()
{
	int	i;

	mem_80store = 0;
	mem_auxrd = 0;
	mem_auxwrt = 0;
	mem_altzp = 0;

	mem_lcbank2 = 0;
	mem_lcread = 0;
	mem_lcwrite = 0;
	mem_lcsecond = 0;

	mem_shadow_text = 1;
	mem_shadow_hires1 = 1;
	mem_shadow_hires2 = 1;
	mem_shadow_super = 0;
	mem_shadow_aux = 1;
	mem_shadow_lc = 1;

	for (i = 0 ; i < 5 ; i++) mem_slot_reg[i] = 0;
	mem_slot_reg[5] = 1;
	mem_slot_reg[6] = 1;
	mem_slot_reg[7] = 1;

	MEM_rebuildMainMem();
	MEM_rebuildAltZpMem();
}

void MEM_shutdown()
{
	printf("\nShutting down emulator memory\n");
}

void MEM_rebuildMainMem()
{
	int	i,offset1,offset2;

	offset1 = mem_auxrd? 0x10000 : 0;
	offset2 = mem_auxwrt? 0x10000 : 0;

	for (i = 0x0002 ; i < 0x0004 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}

	if (mem_80store) {
		offset1 = offset2 = vid_page2? 0x10000 : 0;
	}

	for (i = 0x0004 ; i < 0x0008 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}

	offset1 = mem_auxrd? 0x10000 : 0;
	offset2 = mem_auxwrt? 0x10000 : 0;

	for (i = 0x0008 ; i < 0x0020 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}

	if (mem_80store) {
		offset1 = offset2 = vid_page2? 0x10000 : 0;
	}

	for (i = 0x0020 ; i < 0x0040 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}

	offset1 = mem_auxrd? 0x10000 : 0;
	offset2 = mem_auxwrt? 0x10000 : 0;

	for (i = 0x0040 ; i < 0x00C0 ; i++) {
		mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
		mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
	}
	MEM_rebuildShadowMem();
}

void MEM_rebuildAltZpMem()
{
	int	i,offset;

	offset = mem_altzp? 0x10000 : 0;

	for (i = 0x0000 ; i < 0x0002 ; i++) {
		mem_pages[i].readPtr = mem_pages[i].writePtr = fast_memory + (i * 256) + offset;
	}
	MEM_rebuildLangCardMem();
}

void MEM_rebuildLangCardMem()
{
	int	i,offset1,offset2;

	MEM_buildLanguageCard(0xE000,0,slow_memory);
	MEM_buildLanguageCard(0xE100,0,slow_memory+0x10000);
	offset1 = mem_altzp? 0x10000 : 0;
	if (mem_shadow_lc) {
		MEM_buildLanguageCard(0x0000,offset1,fast_memory);
		MEM_buildLanguageCard(0x0100,0x10000,fast_memory);
	} else {
		offset1 = mem_auxrd? 0x10000 : 0;
		offset2 = mem_auxwrt? 0x10000 : 0;
		for (i = 0x00C0 ; i < 0x0100 ; i++) {
			mem_pages[i].readPtr = fast_memory + (i*256) + offset1;
			mem_pages[i].writePtr = fast_memory + (i*256) + offset2;
		}
		for (i = 0x01C0 ; i < 0x0200 ; i++) {
			mem_pages[i].readPtr = fast_memory + (i*256);
			mem_pages[i].writePtr = fast_memory + (i*256);
		}
	}
}

void MEM_rebuildShadowMem()
{
	int	i,flag;

	if (mem_80store) {
		flag = vid_page2? MEM_FLAG_SHADOW_E1 : MEM_FLAG_SHADOW_E0;
	} else {
		flag = mem_auxwrt? MEM_FLAG_SHADOW_E1 : MEM_FLAG_SHADOW_E0;
	}

	for (i = 0x0004 ; i < 0x0008 ; i++) {
		mem_pages[i].writeFlags = mem_shadow_text? flag : 0;
	}

	for (i = 0x0020 ; i < 0x0040 ; i++) {
		mem_pages[i].writeFlags = mem_shadow_hires1? flag : 0;
	}

	flag = mem_auxwrt? MEM_FLAG_SHADOW_E1 : MEM_FLAG_SHADOW_E0;

	for (i = 0x0040 ; i < 0x0060 ; i++) {
		mem_pages[i].writeFlags = mem_shadow_hires2? flag : 0;
	}

	for (i = 0x0104 ; i < 0x0108 ; i++) {
		mem_pages[i].writeFlags = mem_shadow_text? MEM_FLAG_SHADOW_E1 : 0;
	}

	flag = (mem_shadow_hires1 && mem_shadow_aux) || mem_shadow_super;

	for (i = 0x0120 ; i < 0x0140 ; i++) {
		mem_pages[i].writeFlags = flag? MEM_FLAG_SHADOW_E1 : 0;
	}

	flag = (mem_shadow_hires2 && mem_shadow_aux) || mem_shadow_super;

	for (i = 0x0140 ; i < 0x0160 ; i++) {
		mem_pages[i].writeFlags = flag? MEM_FLAG_SHADOW_E1 : 0;
	}

	for (i = 0x0160 ; i < 0x01A0 ; i++) {
		mem_pages[i].writeFlags = mem_shadow_super? MEM_FLAG_SHADOW_E1 : 0;
	}
}

/* This routine builds a language card in the given bank.  This	*/
/* means building all mapping info from $C000 up to $FFFF.	*/

void MEM_buildLanguageCard(int bank,int offset,byte *ram)
{
	int	i,lcbank;

	mem_pages[bank+0xC0].readFlags = MEM_FLAG_IO;
	mem_pages[bank+0xC0].writeFlags = MEM_FLAG_IO;

	for (i = 0xC1 ; i < 0xD0 ; i++) {
#ifdef ROM_03
		mem_pages[bank+i].readPtr = rom_memory + 0x030000 + (i * 256);
#else
		mem_pages[bank+i].readPtr = rom_memory + 0x010000 + (i * 256);
#endif
		mem_pages[bank+i].readFlags = 0;
		mem_pages[bank+i].writePtr = junk_memory;
		mem_pages[bank+i].writeFlags = MEM_FLAG_INVALID;
	}

	mem_pages[bank+0xC5].readPtr = dsk_slot5_rom;
	mem_pages[bank+0xC7].readPtr = dsk_slot7_rom;

	lcbank = mem_lcbank2? 0x1000 : 0;

	for (i = 0xD0 ; i < 0xE0 ; i++) {
		if (mem_lcread) {
			mem_pages[bank+i].readPtr = ram + (i*256) + offset - lcbank;
		} else {
#ifdef ROM_03
			mem_pages[bank+i].readPtr = rom_memory + 0x030000 + (i * 256);
#else
			mem_pages[bank+i].readPtr = rom_memory + 0x010000 + (i * 256);
#endif
		}
		if (mem_lcwrite) {
			mem_pages[bank+i].writePtr = ram + (i*256) + offset - lcbank;
			mem_pages[bank+i].writeFlags = 0;
		} else {
			mem_pages[bank+i].writePtr = junk_memory;
			mem_pages[bank+i].writeFlags = MEM_FLAG_INVALID;
		}
	}
	for (i = 0xE0 ; i < 0x100 ; i++) {
		if (mem_lcread) {
			mem_pages[bank+i].readPtr = ram + (i*256) + offset;
		} else {
#ifdef ROM_03
			mem_pages[bank+i].readPtr = rom_memory + 0x030000 + (i * 256);
#else
			mem_pages[bank+i].readPtr = rom_memory + 0x010000 + (i * 256);
#endif
		}
		if (mem_lcwrite) {
			mem_pages[bank+i].writePtr = ram + (i*256) + offset;
			mem_pages[bank+i].writeFlags = 0;
		} else {
			mem_pages[bank+i].writePtr = junk_memory;
			mem_pages[bank+i].writeFlags = MEM_FLAG_INVALID;
		}
	}
}
