/*********************************************************************
 *                                                                   *
 *                     XGS : Apple IIGS Emulator                     *
 *                                                                   *
 *        Written and Copyright (C)1996 by Joshua M. Thompson        *
 *                                                                   *
 *  You are free to distribute this code for non-commercial purposes *
 * I ask only that you notify me of any changes you make to the code *
 *     Commercial use is prohibited without my written permission    *
 *                                                                   *
 *********************************************************************/

/*
 * File: emul.c
 *
 * Startup and shutdown code for the emulator, as well as the update
 * routine called 50 times per second.
 */

#include "xgs.h"
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>

#include "adb.h"
#include "clock.h"
#include "disks.h"
#include "emul.h"
#include "sound.h"
#include "video.h"

int	emul_vbl;
int	emul_vblirq;
int	emul_qtrsecirq;
int	emul_onesecirq;

int	emul_speed;
int	emul_speed2;

int	emul_tick;

char	emul_buffer[256];

unsigned long	emul_last_cycles;
unsigned long	emul_target_cycles;

struct itimerval	emul_timer;

int EMUL_init()
{
	int			err;
	struct sigaction	action;

	printf("%s\n",VERSION);
	printf("Written and (C) 1996 by Joshua M. Thompson\n");

	if (chdir(XGS_DIR)) {
		perror("Can't change directory to XGS directory");
		return 1;
	}

	if ((err = MEM_init())) return err;
	if ((err = VID_init())) {
		MEM_shutdown();
		return err;
	}
	if ((err = SND_init())) {
		VID_shutdown();
		MEM_shutdown();
		return err;
	}
	if ((err = ADB_init())) {
		SND_shutdown();
		VID_shutdown();
		MEM_shutdown();
		return err;
	}
	if ((err = CLK_init())) {
		ADB_shutdown();
		SND_shutdown();
		VID_shutdown();
		MEM_shutdown();
		return err;
	}
	if ((err = DSK_init())) {
		CLK_shutdown();
		ADB_shutdown();
		SND_shutdown();
		VID_shutdown();
		MEM_shutdown();
		return err;
	}

	printf("\nInitialzing emulator core\n");

	emul_vbl = 0;
	emul_vblirq = 0;
	emul_qtrsecirq = 0;
	emul_onesecirq = 0;
	emul_speed = 1;
	emul_speed2 = 1;
	emul_tick = 0;
	emul_last_cycles = 0;
	emul_target_cycles = 2500000;

	printf("    - Setting interval timer to %d Hz : ",TIMER_FREQ);
	action.sa_handler = EMUL_update;
	action.sa_flags = SA_RESTART;
	sigemptyset(&action.sa_mask);
	if (sigaction(SIGALRM, &action, NULL)) {
		printf("Failed\n");
		CLK_shutdown();
		ADB_shutdown();
		SND_shutdown();
		VID_shutdown();
		MEM_shutdown();
		return 1;
	}
	emul_timer.it_interval.tv_sec = emul_timer.it_value.tv_sec = 0;
	emul_timer.it_interval.tv_usec = emul_timer.it_value.tv_usec = 1000000/TIMER_FREQ;
	if (setitimer(ITIMER_REAL, &emul_timer, NULL)) {
		printf("Failed\n");
		CLK_shutdown();
		ADB_shutdown();
		SND_shutdown();
		VID_shutdown();
		MEM_shutdown();
		return 1;
	}
	printf("Done\n");

	return 0;
}

void EMUL_update(int parm)
{
	int		diff;

	if (emul_tick & 0x01) {
		emul_vbl ^= 0x80;
		if (emul_vbl == 0x00) {
			if (emul_vblirq) {
				mem_diagtype |= 0x08;
				CPU_irq();
			}
			MEM_update();
			ADB_update();
			CLK_update();
			DSK_update();
		}
	}

	if ((emul_tick == 0) || (emul_tick == 25) || (emul_tick == 50) || (emul_tick == 75)) {
		SND_update();
		VID_update();
		if (emul_qtrsecirq) {
			CPU_irq();
			mem_diagtype |= 0x10;
		}
	}

	if (!emul_tick) {
		diff = cpu_cycle_count - emul_last_cycles;
		emul_last_cycles = cpu_cycle_count;

		sprintf(emul_buffer,"Target speed = %2.3f MHz, Actual speed = %2.3f MHz",
					emul_target_cycles / 1000000.0, diff / 1000000.0);
		VID_drawStatus2(emul_buffer);
	}

	if (emul_tick++ == TIMER_FREQ) emul_tick = 0;
}

void EMUL_reset()
{
	MEM_reset();
	VID_reset();
	SND_reset();
	ADB_reset();
	CLK_reset();
	DSK_reset();
	CPU_reset();
}

void EMUL_shutdown()
{
	DSK_shutdown();
	CLK_shutdown();
	ADB_shutdown();
	SND_shutdown();
	VID_shutdown();
	MEM_shutdown();
	exit(0);
}

void EMUL_trace(int parm)
{
	CPU_setTrace(parm);
}

void EMUL_handleWDM(byte parm)
{
	switch(parm) {
		case 0xC5 :	DSK_handleSlot5();
				break;
		case 0xC7 :	DSK_handleSlot7();
				break;
		case 0xFD :	CPU_setTrace(0);
				break;
		case 0xFE :	CPU_setTrace(1);
				break;
		case 0xFF :	EMUL_shutdown();
				break;
		default :	break;
	}
}

byte EMUL_getVBL(byte val)
{
	return emul_vbl;
}

void usage(char *myname)
{
	fprintf(stderr,"\nUsage: %s [-s5d1 filename ] [-s5d2 filename ] [-s6d1 filename ] [-s6d2 filename ] [-s7d1 filename ] [-s7d2 filename ] [-trace]\n",myname);
	exit(1);
}

void main(int argc, char *argv[])
{
	int		err,i;

	i = 0;
	while (++i < argc) {
		if (!strcmp(argv[i],"-trace")) {
			EMUL_trace(1);
		} else if (!strcmp(argv[i],"-s5d1")) {
			if (++i >= argc) usage(argv[0]);
			s5d1_startup = argv[i];
		} else if (!strcmp(argv[i],"-s5d2")) {
			if (++i >= argc) usage(argv[0]);
			s5d2_startup = argv[i];
		} else if (!strcmp(argv[i],"-s6d1")) {
			if (++i >= argc) usage(argv[0]);
			s6d1_startup = argv[i];
		} else if (!strcmp(argv[i],"-s6d2")) {
			if (++i >= argc) usage(argv[0]);
			s6d2_startup = argv[i];
		} else if (!strcmp(argv[i],"-s7d1")) {
			if (++i >= argc) usage(argv[0]);
			s7d1_startup = argv[i];
		} else if (!strcmp(argv[i],"-s7d2")) {
			if (++i >= argc) usage(argv[0]);
			s7d2_startup = argv[i];
		} else usage(argv[0]);
	}
	if ((err = EMUL_init())) exit(err);

	printf("\nEMULATOR IS RUNNING. PRESS CONTROL-END TO EXIT\n");

	EMUL_reset();

	CPU_setUpdatePeriod(0);
	CPU_run();

	EMUL_shutdown();
}
