/*********************************************************************
 *                                                                   *
 *                     XGS : Apple IIGS Emulator                     *
 *                                                                   *
 *        Written and Copyright (C)1996 by Joshua M. Thompson        *
 *                                                                   *
 *  You are free to distribute this code for non-commercial purposes *
 * I ask only that you notify me of any changes you make to the code *
 *     Commercial use is prohibited without my written permission    *
 *                                                                   *
 *********************************************************************/

/*
 * File: clock.c
 *
 * Emulates the clock registers at $C033 and $C034 (which also has the
 * screen border color as well).
 *
 * This is a partial implementation of the clock chip. We don't
 * support setting the time registers, and we don't support accessing 
 * the BRAM through the old single-byte commands (since the IIGS
 * doesn't seem to use those commands anyway).
 */

#include <time.h>
#include "xgs.h"
#include "clock.h"
#include "video.h"

/* The clock chip data and control registers */

int	clk_data_reg;
int	clk_ctl_reg;

byte	bram[256];

union {
	byte		B[4];
	time_t		L;
} clk_curr_time;

/* Clock state. 0 = waiting for command, 1 = waiting for second	*/
/* byte of two-part command, 2 = waiting for data real/write	*/
/* Bit 7 is set for read and clear for write. Bit 6 is set if	*/
/* BRAM is being accessed, or clear if clock register.		*/

int	clk_state;

/* Clock register/BRAM Location  being accessed. */

int	clk_addr;

int CLK_init()
{
	FILE	*fp;

	printf("\nInitialzing clock and battery RAM\n");
	printf("    - Loading battery RAM from file \"%s\" : ",BRAM_FILE);
	fp = fopen(BRAM_FILE,"r");
	if (fp == NULL) {
		printf("Failed\n");
	} else {
		if (fread(bram,1,256,fp) != 256) {
			printf("Failed\n");
		} else {
			printf("Done\n");
		}
		fclose(fp);
	}
	return 0;
}

void CLK_update()
{
}

void CLK_reset()
{
	clk_data_reg = 0;
	clk_ctl_reg &= 0x3F;
	clk_state = 0;
}

void CLK_shutdown()
{
	FILE	*fp;

	printf("\nShutting down the clock and battery RAM\n");
	printf("    - Saving battery RAM to file \"%s\" : ",BRAM_FILE);
	fp = fopen(BRAM_FILE,"w");
	if (fp == NULL) {
		printf("Failed\n");
		return;
	}
	if (fwrite(fp,1,256,fp) != 256) {
		printf("Failed\n");
		return;
	}
	fclose(fp);
	printf("Done\n");
}

byte CLK_getData(byte val)
{
	return clk_data_reg;
}

byte CLK_setData(byte val)
{
	clk_data_reg = val;
	return 0;
}

byte CLK_getControl(byte val)
{
	return clk_ctl_reg | VID_getBorder(0);
}

byte CLK_setControl(byte val)
{
	VID_setBorder(val & 0x0F);
	clk_ctl_reg = val & 0xF0;
	switch (clk_state & 0x0F) {
		case 0 :	if (val & 0x40) break;
				if ((clk_data_reg & 0x78) == 0x38) {
					clk_state = (clk_data_reg & 0x80) | 0x41;
					clk_addr = (clk_data_reg & 0x07) << 5;
				} else if ((clk_data_reg & 0x73) == 0x01) {
					clk_state = (clk_data_reg & 0x80) | 0x02;
					clk_addr = (clk_data_reg >> 2) & 0x03;
				}
				break;
		case 1 :	if ((val & 0x40) || (clk_data_reg & 0x03)) break;
				clk_addr |= ((clk_data_reg >> 2) & 0x1F);
				clk_state++;
				break;
		case 2 :	if (clk_state & 0x40) {
					bram[clk_addr] = clk_data_reg;
				} else {
					/* Can't set the system time */
				}
				clk_state = 0;
				break;
		default :	clk_state = 0;
				break;
	}
	if ((clk_state & 0x8F) == 0x82) {
		if (clk_state & 0x40) {
			clk_data_reg = bram[clk_addr];
		} else {
			if (!clk_addr) clk_curr_time.L = time(NULL) + CLK_OFFSET;
#ifdef LSB_FIRST
			clk_data_reg = clk_curr_time.B[clk_addr];
#else
			clk_data_reg = clk_curr_time.B[3-clk_addr];
#endif
		}
		clk_state = 0;
	}
	clk_ctl_reg &= 0x7F;	/* Clear transaction bit since we're done */
	return 0;
}
