/* */
/*	pro.c - Apple ProDOS File System Functions */
/*	 */
/*	Written by Galen C. Hunt [email: gchunt@cc.dixie.edu] */
/*	Released into the public domain with no warranties. */
/* */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "aftp.h"

#define	ACCESS_READ		0x01	/* File Access Bit Flags */
#define ACCESS_WRITE	0x02
#define ACCESS_HIDDEN	0x04
#define ACCESS_DEFAULT	0xe3

#define	STORAGE_DELETED	0x0		/* File Storage Types */
#define STORAGE_LEAF	0x1
#define	STORAGE_BRANCH	0x2
#define	STORAGE_TREE	0x3
#define STORAGE_PASCAL	0x4
#define STORAGE_DIR		0xd
#define	STORAGE_VOLUME	0xf

#define	MAX_ITEM_LEN	15
#define	MAX_PATH_LEN	255

typedef struct _proinfo
{
	word	free_block;			/* First block of free block bitmap */
	word 	directory_block;	/* key block number of current working directory */
	byte	volume_version;		/* ProDOS file system version */
	char	volume_name[16];	/* Volume Name */
	char	last_item[16];		/* Name of file on last attempt to find. */
	word	last_directory;		/* Directory of last attempt to find. */
	long	blocks_free;		/* Number of free blocks on disk. */
} * proinfo;

typedef struct _dir_it
{
	disk	d;
	char	name[16];
    int 	key;
    int 	ent_len;
	int		ent_per_blk;
    int 	ent_left;
    int 	ent_offset;
	word	blk;
} dir_it;

typedef struct _filent
{
	disk	d;					/* File's Directory Entry Location */
	word	der_block;
	int		der_offset;
	int		der_length;
    byte	de_storage_type;	/* File's Directory Entry Information */
    char 	de_file_name[16];
    byte 	de_file_type;
    word	de_key_pointer;
    int 	de_blocks_used;
    long 	de_eof;
    byte 	de_creation_date[4];
    byte 	de_version;
    byte 	de_min_version;
	byte 	de_access;
    word 	de_sub_type;
    byte	de_modified_date[4];
    word	de_header_pointer;
} *filent;

typedef struct _pfile
{
	filent	fent;
	word	data_blk;
	long 	data_offset;
	long	file_eof;
	int 	data_pos;
	int 	data_left;
	int		writing;
} *pfile;


/*	Static Variables */
static	byte	free_mask[8] = { 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01 };

/*	Local Prototypes */
static 	pfile	pfile_create(disk d, char* file_name, byte type, word sub_type);
static 	pfile	pfile_open(disk d, char* file_name, char* mode);
static 	int 	pfile_read(pfile pf, byte* buf, int size);
static 	int 	pfile_write(pfile pf, byte* buf, int size);
static 	int 	pfile_seek(pfile pf, long offset);
static	void	pfile_close(pfile pf);
static 	int 	dir_open(disk d, dir_it* dit, int blk);
static	int		dir_reset(dir_it* dit);
static	word	dir_block_add(dir_it *dit);
static	int		dir_entry_get(filent fent);
static	int		dir_entry_put(filent fent);
static 	int 	dir_read(dir_it* dit, filent fent);


/*	date_to_str: */
/*		Convert a ProDOS date to a string. */
static	char *	months[13] = { "???", "JAN", "FEB", "MAR", "APR", "MAY", "JUN", "JUL", "AUG", "SEP", "OCT", "NOV", "DEC" };
static	char *	date_to_str(char *out, byte * dt)
{
	int day = dt[0] & 0x1f;
    int month = ((dt[1] & 1) << 3) | (dt[0] >> 5);
    int	year = dt[1] >> 1;
    int minute = dt[2] /*& 0x1f*/;
    int hour = dt[3] /*& 0x1f*/;

	if (dt[0] || dt[1] || dt[2] || dt[3])
		sprintf(out, "%2d-%3s-%2d %2d:%02d", day, months[month], year, hour, minute);
	else
		strcpy(out, "<NO DATE>      ");
	return out;
}

static	void	date_set(byte *dt)
{
	struct	tm *t;
	time_t	lt;

	memset(dt, 0, 4);
	lt = time(NULL);
	if (t = localtime(&lt))
	{
		t->tm_mon++;
		dt[0] = t->tm_mday & 0x1f;
		dt[0] |= (t->tm_mon & 0x7) << 5;
		dt[1] = t->tm_year << 1;
		dt[1] |= (t->tm_mon & 0x8);
		dt[2] = t->tm_min;
		dt[3] = t->tm_hour;
	}
}

/*	name_to_str: */
/*		Return the name as used in volumes and directories as a c-string. */
static 	char *	name_to_str(char *out, byte* in)
{
    int 	inlen = *in & 0xf;

	memcpy(out, in + 1, inlen);
	out[inlen] = '\0';
	return out;
}

/*	type_to_str: */
/*		Converts a numeric file type to a string */
static	char *	type_to_str(char *out, byte type)
{
	char *	sp = NULL;

	switch (type)
	{
	case 0x01: sp = "BAD"; break;
	case 0x04: sp = "TXT"; break;
	case 0x05: sp = "DIC"; break;
	case 0x06: sp = "BIN"; break;
	case 0x07: sp = "FNT"; break;
	case 0x08: sp = "IMG"; break;
	case 0x0c: sp = "SOS"; break;
	case 0x0f: sp = "DIR"; break;
	case 0x19: sp = "ADB"; break;
	case 0x1a: sp = "AWP"; break;
	case 0x1b: sp = "ASP"; break;
	case 0xf0: sp = "COM"; break;
	case 0xf1: sp = "#1 "; break;
	case 0xf2: sp = "#2 "; break;
	case 0xf3: sp = "#3 "; break;
	case 0xf4: sp = "#4 "; break;
	case 0xf5: sp = "#5 "; break;
	case 0xf6: sp = "#6 "; break;
	case 0xf7: sp = "#7 "; break;
	case 0xf8: sp = "#8 "; break;
	case 0xf9: sp = "RES"; break;
	case 0xfa: sp = "INT"; break;
	case 0xfb: sp = "IVR"; break;
	case 0xfc: sp = "BAS"; break;
	case 0xfd: sp = "VAR"; break;
	case 0xfe: sp = "REL"; break;
	case 0xff: sp = "SYS"; break;
	default: sprintf(out, "$%02X", type);
	}
	if (sp)
		strcpy(out, sp);
	return out;
}


/*	freeblock_size: */
/*		Returns the number of free blocks on the disk. */
static	long		freeblock_size(disk d)
{
	long	cnt = 0;
	proinfo	p = (proinfo)d->os_data;
	word	i, j;
	int		num_blks;
	byte	b;

	num_blks = 1 + ((d->blocks - 1) / (8 * BYTES_PER_BLOCK));
	for (i = 0; i < num_blks; i++)
	{
		disk_read_block(d, p->free_block + i);
		for (j = 0; j < BYTES_PER_BLOCK; j++)
			if (b = d->buffer[j])
				for (; b; b >>= 1)
					cnt++;
	}
	return (p->blocks_free = cnt);
}

/*	freeblock_alloc: */
/*		Return the next free block or 0 if failed. (Marks block as used) */
static	word		freeblock_alloc(disk d)
{
	proinfo	p = (proinfo)d->os_data;
	word	i, j, k;
	word	blk;
	word	blks = d->blocks;
	int		num_blks;

	num_blks = 1 + ((d->blocks - 1) / (8 * BYTES_PER_BLOCK));
	for (blk = 0, i = 0; i < num_blks; i++)
	{
		disk_read_block(d, p->free_block + i);
		for (j = 0; j < BYTES_PER_BLOCK && blk < blks; j++, blk += 8)
			if (d->buffer[j])
			{
				for (k = 0; k < 8; k++)
					if (d->buffer[j] & free_mask[k])
						break;
				d->buffer[j] &= ~free_mask[k];
				disk_dirty(d);
				p->blocks_free--;
				return blk + k;
			}
	}
	printf("Disk Full.\n");
	return 0;
}

/*	freeblock_free: */
/*		Frees the given block. */
static	int		freeblock_free(disk d, word block)
{
	proinfo	p = (proinfo)d->os_data;
	word	i, j, k;

	if (block > d->blocks)
		return -1;

	i = block / (BYTES_PER_BLOCK * 8);
	j = (block % (BYTES_PER_BLOCK * 8)) / 8;
	k = block % 8;

	disk_read_block(d, p->free_block + i);
	d->buffer[j] |= free_mask[k];
	disk_dirty(d);
	p->blocks_free++;
	return 0;
}

/*	find_item: */
/*		Given a pathname, return its next item. */
static	char *	find_item(char *path, char *item)
{
	char *	sp;

	if (sp = strchr(path, '/'))
	{
		*sp++ = '\0';
		strcpy(item, path);
		path = sp;
	}
	else
	{
		strcpy(item, path);
		while (*path)
			path++;
	}
	return path;
}

/*	find_file: */
/*		Given a file name, locate its directory entry. */
/*		Returns the directory block number if good. */
static 	filent	find_file(disk d, char *path)
{
	filent	fent = NULL;
	proinfo	p = (proinfo)d->os_data;
	dir_it 	dit;
	char	name[256];
	char *	item;
	int		found;
	int		dir_blk;

	if (!(fent = alloc(sizeof(*fent))))
	{
		printf("Unable to allocate memory for file entry.\n");
		return NULL;
	}
	item = p->last_item;

	path = strcpy(name, path);
	memset(fent, 0, sizeof(*fent));
	dir_blk = p->directory_block;
	p->last_directory = 0;
	item[0] = '\0';
	if (path[0] == '/')
	{
		dir_blk = 2;
		path++;
	}
	while (path[0])
	{
		path = find_item(path, item);
		if (item[0] == '\0' || !strcmp(item, "."))	/* Current */
			continue;
		if (!strcmp(item, ".."))				/* Parent */
		{
			if (dir_blk != 2)
			{
				disk_read_block(d, dir_blk);
				dir_blk = word_at(d->buffer + 39);
			}
			continue;
		}

    	if (!dir_open(d, &dit, dir_blk))
		{
			free(fent);
			return NULL;
		}
    	for (found = 0; dir_read(&dit, fent);)
		{
			if (fent->de_storage_type == STORAGE_DELETED)
				continue;
			if (!strcmp(fent->de_file_name, item))
			{
				found = 1;
				break;
			}
		}
		if (!found)
		{
			if (path[0] == '\0')
				p->last_directory = dir_blk;
			free(fent);
			return NULL;
		}
		if (fent->de_file_type == 0xf && fent->de_storage_type == STORAGE_DIR)
			dir_blk = fent->de_key_pointer;
	}
	p->last_directory = dir_blk;
	return fent;
}

/*	pfile_from_fent: */
/*		Returns a valid pfile from a fent. */
static 	pfile	pfile_from_fent(filent fent, char *mode)
{
	pfile	pf = NULL;

	if (fent->de_storage_type != STORAGE_LEAF && fent->de_storage_type != STORAGE_BRANCH && fent->de_storage_type != STORAGE_TREE)
	{
		printf("Unsupported storage type: 0x%x\n", fent->de_storage_type);
		return NULL;
	}
	if (*mode == 'w' && !(fent->de_access & ACCESS_WRITE))
	{
		printf("Unable to write to locked file.\n");
		return NULL;
	}
    if (pf = alloc(sizeof(*pf)))
	{
		pf->fent = fent;
    	pf->data_offset = 0;
		pf->writing = *mode == 'w' ? 1 : 0;
		pfile_seek(pf, 0);
	}
    return pf;
}


/*	pfile_create: */
/*		Creates an entry for the given file and opens it for writing.   */
/*		Returns a pfile if successful, or NULL if failed. */
static 	pfile	pfile_create(disk d, char* file_name, byte type, word sub_type)
{
	proinfo	p = (proinfo)d->os_data;
	dir_it	dit;
	filent	fent;
	int		found;
	word	blk;

	if (fent = find_file(d, file_name))			/* Also need to get directory */
	{
		free(fent);
		printf("File already exists\n");
		return NULL;
	}

	if (!p->last_directory)
	{
		printf("Internal Error: last_directory = %d\n", p->last_directory);
		return NULL;
	}

	if (!dir_open(d, &dit, p->last_directory))
		return NULL;
	if (!(fent = alloc(sizeof(*fent))))
		return NULL;

    for (found = 0; dir_read(&dit, fent);)
	{
		if (fent->de_storage_type == STORAGE_DELETED)
		{
			found = 1;
			break;
		}
	}
	if (!found)
	{
		if (!(fent->der_block = dir_block_add(&dit)))
		{
			free(fent);
			return NULL;
		}
		fent->der_offset = 4;
		fent->der_length = dit.ent_len;
	}

	if (!(blk = freeblock_alloc(d)))
	{
		free(fent);
		return NULL;
	}	
	fent->de_storage_type = STORAGE_LEAF;
	strcpy(fent->de_file_name, file_name);
	fent->de_file_type = type;
	fent->de_key_pointer = blk;
	fent->de_blocks_used = 1;
	fent->de_eof = 0;
	date_set(fent->de_creation_date);
	fent->de_version = 0;
    fent->de_min_version = 0;
	fent->de_access = ACCESS_DEFAULT;
	fent->de_sub_type = sub_type;
	fent->de_header_pointer = dit.key;
    date_set(fent->de_modified_date);
	dir_entry_put(fent);

	disk_read_block(d, fent->de_header_pointer);
	disk_dirty(d);
	word_set(d->buffer + 37, word_at(d->buffer + 37) + 1);

	return pfile_from_fent(fent, "w");
}

/*	pfile_open: */
/*		Open a prodos file. */
static 	pfile	pfile_open(disk d, char* file_name, char* mode)
{
	pfile	pf = NULL;
	filent	fent;

	if (fent = find_file(d, file_name))
	{
		if (fent->de_storage_type != STORAGE_LEAF && fent->de_storage_type != STORAGE_BRANCH && fent->de_storage_type != STORAGE_TREE)
		{
			free(fent);
			printf("Unsupported storage type: 0x%x\n", fent->de_storage_type);
			return NULL;
		}
		if (*mode == 'w' && !(fent->de_access & ACCESS_WRITE))
		{
			free(fent);
			printf("Unable to write to locked file.\n");
			return NULL;
		}
		pf = pfile_from_fent(fent, mode);
	}
    return pf;
}


word	reference_get(disk d, int ref)
{
	return (word)d->buffer[ref] | ((word)d->buffer[ref + 256] << 8);
}

void	reference_set(disk d, int ref, word block)
{
	d->buffer[ref] = (block & 0xff);
	d->buffer[ref + 256] = (block & 0xff00) >> 8;
	disk_dirty(d);
}

/*	pfile_seek: */
/* */
static 	int 	pfile_seek(pfile pf, long offset)
{
	filent	fent = pf->fent;
	disk	d = fent->d;
    word 	master, index, data;
	int		i;

	if (offset <0)
		return 0;
	if (!pf->writing && offset > fent->de_eof)
		return 0;

	if (pf->writing)
	{
		if (offset >= (1L*BYTES_PER_BLOCK) && fent->de_storage_type == STORAGE_LEAF)
		{										/* Expand to BRANCH */
			if (!(data = freeblock_alloc(d)))
				return 0;
			fent->de_blocks_used++;
			disk_read_block_zero(d, data);
			reference_set(d, 0, fent->de_key_pointer);
			fent->de_key_pointer = data;
			fent->de_storage_type = STORAGE_BRANCH;
		}
		if (offset >= (256L*BYTES_PER_BLOCK) && fent->de_storage_type == STORAGE_BRANCH)
		{										/* Expand to TREE */
			if (!(data = freeblock_alloc(d)))
				return 0;
			fent->de_blocks_used++;
			disk_read_block_zero(d, data);
			reference_set(d, 0, fent->de_key_pointer);
			fent->de_key_pointer = data;
			fent->de_storage_type = STORAGE_TREE;
		}
	}

    pf->data_offset = offset;
    master = index = data = fent->de_key_pointer;

    switch (fent->de_storage_type)
	{
    case STORAGE_TREE:
		disk_read_block(d, master);
		i = offset >> 17;
		index = reference_get(d, i);			/* fall through... */
		if (!index && pf->writing)
		{
			if (!(index = freeblock_alloc(d)))
				return 0;
			fent->de_blocks_used++;
			disk_read_block(d, master);
			reference_set(d, i, index);
			disk_read_block_zero(d, index);
		}
    case STORAGE_BRANCH:
		if (index)								/* Assume 0 : sparse area */
		{
			disk_read_block(d, index);
			i = (offset >> 9) & 255;
			data = reference_get(d, i);
			if (!data && pf->writing)
			{
				if (!(data = freeblock_alloc(d)))
					return 0;
				fent->de_blocks_used++;
				disk_read_block(d, index);
				reference_set(d, i, data);
				disk_read_block_zero(d, data);
			}
		}
		else
			data = 0;							/* fall through... */
    case STORAGE_LEAF:
		;										/* fall through... */
    }
	/* Note: a data block of zero signals the read routine that it is in */
	/* a sparse block. */
    pf->data_blk = data;
    pf->data_pos = offset & 511;
    pf->data_left = 512 - pf->data_pos;
    if (!pf->writing && pf->data_left > fent->de_eof - pf->data_offset)
		pf->data_left = fent->de_eof - pf->data_offset;
    return 1;
}

static 	int 	pfile_read(pfile pf, byte* buf, int size)
{
	filent	fent = pf->fent;
	disk	d = fent->d;
    int 	bytes_read, step;

    if (pf->data_offset >= fent->de_eof)
		return 0;

    for (bytes_read = 0; bytes_read < size; )
	{
		if (!pfile_seek(pf, pf->data_offset))
			break;
		if (pf->data_blk)
			disk_read_block(d, pf->data_blk);
		else
			memset(d->buffer, 0, sizeof(d->buffer));
		step = pf->data_left;
		if (step > size - bytes_read)
	    	step = size - bytes_read;
		memcpy(buf + bytes_read, d->buffer + pf->data_pos, step);
		pf->data_pos += step;
		pf->data_left -= step;
		pf->data_offset += step;
		bytes_read += step;
		if (pf->data_offset >= fent->de_eof)
	    	break;
    }
    return bytes_read;
}

static 	int 	pfile_write(pfile pf, byte* buf, int size)
{
	filent	fent = pf->fent;
	disk	d = fent->d;
    int 	bytes_wrote, step;

    for (bytes_wrote = 0; bytes_wrote < size; )
	{
		if (!pfile_seek(pf, pf->data_offset))
			break;
		step = pf->data_left;
		if (step > size - bytes_wrote)
	    	step = size - bytes_wrote;
		if (!pf->data_blk)
			break;
		disk_read_block(d, pf->data_blk);
		disk_dirty(d);
		memcpy(d->buffer + pf->data_pos, buf + bytes_wrote, step);
		pf->data_pos += step;
		pf->data_left -= step;
		pf->data_offset += step;
		bytes_wrote += step;
		if (pf->data_offset > fent->de_eof)
			fent->de_eof = pf->data_offset;
    }
    return bytes_wrote;
}

static	void	pfile_close(pfile pf)
{
	if (pf)
	{
		if (pf->writing)
		{
			date_set(pf->fent->de_modified_date);
			dir_entry_put(pf->fent);
		}
		free(pf->fent);
		free(pf);
	}
}

/*//////////////////////////////////////////////////////////////////////////// */
/*	Directory Search Functions */

static 	int 	dir_open(disk d, dir_it* dit, int blk)
{
    disk_read_block(d, blk);
    dit->d = d;
	name_to_str(dit->name, d->buffer + 4);
    dit->key = blk;
    dit->ent_len = d->buffer[0x23];
    dit->ent_per_blk = d->buffer[0x24];
	dir_reset(dit);
    return 1;
}

static	int		dir_reset(dir_it* dit)
{
	disk_read_block(dit->d, dit->key);
	dit->blk = dit->key;
    dit->ent_offset = 4 + dit->ent_len;
    dit->ent_left = dit->ent_per_blk - 1;
}

static	word	dir_block_add(dir_it *dit)
{
	disk	d = dit->d;
	word	next;
	word	prev;

	for (next = dit->key; next; next = word_at(d->buffer + 2))
	{
		disk_read_block(d, next);
		prev = next;
	}

	if (next = freeblock_alloc(d))
	{
		disk_read_block(d, prev);
		disk_dirty(d);
		word_set(d->buffer + 2, next);

		disk_read_block_zero(d, next);
		word_set(d->buffer + 0, prev);
	}
	return next;
}

static	int		dir_entry_get(filent fent)
{
	disk	d = fent->d;
	byte *	ep;

	disk_read_block(d, fent->der_block);
	ep = d->buffer + fent->der_offset;

    fent->de_storage_type = ep[0] >> 4;
	name_to_str(fent->de_file_name, ep);
    fent->de_file_type = ep[0x10];
    fent->de_key_pointer = word_at(ep + 0x11);
    fent->de_blocks_used = word_at(ep + 0x13);
    fent->de_eof = (long)ep[0x17] * 65536 + word_at(ep + 0x15);
	memcpy(fent->de_creation_date, ep + 0x18, 4);
    fent->de_version = ep[0x1c];
    fent->de_min_version = ep[0x1d];
    fent->de_access = ep[0x1e];
    fent->de_sub_type = word_at(ep + 0x1f);
	memcpy(fent->de_modified_date, ep + 0x21, 4);
    fent->de_header_pointer = word_at(ep + 0x25);
	if (!fent->de_creation_date[0] && !fent->de_creation_date[1] && !fent->de_creation_date[2] && !fent->de_creation_date[3])
		memcpy(fent->de_creation_date, fent->de_modified_date, 4);
}

static	int		dir_entry_put(filent fent)
{
	disk	d = fent->d;
	byte *	ep;
	int		namelen;

	disk_read_block(d, fent->der_block);
	disk_dirty(d);
	ep = d->buffer + fent->der_offset;

	memset(ep, 0, fent->der_length);
	namelen = strlen(fent->de_file_name);

	ep[0] = (fent->de_storage_type << 4) | namelen;
	memcpy(ep + 1, fent->de_file_name, namelen);
	ep[0x10] = fent->de_file_type;
	word_set(ep + 0x11, fent->de_key_pointer);
	word_set(ep + 0x13, fent->de_blocks_used);
	word_set(ep + 0x15, fent->de_eof);
	ep[0x17] = fent->de_eof / 65536;
	memcpy(ep + 0x18, fent->de_creation_date, 4);
    ep[0x1c] = fent->de_version;
    ep[0x1d] = fent->de_min_version;
    ep[0x1e] = fent->de_access;
    word_set(ep + 0x1f, fent->de_sub_type);
	memcpy(ep + 0x21, fent->de_modified_date, 4);
    word_set(ep + 0x25, fent->de_header_pointer);
}

static 	int 	dir_read(dir_it* dit, filent fent)
{
	disk	d = dit->d;

	disk_read_block(d, dit->blk);
    if (dit->ent_left == 0)
	{
		dit->blk = word_at(d->buffer + 2);
		if (dit->blk == 0)
	    	return 0;
		disk_read_block(d, dit->blk);
		dit->ent_offset = 4;
		dit->ent_left = dit->ent_per_blk;
    }

	fent->d = d;
	fent->der_block = dit->blk;
	fent->der_offset = dit->ent_offset;
	fent->der_length = dit->ent_len;

	dir_entry_get(fent);

    dit->ent_offset += dit->ent_len;
    dit->ent_left--;

    return 1;
}

static	int		free_branch(disk d, word key)
{
	int		i;
	word	blks[256];

	disk_read_block(d, key);
	for (i = 0; i < 256; i++)
		blks[i] = reference_get(d, i);
	for (i = 0; i < 256; i++)
		if (blks[i])
			freeblock_free(d, blks[i]);
	freeblock_free(d, key);
    return 1;
}

static	int		free_tree(disk d, word key)
{
	int		i;
	word	blks[256];

	disk_read_block(d, key);
	for (i = 0; i < 256; i++)
		blks[i] = reference_get(d, i);
	for (i = 0; i < 256; i++)
		if (blks[i])
			free_branch(d, blks[i]);
	freeblock_free(d, key);
    return 1;
}


/*//////////////////////////////////////////////////////////////////////////// */
/* */
/*	OS Interface Functions */

static int	i_init(disk d)
{
	proinfo	p = (proinfo)d->os_data;

    /* peek at the volume key block to see if things are OK */
    /* a more robust check would be nice */
	disk_read_block(d, 2);
    if ((d->buffer[4] >> 4) != STORAGE_VOLUME)
	{
		printf("Bad ProDOS volume key: 0x%x\n", d->buffer[4] >> 4);
        return -1;
	}

	if (d->blocks < word_at(d->buffer + 41))
		d->blocks = word_at(d->buffer + 41);
	p->directory_block = 2;
	p->free_block = word_at(d->buffer + 39);
	p->volume_version = d->buffer[0x20];
	name_to_str(p->volume_name, d->buffer + 4);
	freeblock_size(d);

    return 1;
}

static int	i_dir(disk d, char *name)
{
	filent	fent;
	dir_it 	dit;
	int		blk;
	proinfo	p = (proinfo)d->os_data;
	char	cdate[24];
	char	mdate[24];
	char	type[4];

	if (!(fent = find_file(d, name)))
	{
		printf("Directory not found.\n");
		return -1;
	}

	blk = p->last_directory;

    dir_open(d, &dit, blk);
	printf("%s: %s%s\n", d->name, blk== 2 ? "/" :  "", dit.name);
	printf(" Name           Type Blks  Modified        Created       End File SubFile  Key\n");
    while (dir_read(&dit, fent))
	{
		if (fent->de_storage_type == STORAGE_DELETED)
	    	continue;
	
		date_to_str(mdate, fent->de_modified_date);
		date_to_str(cdate, fent->de_creation_date);
		type_to_str(type, fent->de_file_type);
		printf("%c%-15.15s %s%5d %s %s %7ld $%4x  %5d\n"
			, fent->de_access & ACCESS_WRITE ? ' ' : '*'
			, fent->de_file_name, type, fent->de_blocks_used
			, mdate, cdate, fent->de_eof, fent->de_sub_type, fent->de_key_pointer);
    }
	free(fent);

	printf("%ld blocks, %ld used, %ld free.\n", d->blocks, d->blocks - p->blocks_free, p->blocks_free);
	return 0;
}

static int	i_get(disk d, char *name, FILE *fp)
{
	pfile	pf;
	byte	buffer[512];
	int		step;
	long	len;

	if (pf = pfile_open(d, name, "r"))
	{
		for (len = 0;; len += step)
		{
			if ((step = pfile_read(pf, buffer, sizeof(buffer))) <= 0)
				break;
			if (text_mode == MODE_TEXT)
			{
				int		i;

				for (i = 0; i < step; i++)
					if (buffer[i] == '\r')
						buffer[i] = '\n';
			}
			if (fwrite(buffer, 1, step, fp) <= 0)
				break;
		}
		pfile_close(pf);
	}
	else
	{
		printf("Couldn't open remote file: %s\n", name);
		return -2;
	}
	printf("Read %ld bytes.\n", len);
	return 0;
}

static int	i_put(disk d, char *name, FILE *fp)
{
	pfile	pf;
	byte	buffer[512];
	int		step;
	long	len;
	byte	type;
	word	sub_type;

	if (text_mode == MODE_TEXT)
	{
		type = 4;
		sub_type = 0;
	}
	else
	{
		type = 6;
		sub_type = 0x2000;
	}
	if (!(pf = pfile_open(d, name, "w")))
		pf = pfile_create(d, name, type, sub_type);

	if (!pf)
	{
		printf("Couldn't create remote file: %s\n", name);
		return -2;
	}

	for (len = 0;; len += step)
	{
		if ((step = fread(buffer, 1, sizeof(buffer), fp)) <= 0)
			break;

		if (text_mode == MODE_TEXT)
		{
			int		i;

			for (i = 0; i < step; i++)
				if (buffer[i] == '\n')
					buffer[i] = '\r';
		}
		if ((step = pfile_write(pf, buffer, step)) <= 0)
			break;
	}
	pfile_close(pf);
	printf("Wrote %ld bytes.\n", len);

	return 0;
}

static int	i_del(disk d, char *name)
{
	filent	fent;
	byte	type;

	if (!(fent = find_file(d, name)))
	{
		printf("File not found.\n");
		return 0;
	}
	switch (type = fent->de_storage_type)
	{
	case STORAGE_DIR:
		printf("Use RMDIR to delete a directory.\n");
		return 0;
	case STORAGE_TREE:
	case STORAGE_BRANCH:
	case STORAGE_LEAF:
		fent->de_storage_type = STORAGE_DELETED;
		dir_entry_put(fent);
		disk_read_block(d, fent->de_header_pointer);
		disk_dirty(d);
		word_set(d->buffer + 37, word_at(d->buffer + 37) + 1);
		if (type == STORAGE_TREE)
			free_tree(d, fent->de_key_pointer);
		else if (type == STORAGE_BRANCH)
			free_branch(d, fent->de_key_pointer);
		else
			freeblock_free(d, fent->de_key_pointer);
		return 1;
	default:
		printf("DEL can't be used to delete: %s\n", name);
	}
	return 0;
}

static int	i_cd(disk d, char *name)
{
	proinfo	p = (proinfo)d->os_data;
	filent	fent;

	if (!(fent = find_file(d, name)))
	{
		printf("Directory not found.\n", name);
		return -1;
	}
	p->directory_block = p->last_directory;
	return 0;
}

static int	i_mkdir(disk d, char *name)
{
	return -1;
}

static int	i_rmdir(disk d, char *name)
{
	return -1;
}

static int	i_name(char *name, int maxlen)
{
	char *	start = name;
	char *	op;
	int		ilen;								/* Item Length */
	int		plen;								/* Path Length */
	int		c;

	for (plen = ilen = 0, op = start; (c = *name++) && plen < MAX_PATH_LEN;)
	{
		if (c == '\\')							/* Convert MS-DOS separators */
			c = '/';
		if (c == ':')							/* Skip drive designations */
		{
			plen = ilen = 0;
			op = start;
			continue;
		}

		if (c == '/')
			ilen = 0;
		else
		{
			if (ilen >= MAX_ITEM_LEN)
				continue;
			c = toupper(c);
		}
		*op++ = c;
		ilen++;
		plen++;
	}
	*op = '\0';
	return 0;
}

static int	i_close(disk d)
{
	return 0;
}

struct _osys	os_pro = { sizeof(struct _proinfo), i_init, i_dir, i_get, i_put, i_del, i_cd, i_mkdir, i_rmdir, i_name, i_close };

/*	End of File */
