' ************************************************************************
' **
' **  synctime.b 	Atomic clock time synchronizer
' **			(C)opyright 1994 Morgan Davis
' **
' ** History:
' **
' ** When    Who Ver	What
' ** ======= === ======	==============================================
' ** 11feb90 mwd	Creation.
' ** 17mar90 mwd	Added time zone switching logic.
' ** 18mar90 mwd	Reorganized program layout.
' ** 24mar90 mwd 1.0	Reworked resources to allow for user-editable
' **			time period table.
' ** 31mar90 mwd	Fixed IsLeapYear to be 100% right. Required
' **			change to tableYear resource to be a full year
' ** 01apr90 mwd 1.2	Fixed bug in theDate range check.
' ** 02apr90 mwd	Compensated for standard times given during
' **			daylight savings.
' ** 04apr90 mwd 1.3	Rewrote connection routines to use the National
' **			Bureau of Standards clock.  Added support for
' **			ThunderClock, Serial Pro, ProClock and
' **			TimeMaster clocks.
' ** 05apr90 mwd	Fixed screen settings after card set.
' ** 14apr90 mwd 1.4	Added round() macro, fixed mod() to act like
' **			the real MOD, removed day-of-week hack and put
' **			a real algorithm in.
' ** 18apr90 mwd 1.5	Fixed clock lookup routine (failed to skip over
' **			pattern data when a matched failed).
' ** 22apr90 dzd 1.6	Verified that time received was time data, not
' **			NIST header info.  Includes error checking.
' **			Pads ints to two digits when setting slot-based
' **			clock cards. (Danield Z. Davidson)
' ** 24apr90 mwd	Changed theDay to be within 0-6 range.	Fixed
' **			bug in slot-based clock formats (TimeMaster
' **			uses a leading year value, but Thunder and Pro-
' **			clocks do not).
' ** 26apr90 mwd	Changed a few lines to work with new basic.h
' ** 29apr90 dzd/mwd	Does day of week calc before calling SetNewTime
' ** 07may90 mwd	Moved jdate calcs before DOW calc!
' ** 07oct90 mwd	Stripped out apr/oct dates from resource file
' **			now that I know that the NIST header has codes
' **			to determine ST and DST times.
' ** 05mar91 mwd 1.7	Fixed a bogus "dtsCode" (should be "dstCode").
' ** 07apr91 mwd 1.8	dstCode was not being adjusted as is the date
' **			when the GMT time propels us into a new date.
' **			Caused reporting (and changing) zones to be off.
' ** 12feb92 mwd 1.9	Fixed day-of-week calc -- was really broken
' **			in a leap year.  Removed Julian date calc stuff
' **			as is not needed.  Fixed negating dstCode when
' **			GMT time moves us back a day.
' ** 25jan94 mwd 3.0	DST reporting checks dstCode for > 0 validity
' **
' ************************************************************************

#define	IDENT_PROG "synctime"
#define	IDENT_VERS "3.0"
#define	IDENT_DATE "25jan94"
#define	IDENT_NAME "Morgan_Davis"

#include <basic.h>
#include <prodos.h>
#include <appleio.h>
#include <proline/proline.h>

#define	ASM_STUFF	$300
#define	ZPAGE_REG	$00
#define	STDTIME		0
#define DLSTIME		1
#define	MAX_BAD_LINES	20	' Number of bogus lines to skip before error

#define	UNKNOWN_CLOCK	-1	' Clock types
#define	IIGS_CLOCK	0
#define	THUNDERCLOCK	1
#define	PROCLOCK	2
#define	TIMEMASTER	3
#define	SERIAL_PRO	4


	' ====================================================
	' RangeCheck macro compares v to n and x returning -1
	' if less than n, 1 if greater than x, or 0 if between
	' ====================================================

#define	RangeCheck(v,n,x)	((v < n) * -1) + (v > x)

#define	IsLeapYear(y)	not mod(y,4) and mod(y,100) or not mod(y,400)

#define	MonthDays(m)	val(mid$(DaysInMonth$, m * 2, 2))

#define	PostMsg(str)	msg$ = str : gosub _PostMsg
#define	PostExit(str)	PostMsg(str) : goto _PostExit


' **********************************************************************
' **
' ** Main
' **
	gosub AppInit		' Set up ProLine application environment
	gosub ValidateRun	' See if synctime can run
	gosub InitGlobals	' Init global variables
	gosub GetResources	' Get resources
	gosub DialClockService	' Call the clock service, get and set time
	gosub ChangeZone	' Fix the time zone (if needed)
	gosub ShowOffTime	' Show how far off the clock was
_PostExit:
	& in ioConsole		' Reset...
	& pr ioConsole		' ...I/O hooks.
	&hangup			' Poof, get offline
	&usr uResetModem
	gosub SendMsgFile	' Set up message for sendmail posting
	&clear
goto Exit			' All done!


' ***************************************************
' **
' ** ValidateRun - Determine if synctime can even run
' **

ValidateRun:
	if not SuperUser then
		PostMsg ("can't run")
		goto Exit
	endif

	gosub FindClock
	if clockID = UNKNOWN_CLOCK then
		PostMsg ("doesn't know about your clock")
		goto Exit
	endif

	&fn fnCarrier, online
	if online then
		PostMsg ("port in use")
		goto Exit
	endif
return


' *****************************************************
' **
' ** InitGlobals - Initialize global variables and data

InitGlobals:
	spoolDir$	= SPOOL_MAIL_PATH
	resourceFile$	= RSRC_PATH + "synctime.rsrc"
	configFile$	= RSRC_PATH + "startup.rsrc"
	tempFile$	= SysInfo$[plTempDir] + "sursrctmp"
	msgFile$	= SysInfo$[plTempDir] + "synctime.log"

	DaysInMonth$	= "?312831303130313130313031"
	dim zone$[2]
return


' **************************************************
' **
' ** GetResources - Get resources from resource file
' **

GetResources:
	&GETINFO resourceFile$, info$
	if info$ = "" then
		PostMsg ("can't find " + resourceFile$)
		goto Exit
	endif

        onerr goto rsrcErr
        	fOpen resourceFile$
	        fRead resourceFile$

		input   phoneNumber$, \
		        baudRate, \
		        utcDiff, \
			zone$[STDTIME], \
			zone$[DLSTIME], \
			baseYear

		fClose

		&spc (phoneNumber$), phoneNumber$	' Strip any spaces
		&spc (zone$[STDTIME]), zone$[STDTIME]
		&spc (zone$[DLSTIME]), zone$[DLSTIME]

		' Fix DaysInMonth$ for February if it's a leap year

		&time (oldTime$)
		oldYear = val(mid$(oldTime$, 13, 2)) + baseYear
		if IsLeapYear(oldYear) then
			&mid$ (DaysInMonth$, 4) = "29"
		endif
	        onerr goto HandleError
		return
		
	rsrcErr:
	onerr goto HandleError
	fClose
	PostMsg ("error reading resources")
goto Exit


' ******************************************************************
' **
' ** DialClockService - Call it, get online, and get the date & time
' **

DialClockService:
	&pr ioConsole
	PostMsg ("dialing " + phoneNumber$)
	&speed (baudRate > 300) + (baudRate > 1200) + (baudRate > 2400) + \
		(baudRate > 9600)
	&call phoneNumber$
	&wait for carrier, result
	if result <> wfcConnect then
		PostExit ("no connect")
	endif

	PostMsg ("connected, setting the " + clockName$)

	' =================================================
	' Skip past NIST header junk by reading lines until
	' we get one that appears to contain time/date info
	' =================================================

	&clear				' Flush input buffer to get ready
	&wait -20 for ") *^M", x	' Wait for a valid time info line
	if not x then
		PostExit ("couldn't get past NIST header")
	endif

	i = 0				' Clear counter
	& in ioBoth
	repeat
		& pr ioModem
		& read a$		' get and echo a line
		& pr ioConsole
		i = i + 1
	until right$(a$,1) = "#" or i = 15
	if i = 15 then
		PostExit ("delay correction failed")
	endif

GetDateAndTime:
	&clear				' Flush input
	&wait -10 for "#^M^J", x	' Ensure we get delay-modified time
	if not x then
		PostExit ("expected delay-correct time")
	endif

	' ===========================================
	' Read the first 26 characters of time info
	' and begin processing it immediately so that
	' we're ready to set the time on the mark.
	' ===========================================

	&get (26), timeLine$		' Need first 26 characters

	theYear   = val(mid$(timeLine$,  7, 2))
	theMonth  = val(mid$(timeLine$, 10, 2))
	theDate   = val(mid$(timeLine$, 13, 2))
	theHour   = val(mid$(timeLine$, 16, 2))
	theMinute = val(mid$(timeLine$, 19, 2))
	theSecond = val(mid$(timeLine$, 22, 2))
	dstCode   = val(mid$(timeLine$, 25, 2))
	
	' ==================================================
	' This routine must check to see if the difference
	' from the UTC time propels us into a different date
	' (either forward or backward in time).
	' ==================================================

	theHour = theHour + utcDiff + (SysInfo$[plZone] = zone$[DLSTIME])
	delta = RangeCheck (theHour, 0, 23)
	if sgn(delta) then
		theHour = (-24 * delta) + theHour
		theDate = theDate + delta		' Adjust date
		dstCode = dstCode - delta		' Adjust DST count
		delta = RangeCheck (theDate, 1, MonthDays(theMonth))
		if sgn(delta) then
			theMonth = theMonth + delta
			delta = RangeCheck (theMonth, 1, 12)
			if sgn(delta) then
				theMonth = (-12 * delta) + theMonth
				theYear = theYear + delta
				if theYear > 99 then
					theYear = 0
					baseYear = baseYear + 100
				endif
			endif
			theDate = MonthDays(theMonth)
		endif
	endif

	' =============================================
	' Determine the day of the week (0=Sun...6=Sat)
	' =============================================

	K = int(1 / theMonth + .6)
	L = theYear + baseYear - K
	delta = int(13 * (12 * K + theMonth + 1) / 5) \
		+ int(5 * L / 4) \
		- int(L / 100) \
		+ int(L / 400) \
		+ theDate - 1
	theDay = delta - 7 * int(delta / 7)

	' ==============================================
	' Get remainder of time info from service, which
	' also provides us with "on the mark" timing.
	' Get the current time for difference reporting.
	' Set the real time, and then get the new time.
	' ==============================================

	&get timeLine2$			' Get end of line from service
	&time (oldTime$)		' When we were
	gosub SetNewTime		' Set new time
	&time (curTime$)		' When we are
return


' ************************************************************
' **
' ** ShowOffTime - Report how far off the clock was in seconds
' **

ShowOffTime:
	oldHour = val(mid$(oldTime$,16))
	oldMinute = val(mid$(oldTime$,19,2))
	oldSecond = val(right$(oldTime$,2))

	oldTotal = oldHour * 3600 + oldMinute * 60 + oldSecond
	newTotal = theHour * 3600 + theMinute * 60 + theSecond
	delta = newTotal - oldTotal

	PostMsg (timeLine$ + timeLine2$)
	a$ = "clock was "
	if not delta then
		a$ = a$ + "right on time!"
	else
		if delta < 0 then
			a$ = a$ + "ahead"
		else
			a$ = a$ + "behind"
		endif
		a$ = a$ + " by " + str$(abs(delta)) +  " seconds"
	endif
	PostMsg (a$)

	' ==============================
	' Post DST/ST count downs
	' ==============================

	if dstCode > 0 and dstCode <> 50 then
		if dstCode > 50 then
			delta = dstCode - 50
			a$ = "Daylight Savings Time"
		else
			delta = dstCode
			a$ = "Standard Time"
		endif
		j$ = " begins "
		i$ = "in " + str$(delta) + " days"
		if delta = 1 then
			i$ = "today at 2AM"
			if theHour >= 2 then j$ = " began "
		endif
		if delta = 2 then i$ = "tomorrow at 2AM"
		PostMsg (a$ + j$ + i$)
	endif

	PostMsg ("it is now " + curTime$ + " " + SysInfo$[plZone])
return


' ************************************************************
' **
' ** ChangeZone - Change the time zone in startup.rsrc file and RAM
' **
' ** Determine which part of the year we're in (dstCode) to see
' ** if it is governed by standard time or daylight savings time. 
' ** Checks to see if the system's currently set zone matches the
' ** one we're supposed to be in.  If not, the zone is changed in
' ** config RAM as well as in the startup.rsrc file. 
' **
' ** The dstCode has the following values:
' **
' **       00 = We are on standard time (xST)
' ** 99 to 51 = Now on xST, when count is 51 go to xDT at 2:00am local time
' **       50 = We are on daylight savings time (xDT).
' ** 49 to 01 = Now on xDT, when count is 01 go to xST at 2:00am local time
' **

ChangeZone:
	dlsFlag = dstCode > 0 and dstCode < 51
	if (dstCode = 1 or dstCode = 51) and theHour >= 2 then
		dlsFlag = dstCode = 51
	endif
	if zone$[dlsFlag] <> SysInfo$[plZone] then
		oldZone$ = SysInfo$[plZone]
		SysInfo$[plZone] = zone$[dlsFlag]
		PostMsg ("changing from "+ oldZone$ + " to "+ zone$[dlsFlag])

		&restore SysInfo_Cell to info$		' Change it in RAM
		&pos (info$, oldZone$ + ":"), p
		&mid$ (info$, p) = zone$[dlsFlag]
		&store info$ to SysInfo_Cell

		fOpen configFile$			' Change config file
		fOpen tempFile$
		onerr goto configEOF
		do
			fRead configFile$
			&get info$
			if info$ = oldZone$ then info$ = zone$[dlsFlag]
			fWrite tempFile$
			print info$
		loop
		configEOF:
		&onerr
		onerr goto HandleError
		fClose
		&copy (tempFile$ to configFile$)
		fDelete tempFile$

		' ===================
		' Now, readjust clock
		' ===================

		gosub GetDateAndTime
	endif
return


' ***************************************************
' **
' ** SendMsgFile - Create mailbox header and send log
' **
' ** The letter is addressed to "root" so that it is
' ** sent to the root alias (which could be many users).
' **

SendMsgFile:
	gosub MakeUniqueName
	fAppend spoolDir$ theFile$
	print "From " ID$[uName] " " left$(theTime$,3) mid$(theTime$,8,5) \
	      mid$(theTime$,6,3) right$(theTime$,8) " 19" mid$(theTime$,13,2)
	print "Date: " left$(theTime$,5) val(mid$(theTime$,6)) \
	      mid$(theTime$,8) " " SysInfo$[plZone]
	print "From: " ID$[uName] " (" ID$[uFullName] ")"
	print "To: root"
	print "Subject: " argv$[0] " log"
	print 
	fClose
	&add (msgFile$ to spoolDir$ + theFile$)
	fDelete msgFile$
return


' *************************************************************
' **
' ** FindClock - Returns clockID of installed clock
' **
' ** First checks to see if machine is a IIGS.  If not, it reads
' ** data containing an offset from a peripheral card slot
' ** followed by a byte to make a comparison of the byte in memory.
' ** The matching continues until a complete pattern is found,
' ** and there's the clock.  It does this for seven slots.
' **
' ** Returns the clock's ID in "clockID", slot of clock (if
' ** not a IIGS) in "clockSlot", and clockName$.  The data table
' ** makes it easy to add new clocks.
' **

data	"Serial Pro", SERIAL_PRO, \
		0,44, 1,88, 2,255, 3,112, 5,56, 6,144, \
		7,24, 8,184, 253,193, 254,197, \
		-1,0
data	"TimeMaster", TIMEMASTER, \
		0,8, 1,120, 254,178, \
		-1,0
data	"ThunderClock", THUNDERCLOCK, \
		0,8, 1,120, 2,40, \
		-1,0
data	"ProClock", PROCLOCK, \
		34,10, 32,141, \
		-1,0
data	"", UNKNOWN_CLOCK


FindClock:
	' =========================
	' Test for Apple IIGS first
	' =========================

	&poke ASM_STUFF, $38, $20, $1F, $FE, $66, ZPAGE_REG, $60
	call ASM_STUFF	
	if peek(ZPAGE_REG) < $80 then
		clockID = IIGS_CLOCK
		clockName$ = "IIGS Clock"
		return
	endif

	clockSlot = 0
	repeat
		clockSlot = clockSlot + 1
		i = $C000 + (clockSlot * $100)
		restore
		repeat
			read clockName$, clockID
			if clockID <> UNKNOWN_CLOCK then
				repeat
					read offset, match
					found = offset = -1
				until found or peek(i + offset) <> match
				while offset > -1
					read offset, match
				wend
			endif
		until found or clockID = UNKNOWN_CLOCK
	until found or clockSlot = 7
return


' *****************************************
' **
' ** SetNewTime - Sets System Clock
' **

SetNewTime:
	if clockID = IIGS_CLOCK then
		&poke ASM_STUFF, $18, $FB, $C2, $30, \
			$F4, theDate - 1, theMonth - 1, \
			$F4, theHour, theYear, \
			$F4, theSecond, theMinute, \
			$A2, $03, $0E, $22, $00, $00, $E1, \
			$38, $FB, $60
		call ASM_STUFF		' WriteTimeHex toolbox call
		return			' Ah, the IIGS is so simple
	endif

	' =========================
	' Now set the date and time
	' =========================

	&right$(str$(theYear), 2, 48), year$
	&right$(str$(theMonth), 2, 48), month$
	&right$(str$(theDate), 2, 48), date$
	&right$(str$(theHour), 2, 48), hour$
	&right$(str$(theMinute), 2, 48), minute$
	&right$(str$(theSecond), 2, 48), second$
	day$ = str$(theDay)

	if clockID = SERIAL_PRO or clockID = TIMEMASTER then
		tm$ = year$ + "/" + month$ + "/" + date$ + " " + day$ + \
                        " " + hour$ + ":" + minute$ + ":" + second$
	else
		tm$ = month$ + " " + day$ + " " + date$ + \
			" " + hour$ + " " + minute$ + " " + second$
	endif

	fm$ = "!"
	if clockID = SERIAL_PRO then
		call $C0F7 + (clockSlot * $100), fm$, tm$
	else
		& scrn(stdAppleIO)
		fOutPort clockSlot
		print fm$ tm$
		& scrn(stdMWIO)
	endif
return


' ***********************************************
' **
' ** _PostMsg - Msg(msg$) handler -- posts a message

_PostMsg:
	if msgFile$ > "" then
		fAppend msgFile$
		print argv$[0] ": " msg$
		fClose msgFile$
	endif
	print argv$[0] ": " msg$
return


' *****************************************************************
' **
' ** Build a filename (theFile$) from theTime$ in the format:
' **
' **	ddSSSSS
' **
' ** Where dd is an alphabetic variant of the combined month and date
' ** values, and SSSSS are the number of seconds that have elapsed since
' ** midnight.

MakeUniqueName:
	&time (theTime$)
	&pos ("?anebarprayunulugepctovec", mid$(theTime$, 10, 2)), i
	i = i / 2
	d = val (mid$ (theTime$,6))
	&right$ (str$(val(mid$(theTime$,16)) * 3600 + \
		val(mid$(theTime$, 19)) * 60 + \
		val(right$(theTime$, 2))), 5, 48), theFile$
	theFile$ = spoolMailDir$ + \
		chr$(64 + i) + chr$(48 + d + 7 * (d > 9)) + theFile$
return


' ***********************************************
' **
' ** Include standard ProLine Application Library

#include <proline/proline.lib>
