/*
 * EMU][ Apple ][-class emulator
 * Copyright (C) 2002, 2003 by the EMU][ Project/Dapple ][ Team
 *
 * Component:  DISK: routines for emulating two disk drives
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  Current exception for ASMLIB.O linkage, if Z80.C is not used
 *
 * 20040428  New headers generated for v0.4 release
 *           Z80 core development
 *
 */


/*----------------------------------------


        disk.c


        see include file disk.h for definitions


----------------------------------------*/


#ifndef DEF_INC_DISK_C
#define DEF_INC_DISK_C


// **** include general definition files

//#include <stdio.h>            // included by disk.h
#include <stdlib.h>             // used for malloc(), free()
//#include <string.h>           // included by disk.h
#include <unistd.h>             // used for chdir()
//#include "..\libs\general.h"  // included by disk.h
//#include "..\libs\asmlib.h"   // included by disk.h
//#include "dapple.h"           // included by disk.h
#define false   0
#define true    1


// **** include definition file of this source

#include "disk.h"


// **** include further Emu][ specific files

#include "cpu65c02.h"
#include "gui.h"
#include "memory.h"


// **** variables

static unsigned char    driverom[256];
static unsigned char    driveromexists;
unsigned char           drivepath[260]; // also used by memory.c to bload files
static unsigned char    driveselect;
static unsigned int     drivelatch;
static unsigned int     drivelatch0ee;
drivedisk               drivedisk1;
drivedisk               drivedisk2;
static unsigned char    drivemotorflag1;
static unsigned char    drivemotorflag2;
unsigned char           drivefastmode;
unsigned char           drivefastflag;
unsigned char           drivemessageflag;
static unsigned int     driveoldlinecycle;
static unsigned int     driveolddelay;
static unsigned char    driveselectmenu;






/*--------------------------------------*/


      void driveinit(void) {

        memset(driverom, 0x0aa, sizeof(driverom));      // clear rom with $aa
        strcpy(drivepath, callingpath);
        driveromexists = 0;
        drivedisk1.slot = 6;
        drivedisk2.slot = 6;
        drivedisk1.drive= 1;
        drivedisk2.drive= 2;
        drivedisk1.volume = 254;
        drivedisk2.volume = 254;
        driveempty(&drivedisk1);
        driveempty(&drivedisk2);
        drivelatch      = 0xaa;
        drivelatch0ee   = 0;
        drivedisk1.track        = 34;
        drivedisk2.track        = 34;
        drivedisk1.phase        = 35*2-1;
        drivedisk2.phase        = 35*2-1;
        drivedisk1.bytepointer  = 0;
        drivedisk2.bytepointer  = 0;
        drivedisk1.writeprotected = false;
        drivedisk2.writeprotected = false;
        drivefastmode   = false;
        drivefastflag   = true;
        drivemessageflag = false;
        driveselectmenu = 0;

      } // driveinit


/*--------------------------------------*/


      void driveloadrom(unsigned int keyboard, unsigned int window) {
        FILE *file;
        unsigned char filepath[260];
        unsigned char oldcwd[260];


        cwdxchgslash(oldcwd, 256);                              // read old directory path
        chdir(inirompath);                                      // set rom directory

        strcpy(filepath, "slotdisk.a65");                       // search for source code
        file=fopen(filepath,"rb");
        if (!file) {
          strcpy(filepath, "slotdisk.s");
          file=fopen(filepath,"rb");
          if (!file) {
            strcpy(filepath, "slotdisk.asm");
            file=fopen(filepath,"rb");
          }
        }
        if (file) {
          fclose(file);                                         // close file again
          stringwritemessage(window,
"!\
EAssemble source code for disk ][ slot rom...\r;\
GAssembliere Source Code fr das Disk ][-Slotrom...\r;\
;");
          screenupdate = 1;
          taskswitch();
          memset(driverom, 0x0aa, sizeof(driverom));
          assembler(driverom, sizeof(driverom), filepath);      // assemble source code
          driveromexists = 1;
        }
        else {
          strcpy(filepath, "disk.rom");;                        // search binary rom file
          file=fopen(filepath,"rb");
          if (file) {
            stringwritemessage(window,
"!\
EReading disk ][ slot rom from file...\r;\
GLese Disk ][-Slotrom aus Datei...\r;\
;");
            screenupdate = 1;
            taskswitch();
            fread(driverom, sizeof(driverom), 1, file);         // read binary rom file
            fclose(file);
            driveromexists = 1;
          }
          else {
            stringwritemessage(window,
"!\
ENo separate disk ][ slot rom found\r;\
GKein gesondertes Disk ][-Slotrom vorhanden\r;\
;");
            screenupdate = 1;
            taskswitch();
          }
        }
        chdir(oldcwd);                                          // restore original directory
//      windowpresskey(keyboard, window);
//      debugger(driverom, sizeof(driverom));

      } // driveloadrom


/*--------------------------------------*/


      unsigned char *drivestore(void *slotdata, unsigned int winprotocol, FILE *file, unsigned int percent) {
        unsigned char header[32];

        memset(header, 0x00, sizeof(header));
        strcpy(header, "DRIVE STATE V0.27");
        fwrite(header,                  sizeof(header),                 1,file);

        fwrite(driverom,                sizeof(driverom),               1,file);
        fwrite(&driveromexists,         sizeof(driveromexists),         1,file);
        fwrite(drivepath,               sizeof(drivepath),              1,file);
        fwrite(&drivedisk1,             sizeof(drivedisk1),             1,file);
        fwrite(&drivedisk2,             sizeof(drivedisk2),             1,file);
        fwrite(&driveselect,            sizeof(driveselect),            1,file);
        fwrite(&drivelatch,             sizeof(drivelatch),             1,file);
        fwrite(&drivelatch0ee,          sizeof(drivelatch0ee),          1,file);
        fwrite(&drivemotorflag1,        sizeof(drivemotorflag1),        1,file);
        fwrite(&drivemotorflag2,        sizeof(drivemotorflag2),        1,file);
        fwrite(&drivefastmode,          sizeof(drivefastmode),          1,file);
        fwrite(&drivefastflag,          sizeof(drivefastflag),          1,file);
        fwrite(&drivemessageflag,       sizeof(drivemessageflag),       1,file);
        fwrite(&driveoldlinecycle,      sizeof(driveoldlinecycle),      1,file);
        fwrite(&driveolddelay,          sizeof(driveolddelay),          1,file);
        fwrite(&driveselectmenu,        sizeof(driveselectmenu),        1,file);

        guipercent(winprotocol, percent,
"!\
EDrive stored.;\
GLaufwerk gespeichert.;\
");

        return NULL;            // no error

      } // drivestore


/*--------------------------------------*/


      unsigned char *driverestore(void *slotdata, unsigned int winprotocol, FILE *file, unsigned int percent) {
        unsigned char header[32];

        fread(header,                   sizeof(header),                 1,file);
        if (strcmp(header, "DRIVE STATE V0.27")) {
          stringwrite(winprotocol, "Drive emulation data not found.\r");
          return taskseterror("Drive emulation data not found.");
        }

        fread(driverom,                 sizeof(driverom),               1,file);
        fread(&driveromexists,          sizeof(driveromexists),         1,file);
        fread(drivepath,                sizeof(drivepath),              1,file);
        fread(&drivedisk1,              sizeof(drivedisk1),             1,file);
        fread(&drivedisk2,              sizeof(drivedisk2),             1,file);
        fread(&driveselect,             sizeof(driveselect),            1,file);
        fread(&drivelatch,              sizeof(drivelatch),             1,file);
        fread(&drivelatch0ee,           sizeof(drivelatch0ee),          1,file);
        fread(&drivemotorflag1,         sizeof(drivemotorflag1),        1,file);
        fread(&drivemotorflag2,         sizeof(drivemotorflag2),        1,file);
        fread(&drivefastmode,           sizeof(drivefastmode),          1,file);
        fread(&drivefastflag,           sizeof(drivefastflag),          1,file);
        fread(&drivemessageflag,        sizeof(drivemessageflag),       1,file);
        fread(&driveoldlinecycle,       sizeof(driveoldlinecycle),      1,file);
        fread(&driveolddelay,           sizeof(driveolddelay),          1,file);
        fread(&driveselectmenu,         sizeof(driveselectmenu),        1,file);

        if (driveselect) {
          if (drivemotorflag1) {
            imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTON);
          }
          else {
            imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTOFF);
          }
          imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTOFF);
        }
        else {
          if (drivemotorflag2) {
            imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTON);
          }
          else {
            imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTOFF);
          }
          imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTOFF);
        }

        guipercent(winprotocol, percent,
"!\
EDrive restored.;\
GLaufwerk geladen.;\
");

        return NULL;            // no error

      } // driverestore


/*--------------------------------------*/


      void driveempty(drivedisk *disk) {        // set all bytes of emulated disk to 0xaa


        strcpy(disk->path, "");                 // clear filename
        disk->type = DISKTYPENONE;              // set type to no disk present
        disk->changed = false;

        memset(disk->data, 0xaa, sizeof(disk->data));
        disk->tracksize = 0x186a;

      } // driveempty


/*--------------------------------------*/


      void driveconvertsecnib(unsigned char source[144360], drivedisk *disk, unsigned int volume) {
        unsigned int    track;
        unsigned int    sector;
        unsigned int    sectorsit;
        unsigned int    secpointer;
        unsigned char   *nibpointer;
        unsigned int    zappointer;
        unsigned int    happointer;
        unsigned char   diskbyte;
        unsigned char   diskzap[256];
        unsigned char   diskhap[256];
        static unsigned int dospos [] = {
          0x00, 0x0d, 0x0b, 0x09, 0x07, 0x05, 0x03, 0x01,
          0x0e, 0x0c, 0x0a, 0x08, 0x06, 0x04, 0x02, 0x0f
        };
        static unsigned int prodospos [] = {
          0x00, 0x02, 0x04, 0x06, 0x08, 0x0a, 0x0c, 0x0e,
          0x01, 0x03, 0x05, 0x07, 0x09, 0x0b, 0x0d, 0x0f
        };
        static unsigned char diskwit [] = {
          0x96, 0x97, 0x9a, 0x9b, 0x9d, 0x9e, 0x9f, 0xa6,
          0xa7, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb2, 0xb3,
          0xb4, 0xb5, 0xb6, 0xb7, 0xb9, 0xba, 0xbb, 0xbc,
          0xbd, 0xbe, 0xbf, 0xcb, 0xcd, 0xce, 0xcf, 0xd3,
          0xd6, 0xd7, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde,
          0xdf, 0xe5, 0xe6, 0xe7, 0xe9, 0xea, 0xeb, 0xec,
          0xed, 0xee, 0xef, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6,
          0xf7, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
        };

        secpointer      = 0;
        for ( track=0; track < 35; track++) {
          for ( sector=0; sector < 16; sector++) {
            if (disk->type == DISKTYPEDOS ) {
              sectorsit = dospos[sector];
            }
            else {
              sectorsit = prodospos[sector];
            }
            if (disk->tracksize == 0x1a00) {
              nibpointer        = &disk->data[(track * 0x1a00) + (sectorsit * 416 )];
            }
            else {
              nibpointer        = &disk->data[(track * 0x1a00) + (sectorsit * 390 )];
            }
            *nibpointer++       = 0x00;         /* write header prolog */
            *nibpointer++       = 0xd5;
            *nibpointer++       = 0xaa;
            *nibpointer++       = 0x96;
            *nibpointer++       = ((volume >> 1) & 0x55) | 0xaa;
            *nibpointer++       = ( volume       & 0x55) | 0xaa;
            *nibpointer++       = ((track >> 1)  & 0x55) | 0xaa;
            *nibpointer++       = ( track        & 0x55) | 0xaa;
            *nibpointer++       = ((sectorsit >> 1) & 0x55) | 0xaa;
            *nibpointer++       = ( sectorsit       & 0x55) | 0xaa;
            *nibpointer++       = (((volume ^ track ^ sectorsit) >> 1) & 0x55) | 0xaa;
            *nibpointer++       = (( volume ^ track ^ sectorsit)       & 0x55) | 0xaa;
            *nibpointer++       = 0xde;         /* write header epilog */
            *nibpointer++       = 0xaa;
            *nibpointer++       = 0xeb;
            *nibpointer++       = 0x00;         /* gap */
            *nibpointer++       = 0x01;
            *nibpointer++       = 0x00;
            *nibpointer++       = 0x01;
            *nibpointer++       = 0x00;
            *nibpointer++       = 0xd5;         /* write data prolog */
            *nibpointer++       = 0xaa;
            *nibpointer++       = 0xad;

            for ( zappointer=0; zappointer < 256; zappointer++) {       /* clear buffers */
              diskzap[zappointer]       = 0;
              diskhap[zappointer]       = 0;
            }

            happointer  = 2;
            do {
              for ( zappointer=0; zappointer < 0x56; zappointer++) {
                happointer              = (happointer - 1) & 0xff;
                diskbyte                = source[secpointer + happointer];
                diskzap[zappointer]     = (diskzap[zappointer] << 2)
                                          | ((diskbyte & 0x1) << 1)
                                          | ((diskbyte & 0x2) >> 1);
                diskhap[happointer]     = diskbyte >> 2;
              }
            }
            while (happointer != 0);

            for ( zappointer=0x56; zappointer > 0; zappointer--) {
              *nibpointer++             = diskwit[diskzap[zappointer] ^ diskzap[zappointer-1]];
            }

            *nibpointer++               = diskwit[diskzap[0] ^ diskhap[0]];

            for (happointer=1; happointer < 256; happointer++) {
              *nibpointer++             = diskwit[diskhap[happointer] ^ diskhap[happointer-1]];
            }

            *nibpointer++               = diskwit[diskhap[255]];

            *nibpointer++       = 0xde;         /* write data epilog */
            *nibpointer++       = 0xaa;
            *nibpointer++       = 0xeb;
            if (disk->tracksize == 0x1a00) {
              *nibpointer++     = 0x00;
              *nibpointer++     = 0x01;
              *nibpointer++     = 0x00;
              for ( zappointer=0; zappointer<11; zappointer++) {        /* write gap */
                *nibpointer++   = 0x01;
                *nibpointer++   = 0x00;
                *nibpointer++   = 0x01;
                *nibpointer++   = 0x00;
              }
            }
            else {
              *nibpointer++     = 0x00;
              for ( zappointer=0; zappointer<5; zappointer++) {         /* write gap */
                *nibpointer++   = 0x01;
                *nibpointer++   = 0x00;
                *nibpointer++   = 0x01;
                *nibpointer++   = 0x00;
              }
            }
            secpointer  = secpointer + 256;
          } /* for sector */
          if (disk->tracksize != 0x1a00) {
            for ( zappointer=0; zappointer<5; zappointer++) {
              *nibpointer++     = 0x01;
              *nibpointer++     = 0x00;
            }
            for ( zappointer=0; zappointer<((0x1a00 - 6250)); zappointer++) {   /* fill rest of track */
              *nibpointer++     = 0xaa;
//            *nibpointer++     = 0x01;
//            *nibpointer++     = 0x00;
            }
          }
        } // for (track)
      } // driveconvertsecnib


/*--------------------------------------*/


static unsigned char *trackpointer;
static unsigned int bytecounter;
static unsigned int errorcounter;


      unsigned int incpointer() {

        trackpointer++;
        bytecounter--;
        if (!bytecounter) {
          trackpointer = trackpointer - 0x1a00;
          bytecounter = 0x1a00;
          errorcounter++;
          if (errorcounter >= 2) {
            return 1;
          }
        }
        return 0;

      } // incpointer


      void drivereaderror(unsigned int window, unsigned char *stringptr,
                          unsigned int track,  unsigned int sector) {

        stringwrite(window, "Warning (track: ");
        int32write(window, track);
        stringwrite(window, ", sector: ");
        int32write(window, sector);
        stringwrite(window, "):\r");
        stringwrite(window, stringptr);
        stringwrite(window, ".\r");

        return;
      } // drivereaderror


      unsigned int drivereadsector(unsigned int window, drivedisk *disk, unsigned char *diskbuffer,
                                   unsigned int track, unsigned int sector) {
        unsigned char headerfound;
        unsigned char rvolume;
        unsigned char rtrack;
        unsigned char rsector;
        unsigned char rchecksum;
        unsigned char diskbyte;
        int zappointer;
        int happointer;
        unsigned char diskzap[256];
        unsigned char diskhap[256];
        static unsigned char luet[] = {
                0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
                0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f,
                0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,
                0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
                0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,
                0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0x5b,0x5c,0x5d,0x5e,0x5f,
                0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f,
                0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7a,0x7b,0x7c,0x7d,0x7e,0x7f,
                0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,
                0x90,0x91,0x92,0x93,0x94,0x95,0x00,0x01,0x98,0x99,0x02,0x03,0x9c,0x04,0x05,0x06,
                0xa0,0xa1,0xa2,0xa3,0xa4,0xa5,0x07,0x08,0xa8,0xa9,0xaa,0x09,0x0a,0x0b,0x0c,0x0d,
                0xb0,0xb1,0x0e,0x0f,0x10,0x11,0x12,0x13,0xb8,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,
                0xc0,0xc1,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0x1b,0xcc,0x1c,0x1d,0x1e,
                0xd0,0xd1,0xd2,0x1f,0xd4,0xd5,0x20,0x21,0xd8,0x22,0x23,0x24,0x25,0x26,0x27,0x28,
                0xe0,0xe1,0xe2,0xe3,0xe4,0x29,0x2a,0x2b,0xe8,0x2c,0x2d,0x2e,0x2f,0x30,0x31,0x32,
                0xf0,0xf1,0x33,0x34,0x35,0x36,0x37,0x38,0xf8,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f
        };

        trackpointer = &disk->data[track * 0x1a00];
        bytecounter = 0x1a00;
        errorcounter = 0;

        if (track > 34) { drivereaderror(window, "Track > 34", track, sector); return 1; }
        if (sector > 15) { drivereaderror(window, "Sektor > 15", track, sector); return 1; }

        do {
          headerfound = 0;
          do {
            if (*trackpointer == 0xd5) {
              if (incpointer()) {
                drivereaderror(window, "Sector header prolog byte AA not found", track, sector);
                return 1;
              }
              if (*trackpointer == 0xaa) {
                if (incpointer()) {
                  drivereaderror(window, "Sector header prolog byte 96 not found", track, sector);
                  return 1;
                }
                if (*trackpointer == 0x96) {
                  headerfound = 1;
                }
              }
            }
            else {
              if (incpointer()) {
                drivereaderror(window, "Sector header prolog byte D5 not found", track, sector);
                return 1;
              }
            }
          } // do
          while (!headerfound);

          incpointer();
          rvolume = ((*trackpointer) << 1) | 1;
          incpointer();
          rvolume = (*trackpointer) & rvolume;
          incpointer();
          rtrack = ((*trackpointer) << 1) | 1;
          incpointer();
          rtrack = (*trackpointer) & rtrack;
          incpointer();
          rsector = ((*trackpointer) << 1) | 1;
          incpointer();
          rsector = (*trackpointer) & rsector;
          incpointer();
          rchecksum = ((*trackpointer) << 1) | 1;
          incpointer();
          rchecksum = (*trackpointer) & rchecksum;

          if (rvolume ^ rtrack ^ rsector ^ rchecksum) {
            drivereaderror(window, "Checksum error in sector header", track, sector);
//          debugger(&disk->data[track * 0x1a00], 0x1a00);
            return 1;
          }

          if (rtrack != track) {
            drivereaderror(window, "Incorrect track number in sector header",track, sector);
            return 1;
          }
        } // do
        while (rsector != sector);

/*
        incpointer();
        if (*trackpointer != 0xde) {
          drivereaderror(window, "Sector header epilog byte DE not found", track, sector);
          return 1;
        }
        incpointer();
        if (*trackpointer != 0xaa) {
          drivereaderror(window, "Sector header epilog byte AA not found", track, sector);
          return 1;
        }
        incpointer();
        if (*trackpointer != 0xeb) {
          drivereaderror(window, "Sector header epilog byte EB not found", track, sector);
//          debugger(&disk->data[track * 0x1a00], 0x1a00);
          return 1;
        }
*/

/* sector header found. Now look for data header */

        headerfound = 0;
        do {
          if (*trackpointer == 0xd5) {
            if (incpointer()) {
              drivereaderror(window, "Data header prolog byte AA not found", track, sector);
              return 1;
            }
            if (*trackpointer == 0xaa) {
              if (incpointer()) {
                drivereaderror(window, "Data header prolog byte AD not found", track, sector);
                return 1;
              }
              if (*trackpointer == 0xad) {
                headerfound = 1;
              }
            }
          }
          else {
            if (incpointer()) {
              drivereaderror(window, "Data header prolog byte D5 not found", track, sector);
              return 1;
            }
          }
        } /* do */
        while (!headerfound);

        diskbyte   = 0;
        for (zappointer=0x55; zappointer >= 0; zappointer--) {
          incpointer();
          diskbyte = diskbyte ^ luet[*trackpointer];
          diskzap[zappointer] = diskbyte;
        } /* for */
        for (happointer=0; happointer < 0x100; happointer++) {
          incpointer();
          diskbyte = diskbyte ^ luet[*trackpointer];
          diskhap[happointer] = diskbyte;
        } /* for */
        incpointer();
        diskbyte = diskbyte ^ luet[*trackpointer];
        if (diskbyte) {
          drivereaderror(window, "Checksum error in data field", track, sector);
          return 1;
        }

        zappointer = 0x56;
        for (happointer=0; happointer < 0x100; happointer++) {
          zappointer--;
          if (zappointer < 0) { zappointer = 0x55; }
          diskbuffer[happointer] = (diskhap[happointer] << 2)
                                 | ((diskzap[zappointer] & 1) << 1)
                                 | ((diskzap[zappointer] & 2) >> 1);
          diskzap[zappointer] = diskzap[zappointer] >> 2;
        }

// data prolog is ignored

        return 0;
      } // drivereadsector


/*--------------------------------------*/


      unsigned int drivesave(drivedisk *disk, unsigned char *filename) {
        unsigned int keyboard;
        unsigned int window;
        unsigned char key;
        unsigned int error;
        FILE *file;
        unsigned char oldcwd[260];
        unsigned char *diskbuffer;
        unsigned int track;
        unsigned int sector;
        static unsigned int dospos [] = {
          0x00, 0x0d, 0x0b, 0x09, 0x07, 0x05, 0x03, 0x01,
          0x0e, 0x0c, 0x0a, 0x08, 0x06, 0x04, 0x02, 0x0f
        };


        error  = 0;
        key    = 0;
        diskbuffer = malloc(143360);
        if (diskbuffer == 0) {
          return -1;
        }

        if (!(windowprotocol(&keyboard, &window,
"!\
ESave Disk Image;\
GSpeichere Diskimage;\
"))) {

          cwdxchgslash(oldcwd, 256);
          chdir(drivepath);                     // restore drive path in case only a file is given

          stringwrite(window, "Attempting to save disk image\r\'");
          stringwrite(window, filename);
          stringwrite(window, "\'...\r");
          screenupdate = 1;
          taskswitch();

          if (disk->type == DISKTYPEDOS) {
            stringwritemessage(window,
"!\
ESave disk as DOS image...\r;\
GSpeichere Diskette als DOS-Image...\r;\
");
            guipercent(window, 0,
"!\
EConverting nibble image to DOS image...;\
GWandle Nibble-Image um in DOS-Image...;\
");
            for (track=0; (track < 35) && (!error); track++) {
              for (sector=0; (sector < 16) && (!error); sector++) {
                error = drivereadsector(window, disk, &diskbuffer[(track*0x1000)+ (sector*0x100)],
                                        track, dospos[sector]);
                if (error) {
                  stringwrite(window, "Could not convert to DOS image.\r");
                  stringwrite(window, "Do you wish to save the image as a nibble image? (Y/N)\r");
                  screenupdate = 1;
                  do {
                    taskswitch();
                    if (windowgetclose(window)) {
                      key = 'n';
                    }
                    else {
                      key = (unsigned char)channelin(keyboard);
                    }
                    if (exitprogram) { key = 0x3; };
                  }
                  while (key == 0);
                  if ((key == 'y') || (key == 'Y')) {
                    disk->type = DISKTYPENIBBLE;
                    if (strright(filename, ".DSK")) {
                      filename[strlen(filename)-3] = 0;
                      strcat(filename, "nib");
                    }
                    else {
                      if (strright(filename, ".DO")) {
                        filename[strlen(filename)-2] = 0;
                        strcat(filename, "nib");
                      }
                    }
                  }
                }
              } // for (sector)
            } // for (track)
            if (!error) {
              file = fopen(filename, "wb");
              if (!file) {
                stringwrite(window, "Error opening file.\r");
                stringwrite(window, stringgeterror());  // display error
                channelout(window, 13);
                error = 1;
              }
              else {
                guipercent(window, 50,
"!\
ESaving data...;\
GSpeichere Daten...;\
");
                fwrite(diskbuffer, 143360, 1, file);
                fclose(file);
                disk->changed = false;
              }
            }
            else {
              if (disk->type == DISKTYPENIBBLE) {
                error = 0;
                key   = 0;
              }
            }
          } // if (disk->type == DISKTYPEDOS

          if (disk->type == DISKTYPEPRODOS) {
            stringwritemessage(window,
"!\
ESave disk as ProDOS image...\r;\
GSpeichere Diskette als ProDOS-Image...\r;\
");
            guipercent(window, 0,
"!\
EConverting nibble image to ProDOS image...;\
GWandle Nibble-Image um in ProDOS-Image...;\
");
            for (track=0; (track < 35) && (!error); track++) {
              for (sector=0; (sector < 16) && (!error); sector++) {
                error = drivereadsector(window, disk, &diskbuffer[(track*0x1000)+ (sector*0x100)],
                                        track, sector);
                if (error) {
                  stringwrite(window, "\rCould not convert to ProDoS image.\r");
                  stringwrite(window, "Do you wish to save image as a nibble image? (Y/N)\r");
                  screenupdate = 1;
                  do {
                    taskswitch();
                    if (windowgetclose(window)) {
                      key = 'n';
                    }
                    else {
                      key = (unsigned char)channelin(keyboard);
                    }
                    if (exitprogram) { key = 0x3; };
                  }
                  while (key != 0);
                  if ((key == 'y') || (key == 'Y')) {
                    disk->type = DISKTYPENIBBLE;
                  }
                }
              } // for (sector)
            } // for (track)
            if (!error) {
              file = fopen(filename, "wb");
              if (!file) {
                stringwrite(window, "Error opening file.\r");
                stringwrite(window, stringgeterror());  // display error
                channelout(window, 13);
                error = 1;
              }
              else {
                guipercent(window, 50,
"!\
ESaving data...;\
GSpeichere Daten...;\
");
                fwrite(diskbuffer, 143360, 1, file);
                fclose(file);
                disk->changed = false;
              }
            }
            else {
              if (disk->type == DISKTYPENIBBLE) {
                error = 0;
                key   = 0;
              }
            }
          } // if (disk->type == DISKTYPEPRODOS

          if (disk->type == DISKTYPENIBBLE) {
            stringwrite(window, "Save disk as nibble image...\r");
            screenupdate = 1;
            taskswitch();
            file = fopen(filename, "wb");
            if (!file) {
              stringwrite(window, "Error opening file.\r");
              stringwrite(window, stringgeterror());    // display error
              channelout(window, 13);
              error = 1;
            }
            else {
              guipercent(window, 50,
"!\
ESaving data...;\
GSpeichere Daten...;\
");
              fwrite(&disk->data, 232960, 1, file);
              fclose(file);
              disk->changed = false;
            }
          } // if (disk->type == DISKTYPENIBBLE

          if (disk->type == DISKTYPENIBBLE2) {
            stringwrite(window, "Save disk as nibble2 image...\r");
            screenupdate = 1;
            taskswitch();
            file = fopen(filename, "wb");
            if (!file) {
              stringwrite(window, "Error opening file.\r");
              stringwrite(window, stringgeterror());    // display error
              channelout(window, 13);
              error = 1;
            }
            else {
              guipercent(window, 50,
"!\
ESaving data...;\
GSpeichere Daten...;\
");
              for (track=0; track<35; track++) {
                fwrite(&disk->data[track*0x1a00], 0x186a, 1, file);
              }
              fclose(file);
              disk->changed = false;
            }
          } // if (disk->type == DISKTYPENIBBLE2

          if ((error) && (key == 0)) {
            windowpresskey(keyboard, window);
          }
          else {
            getpathstring(drivepath, filename);
            guipercent(window, 100,
"!\
EDisk image saved.;\
GDiskimage gespeichert.;\
");
//          windowpresskey(keyboard, window);
          }
          chdir(oldcwd);

          channelclose(keyboard);
          channelclose(window);
        }

        free(diskbuffer);
        return error;

      } // drivesave


/*--------------------------------------*/


      unsigned char *driveeject(drivedisk *disk) {
        register unsigned char *error;
        register unsigned char key;
        unsigned int keyboard;
        unsigned int window;

        if (disk->changed) {
          error = windowprotocol(&keyboard, &window,
"!\
EEject Disk;\
GDiskette auswerfen;\
");
          if (error) {
            return error;
          }

          stringwritemessage(window,
"!\
EThe current disk\r\';\
GDie aktuelle Diskette\r\';\
");
          stringwrite(window, disk->path);
          stringwrite(window, "\'\rin Slot ");
          int32write(window, disk->slot);
          stringwritemessage(window,
"!\
E Drive ;\
G Laufwerk ;\
");
          int32write(window, disk->drive);
          stringwritemessage(window,
"!\
E was changed.\rDo you wish to save the disk? (Y/N)\r;\
G wurde gendert.\rMchten Sie die Diskette speichern? (J/N)\r;\
");
          do {
            key = keyboardyesno(keyboard, window);
            if (key == 1) {
              if (!(drivesave(disk, disk->path))) {
                key = 0;
              }
            }
          }
          while (key != 0);
          channelclose(keyboard);
          channelclose(window);
        } // if (disk->changed)
        driveempty(disk);

        return NULL;            // no error

      } // driveeject


/*--------------------------------------*/


      // load standard DOS image (.DSK) or Nibble image (.NIB)
      unsigned char *driveload(drivedisk *disk, unsigned char *filename) {
        register unsigned char  *error;
        card8           *diskbuffer;
        unsigned int    keyboard;
        unsigned int    window;
        unsigned int    track;
        unsigned int    filelength;
        unsigned char   oldcwd[260];


        error = driveeject(disk);
        if (error) {
          return error;
        }

        error = windowprotocol(&keyboard, &window,
"!\
ELoad Disk Image;\
GLade Diskimage;\
");
        if (error) {
          return error;
        }

        cwdxchgslash(oldcwd, 256);
        chdir(drivepath);                               // restore disk path in case only a file is given

        stringwritemessage(window,
"!\
EAttempting to load disk image:\r\';\
GVersuche, Diskimage zu laden:\r\';\
");
        stringwrite(window, filename);
        stringwrite(window, "'\r");
        screenupdate = 1;
        taskswitch();

        error = fileresident(filename, (void *)&diskbuffer, &filelength);
        if (error) {
          stringwritemessage(window,
"!\
EDiskimage could not be loaded.\r;\
GDiskimage konnte nicht geladen werden.\r;\
");
          stringwrite(window, stringgeterror());        // display error
          channelout(window, 13);
        }
        else {
          if (filelength < 143360) {
            error =
"!\
EFile is not an apple disk image.\r;\
GDatei ist keine Applediskette.\r;\
";
          }
          else {
            guipercent(window, 10,
"!\
ELoading disk image...;\
GLade Diskimage...;\
");

            if (filelength < 218750 /*==143360*/) {
              if (strright(filename, ".PO")) {
                guipercent(window, 50,
"!\
EConverting ProDos image...;\
GKonvertiere ProDos Image...;\
");
                screenupdate = 1;
                taskswitch();
                disk->type      = DISKTYPEPRODOS;
                disk->tracksize = 0x186a;
                driveconvertsecnib(diskbuffer, disk, disk->volume);
              }
              else {
                guipercent(window, 50,
"!\
EConverting DOS image...;\
GKonvertiere DOS Image...;\
");
                screenupdate = 1;
                taskswitch();
                disk->type      = DISKTYPEDOS;
                disk->tracksize = 0x186a;
                driveconvertsecnib(diskbuffer, disk, disk->volume);
              }
              strcpy(disk->path, filename);
              disk->changed = false;
            }
            else {
              if (filelength==232960) {
                memcpy(&disk->data[0], &diskbuffer[0], 232960);
                disk->type      = DISKTYPENIBBLE;
                disk->tracksize = 0x1a00;
                strcpy(disk->path, filename);
                disk->changed   = false;
              }
              else {
                if (filelength==218750) {
                  for (track=0; track<35; track++) {
                    memcpy(&disk->data[track*0x1a00], &diskbuffer[track*0x186a], 0x186a);
                    memset(&disk->data[track*0x1a00+0x186a], 0xaa, 196);
                  }
                  disk->type = DISKTYPENIBBLE2;
                  disk->tracksize       = 0x186a;
                  strcpy(disk->path, filename);
                  disk->changed = false;
                }
                else {
                  error =
"!\
EFile is not an apple disk image.\r;\
GDatei ist keine Applediskette.\r;\
";
                }
              } // else if (filelength == 232960)
            } // else if (filelength < 218750)
          } // else if (!file)

          if (error) {
            stringwritemessage(window, error);
            windowpresskey(keyboard, window);
          }
          else {
            getpathstring(drivepath, filename);
            guipercent(window, 100,
"!\
EDisk image loaded.;\
GDiskimage geladen.;\
");
//          windowpresskey(keyboard, window);
          }

          free(diskbuffer);

        } // if (error=fileresident)

        channelclose(keyboard);
        channelclose(window);

        chdir(oldcwd);

        return error;

      } // driveload


/*-------------------------------------*/


      void drivetrackmessage(void) {
        unsigned char message[256];
        register unsigned int i;

        if (drivemessageflag) {
          strcpy (message, "Drive 1 Track: xx / Drive 2 Track: xx");
          i = drivedisk1.track / 10;
          if (i == 0) { message[15] = ' '; }
          else        { message[15] = i + '0'; }
          message[16] = (drivedisk1.track % 10) + '0';
          i = drivedisk2.track / 10;
          if (i == 0) { message[35] = ' '; }
          else        { message[35] = i + '0'; }
          message[36] = (drivedisk2.track % 10) + '0';
          setmessage(message);
        }
      } // drivetrackmessage


/*--------------------------------------*/


      unsigned char *drivenew(slot *slotpointer, unsigned int slotnumber) {

        slotpointer->slotclose          = (void *)&driveclose;
        slotpointer->slotreset          = (void *)&drivereset;
        slotpointer->slotstore          = (void *)&drivestore;
        slotpointer->slotrestore        = (void *)&driverestore;
        slotpointer->slotget            = (void *)&driveread;
        slotpointer->slotset            = (void *)&drivewrite;
        slotpointer->slotromget         = (void *)&driveromread;
        slotpointer->slotromset         = (void *)&driveromwrite;
        slotpointer->slotmenu           = (void *)&drivemenu;
        slotpointer->slotshift          = (void *)&slotnofunction;
        slotpointer->slotctrl           = (void *)&slotnofunction;
        slotpointer->slotshiftctrl      = (void *)&slotnofunction;
        slotpointer->slottype           = SLOTTYPEDISKDRIVE;
        slotpointer->slotdata           = NULL;
        slotpointer->slotname           =
"!\
EDisk drive;\
GDiskettenlaufwerk;\
";

        return NULL;            // no error

      } // drivenew


/*--------------------------------------*/


      unsigned char *driveclose(void *slotdata) {

        strcpy(inidiskpath1, drivedisk1.path);
        strcpy(inidiskpath2, drivedisk2.path);
        driveeject(&drivedisk1);
        driveeject(&drivedisk2);
        return NULL;            // no error

      } // driveclose


/*--------------------------------------*/


      unsigned char *drivereset(void *slotdata) {

        driveselect             = true;         // drive 1
        drivedisk1.access       = true;         // read
        drivedisk2.access       = true;         // read
        drivemotorflag1         = false;        // off
        drivemotorflag2         = false;        // off
        imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTOFF);
        imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTOFF);
        if (drivefastmode) {
          cpusetlinecycle(driveoldlinecycle);
          cpusetdelay(driveolddelay);
          drivefastmode = false;
        }
        return NULL;    // no error

      } // drivereset


/*--------------------------------------*/


      unsigned char driveread(void *slotdata, unsigned int addr) {
        unsigned int i;
        unsigned char bytebuffer;

        switch (addr & 0xf) {
          case 0x0 :                    // ---
          case 0x2 :
          case 0x4 :
          case 0x6 :
            return 0x0d;
          case 0x1 :                    // set phase 0
          case 0x3 :                    // set phase 1
          case 0x5 :                    // set phase 2
          case 0x7 :                    // set phase 3
            addr = (addr &0xf) >> 1;
            if (driveselect) {
              i = (drivedisk1.phase + 1) & 3;
              if (i == addr) {
                if (drivedisk1.phase < (35*2-1)) {
                  drivedisk1.phase++;
                  drivedisk1.track = drivedisk1.phase >> 1;
                  drivetrackmessage();
                }
              }
              else {
                if ( ((i+2)&3) == addr) {
                  if (drivedisk1.phase > 0) {
                    drivedisk1.phase--;
                    drivedisk1.track = drivedisk1.phase >> 1;
                    drivetrackmessage();
                  }
                }
              }
            }
            else {
              i = (drivedisk2.phase + 1) & 3;
              if (i == addr) {
                if (drivedisk2.phase < (35*2-1)) {
                  drivedisk2.phase++;
                  drivedisk2.track = drivedisk2.phase >> 1;
                  drivetrackmessage();
                }
              }
              else {
                if ( ((i+2)&3) == addr) {
                  if (drivedisk2.phase > 0) {
                    drivedisk2.phase--;
                    drivedisk2.track = drivedisk2.phase >> 1;
                    drivetrackmessage();
                  }
                }
              }
            }
            return 0xd;
          case 0x8 :    // motor off
            if (driveselect) {
              drivemotorflag1   = false;
              imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTOFF);
              drivemotorflag2   = false;
              imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTOFF);
              screenupdate = 1;
            }
            else {
              drivemotorflag1   = false;
              imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTOFF);
              drivemotorflag2   = false;
              imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTOFF);
              screenupdate = 1;
            }
            if (drivefastmode) {
              cpusetlinecycle(driveoldlinecycle);
              cpusetdelay(driveolddelay);
              drivefastmode     = false;
            }
            return 0xa0;
          case 0x9 :    // motor on
            if (driveselect) {
              drivemotorflag1   = true;
              imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTON);
              drivemotorflag2   = false;
              imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTOFF);
              screenupdate = 1;
            }
            else {
              drivemotorflag1   = false;
              imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTOFF);
              drivemotorflag2   = true;
              imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTON);
              screenupdate = 1;
            }
            if (drivefastflag) {
              if (!drivefastmode) {
                driveoldlinecycle       = cpugetlinecycle();
                driveolddelay           = cpugetdelay();
                cpusetdelay(0);
                cpusetlinecycle(65*8);
                drivefastmode   = true;
              }
            }
            return 0xa0;
          case 0xa :                    // set drive 1
            driveselect = true;
            return 0xa0;
          case 0xb :                    // set drive 2
            driveselect = false;
            return 0xa0;
          case 0xc :                    // read/write byte to/from disk
            if (driveselect) {
              if (drivedisk1.access) {
                bytebuffer = drivedisk1.data[drivedisk1.track * 0x1a00 + drivedisk1.bytepointer];
              }
              else {
                if ((drivedisk1.type != DISKTYPENONE) && (!drivedisk1.writeprotected)) {
                  drivedisk1.data[drivedisk1.track * 0x1a00 + drivedisk1.bytepointer] = drivelatch;
                  drivedisk1.changed = true;
                }
                bytebuffer = drivelatch;
              }
//            if (drivemotorflag1) {
                drivedisk1.bytepointer++;
                if (drivedisk1.bytepointer >= drivedisk1.tracksize) {
                  drivedisk1.bytepointer = 0;
                }
//            }
              return bytebuffer;
            }
            else {
              if (drivedisk2.access) {
                bytebuffer = drivedisk2.data[drivedisk2.track * 0x1a00 + drivedisk2.bytepointer];
              }
              else {
                if ((drivedisk2.type != DISKTYPENONE) && (!drivedisk2.writeprotected)) {
                  drivedisk2.data[drivedisk2.track * 0x1a00 + drivedisk2.bytepointer] = drivelatch;
                  drivedisk2.changed = true;
                }
                bytebuffer      = drivelatch;
              }
//            if (drivemotorflag2) {
                drivedisk2.bytepointer++;
                if (drivedisk2.bytepointer >= drivedisk2.tracksize) {
                  drivedisk2.bytepointer = 0;
                }
//            }
              return bytebuffer;
            }
          case 0xd :
            return 0xa0;
          case 0xe :                    // set disk read
            if (driveselect) {
              drivedisk1.access = true;
              return (((drivedisk1.writeprotected)? 0x80 : 0x00) | drivelatch0ee);
            }
            else {
              drivedisk2.access = true;
              return (((drivedisk2.writeprotected)? 0x80 : 0x00) | drivelatch0ee);
            }
          case 0xf :                    // set disk write
            if (driveselect) {
              drivedisk1.access = false;
              return 0;
            }
            else {
              drivedisk2.access = false;
              return 1;
            }
        } // switch (addr & 0xf)

        return 0xff;    // not needed, just here to avoid compiler warnings

      } // driveread


/*--------------------------------------*/


      void drivewrite(void *slotdata, unsigned int addr, unsigned char value) {

        switch (addr&0xf) {

          case 0x0 :
          case 0x1 :
          case 0x2 :
          case 0x3 :
          case 0x4 :
          case 0x5 :
          case 0x6 :
          case 0x7 :
            return;
          case 0x8 :
            if (driveselect) {
              drivemotorflag1   = false;
              imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTOFF);
              drivemotorflag2   = false;
              imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTOFF);
              screenupdate = 1;
            }
            else {
              drivemotorflag1   = false;
              imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTOFF);
              drivemotorflag2   = false;
              imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTOFF);
              screenupdate = 1;
            }
            if (drivefastmode) {
              cpusetlinecycle(driveoldlinecycle);
              cpusetdelay(driveolddelay);
              drivefastmode     = false;
            }
            return;
          case 0x9 :
            if (driveselect) {
              drivemotorflag1   = true;
              imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTON);
              drivemotorflag2   = false;
              imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTOFF);
              screenupdate = 1;
            }
            else {
              drivemotorflag1   = false;
              imagefillbox(window, SLOTX1, SLOT6Y, SLOTX1+1, SLOT6Y+1, RGBLGHTOFF);
              drivemotorflag2   = true;
              imagefillbox(window, SLOTX2, SLOT6Y, SLOTX2+1, SLOT6Y+1, RGBLGHTON);
              screenupdate = 1;
            }
            if (drivefastflag) {
              if (!drivefastmode) {
                driveoldlinecycle       = cpugetlinecycle();
                driveolddelay           = cpugetdelay();
                cpusetdelay(0);
                cpusetlinecycle(65*8);
                drivefastmode   = true;
              }
            }
            return;
          case 0xa :
            driveselect = true;         // set drive 1
            return;
          case 0xb :
            driveselect = false;        // set drive 2
            return;
          case 0xc :
            return;
          case 0xd :
            drivelatch  = value;
            return;
          case 0xe :                    // set disk read
            if (driveselect) {
              drivedisk1.access = true;
            }
            else {
              drivedisk2.access = true;
            }
            return;
          case 0xf :                    // set disk write
            drivelatch0ee       = value & 0x1f;
            if (driveselect) {
              drivedisk1.access = false;
            }
            else {
              drivedisk2.access = false;
            }
            return;
        } // switch (addr&0xf)

      } // drivewrite


/*--------------------------------------*/


      unsigned char driveromread(void *slotdata, unsigned int addr) {

        if (driveromexists) {
          return driverom[addr & 0xff];         // read from the rom which was separately loaded
        }
        else {
          return memram[0x20000+0x600+(addr&0xff)];
        }

      } // driveromread


/*--------------------------------------*/


      void driveromwrite(void *slotdata, unsigned int addr, unsigned char value) {

        return;         // Nothing happens here

      } // driveromwrite


/*--------------------------------------*/


      unsigned char *drivemenu(void *slotdata) {
        register unsigned char *error;
        unsigned int    keyboard;
        unsigned int    window;
        unsigned char   key;
        unsigned int    update;
        unsigned char   filename[260];
        drivedisk       *selecteddisk;
        unsigned char   volumestring[32];
        int             newvolume;


        error = windowaddio( -1, -1, WINDOWXSIZE, WINDOWYSIZE, -1, 1,
"!\
EDrive Options;\
GDiskettenlaufwerk Optionen;\
", 0, &keyboard, &window);
        if (error) {
          return error;
        }

        selecteddisk = &drivedisk1;     // not necessary. Just here to avoid compiler warning

        update = 1;
        do {
          if (update) {

            if (driveselectmenu) {
              selecteddisk = &drivedisk2;
            }
            else {
              selecteddisk = &drivedisk1;
            }

            channelout(window, 12);             // clear window
            stringwritemessage(window,
"!\
E\r[ESC] - Quit\r\r;\
G\r[ESC] - Verlasse Men\r\r;\
");

            stringwritemessage(window,
"!\
E\rCurrent drive is drive ;\
G\rAktuelles Laufwerk ist Laufwerk ;\
");
            channelout(window, driveselectmenu + '1');
            stringwritemessage(window,
"!\
E.\rFilename:\r;\
G.\rDateiname:\r;\
");
            if (! selecteddisk->path[0]) {
              stringwritemessage(window,
"!\
E<empty>;\
G<leer>;\
");
            }
            else {
              stringwrite(window, selecteddisk->path);
            }

            stringwritemessage(window,
"!\
E\r\r[V] - Volume number: ;\
G\r\r[V] - Volumenummer: ;\
");
            sprintf(volumestring, "%d", selecteddisk->volume);
            stringwrite(window, volumestring);
            channelout(window, 13);

            stringwritemessage(window,
"!\
E\
\r[1] - Select Drive 1\
\r[2] - Select Drive 2\
\r[E] - Eject Disk\
\r[L] - Load disk image\
\r[S] - Save disk image\
\r\r[M] - Drive Message: \
;\
G\
\r[1] - Whle Laufwerk 1\
\r[2] - Whle Laufwerk 2\
\r[E] - Diskette auswerfen\
\r[L] - Lade eine Diskette\
\r[S] - Speichere eine Diskette\
\r\r[M] - Laufwerkmeldungen: \
;\
");


            stringwriteonoff(window, drivemessageflag);
            stringwritemessage(window,
"!\
E\r[F] - Drive Fast Access: ;\
G\r[F] - Schnelle Laufwerkemulation: ;\
");
            stringwriteonoff(window, drivefastflag);
            screenupdate = 1;
            update = 0;
          }
          do {
            taskswitch();
            if (windowgetclose(window)) {
              key = 27;
            }
            else {
              key = (unsigned char)channelin(keyboard);
            }
          }
          while ((key == 0) && (!exitprogram));
          switch (key) {
            case '1' :
                driveselectmenu = 0;
                update = 1;
                break;
            case '2' :
                driveselectmenu = 1;
                update = 1;
                break;
            case 'd' :
            case 'D' :
                debugger(selecteddisk->data, sizeof(selecteddisk->data));
                break;
            case 'e' :
            case 'E' :
                driveeject(selecteddisk);
                update = 1;
                break;
            case 'f' :
            case 'F' :
                drivefastflag = !drivefastflag;
                update = 1;
                break;
            case 'l' :
            case 'L' :
                if (selecteddisk->drive == 1) {
                  if (fileselectmenu(
"!\
ELoad Disk Image Drive 1;\
GLade Diskette Laufwerk 1;\
",
"!\
EDisk image (*.DSK, *.NIB, *.NB2, *.DO, *.PO);\
GDiskimage (*.DSK, *.NIB, *.NB2, *.DO, *.PO);\
",
                                     filename, drivepath, "._img_", 0)) {
                    driveload(selecteddisk, filename);
                  }
                }
                else {
                  if (fileselectmenu(
"!\
ELoad Disk Image Drive 2;\
GLade Diskette Laufwerk 2;\
",
"!\
EDisk image (*.DSK, *.NIB, *.NB2, *.DO, *.PO);\
GDiskimage (*.DSK, *.NIB, *.NB2, *.DO, *.PO);\
",
                                     filename, drivepath, "._img_", 0)) {
                    driveload(selecteddisk, filename);
                  }
                }
                update = 1;
                break;
            case 'm' :
            case 'M' :
                drivemessageflag = !drivemessageflag;
                update = 1;
                break;
            case 's' :
            case 'S' :
                strcpy(filename, drivepath);
                if (selecteddisk->type == DISKTYPENONE) {
                  stringwrite(window, "\r\rNo disk in drive to save.\r");
                  windowpresskey(keyboard, window);
                }
                else {
                  if (selecteddisk->drive == 1) {
                    if (fileselectmenu(
"!\
ESave Disk Image Drive 1;\
GSpeichere Diskette Laufwerk 1;\
",
"!\
EDisk image (*.DSK, *.NIB, *.NB2, *.DO, *.PO);\
GDiskimage (*.DSK, *.NIB, *.NB2, *.DO, *.PO);\
",
                                       selecteddisk->path, drivepath, "._img_", 1)) {
                      drivesave(selecteddisk, selecteddisk->path);
                    }
                  }
                  else {
                    if (fileselectmenu(
"!\
ESave Disk Image Drive 2;\
GSpeichere Diskette Laufwerk 2;\
",
"!\
EDisk image (*.DSK, *.NIB, *.NB2, *.DO, *.PO);\
GDiskimage (*.DSK, *.NIB, *.NB2, *.DO, *.PO);\
",
                                       selecteddisk->path, drivepath, "._img_", 1)) {
                      drivesave(selecteddisk, selecteddisk->path);
                    }
                  }
                }
                update = 1;
                break;
            case 'v' :
            case 'V' :
                stringwritemessage(window,
"!\
E\
\r\rPlease enter a volume number (<ESC> to cancel).\r\
The standard value for DOS 3.3 disk is 254.\r\
(Note: Changing the volume number will only take effect\r\
after a new .DSK- or .PO-disk image has been loaded.)\r\
>\
;\
G\
\r\rBitte geben Sie eine Volumenummer ein. (<ESC> zum Abbrechen)\r\
Der Standardwert fr eine DOS-Diskette betrgt 254.\r\
(Bitte beachten Sie: Eine Vernderung der Volumenummer\r\
wirkt sich erst aus, wenn eine .DSK- oder .PO-Diskette\r\
neu geladen wurde.)\r\
>\
;\
");
                imagesettextcolor(window, RGBBLACK, RGBWHITE);
                if (imagestringread(window, keyboard, volumestring, 9)) {
                  imagesettextcolor(window, RGBLBLUE, RGBWHITE);
                  newvolume = stringtoint(volumestring);
                  if ((newvolume < 0) || (newvolume > 255)) {
                    stringwritemessage(window,
"!\
E\rA volume numer must have a value between 0 and 255.\r;\
G\rEine Volumenummer mu zwischen 0 und 255 liegen.\r;\
");
                    windowpresskey(keyboard, window);
                  }
                  else {
                    selecteddisk->volume = newvolume;
                  }
                }
                else {
                  imagesettextcolor(window, RGBLBLUE, RGBWHITE);
                }
                update = 1;
                break;
          } // switch (key)
        }
        while ((key != 27) && (key != 32) && (key != 13) && (!exitprogram));
        channelclose(keyboard);
        channelclose(window);

        return NULL;            // no error
      } // drivemenu


// --> #ifndef DEF_INC_DISK_C
#endif
