/*
 * EMU][ Apple ][-class emulator
 * Copyright (C) 2002, 2003 by the EMU][ Project/Dapple ][ Team
 *
 * Component: CPU65C02: CPU emulation
 * ADC and SBC instructions contributed by Scott Hemphill.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  Current exception for ASMLIB.O linkage, if Z80.C is not used
 *
 * 20040428  New headers generated for v0.4 release
 *           Z80 core development
 *
 *
 */


/*----------------------------------------


        cpu65c02.c


        see include file cpu65c02.h for definitions


----------------------------------------*/


#ifndef DEF_INC_CPU65C02_C
#define DEF_INC_CPU65C02_C


// **** include general definition files

//#include <stdio.h>                    // included by cpu65c02.h
//#include <string.h>                   // included by cpu65c02.h
//#include "..\libs\general.h"          // included by cpu65c02.h
//#include "..\libs\asmlib.h"           // included by cpu65c02.h
//#include "dapple.h"                   // included by cpu65c02.h


// **** include definition file of this source

#include "cpu65c02.h"


// **** include further Emu][ specific files

#include "disk.h"
#include "gui.h"
#include "memory.h"
#include "mouse.h"
#include "video.h"
#ifdef EMUZ80
#include "z80.h"
#endif


#ifdef CPU_CYCLEEXACT
#include "..\libs\raf.h"
#include "..\c64emu\cpu6502.h"
#endif


// rom
#ifdef REECOP
void romexecute(unsigned int addr);
#endif


// **** variables

//  6502 registers
card8           areg;           // Accumulator
card8           xreg;           // X-Register
card8           yreg;           // Y-Register
card8           stack;          // Stack-Register
card16          pc;             // Program counter

card8           nflag;          // N-Flag
card8           vflag;          // Overflow-Flag
card8           iflag;          // Interrupt-Flag
card8           bflag;          // Break-Flag
card8           dflag;          // Decimal-Flag
card8           zflag;          // Zero-Flag
card8           cflag;          // Carry-Flag


#ifndef CPU_ASM

// state of the processor
unsigned int            stateflags;

static unsigned char    bytebuffer;
static unsigned int     address;        // 16-bit address

static unsigned char    cputype;

static unsigned int     cycle;          // Number of cycles
static unsigned int     lastcycle;
static unsigned int     linecycles;
static unsigned int     cpudelay;

static unsigned int     breakpoint;
static unsigned char    traceflag;      // set tracemode
// --> ifdef CPU_ASM
#endif

unsigned char           cpumessageflag;



// **** wrapper code for cycle exact emulation

#ifdef CPU_CYCLEEXACT

cpu6502         cpu;
raf             cpuraf;

      unsigned char *wrapramget(raf *rafptr, card32 index, card8 *result) {

        *result = memoryread(index);
        return NULL;

      } // wrapramget

      unsigned char *wrapramset(raf *rafptr, card32 index, card8 value) {

        memorywrite(index, value);
        return NULL;

      } // wrapramset


#endif






/*----------------------------------------

 error  *cpuinit        (void)

        initialize cpu

-----------------------------------------*/

      unsigned char *cpuinit(void) {


#ifdef CPU_CYCLEEXACT
        rafinit(&cpuraf, &memram[0], 65536);
        cpuraf.rafin    = (void *)&rafnoin;
        cpuraf.rafout   = (void *)&rafnoout;
        cpuraf.rafget   = (void *)&*wrapramget;
        cpuraf.rafset   = (void *)&*wrapramset;
        cpu6502init(&cpu, &cpuraf);
#endif

#ifdef CPU_ASM
        cpu6502init();
#else
        areg    = 0;                    // clear register
        xreg    = 0;
        yreg    = 0;
        stack   = 0xff;                 // + 0x100 ! ==> cpupullstack/cpupushstack
        pc      = 0;
        zflag   = 0;                    // clear flags
        nflag   = 0;
        cflag   = 0;
        vflag   = 0;
        iflag   = 0;
        dflag   = 0;
        bflag   = 0;

        cycle   = 0;                    // clear cycle count
        lastcycle = 0;

        stateflags = 0;
#endif
        cpusettracemode(0);             // trace mode off
        cpuclearbreakpoint();           // no breakpoint

        cpumessageflag = 0;
        cpusettype(CPU_65C02);

        cpusetdelay(inicpudelay);
        cpusetlinecycle(65);

        cpureset();

        return NULL;

      } // cpuinit


/*----------------------------------------

 error  *cpuclose       (void)

        finalize cpu

-----------------------------------------*/


      unsigned char *cpuclose(void) {

        return NULL;            // no error

      } // cpuclose


/*----------------------------------------

 error  *cpustore       (unsigned int winprotocol, FILE *file, unsigned int percent)

        write all necessary data about cpu to file

-----------------------------------------*/

      unsigned char *cpustore(unsigned int winprotocol, FILE *file, unsigned int percent) {
        unsigned char flags;
        unsigned int  reg;
        unsigned char header[32];

        memset(header, 0x00, sizeof(header));
        strcpy(header, "CPU STATE V0.31");
        fwrite(header,          sizeof(header), 1, file);

        reg = cpugettype();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugeta();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugetx();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugety();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugetpc();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugetsp();
        fwrite(&reg,             sizeof(reg),           1,file);
        flags = cpugetflags();
        fwrite(&flags,           sizeof(flags),         1,file);
        reg = cpugetstate();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugetcycle();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugetlastcycle();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugetlinecycle();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugetdelay();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugetbreakpoint();
        fwrite(&reg,             sizeof(reg),           1,file);
        reg = cpugettracemode();
        fwrite(&reg,             sizeof(reg),           1,file);

        fwrite(&inicpudelay,     sizeof(inicpudelay),   1,file);
#ifdef DEF_INC_ASMLIB_H
        if (cpuregswindow) {
          flags = windowgetvisibility(cpuregswindow);
        }
        else {
          flags = 0;
        }
#else
        flags = 0;
#endif
        fwrite(&flags,           sizeof(flags),         1,file);
        fwrite(&cpumessageflag,  sizeof(cpumessageflag),1,file);

#ifdef DEF_INC_DAPPLE_H
        guipercent(winprotocol, percent,
"!\
ECPU stored.;\
GCPU gespeichert.;\
");
#endif

        return NULL;            // no error

      } // cpustore


/*----------------------------------------

 error  *cpurestore     (unsigned int winprotocol, FILE *file, unsigned int percent)

        read all necessary data about cpu from file

-----------------------------------------*/

      unsigned char *cpurestore(unsigned int winprotocol, FILE *file, unsigned int percent) {
        unsigned char flags;
        unsigned int  reg;
        unsigned char header[32];

        fread(header,           sizeof(header),         1,file);
        if (strcmp(header, "CPU STATE V0.31")) {
#ifdef DEF_INC_ASMLIB_H
          stringwrite(winprotocol, "CPU emulation data not found.\r");
#endif
          return "CPU emulation data not found.\r";
        }

        fread(&reg,             sizeof(reg),            1,file);
        cpusettype(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpuseta(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpusetx(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpusety(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpusetpc(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpusetsp(reg);
        fread(&flags,           sizeof(flags),          1,file);
        cpusetflags(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpuclearstate(-1);      // clear all
        cpusetstate(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpusetcycle(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpusetlastcycle(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpusetlinecycle(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpusetdelay(reg);
        fread(&reg,             sizeof(reg),            1,file);
        cpusetbreakpoint(reg);
#ifdef EMUZ80
        z80setbreakpoint(reg);
#endif
        fread(&reg,             sizeof(reg),            1,file);
        cpusettracemode(reg);

        fread(&inicpudelay,     sizeof(inicpudelay),    1,file);
        fread(&flags,           sizeof(flags),          1,file);

#ifdef DEF_INC_ASMLIB_H
        if (cpuregswindow) {
          channelout(cpuregswindow, 12);
#ifdef EMUZ80
          if (!cpuflag) {
#endif
            cpuwriteregs(cpuregswindow);
#ifdef EMUZ80
          }
          else {
            z80writeregs(cpuregswindow);
          }
#endif
          windowsetvisibility(cpuregswindow, flags);
        }
#endif
        fread(&cpumessageflag,  sizeof(cpumessageflag), 1,file);

        if (cpugetstate() & CPU_STATEHALT) {
          cpulight(0);
        }
        else {
          cpulight(1);
        }

#ifdef DEF_INC_DAPPLE_H
        guipercent(winprotocol, percent,
"!\
ECPU restored.;\
GCPU geladen.;\
");
#endif

        return NULL;            // no error

      } // cpurestore


/*----------------------------------------

 error  *cpureset       (void)

        reset 6502

-----------------------------------------*/


      unsigned char *cpureset(void) {

        if (appletype & (APPLEIIC | LASER128)) {
          if (cpugettype() == CPU_6502) {       /* the Apple//c and the Laser 128 demand at least a 65c02 */
            cpusettype(CPU_65C02);
          }
        }

#ifdef CPU_CYCLEEXACT
        cpu6502reset(&cpu);
#endif

#ifdef CPU_ASM
        cpu6502reset();
#else
        stateflags = stateflags | CPU_STATERESET;
#endif
        cpulight(1);

        return NULL;            // no error

      } // cpureset


/*--------------------------------------*/


      void hex8tostringf(unsigned int value, unsigned char *stringpointer) {
        register unsigned char c;

        c = (unsigned char)value >> 4;
        if (c > 9) { c = c + 'A' - 10; }
        else { c = c + '0'; }
        *stringpointer = c;
        c = (unsigned char)value & 0xf;
        if (c > 9) { c = c + 'A' - 10; }
        else { c = c + '0'; }
        *(stringpointer+1) = c;

      }

#ifndef CPU_ASM
#ifdef DEF_INC_ASMLIB_H

      void cpuwriteregs(unsigned int window) {
        unsigned char message[80];

        channelout(window, 12);
//                       012345678901234567890123 456789012345678901234 567890123456789
        strcpy(message, "A: $xx, X: $xx, Y: $xx\rPC: $xxxx, SP: $01xx\rFlags: NV1BDIZC");
        hex8tostringf(cpugeta(), &message[4]);
        hex8tostringf(cpugetx(), &message[12]);
        hex8tostringf(cpugety(), &message[20]);
        hex8tostringf(cpugetpc() >> 8,   &message[28]);
        hex8tostringf(cpugetpc() & 0xff, &message[30]);
        hex8tostringf(stack &0xff, &message[41]);
        cpuflagstostring( cpugetflags(), &message[51]);
        stringwrite(window, message);

      } // cpuwriteregs
#endif
#endif


/*--------------------------------------*/


      void cpulight(unsigned char value) {

#ifdef DEF_INC_ASMLIB_H
        if (value) {
          imagefillbox(window, SLOTX1, 388, SLOTX1+1,389, RGBLGHTON);
        }
        else {
          imagefillbox(window, SLOTX1, 388, SLOTX1+1,389, RGBLGHTOFF);
        }
        screenupdate = 1;
#endif

      } // cpulight


/*--------------------------------------*/

#ifndef CPU_ASM



      unsigned int cpugettype (void) {

        return cputype;

      } // cpugettype


/*--------------------------------------*/


      void cpusettype (unsigned int value) {

        switch (value) {
          case CPU_6502 :
          case CPU_65C02 :
          case CPU_65SC02 :
            cputype = value;
            break;
        } // switch (value)

      } // cpusettype


/*--------------------------------------*/


      unsigned int cpugetstate(void) {

        return stateflags;

      } // cpugetstate


/*--------------------------------------*/


      void cpusetstate(unsigned int value) {

        value = value & (CPU_STATEHALT | CPU_STATERESET | CPU_STATETRACE
                       | CPU_STATEIRQ1 | CPU_STATEIRQ2 | CPU_STATENMI1 | CPU_STATENMI2)
                       & (~stateflags);

        if (value & CPU_STATERESET) {
          setmessage("CPU Reset triggered");
        }
        if (value & CPU_STATEHALT) {
          setmessage("CPU halted");
          cpulight(0);
        }
        if (value & CPU_STATETRACE) {
          setmessage("CPU trace mode set");
        }

/*
        Don't put a message on screen with Dapple, because mouse VBL just generates too many interrupts ...

        if (value & CPU_STATEIRQ1) {
          setmessage("CPU IRQ1 triggered");
        }
        if (value & CPU_STATEIRQ2) {
          setmessage("CPU IRQ2 triggered");
        }
*/
        if (value & CPU_STATENMI1) {
          setmessage("CPU NMI1 triggered");
        }
        if (value & CPU_STATENMI2) {
          setmessage("CPU NMI2 triggered");
        }
        stateflags = stateflags | value;

      } // cpusetstate


/*--------------------------------------*/


      void cpuclearstate(unsigned int value) {

        value = value
                & (CPU_STATEHALT | CPU_STATERESET | CPU_STATETRACE | CPU_STATEBPT | CPU_STATEGURU
                 | CPU_STATEIRQ1 | CPU_STATEIRQ2  | CPU_STATENMI1  | CPU_STATENMI2)
                & (stateflags);

        if (value & CPU_STATEHALT) {
          setmessage("CPU started");
          cpulight(1);
        }
        if (value & CPU_STATERESET) {
          setmessage("CPU Reset cleared");
        }
        if (value & CPU_STATEGURU) {    // if it was an illegal opcode ==> reset whole machine
          applereset();
        }
// don't show message because of too many trace exceptions
/*      if (value & STATETRACE) {
          setmessage("CPU trace mode cleared");
        }
*/

        if (value & CPU_STATEIRQ1) {
          setmessage("CPU IRQ1 cleared");
        }
        if (value & CPU_STATEIRQ2) {
          setmessage("CPU IRQ1 cleared");
        }
        if (value & CPU_STATENMI1) {
          setmessage("CPU NMI1 cleared");
        }
        if (value & CPU_STATENMI2) {
          setmessage("CPU NMI1 cleared");
        }

        stateflags = stateflags & (~(value));

      } // cpuclearstate


/*--------------------------------------*/


      void cpufastsethalt(void) {

        stateflags = stateflags | CPU_STATEHALT;
        cpulight(0);

      } // cpufastsethalt


/*--------------------------------------*/


      void cpufastclearhalt(void) {

        stateflags = stateflags & ~(CPU_STATEHALT);
        cpulight(1);

      } // cpufastclearhalt


/*--------------------------------------*/


      unsigned int cpugetbreakpoint(void) {

        return breakpoint;

      } // cpugetbreakpoint


/*--------------------------------------*/


      void cpusetbreakpoint (unsigned int value) {

        if (value > 0xffff) {
          breakpoint = 0xffffffff;
        }
        else {
          breakpoint = value;
        }
        stateflags = stateflags & ~CPU_STATEBPT;

      } // cpusetbreakpoint


/*-------------------------------------*/


      void cpuclearbreakpoint(void) {

        breakpoint = 0xffffffff;
        stateflags = stateflags & ~CPU_STATEBPT;

      } // cpuclearbreakpoint


/*--------------------------------------*/


      unsigned int cpugettracemode(void) {

        return traceflag;

      } // cpugettracemode


/*-------------------------------------*/


      void cpusettracemode(unsigned int value) {

        if (value < 3) {
          traceflag = value;
        }

      } // cpusettracemode


/*--------------------------------------*/


      unsigned int cpugeta(void) {

#ifdef CPU_CYCLEEXACT
        return cpu.areg;
#else
        return areg;
#endif

      } // cpugeta


/*--------------------------------------*/


      void cpuseta(unsigned int value) {

#ifdef CPU_CYCLEEXACT
        cpu.areg = value & 0xff;
#else
        areg = value & 0xff;
#endif

      } // cpuseta


/*--------------------------------------*/


      unsigned int cpugetx(void) {

#ifdef CPU_CYCLEEXACT
        return cpu.xreg;
#else
        return xreg;
#endif

      } // cpugetx


/*--------------------------------------*/


      void cpusetx(unsigned int value) {

#ifdef CPU_CYCLEEXACT
        cpu.xreg = value & 0xff;
#else
        xreg = value & 0xff;
#endif

      } // cpusetx


/*--------------------------------------*/


      unsigned int cpugety(void) {

#ifdef CPU_CYCLEEXACT
        return cpu.yreg;
#else
        return yreg;
#endif

      } // cpugety


/*--------------------------------------*/


      void cpusety(unsigned int value) {

#ifdef CPU_CYCLEEXACT
        cpu.yreg = value &0xff;
#else
        yreg = value & 0xff;
#endif

      } // cpusety


/*--------------------------------------*/


      unsigned int cpugetpc(void) {

#ifdef CPU_CYCLEEXACT
        return cpu.pcreg;
#else
        return pc;
#endif

      } // cpugetpc


/*--------------------------------------*/


      void cpusetpc(unsigned int address) {

#ifdef CPU_CYCLEEXACT
        cpu.pcreg = address &0xffff;
#else
        pc = address & 0xffff;
#endif

      } // cpusetpc


/*--------------------------------------*/


      unsigned int cpugetsp(void) {

#ifdef CPU_CYCLEEXACT
        return cpu.spreg | 0x100;
#else
        return (stack | 0x100);
#endif

      } // cpugetsp


/*--------------------------------------*/


      void cpusetsp(unsigned int address) {

#ifdef CPU_CYCLEEXACT
        cpu.spreg = address & 0xff;
#else
        stack = address & 0xff;
#endif

      } // cpusetsp


/*--------------------------------------*/


      unsigned int cpugetcycle(void) {

        return cycle;

      } // cpugetcycle


/*--------------------------------------*/


      void cpusetcycle(unsigned int value) {

        cycle = value;

      } // cpusetcycle


/*--------------------------------------*/


      unsigned int cpugetlastcycle(void) {

        return lastcycle;

      } // cpugetlastcycle


/*--------------------------------------*/


      void cpusetlastcycle(unsigned int value) {

        lastcycle = value;

      } // cpusetlastcycle


/*--------------------------------------*/


      unsigned int cpugetlinecycle(void) {

        return linecycles;

      } // cpugetlinecycle


/*--------------------------------------*/


      void cpusetlinecycle(unsigned int value) {

        linecycles = value;

      } // cpusetlinecycle


/*--------------------------------------*/


      unsigned int cpugetdelay(void) {

        return cpudelay;

      } // cpugetdelay


/*--------------------------------------*/


      void cpusetdelay(unsigned int value) {

        cpudelay = value;

      } // cpusetdelay

// --> ifndef CPU_ASM
#endif


/*--------------------------------------*/


      unsigned char cpuinuse(void) {
        register unsigned int state;

        state = cpugetstate();
        return ((state & CPU_STATEHALT)
              ||(state & CPU_STATEBPT)
              ||(state & CPU_STATEGURU)) ? 0 : -1;

      } /// cpuinuse


/*--------------------------------------*/



/*----------------------------------------

        flags handling

        - cpuflagssetnz
        - cpugetflags
        - cpusetflags
        - cpuflagstostring

----------------------------------------*/



/*--------------------------------------*/


      void cpuflagssetnz(unsigned int value) {

        zflag = (value == 0) ? 1 : 0;
        nflag = (value > 0x7f) ? 1 : 0;

      } // cpuflagssetnz


/*--------------------------------------*/


      card8 cpugetflags(void) {

        return
            ((nflag) ? 0x80 : 0)
          | ((vflag) ? 0x40 : 0)
          | 0x20
          | ((bflag) ? 0x10 : 0)
          | ((dflag) ? 0x08 : 0)
          | ((iflag) ? 0x04 : 0)
          | ((zflag) ? 0x02 : 0)
          | ((cflag) ? 0x01 : 0);

      } // cpugetflags


/*--------------------------------------*/


      void cpusetflags(unsigned int value) {

        nflag = (value & 0x80) ? 1 : 0;
        vflag = (value & 0x40) ? 1 : 0;
//      bflag = (value & 0x10) ? 1 : 0;
        dflag = (value & 0x08) ? 1 : 0;
        iflag = (value & 0x04) ? 1 : 0;
        zflag = (value & 0x02) ? 1 : 0;
        cflag = (value & 0x01) ? 1 : 0;

      } // cpusetflags


/*--------------------------------------*/


      void cpuflagstostring(unsigned int value, unsigned char *stringptr) {

        if (value & 0x80) { *stringptr++ = 'N'; }
        else              { *stringptr++ = '-'; }
        if (value & 0x40) { *stringptr++ = 'V'; }
        else              { *stringptr++ = '-'; }
        if (value & 0x20) { *stringptr++ = '1'; }
        else              { *stringptr++ = '-'; }
        if (value & 0x10) { *stringptr++ = 'B'; }
        else              { *stringptr++ = '-'; }
        if (value & 0x08) { *stringptr++ = 'D'; }
        else              { *stringptr++ = '-'; }
        if (value & 0x04) { *stringptr++ = 'I'; }
        else              { *stringptr++ = '-'; }
        if (value & 0x02) { *stringptr++ = 'Z'; }
        else              { *stringptr++ = '-'; }
        if (value & 0x01) { *stringptr   = 'C'; }
        else              { *stringptr   = '-'; }

      } /* cpuflagstostring */



#ifndef CPU_ASM

/*----------------------------------------

        addressing modes

        - readzp

----------------------------------------*/


#define incpc pc++


/* standard read zero page instruction */
      unsigned char readzp(void) {
        address = memoryread(pc);
        incpc;
        return memoryread(address);
      } /* readzp */

/* save address for read-and-modify instructions */
      unsigned char readzpm(void) {
        address = memoryread(pc);
        incpc;
        return memoryread(address);
      } /* readzp */

/* standard read zero page,x instruction */
      unsigned char readzpx(void) {
        address = (memoryread(pc) + xreg) & 0xff;
        incpc;
        return memoryread(address);
      } /* readzpx */

/* save address for read-and-modify instructions */
      unsigned char readzpxm(void) {
        address = (memoryread(pc) + xreg) & 0xff;
        incpc;
        return memoryread(address);
      } /* readzpx */

/* standard read zero page,y instruction */
      unsigned char readzpy(void) {
        address = (memoryread(pc) + yreg) & 0xff;
        incpc;
        return memoryread(address);
      } /* readzpy */


/* standard read abs instruction */
      unsigned char readabs(void) {
        address = memoryread(pc);
        incpc;
        address = address + ( memoryread(pc) << 8 );
        incpc;
        return memoryread(address);
      } /* readabs */

/* save address for read-and-modify instructions */
      unsigned char readabsm(void) {
        address = memoryread(pc);
        incpc;
        address = address + ( memoryread(pc) << 8 );
        incpc;
        return memoryread(address);
      } /* readabs */

/* standard read abs,x instruction */
      unsigned char readabsx(void) {
        address = memoryread(pc);
        incpc;
        address = (address + ( memoryread(pc) << 8 ) + xreg) & 0xffff;
        incpc;
        return memoryread(address);
      } /* readabsx */

/* save address for read-and-modify instructions */
      unsigned char readabsxm(void) {
        address = memoryread(pc);
        incpc;
        address = (address + ( memoryread(pc) << 8 ) + xreg) & 0xffff;
        incpc;
        return memoryread(address);
      } /* readabsxm */

/* standard read abs,y instruction */
      unsigned char readabsy(void) {
        address = memoryread(pc);
        incpc;
        address = (address + ( memoryread(pc) << 8 ) + yreg) & 0xffff;
        incpc;
        return memoryread(address);
      } /* readabsy */

/* save address for read-and-modify instructions */
      unsigned char readabsym(void) {
        address = memoryread(pc);
        incpc;
        address = (address + ( memoryread(pc) << 8 ) + yreg) & 0xffff;
        incpc;
        return memoryread(address);
      } /* readabsym */

      unsigned char readindzp(void) {
        address = memoryread(pc);
        incpc;
        address = memoryread(address) + ( memoryread((address + 1) & 0xff) << 8);
        return memoryread(address);
      } /* readindzp */


      unsigned char readindzpx(void) {
        address = ( memoryread(pc) + xreg ) & 0xff;
        incpc;
        address = memoryread(address) + ( memoryread((address + 1) & 0xff) << 8);
        return memoryread(address);
      } /* readindzpx */


      unsigned char readindzpy(void) {
        address = memoryread(pc);
        incpc;
        address = ( memoryread(address) + ( memoryread((address + 1) & 0xff) << 8) + yreg) & 0xffff;
        return memoryread(address);
      } /* readindzpy */



/* standard write zero page instruction */
      void writezp (unsigned int value) {
        address = memoryread(pc);
        incpc;
        memorywrite(address, value);
      } /* writezp */


/* standard write zero page,x instruction */
      void writezpx (unsigned int value) {
        address = ( memoryread(pc) + xreg ) & 0xff;
        incpc;
        memorywrite(address, value);
      } /* writezpx */


/* standard write zero page,y instruction */
      void writezpy (unsigned int value) {
        address = ( memoryread(pc) + yreg ) & 0xff;
        incpc;
        memorywrite(address, value);
      } /* writezpy */


/* standard write abs instruction */
      void writeabs (unsigned int value) {
        address = memoryread(pc);
        incpc;
        address = address + ( memoryread(pc) << 8 );
        incpc;
        memorywrite(address, value);
      } /* writeabs */


/* standard write abs,x instruction */
      void writeabsx (unsigned int value) {
        address = memoryread(pc);
        incpc;
        address = (address + ( memoryread(pc) << 8 ) + xreg) & 0xffff;
        incpc;
        memorywrite(address, value);
      } /* writeabsx */


/* standard write abs,x instruction */
      void writeabsy (unsigned int value) {
        address = memoryread(pc);
        incpc;
        address = (address + ( memoryread(pc) << 8 ) + yreg) & 0xffff;
        incpc;
        memorywrite(address, value);
      } /* writeabsy */

      void writeindzp (unsigned int value) {
        address = memoryread(pc);
        incpc;
        address = memoryread(address) + ( memoryread((address + 1) & 0xff) << 8);
        memorywrite(address, value);
      } /* writeindzp */


      void writeindzpx (unsigned int value) {
        address = ( memoryread(pc) + xreg ) & 0xff;
        incpc;
        address = memoryread(address) + ( memoryread((address + 1) & 0xff) << 8);
        memorywrite(address, value);
      } /* writeindzpx */


/* standard write (zp),y instruction */
      void writeindzpy (unsigned int value) {
        address = memoryread(pc);
        incpc;
        address = ( memoryread(address) + ( memoryread((address + 1) & 0xff) << 8 ) + yreg) & 0xffff;
        memorywrite(address, value);
      } /* writeindzpy */
#endif

/*----------------------------------------

        stack handling

        - cpupullstack
        - cpupushstack

----------------------------------------*/


      unsigned char cpupullstack(void) {

        stack = (stack + 1) & 0xff;
        return memoryread(stack | 0x100);

      } // cpupullstack


/*--------------------------------------*/


      void cpupushstack(unsigned int value) {

        memorywrite(stack | 0x100, value);
        stack = (stack - 1) & 0xff;

      } // cpupushstack


/*--------------------------------------*/


/* Adc */

/* The following code was provided by Mr. Scott Hemphill. */
/* Thanks a lot! */

      void adc(unsigned int value) {
        register unsigned int w;

        if ((areg ^ value) & 0x80) {
          vflag = 0;
        }
        else {
          vflag = 1;
        }

        if (dflag) {
          w = (areg & 0xf) + (value & 0xf) + cflag;
          if (w >= 10) {
            w = 0x10 | ((w+6)&0xf);
          }
          w += (areg & 0xf0) + (value & 0xf0);
          if (w >= 160) {
            cflag = 1;
            if (vflag && w >= 0x180) vflag = 0;
            w += 0x60;
          }
          else {
            cflag = 0;
            if (vflag && w < 0x80) vflag = 0;
          }
        }
        else {
          w = areg + value + cflag;
          if (w >= 0x100) {
            cflag = 1;
            if (vflag && w >= 0x180) vflag = 0;
          }
          else {
            cflag = 0;
            if (vflag && w < 0x80) vflag = 0;
          }
        }
        areg = (unsigned char)w;
        nflag = (areg >= 0x80) ? 1 : 0;
        zflag = (areg == 0)    ? 1 : 0;
      } /* adc */


/* Sbc */

/* The following code was provided by Mr. Scott Hemphill. */
/* Thanks a lot again! */

      void sbc(unsigned int value) {
        register unsigned int w;
        register unsigned char temp;

        if ((areg ^ value) & 0x80) {
          vflag = 1;
        }
        else {
          vflag = 0;
        }

        if (dflag) {            /* decimal subtraction */
          temp = 0xf + (areg & 0xf) - (value & 0xf) + (cflag);
          if (temp < 0x10) {
            w = 0;
            temp -= 6;
          }
          else {
            w = 0x10;
            temp -= 0x10;
          }
          w += 0xf0 + (areg & 0xf0) - (value & 0xf0);
          if (w < 0x100) {
            cflag = 0;
            if (vflag && w < 0x80) vflag = 0;
            w -= 0x60;
          }
          else {
            cflag = 1;
            if ((vflag) && w >= 0x180) vflag = 0;
          }
          w += temp;
        }
        else {                  /* standard binary subtraction */
          w = 0xff + areg - value + cflag;
          if (w < 0x100) {
            cflag = 0;
            if (vflag && (w < 0x80)) vflag = 0;
          }
          else {
            cflag = 1;
            if (vflag && (w >= 0x180)) vflag = 0;
          }
        }
        areg = (unsigned char)w;
        nflag = (areg >= 0x80) ? 1 : 0;
        zflag = (areg == 0)    ? 1 : 0;
      } /* sbc */


/*--------------------------------------*/


/* Cmp */

      void cmp(unsigned int value) {

        cflag = (areg >= value) ? 1 : 0;
        nflag = (((areg - value) & 0x80) == 0) ? 0 : 1;
        zflag = (areg == value) ? 1 : 0;

      } /* cmp */

/* Cpx */

      void cpx(unsigned int value) {

        cflag = (xreg >= value) ? 1 : 0;
        nflag = (((xreg - value) & 0x80) == 0) ? 0 : 1;
        zflag = (xreg == value) ? 1 : 0;

      } /* cpx */

/* Cpy */

      void cpy(unsigned int value) {

        cflag = (yreg >= value) ? 1 : 0;
        nflag = (((yreg - value) & 0x80) == 0) ? 0 : 1;
        zflag = (yreg == value) ? 1 : 0;

      } // cpy


/*--------------------------------------*/

#ifndef CPU_ASM
      void cpubrk() {
        unsigned char message[80];

        if (cpumessageflag) {
          strcpy(message, "BRK instruction hit at address $xxxx");
          hex8tostringf((pc - 1) >> 8,   &message[32]);
          hex8tostringf((pc - 1) & 0xff, &message[34]);
          setmessage(message);
        }
        stateflags = stateflags | CPU_STATEBRK; /* give the surrounding environment     */
        incpc;                                  /* the chance to respond to a BRK       */
        cpupushstack(pc >> 8);                  /* e.g. with a debugger                 */
        cpupushstack(pc & 0xff);
        bflag = 1;
        cpupushstack( cpugetflags() );
        iflag = 1;

      } /* cpubrk */


      void cpuillegal(unsigned int opcode) {
        unsigned char message[80];

        pc = (pc - 1) & 0xffff;
        stateflags = stateflags | CPU_STATEGURU | CPU_STATEHALT;
        cpulight(0);

        strcpy(message, "Illegal instruction $xx hit at address $xxxx");
        hex8tostringf(opcode,    &message[21]);
        hex8tostringf(pc >> 8,   &message[40]);
        hex8tostringf(pc & 0xff, &message[42]);
        setmessage(message);

      } // cpuillegal


/* the following function will only show that an illegal opcode was executed,
   but will not halt the CPU */
      void cpushowillegal(unsigned int opcode) {
        unsigned char message[80];

        if (cpumessageflag) {
          strcpy(message, "Illegal instruction $xx hit at address $xxxx");
          hex8tostringf(opcode,    &message[21]);
          hex8tostringf((pc - 1) >> 8,   &message[40]);
          hex8tostringf((pc - 1) & 0xff, &message[42]);
          setmessage(message);
        }

      } // cpushowillegal
// --> #ifndef CPU_ASM
#endif


/*--------------------------------------*/

#ifndef CPU_ASM

      unsigned int cpuline(void) {


#ifdef CPU_CYCLEEXACT

// draw line on screen
        do {
          if ((cycle - lastcycle) >= linecycles) {
            lastcycle = lastcycle + linecycles;
            virtline();
            if (!virtrasterline) {
              if (traceflag == 2) {
                stateflags = stateflags | CPU_STATETRACE;
              }
              return cpu6502getstate(&cpu);
            }
          }

          cpu6502step(&cpu);
          cycle = cycle + 1;

        } while (1);

#else
        do {    // ... while (!tracemode)

// test interrupt signals et al.
          if (stateflags) {
            if (stateflags & CPU_STATECOP) {
              stateflags = stateflags & ~(CPU_STATECOP | CPU_STATEGURU | CPU_STATEHALT);
            }
            if (stateflags & CPU_STATERESET) {
              stateflags = stateflags & ~(CPU_STATEHALT | CPU_STATERESET | CPU_STATEGURU | CPU_STATEBPT | CPU_STATEBRK);
              if (cputype & (CPU_65C02 | CPU_65SC02)) {
                dflag = 0;                      // on a 65c02/65sc02 the dflag gets cleared on reset
              }
              pc = memoryread(0xfffc) | (memoryread(0xfffd) << 8);
              cycle = cycle + 5;
            }
            if (stateflags & CPU_STATEHALT) {   // processor still halted?
#ifdef DEF_INC_ASMLIB_H
              screenupdate = 1;
#endif
              taskswitch();
              return stateflags;
            }
            if (stateflags & CPU_STATEBRK) {
              stateflags = stateflags & ~CPU_STATEBRK;
              if (cputype & (CPU_65C02 | CPU_65SC02)) {
                dflag = 0;                      // on a 65c02/65sc02 the dflag gets cleared on brk
              }
              pc    = memoryread(0xfffe) | (memoryread(0xffff) << 8);
              cycle = cycle + 7;
            }
            if (stateflags & (CPU_STATENMI1 | CPU_STATENMI2)) {
              stateflags = stateflags & ~(CPU_STATENMI1 | CPU_STATENMI2);
              cpupushstack(pc >> 8);
              cpupushstack(pc & 0xff);
              cpupushstack( cpugetflags() );
              if (cputype & (CPU_65C02 | CPU_65SC02)) {
                dflag = 0;                      // on a 65c02/65sc02 the dflag gets cleared on nmi
              }
              pc    = memoryread(0xfffa) | (memoryread(0xfffb) << 8);
              cycle = cycle + 6;
            }
            else {
              if (stateflags & (CPU_STATEIRQ1 | CPU_STATEIRQ2)) {
                if (!iflag) {
                  stateflags = stateflags & (~CPU_STATEIRQ1 | CPU_STATEIRQ2);
                  cpupushstack(pc >> 8);
                  cpupushstack(pc & 0xff);
                  cpupushstack( cpugetflags() );
                  iflag = 1;
                  if (cputype & (CPU_65C02 | CPU_65SC02)) {
                    dflag = 0;          // on a 65c02/65sc02 the dflag gets cleared on irq
                  }
                  pc    = memoryread(0xfffe) | (memoryread(0xffff) << 8);
                  cycle = cycle + 6;
                } // if (!iflag)
              }
            }
            stateflags = stateflags &  ~(CPU_STATETRACE | CPU_STATEGURU);       // remove these
          } // if (stateflags)

// test for breakpoint
          if (pc == breakpoint) {
            stateflags = stateflags ^ CPU_STATEBPT;
            if (stateflags & CPU_STATEBPT) {
              setmessage("CPU stopped because breakpoint was reached");
              cpulight(0);
              return stateflags;
            }
          }

/* draw line on screen */
          if ((cycle - lastcycle) >= linecycles) {
            lastcycle = lastcycle + linecycles;
            virtline();
            if (!virtrasterline) {
              if (traceflag == 2) {
                stateflags = stateflags | CPU_STATETRACE;
              }
              return stateflags;
            }
          }

// interprete next opcode
          bytebuffer    = memoryread(pc);
          incpc;
          switch (bytebuffer) {
                case 0x00 :                                             // BRK
                        cpubrk();
                        return stateflags;
                case 0x01 :                                             // ORA (zp,x)
                        areg    = areg | readindzpx();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 6;
                        break;
                case 0x02 :                                             // COP
#ifdef REECOP
                        stateflags      = stateflags | CPU_STATECOP;
                        stateflags      = stateflags | CPU_STATEHALT;
                        stateflags      = stateflags | CPU_STATEGURU;
                        pc              = (pc - 1) & 0xffff;
//                      taskloghex32 (LOG_VERBOSE, pc);
//                      tasklogstring(LOG_VERBOSE, " | COP instruction");
//                      tasklogreturn(LOG_VERBOSE, );
                        return stateflags;
#else
                        if (cputype == CPU_6502) {
                          cpuillegal(0x02);                             // illegal 6502
                        }
                        else {                                          // NOP2 65C02
                          incpc;
                          cycle = cycle + 2;
                        }
#endif
                        break;
                case 0x03 :                                             // NOP 65C02
                        cycle   = cycle + 2;
                        break;
                case 0x04 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0x04);                         // illegal SKB 6502
                          incpc;
                          cycle = cycle + 2;
                        }
                        else {
                          bytebuffer    = readzpm();                    // TSB zp 65C02
                          zflag = ((bytebuffer & areg) == 0) ? 1 : 0;
                          memorywrite(address, areg | bytebuffer);
                          cycle = cycle + 5;
                        }
                        break;
                 case 0x05 :                                            // ORA zp
                        areg    = areg | readzp();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 3;
                        break;
                case 0x06 :                                             // ASL zp
                        bytebuffer      = readzpm();
                        if ( bytebuffer > 0x7f ) {
                          bytebuffer    = (bytebuffer << 1) & 0xff;
                          cflag         = 1;
                        }
                        else {
                          bytebuffer  = bytebuffer << 1;
                          cflag         = 0;
                        }
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 5;
                        break;
                case 0x07 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x07);               /* illegal SLO (ASO) zp 6502 */
                    bytebuffer    = readzpm();
                    if ( bytebuffer > 0x7f ) {
                      bytebuffer  = (bytebuffer << 1) & 0xff;
                      cflag       = 1;
                    }
                    else {
                      bytebuffer  = bytebuffer << 1;
                      cflag       = 0;
                    }
                    memorywrite(address, bytebuffer);
                    areg = areg | bytebuffer;
                    cpuflagssetnz(areg);
                    cycle = cycle + 5;
                  }
                  else {                                /* RMB0 zp */
                    bytebuffer    = readzpm() & 0xfe;
                    memorywrite(address, bytebuffer);
                    cycle = cycle + 5;
                  }
                  break;
                case 0x08 :                                             // PHP
                        cpupushstack( cpugetflags() );
                        cycle = cycle + 3;
                        break;
                case 0x09 :                                             // ORA #
                        areg  = areg | memoryread(pc);
                        incpc;
                        cpuflagssetnz(areg);
                        cycle = cycle + 2;
                        break;
                case 0x0a :                                             // ASL
                        if ( areg > 0x7f ) {
                          areg  = (areg << 1) & 0xff;
                          cflag = 1;
                        }
                        else {
                          areg  = areg << 1;
                          cflag = 0;
                        }
                        cpuflagssetnz(areg);
                        cycle   = cycle + 2;
                        break;
                case 0x0b :                                             // NOP 65C02
                        cycle = cycle + 1;
                        break;
                case 0x0c :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0x0c);                         // illegal SKW 6502
                          pc    = (pc + 2) & 0xffff;
                          cycle = cycle + 4;
                        }
                        else {
                          bytebuffer = readabsm();                      // TSB abs 65C02
                          zflag = ((bytebuffer & areg) == 0) ? 1 : 0;
                          memorywrite(address, areg | bytebuffer);
                          cycle = cycle + 6;
                        }
                        break;
                case 0x0d :                                             // ORA abs
                        areg  = areg | readabs();
                        cpuflagssetnz(areg);
                        cycle = cycle + 4;
                        break;
                case 0x0e :                                             // ASL abs
                  bytebuffer    = readabsm();
                  if ( bytebuffer > 0x7f ) {
                    bytebuffer  = (bytebuffer << 1) & 0xff;
                    cflag       = 1;
                  }
                  else {
                    bytebuffer  = bytebuffer << 1;
                    cflag       = 0;
                  }
                  cpuflagssetnz(bytebuffer);
                  memorywrite(address, bytebuffer);
                  cycle = cycle + 6;
                  break;
                case 0x0f :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x0f);               /* illegal SLO (ASO) abs 6502 */
                    bytebuffer    = readabsm();
                    if ( bytebuffer > 0x7f ) {
                      bytebuffer  = (bytebuffer << 1) & 0xff;
                      cflag       = 1;
                    }
                    else {
                      bytebuffer  = bytebuffer << 1;
                      cflag       = 0;
                    }
                    memorywrite(address, bytebuffer);
                    areg = areg | bytebuffer;
                    cpuflagssetnz(areg);
                    cycle = cycle + 6;
                  }
                  else {
                    if (cputype == CPU_65C02) {
                      pc        = (pc + 2) & 0xffff;    /* NOP3 65C02 */
                      cycle     = cycle + 2;
                    }
                    else {
                      if (!(readzp() & 0x01)) {         /* BBR0 65SC02 */
                        bytebuffer= memoryread(pc);
                        if ( bytebuffer > 0x7f) {
                          pc    = (pc - 255 + bytebuffer) & 0xffff;
                        }
                        else {
                          pc    = (pc + 1 + bytebuffer) & 0xffff;
                        }
                        cycle   = cycle + 5;
                      }
                      else {
                        incpc;
                        cycle   = cycle + 3;
                      }
                    }
                  }
                  break;
                case 0x10 :                                             // BPL
                  if ( nflag == 0) {
                    bytebuffer= memoryread(pc);
                    if ( bytebuffer > 0x7f) {
                      pc        = (pc - 255 + bytebuffer) & 0xffff;
                    }
                    else {
                      pc        = (pc + 1 + bytebuffer) & 0xffff;
                    }
                    cycle       = cycle + 3;
                  }
                  else {
                    pc          = (pc + 1) & 0xffff;
                    cycle       = cycle + 2;
                  }
                  break;
                case 0x11 :                                             // ORA (zp),y
                        areg    = areg | readindzpy();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 5;
                        break;
                case 0x12 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x12);                   /* illegal 6502 */
                  }
                  else {                                /* ORA (zp) 65C02 */
                    areg  = areg | readindzp();
                    cpuflagssetnz(areg);
                    cycle = cycle + 5;
                  }
                  break;
                case 0x13 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x12);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x14 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x14);               /* illegal SKB 6502 */
                    incpc;
                    cycle = cycle + 3;
                  }
                  else {                                /* TRB zp 65C02 */
                    bytebuffer    = readzpm();
                    zflag = ((bytebuffer & areg) == 0) ? 1 : 0;
                    memorywrite(address, (~areg) & bytebuffer);
                    cycle = cycle + 5;
                  }
                  break;
                case 0x15 :                                             // ORA zp,x
                        areg  = areg | readzpx();
                        cpuflagssetnz(areg);
                        cycle = cycle + 4;
                        break;
                case 0x16 :                                             // ASL zp,x
                        bytebuffer    = readzpxm();
                        if ( bytebuffer > 0x7f ) {
                          bytebuffer  = (bytebuffer << 1) & 0xff;
                          cflag       = 1;
                        }
                        else {
                          bytebuffer  = bytebuffer << 1;
                          cflag       = 0;
                        }
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle = cycle + 6;
                        break;
                case 0x17 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x17);               /* illegal SLO (ASO) zp,x 6502 */
                    bytebuffer    = readzpxm();
                    if ( bytebuffer > 0x7f ) {
                      bytebuffer  = (bytebuffer << 1) & 0xff;
                      cflag       = 1;
                    }
                    else {
                      bytebuffer  = bytebuffer << 1;
                      cflag       = 0;
                    }
                    memorywrite(address, bytebuffer);
                    areg = areg | bytebuffer;
                    cpuflagssetnz(areg);
                    cycle = cycle + 6;
                  }
                  else {                                /* RMB1 zp 65SC02 */
                    bytebuffer    = readzpm() & 0xfd;
                    memorywrite(address, bytebuffer);
                    cycle = cycle + 5;
                  }
                  break;
                case 0x18 :                                             // CLC
                        cflag   = 0;
                        cycle   = cycle + 2;
                        break;
                case 0x19 :                                             // ORA abs,y
                        areg    = areg | readabsy();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x1a :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0x1a);                         // illegal NOP 6502
                          cycle = cycle + 2;
                        }
                        else {                                          // INA 65C02
                          areg  = (areg + 1) & 0xff;
                          cpuflagssetnz(xreg);
                          cycle = cycle + 2;
                        }
                        break;
                case 0x1b :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x1b);               /* illegal SLO (ASO) abs,y 6502 */
                    bytebuffer    = readabsym();
                    if ( bytebuffer > 0x7f ) {
                      bytebuffer  = (bytebuffer << 1) & 0xff;
                      cflag       = 1;
                    }
                    else {
                      bytebuffer  = bytebuffer << 1;
                      cflag       = 0;
                    }
                    memorywrite(address, bytebuffer);
                    areg = areg | bytebuffer;
                    cpuflagssetnz(areg);
                    cycle = cycle + 7;
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x1c :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0x1c);                         // illegal SKW 6502
                          pc = (pc + 2) & 0xffff;
                          cycle = cycle + 4;
                        }
                        else {                                          // TRB abs 65C02
                          bytebuffer    = readabsm();
                          zflag = ((bytebuffer & areg) == 0) ? 1 : 0;
                          memorywrite(address, (~areg) & bytebuffer);
                          cycle = cycle + 6;
                        }
                        break;
                case 0x1d :                                             // ORA abs,x
                        areg    = areg | readabsx();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x1e :                                             // ASL abs,x
                        bytebuffer      = readabsxm();
                        if ( bytebuffer > 0x7f ) {
                          bytebuffer    = (bytebuffer << 1) & 0xff;
                          cflag = 1;
                        }
                        else {
                          bytebuffer    = bytebuffer << 1;
                          cflag = 0;
                        }
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 7;
                        break;
                case 0x1f :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x1f);               /* illegal SLO (ASO) abs,x 6502 */
                    bytebuffer    = readabsxm();
                    if ( bytebuffer > 0x7f ) {
                      bytebuffer  = (bytebuffer << 1) & 0xff;
                      cflag       = 1;
                    }
                    else {
                      bytebuffer  = bytebuffer << 1;
                      cflag       = 0;
                    }
                    memorywrite(address, bytebuffer);
                    areg = areg | bytebuffer;
                    cpuflagssetnz(areg);
                    cycle = cycle + 7;
                  }
                  else {
                    if (cputype == CPU_65C02) {
                      pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                      cycle = cycle + 3;
                    }
                    else {
                      if (!(readzp() & 0x02)) {           /* BBR1 65SC02 */
                        bytebuffer= memoryread(pc);
                        if ( bytebuffer > 0x7f) {
                          pc    = (pc - 255 + bytebuffer) & 0xffff;
                        }
                        else {
                          pc    = (pc + 1 + bytebuffer) & 0xffff;
                        }
                        cycle   = cycle + 5;
                      }
                      else {
                        incpc;
                        cycle   = cycle + 3;
                      }
                    }
                  }
                  break;
                case 0x20 :                                             // JSR abs
                        address = memoryread(pc);
                        incpc;
                        cpupushstack(pc >> 8);
                        cpupushstack(pc & 0xff);
                        pc      = address + ( memoryread(pc) << 8 );
                        cycle   = cycle + 6;
                        break;
                case 0x21 :                                             // AND (zp,x)
                        areg  = areg & readindzpx();
                        cpuflagssetnz(areg);
                        cycle = cycle + 6;
                        break;
                case 0x22 :
                        if (cputype == CPU_6502) {
                          cpuillegal(0x22);                             // illegal 6502
                        }
                        else {                                          // NOP2 65C02
                          pc    = (pc + 1) & 0xffff;
                          cycle = cycle + 2;
                        }
                        break;
                case 0x23 :
                        if (cputype == CPU_6502) {
                          cpuillegal(0x23);                             // illegal 6502
                        }
                        else {                                          // NOP 65C02
                          cycle = cycle + 2;
                        }
                        break;
                case 0x24 :                                             // BIT zp
                        bytebuffer = readzp();
                        vflag   = ((bytebuffer & 0x40)) ? 1 : 0;
                        nflag   = ((bytebuffer & 0x80)) ? 1 : 0;
                        zflag   = ((bytebuffer & areg)) ? 0 : 1;
                        cycle   = cycle + 3;
                        break;
                case 0x25 :                                             // AND zp
                        areg    = areg & readzp();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 3;
                        break;
                case 0x26 :                                             // ROL zp
                        bytebuffer      = readzpm();
                        if ( bytebuffer > 0x7f ) {
                          bytebuffer    = ((bytebuffer << 1) | cflag) & 0xff;
                          cflag         = 1;
                        }
                        else {
                          bytebuffer    = (bytebuffer << 1) | cflag;
                          cflag         = 0;
                        }
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 5;
                        break;
                case 0x27 :                                             // RMB2 zp
                        bytebuffer = readzpm() & 0xfb;
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 5;
                        break;
                case 0x28 :                                             // PLP
                        cpusetflags( cpupullstack() );
                        bflag   = 0;                                    // don't set b flag
                        cycle   = cycle + 4;
                        break;
                case 0x29 :                                             // AND #
                        areg    = areg & memoryread(pc);
                        incpc;
                        cpuflagssetnz(areg);
                        cycle   = cycle + 2;
                        break;
                case 0x2a :                                             // ROL
                        if (areg > 0x7f) {
                          areg  = ((areg << 1) | cflag) & 0xff;
                          cflag = 1;
                        }
                        else {
                          areg  = (areg << 1) | cflag;
                          cflag = 0;
                        }
                        cpuflagssetnz(areg);
                        cycle   = cycle + 2;
                        break;
                case 0x2b :
                        if (cputype == CPU_6502) {
                          cpuillegal(0x2b);                             // illegal 6502
                        }
                        else {                                          // NOP 65C02
                          cycle = cycle + 2;
                        }
                        break;
                case 0x2c :                                             // BIT abs
                        bytebuffer = readabs();
                        vflag   = ((bytebuffer & 0x40) == 0) ? 0 : 1;
                        nflag   = ((bytebuffer & 0x80) == 0) ? 0 : 1;
                        zflag   = ((bytebuffer & areg) == 0) ? 1 : 0;
                        cycle   = cycle + 4;
                        break;
                case 0x2d :                                             // AND abs
                        areg    = areg & readabs();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x2e :                             /* ROL abs */
                  bytebuffer    = readabsm();
                  if ( bytebuffer > 0x7f ) {
                    bytebuffer  = ((bytebuffer << 1) | cflag) & 0xff;
                    cflag       = 1;
                  }
                  else {
                    bytebuffer  = (bytebuffer << 1) | cflag;
                    cflag       = 0;
                  }
                  cpuflagssetnz(bytebuffer);
                  memorywrite(address, bytebuffer);
                  cycle = cycle + 6;
                  break;
                case 0x2f :
                  if (cputype == CPU_65SC02) {
                    if (!(readzp() & 0x04)) {           /* BBR2 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      pc        = (pc + 1) & 0xffff;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    cycle = cycle + 3;
                    break;
                  }
                case 0x30 :                             /* BMI */
                  if ( nflag != 0 ) {
                    bytebuffer= memoryread(pc);
                    if ( bytebuffer > 0x7f ) {
                      pc        = (pc - 255 + bytebuffer) & 0xffff;
                    }
                    else {
                      pc        = (pc + 1 + bytebuffer) & 0xffff;
                    }
                    cycle       = cycle + 3;
                  }
                  else {
                    pc          = (pc + 1) & 0xffff;
                    cycle       = cycle + 2;
                  }
                  break;
                case 0x31 :                                             // AND (zp),y
                        areg    = areg & readindzpy();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 5;
                        break;
                case 0x32 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x32);                   /* illegal 6502 */
                  }
                  else {                                /* AND (zp) 65C02 */
                    areg  = areg & readindzp();
                    cpuflagssetnz(areg);
                    cycle = cycle + 5;
                  }
                  break;
                case 0x33 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x33);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x34 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x34);               /* illegal SKB 6502 */
                    incpc;
                    cycle = cycle + 4;
                  }
                  else {                                /* BIT zp,x 65C02 */
                    bytebuffer    = readzpx();
                    vflag = ((bytebuffer & 0x40) == 0) ? 0 : 1;
                    nflag = ((bytebuffer & 0x80) == 0) ? 0 : 1;
                    zflag = ((bytebuffer & areg) == 0) ? 1 : 0;
                    cycle = cycle + 4;
                  }
                  break;
                case 0x35 :                                             // AND zp,x
                        areg    = areg & readzpx();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x36 :                                             // ROL zp,x
                        bytebuffer      = readzpxm();
                        if ( bytebuffer > 0x7f ) {
                          bytebuffer    = ((bytebuffer << 1) | cflag) & 0xff;
                          cflag         = 1;
                        }
                        else {
                          bytebuffer    = (bytebuffer << 1) | cflag;
                          cflag         = 0;
                        }
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 6;
                        break;
                case 0x37 :                                             // RMB3 zp
                        bytebuffer    = readzpm() & 0xf7;
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 5;
                        break;
                case 0x38 :                                             // SEC
                        cflag   = 1;
                        cycle   = cycle + 2;
                        break;
                case 0x39 :                                             // AND abs,y
                        areg    = areg & readabsy();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x3a :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x3a);               /* illegal NOP 6502 */
                    cycle = cycle + 2;
                  }
                  else {                                /* DEA 65C02 */
                    areg  = (areg - 1) & 0xff;
                    cpuflagssetnz(xreg);
                    cycle = cycle + 2;
                  }
                  break;
                case 0x3b :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x3b);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x3c :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x3c);               /* illegal SKW 6502 */
                    pc = (pc + 2) & 0xffff;
                    cycle = cycle + 4;
                  }
                  else {
                    bytebuffer    = readabsx();         /* BIT abs,x 65C02 */
                    vflag = ((bytebuffer & 0x40) == 0) ? 0 : 1;
                    nflag = ((bytebuffer & 0x80) == 0) ? 0 : 1;
                    zflag = ((bytebuffer & areg) == 0) ? 1 : 0;
                    cycle = cycle + 4;
                  }
                  break;
                case 0x3d :                                             // AND abs,x
                        areg    = areg & readabsx();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x3e :                                             // ROL abs,x
                        bytebuffer      = readabsxm();
                        if ( bytebuffer > 0x7f ) {
                          bytebuffer    = ((bytebuffer << 1) | cflag) & 0xff;
                          cflag         = 1;
                        }
                        else {
                          bytebuffer    = (bytebuffer << 1) | cflag;
                          cflag         = 0;
                        }
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle = cycle + 5;
                        break;
                case 0x3f :
                  if (cputype == CPU_65SC02) {
                    if (!(readzp() & 0x08)) {           /* BBR3 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      pc        = (pc + 1) & 0xffff;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    break;
                  }
                case 0x40 :                                             // RTI
                        cpusetflags( cpupullstack() );
                        bflag   = 0;                                    // don't set b flag
                        pc      = cpupullstack() + (cpupullstack() << 8);
                        cycle   = cycle + 6;
                        break;
                case 0x41 :                                             // EOR (zp,x)
                        areg    = areg ^ readindzpx();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 6;
                        break;
                case 0x42 :
                        if (cputype == CPU_6502) {
                          cpuillegal(0x42);                             // illegal 6502
                        }
                        else {                                          // NOP2 65C02
                          incpc;
                          cycle = cycle + 2;
                        }
                        break;
                case 0x43 :
                        if (cputype == CPU_6502) {
                          cpuillegal(0x43);                             // illegal 6502
                        }
                        else {                                          // NOP 65C02
                          cycle = cycle + 2;
                        }
                        break;
                case 0x44 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x44);               /* illegal SKB 6502 */
                    incpc;
                    cycle = cycle + 3;
                  }
                  else {                                /* NOP3 65C02 */
                    pc    = (pc + 1) & 0xffff;
                    cycle = cycle + 3;
                  }
                  break;
                case 0x45 :                                             // EOR zp
                        areg  = areg ^ readzp();
                        cpuflagssetnz(areg);
                        cycle = cycle + 3;
                        break;
                case 0x46 :                                             // LSR zp
                        bytebuffer    = readzpm();
                        cflag         = ((bytebuffer & 1) == 0) ? 0 : 1;
                        bytebuffer    = bytebuffer >> 1;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle         = cycle + 5;
                        break;
                case 0x47 :                                             // RMB4 zp
                        bytebuffer    = readzpm() & 0xef;
                        memorywrite(address, bytebuffer);
                        cycle = cycle + 5;
                        break;
                case 0x48 :                                             // PHA
                        cpupushstack(areg);
                        cycle = cycle + 3;
                        break;
                case 0x49 :                                             // EOR #
                        areg  = areg ^ memoryread(pc);
                        incpc;
                        cpuflagssetnz(areg);
                        cycle = cycle + 2;
                        break;
                case 0x4a :                                             // LSR
                        cflag = ((areg & 1) == 0) ? 0 : 1;
                        areg  = areg >> 1;
                        cpuflagssetnz(areg);
                        cycle = cycle + 2;
                        break;
                case 0x4b :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x4b);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x4c :                                             // JMP abs
                        pc      = memoryread(pc) + ( memoryread( (pc + 1) & 0xffff) << 8 );
                        cycle   = cycle + 3;
                        break;
                case 0x4d :                                             // EOR abs
                        areg    = areg ^ readabs();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x4e :                                             // LSR abs
                        bytebuffer      = readabsm();
                        cflag           = ((bytebuffer & 1) == 0) ? 0 : 1;
                        bytebuffer      = bytebuffer >> 1;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle           = cycle + 6;
                        break;
                case 0x4f :
                  if (cputype == CPU_65SC02) {
                    if (!(readzp() & 0x10)) {           /* BBR4 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      pc        = (pc + 1) & 0xffff;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    break;
                  }
                case 0x50 :                             /* BVC */
                  if ( vflag == 0) {
                    bytebuffer= memoryread(pc);
                    if ( bytebuffer > 0x7f) {
                      pc        = (pc - 255 + bytebuffer) & 0xffff;
                    }
                    else {
                      pc        = (pc + 1 + bytebuffer) & 0xffff;
                    }
                    cycle       = cycle + 3;
                  }
                  else {
                    pc          = (pc + 1) & 0xffff;
                    cycle       = cycle + 2;
                  }
                  break;
                case 0x51 :                                             // EOR (zp),y
                        areg    = areg ^ readindzpy();
                        cpuflagssetnz(areg);
                        cycle = cycle + 5;
                        break;
                case 0x52 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x52);                   /* illegal 6502 */
                  }
                  else {                                /* EOR (zp) 65C02 */
                    areg  = areg ^ readindzp();
                    cpuflagssetnz(areg);
                    cycle = cycle + 5;
                  }
                  break;
                case 0x53 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x53);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x54 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x54);               /* illegal SKB 6502 */
                    incpc;
                    cycle = cycle + 4;
                  }
                  else {                                /* NOP4 65C02 */
                    incpc;
                    cycle = cycle + 4;
                  }
                  break;
                case 0x55 :                                             // EOR zp,x
                        areg  = areg ^ readzpx();
                        cpuflagssetnz(areg);
                        cycle = cycle + 4;
                        break;
                case 0x56 :                                             // LSR zp,x
                        bytebuffer      = readzpxm();
                        cflag           = ((bytebuffer & 1) == 0) ? 0 : 1;
                        bytebuffer      = bytebuffer >> 1;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle           = cycle + 6;
                        break;
                case 0x57 :                                             // RMB5 zp
                        bytebuffer      = readzpm() & 0xdf;
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 5;
                        break;
                case 0x58 :                                             // CLI
                        iflag   = 0;
                        cycle   = cycle + 2;
                        break;
                case 0x59 :                                             // EOR abs,y
                        areg    = areg ^ readabsy();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x5a :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x5a);               /* illegal NOP 6502 */
                    cycle = cycle + 2;
                  }
                  else {                                /* PHY 65C02 */
                    cpupushstack(yreg);
                    cycle = cycle + 3;
                  }
                  break;
                case 0x5b :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x5b);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x5c :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x5c);               /* illegal SKW 6502 */
                    pc = (pc + 2) & 0xffff;
                    cycle = cycle + 4;
                  }
                  else {                                /* NOP8 65C02 */
                    pc    = (pc + 2) & 0xffff;
                    cycle = cycle + 8;
                  }
                  break;
                case 0x5d :                                             // EOR abs,x
                        areg    = areg ^ readabsx();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x5e :                             /* LSR abs,x */
                  bytebuffer    = readabsxm();
                  cflag         = ((bytebuffer & 1) == 0) ? 0 : 1;
                  bytebuffer    = bytebuffer >> 1;
                  cpuflagssetnz(bytebuffer);
                  memorywrite(address, bytebuffer);
                  cycle         = cycle + 7;
                  break;
                case 0x5f :
                  if (cputype == CPU_65SC02) {
                    if (!(readzp() & 0x20)) {           /* BBR5 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      pc        = (pc + 1) & 0xffff;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    break;
                  }
                case 0x60 :                                             // RTS
                        pc      = ((cpupullstack() + (cpupullstack() << 8)) + 1) & 0xffff;
                        cycle   = cycle + 6;
                        break;
                case 0x61 :                                             // ADC (zp,x)
                        adc(readindzpx());
                        cycle = cycle + 6;
                        break;
                case 0x62 :                                             // NOP2 65C02
                        pc      = (pc + 1) & 0xffff;
                        cycle   = cycle + 2;
                        break;
                case 0x63 :                                             // NOP 65C02
                        cycle   = cycle + 2;
                        break;
                case 0x64 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0x64);                         // illegal SKB 6502
                          incpc;
                          cycle = cycle + 3;
                        }
                        else {                                          // STZ zp 65C02
                          writezp(0);
                          cycle = cycle + 3;
                        }
                        break;
                case 0x65 :                                             // ADC zp
                        adc(readzp());
                        cycle   = cycle + 3;
                        break;
                case 0x66 :                                             // ROR zp
                  bytebuffer    = readzpm();
                  if (cflag) {
                    cflag       = (bytebuffer & 1) ? 1 : 0;
                    bytebuffer  = (bytebuffer >> 1) | 0x80;
                  }
                  else {
                    cflag       = (bytebuffer & 1) ? 1 : 0;
                    bytebuffer  = bytebuffer >> 1;
                  }
                  cpuflagssetnz(bytebuffer);
                  memorywrite(address, bytebuffer);
                  cycle = cycle + 5;
                  break;
                case 0x67 :                                             // RMB6 zp
                        bytebuffer = readzpm() & 0xbf;
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 5;
                        break;
                case 0x68 :                                             // PLA
                        areg  = cpupullstack();
                        cpuflagssetnz(areg);
                        cycle = cycle + 4;
                        break;
                case 0x69 :                                             // ADC #
                        adc(memoryread(pc));
                        incpc;
                        cycle = cycle + 2;
                        break;
                case 0x6a :                             /* ROR */
                  if (cflag) {
                    cflag       = (areg & 1) ? 1 : 0;
                    areg        = (areg >> 1) | 0x80;
                  }
                  else {
                    cflag       = (areg & 1) ? 1 : 0;
                    areg        = areg >> 1;
                  }
                  cpuflagssetnz(areg);
                  cycle = cycle + 2;
                  break;
                case 0x6b :                                             // NOP 65C02
                        cycle   = cycle + 2;
                        break;
                case 0x6c :                             /* JMP (abs) */
                  address       = memoryread(pc) + ( memoryread((pc + 1) & 0xffff) << 8 );
                  if (cputype == CPU_6502) {            /* bug in 6502! */
                    pc          = memoryread(address)
                                + ( memoryread(((address + 1) & 0xff) | (address & 0xff00)) << 8 );
                  }
                  else {
                    pc          = memoryread(address)
                                + ( memoryread((address + 1) & 0xffff) << 8 );
                  }
                  cycle = cycle + 5;
                  break;
                case 0x6d :                                             // ADC abs
                        adc(readabs());
                        cycle   = cycle + 4;
                        break;
                case 0x6e :                                             // ROR abs
                        bytebuffer      = readabsm();
                        if (cflag) {
                          cflag         = (bytebuffer & 1) ? 1 : 0;
                          bytebuffer    = (bytebuffer >> 1) | 0x80;
                        }
                        else {
                          cflag         = (bytebuffer & 1) ? 1 : 0;
                          bytebuffer    = bytebuffer >> 1;
                        }
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 6;
                        break;
                case 0x6f :
                  if (cputype == CPU_65SC02) {
                    if (!(readzp() & 0x40)) {           /* BBR6 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      incpc;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    cycle = cycle + 3;
                    break;
                  }
                case 0x70 :                             /* BVS */
                  if ( vflag != 0) {
                    bytebuffer= memoryread(pc);
                    if ( bytebuffer > 0x7f) {
                      pc        = (pc - 255 + bytebuffer) & 0xffff;
                    }
                    else {
                      pc        = (pc + 1 + bytebuffer) & 0xffff;
                    }
                    cycle       = cycle + 3;
                  }
                  else {
                    incpc;
                    cycle       = cycle + 2;
                  }
                  break;
                case 0x71 :                                             // ADC (zp),y
                        adc(readindzpy());
                        cycle   = cycle + 5;
                        break;
                case 0x72 :                                             // ADC (zp)
                         adc(readindzp());
                        cycle = cycle + 5;
                        break;
                case 0x73 :                                             // NOP 65C02
                        cycle = cycle + 2;
                        break;
                case 0x74 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x74);               /* illegal SKB 6502 */
                    incpc;
                    cycle = cycle + 4;
                  }
                  else {                                /* STZ zp,x 65C02 */
                    writezpx(0);
                    cycle = cycle + 4;
                  }
                  break;
                case 0x75 :                                             // ADC zp,x
                        adc(readzpx());
                        cycle   = cycle + 4;
                        break;
                case 0x76 :                             /* ROR zp,x */
                  bytebuffer    = readzpxm();
                  if (cflag) {
                    cflag       = (bytebuffer & 1) ? 1 : 0;
                    bytebuffer  = (bytebuffer >> 1) | 0x80;
                  }
                  else {
                    cflag       = (bytebuffer & 1) ? 1 : 0;
                    bytebuffer  = bytebuffer >> 1;
                  }
                  cpuflagssetnz(bytebuffer);
                  memorywrite(address, bytebuffer);
                  cycle = cycle + 6;
                  break;
                case 0x77 :                                             // RMB7 zp
                        bytebuffer    = readzpm() & 0x7f;
                        memorywrite(address, bytebuffer);
                        cycle = cycle + 5;
                        break;
                case 0x78 :                                             // SEI
                        iflag   = 1;
                        cycle   = cycle + 2;
                        break;
                case 0x79 :                                             // ADC abs,y
                        adc(readabsy());
                        cycle   = cycle + 4;
                        break;
                case 0x7a :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0x7a);                         // illegal NOP 6502
                          cycle = cycle + 2;
                        }
                        else {                                          // PLY
                          yreg  = cpupullstack();
                          cpuflagssetnz(yreg);
                          cycle = cycle + 4;
                        }
                        break;
                case 0x7b :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x7b);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x7c :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x7c);               /* illegal SKW 6502 */
                    pc = (pc + 2) & 0xffff;
                    cycle = cycle + 4;
                  }
                  else {
                    address     = (memoryread(pc) + ( memoryread((pc + 1) & 0xffff) << 8 ) /* JMP (abs,x) 65C02 */
                                 + xreg) & 0xffff;
                    pc          = memoryread(address)
                                 + ( memoryread((address + 1) & 0xffff) << 8 );
                    cycle = cycle + 5;
                  }
                  break;
                case 0x7d :                                             // ADC abs,x
                        adc(readabsx());
                        cycle = cycle + 4;
                        break;
                case 0x7e :                             /* ROR abs,x */
                  bytebuffer    = readabsxm();
                  if (cflag) {
                    cflag       = (bytebuffer & 1) ? 1 : 0;
                    bytebuffer  = (bytebuffer >> 1) | 0x80;
                  }
                  else {
                    cflag       = (bytebuffer & 1) ? 1 : 0;
                    bytebuffer  = bytebuffer >> 1;
                  }
                  cpuflagssetnz(bytebuffer);
                  memorywrite(address, bytebuffer);
                  cycle = cycle + 7;
                  break;
                case 0x7f :
                  if (cputype == CPU_65SC02) {
                    if (!(readzp() & 0x80)) {           /* BBR7 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      pc        = (pc + 1) & 0xffff;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    cycle = cycle + 3;
                    break;
                  }
                case 0x80 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0x80);                         // illegal SKB 6502
                          incpc;
                          cycle = cycle + 2;
                        }
                        else {                                          // BRA 65C02/65SC02
                          bytebuffer= memoryread(pc);
                          if ( bytebuffer > 0x7f) {
                            pc  = (pc - 255 + bytebuffer) & 0xffff;
                          }
                          else {
                            pc  = (pc + 1 + bytebuffer) & 0xffff;
                          }
                          cycle = cycle + 3;
                        }
                        break;
                case 0x81 :                                             // STA (zp,x)
                        writeindzpx(areg);
                        cycle   = cycle + 6;
                        break;
                case 0x82 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x82);               /* illegal SKB 6502 */
                    incpc;
                    cycle = cycle + 2;
                  }
                  else {                                                // NOP2 65C02
                    incpc;
                    cycle = cycle + 2;
                  }
                  break;
                case 0x83 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x83);               /* illegal 6502 AXS (SAX) (zp,x) */
                    writeindzpx(areg & xreg);
                    cycle = cycle + 6;
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x84 :                                             // STY zp
                        writezp(yreg);
                        cycle   = cycle + 3;
                        break;
                case 0x85 :                                             // STA zp
                        writezp(areg);
                        cycle   = cycle + 3;
                        break;
                case 0x86 :                                             // STX zp
                        writezp(xreg);
                        cycle   = cycle + 3;
                        break;
                case 0x87 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0x87);         // illegal AXS (SAX) zp 6502
                          writezp(areg & xreg);
                          cycle = cycle + 3;
                        }
                        else {
                          if (cputype == CPU_65C02) {
                           incpc;                       // NOP2 65C02
                           cycle        = cycle + 3;    // correct?
                          }
                          else {                        // SMB0 zp 65SC02
                            bytebuffer  = readzpm() | 0x01;
                            memorywrite(address, bytebuffer);
                            cycle       = cycle + 5;
                          }
                        }
                        break;
                case 0x88 :                                             // DEY
                        yreg    = (yreg - 1) & 0xff;
                        cpuflagssetnz(yreg);
                        cycle   = cycle + 2;
                        break;
                case 0x89 :
                        if (cputype == CPU_6502) {
                          cpuillegal(0x89);                             // illegal 6502
                        }
                        else {                                          // BIT #
                          bytebuffer    = memoryread(pc);
                          incpc;
                          vflag = ((bytebuffer & 0x40) == 0) ? 0 : 1;
                          nflag = ((bytebuffer & 0x80) == 0) ? 0 : 1;
                          zflag = ((bytebuffer & areg) == 0) ? 1 : 0;
                          cycle = cycle + 2;
                        }
                        break;
                case 0x8a :                                             // TXA
                        areg    = xreg;
                        cpuflagssetnz(areg);
                        cycle   = cycle + 2;
                        break;
                case 0x8b :
                        if (cputype == CPU_6502) {
                          cpuillegal(0x8b);                             // illegal 6502
                        }
                        else {                                          // NOP 65C02
                          cycle = cycle + 2;
                        }
                        break;
                case 0x8c :                                             // STY abs
                        writeabs(yreg);
                        cycle = cycle + 4;
                        break;
                case 0x8d :                                             // STA abs
                        writeabs(areg);
                        cycle = cycle + 4;
                        break;
                 case 0x8e :                                            // STX abs
                        writeabs(xreg);
                        cycle = cycle + 4;
                        break;
                case 0x8f :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0x8f);               /* illegal AXS (SAX) abs 6502 */
                    writeabs(areg & xreg);
                    cycle = cycle + 4;
                  }
                  else {
                    if (cputype == CPU_65C02) {
                      pc  = (pc + 2) & 0xffff;          /* NOP3 65C02 */
                      cycle = cycle + 4;                /* correct? */
                    }
                    else {
                      if (readzp() & 0x01) {            /* BBS0 65SC02 */
                        bytebuffer= memoryread(pc);
                        if ( bytebuffer > 0x7f) {
                          pc      = (pc - 255 + bytebuffer) & 0xffff;
                        }
                        else {
                          pc      = (pc + 1 + bytebuffer) & 0xffff;
                        }
                        cycle     = cycle + 5;
                      }
                      else {
                        incpc;
                        cycle     = cycle + 3;
                      }
                    }
                  }
                  break;
                case 0x90 :                             /* BCC */
                  if ( cflag == 0) {
                    bytebuffer= memoryread(pc);
                    if ( bytebuffer > 0x7f) {
                      pc        = (pc - 255 + bytebuffer) & 0xffff;
                    }
                    else {
                      pc        = (pc + 1 + bytebuffer) & 0xffff;
                    }
                    cycle       = cycle + 3;
                  }
                  else {
                   pc           = (pc + 1) & 0xffff;
                   cycle        = cycle + 2;
                  }
                  break;
                case 0x91 :                                             // STA (zp),y
                        writeindzpy(areg);
                        cycle = cycle + 6;
                        break;
                case 0x92 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x92);                   /* illegal 6502 */
                  }
                  else {                                                // STA (zp)
                    writeindzp(areg);
                    cycle = cycle + 6;
                  }
                  break;
                case 0x93 :
                        if (cputype == CPU_6502) {
                          cpuillegal(0x93);                             // illegal 6502
                        }
                        else {                                          // NOP 65C02
                          cycle = cycle + 2;
                        }
                        break;
                case 0x94 :                                             // STY zp,x
                        writezpx(yreg);
                        cycle   = cycle + 4;
                        break;
                case 0x95 :                                             // STA zp,x
                        writezpx(areg);
                        cycle   = cycle + 4;
                        break;
                case 0x96 :                                             // STX zp,y
                        writezpy(xreg);
                        cycle   = cycle + 4;
                        break;
                case 0x97 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0x97);                         // illegal AXS (SAX) zp,y 6502
                          writezpy(areg & xreg);
                          cycle = cycle + 4;
                        }
                        else {
                          if (cputype == CPU_65C02) {
                            incpc;                                      // NOP2 65C02
                            cycle = cycle + 4;                          // correct?
                          }
                          else {                                        // SMB1 zp 65SC02
                            bytebuffer    = readzpm() | 0x02;
                            memorywrite(address, bytebuffer);
                            cycle = cycle + 5;
                          }
                        }
                        break;
                case 0x98 :                                             // TYA
                        areg  = yreg;
                        cpuflagssetnz(areg);
                        cycle = cycle + 2;
                        break;
                case 0x99 :                                             // STA abs,y
                        writeabsy(areg);
                        cycle = cycle + 5;
                        break;
                case 0x9a :                                             // TXS
                        stack = xreg;
                        cycle = cycle + 2;
                        break;
                case 0x9b :
                  if (cputype == CPU_6502) {
                    cpuillegal(0x9b);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0x9c :
                        if (cputype == CPU_6502) {
                          cpuillegal(0x9c);                             // illegal 6502
                        }
                        else {                                          // STZ abs 65C02
                          writeabs(0);
                          cycle = cycle + 4;
                        }
                        break;
                case 0x9d :                                             // STA abs,x
                        writeabsx(areg);
                        cycle   = cycle + 5;
                        break;
                case 0x9e :                                             // STZ abs,x
                        writeabsx(0);
                        cycle   = cycle + 5;
                        break;
                case 0x9f :
                  if (cputype == CPU_65SC02) {
                    if (readzp() & 0x02) {              /* BBS1 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      pc        = (pc + 1) & 0xffff;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;                            // NOP3 65C02
                    break;
                  }
                case 0xa0 :                                             // LDY #
                        yreg  = memoryread(pc);
                        incpc;
                        cpuflagssetnz(yreg);
                        cycle = cycle + 2;
                        break;
                case 0xa1 :                                             // LDA (zp,x)
                        areg  = readindzpx();
                        cpuflagssetnz(areg);
                        cycle = cycle + 6;
                        break;
                case 0xa2 :                                             // LDX #
                        xreg  = memoryread(pc);
                        incpc;
                        cpuflagssetnz(xreg);
                        cycle = cycle + 2;
                        break;
                case 0xa3 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0xa3);                         // illegal LAX (zp,x) 6502
                          areg = readindzpx();
                          xreg = areg;
                          cpuflagssetnz(areg);
                          cycle = cycle + 6;
                        }
                        else {
                          cycle = cycle + 2;                            // NOP 65C02
                        }
                        break;
                case 0xa4 :                                             // LDY zp
                        yreg  = readzp();
                        cpuflagssetnz(yreg);
                        cycle = cycle + 3;
                        break;
                case 0xa5 :                                             // LDA zp
                        areg  = readzp();
                        cpuflagssetnz(areg);
                        cycle = cycle + 3;
                        break;
                case 0xa6 :                                             // LDX zp
                        xreg  = readzp();
                        cpuflagssetnz(xreg);
                        cycle = cycle + 3;
                        break;
                        case 0xa7 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0xa7);                         // illegal LAX zp 6502
                          areg = readzp();
                          xreg = areg;
                          cpuflagssetnz(areg);
                          cycle = cycle + 3;
                        }
                        else {
                          if (cputype == CPU_65C02) {
                            incpc;                                      // NOP2 65C02
                            cycle = cycle + 3;
                          }
                          else {
                            bytebuffer    = readzpm() | 0x04;           // SMB2 zp 65SC02
                            memorywrite(address, bytebuffer);
                            cycle = cycle + 5;
                          }
                        }
                        break;
                case 0xa8 :                                             // TAY
                        yreg    = areg;
                        cpuflagssetnz(yreg);
                        cycle   = cycle + 2;
                        break;
                case 0xa9 :                                             // LDA #
                        areg    = memoryread(pc);
                        incpc;
                        cpuflagssetnz(areg);
                        cycle   = cycle + 2;
                        break;
                case 0xaa :                                             // TAX
                        xreg    = areg;
                        cpuflagssetnz(xreg);
                        cycle   = cycle + 2;
                        break;
                case 0xab :
                  if (cputype == CPU_6502) {
                    cpuillegal(0xab);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0xac :                                             // LDY abs
                        yreg    = readabs();
                        cpuflagssetnz(yreg);
                        cycle   = cycle + 4;
                        break;
                case 0xad :                                             // LDA abs
                        areg    = readabs();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0xae :                                             // LDX abs
                        xreg    = readabs();
                        cpuflagssetnz(xreg);
                        cycle   = cycle + 4;
                        break;
                case 0xaf :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0xaf);               /* illegal LAX abs 6502 */
                    areg = readabs();
                    xreg = areg;
                    cpuflagssetnz(areg);
                    cycle = cycle + 4;
                  }
                  else {
                    if (cputype == CPU_65C02) {
                      pc = (pc + 2) & 0xffff;           /* NOP3 65C02 */
                      cycle = cycle + 4;                /* correct? */
                    }
                    else {
                      if (readzp() & 0x04) {            /* BBS2 65SC02 */
                        bytebuffer= memoryread(pc);
                         if ( bytebuffer > 0x7f) {
                          pc    = (pc - 255 + bytebuffer) & 0xffff;
                        }
                        else {
                          pc    = (pc + 1 + bytebuffer) & 0xffff;
                        }
                        cycle   = cycle + 5;
                      }
                      else {
                        pc      = (pc + 1) & 0xffff;
                        cycle   = cycle + 3;
                      }
                    }
                  }
                  break;
                case 0xb0 :                                             // BCS
                        if ( cflag != 0) {
                          bytebuffer= memoryread(pc);
                          if ( bytebuffer > 0x7f) {
                            pc  = (pc - 255 + bytebuffer) & 0xffff;
                          }
                          else {
                            pc  = (pc + 1 + bytebuffer) & 0xffff;
                          }
                          cycle         = cycle + 3;
                        }
                        else {
                          pc            = (pc + 1) & 0xffff;
                          cycle         = cycle + 2;
                        }
                        break;
                case 0xb1 :                                             // LDA (zp),y
                        areg    = readindzpy();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 5;
                        break;
                case 0xb2 :
                        if (cputype == CPU_6502) {
                          cpuillegal(0xb2);                             // illegal 6502
                        }
                        else {                                          // LDA (zp) 65C02
                          areg  = readindzp();
                          cpuflagssetnz(areg);
                          cycle = cycle + 5;
                        }
                        break;
                case 0xb3 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0xb3);                         // illegal LAX (zp),y 6502
                          areg  = readindzpy();
                          xreg  = areg;
                          cpuflagssetnz(areg);
                          cycle = cycle + 5;
                        }
                        else {
                          cycle = cycle + 2;                            // NOP 65C02
                        }
                        break;
                case 0xb4 :                                             // LDY zp,x
                        yreg    = readzpx();
                        cpuflagssetnz(yreg);
                        cycle   = cycle + 4;
                        break;
                case 0xb5 :                                             // LDA zp,x
                        areg    = readzpx();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0xb6 :                                             // LDX zp,y
                        xreg    = readzpy();
                        cpuflagssetnz(xreg);
                        cycle   = cycle + 4;
                        break;
                case 0xb7 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0xb7);                         // illegal LAX zp,y 6502
                          areg  = readzpy();
                          xreg  = areg;
                          cpuflagssetnz(areg);
                          cycle = cycle + 4;
                        }
                        else {
                          if (cputype == CPU_65C02) {
                            incpc;                                      // NOP2 65C02
                            cycle = cycle + 4;                          // correct?
                          }
                          else {
                            bytebuffer = readzpm() | 0x08;              // SMB3 zp 65SC02
                            memorywrite(address, bytebuffer);
                            cycle = cycle + 5;
                          }
                        }
                        break;
                case 0xb8 :                                             // CLV
                        vflag   = 0;
                        cycle   = cycle + 2;
                        break;
                case 0xb9 :                                             // LDA abs,y
                        areg    = readabsy();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                 case 0xba :                                            // TSX
                        xreg    = stack;
                        cpuflagssetnz(xreg);
                        cycle   = cycle + 2;
                        break;
                case 0xbb :
                        if (cputype == CPU_6502) {
                          cpuillegal(0xbb);                             // illegal 6502
                        }
                        else {                                          // NOP 65C02
                          cycle = cycle + 2;
                        }
                        break;
                case 0xbc :                                             // LDY abs,x
                        yreg    = readabsx();
                        cpuflagssetnz(yreg);
                        cycle   = cycle + 4;
                        break;
                case 0xbd :                                             // LDA abs,x
                        areg    = readabsx();
                        cpuflagssetnz(areg);
                        cycle   = cycle + 4;
                        break;
                case 0xbe :                                             // LDX abs,y
                        xreg    = readabsy();
                        cpuflagssetnz(xreg);
                        cycle   = cycle + 4;
                        break;
                case 0xbf :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0xbf);                         // illegal LAX abs,y 6502
                          areg  = readabsy();
                          xreg  = areg;
                          cpuflagssetnz(areg);
                          cycle = cycle + 4;
                        }
                        else {
                          if (cputype == CPU_65C02) {
                            pc  = (pc + 2) & 0xffff;                    // NOP3 65C02
                            cycle = cycle + 4;
                          }
                          else {
                            if (readzp() & 0x08) {                      // BBS3 65SC02
                              bytebuffer= memoryread(pc);
                              if ( bytebuffer > 0x7f) {
                                pc      = (pc - 255 + bytebuffer) & 0xffff;
                              }
                              else {
                                pc      = (pc + 1 + bytebuffer) & 0xffff;
                              }
                              cycle     = cycle + 5;
                            }
                            else {
                              incpc;
                              cycle     = cycle + 3;
                            }
                          }
                        }
                        break;
                case 0xc0 :                                             // CPY #
                        cpy(memoryread(pc));
                        incpc;
                        cycle   = cycle + 2;
                        break;
                case 0xc1 :                                             // CMP (zp,x)
                        cmp(readindzpx());
                        cycle   = cycle + 6;
                        break;
                case 0xc2 :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0xc2);                         // illegal SKB 6502
                          incpc;
                          cycle = cycle + 2;
                        }
                        else {                                          // NOP2 65C02
                          incpc;
                          cycle = cycle + 2;
                        }
                        break;
                case 0xc3 :
                        if (cputype == CPU_6502) {
                          cpuillegal(0xc3);                             // illegal 6502
                        }
                        else {                                          // NOP 65C02
                          cycle = cycle + 2;
                        }
                        break;
                case 0xc4 :                                             // CPY zp
                        cpy(readzp());
                        cycle = cycle + 3;
                        break;
                case 0xc5 :                                             // CMP zp
                        cmp(readzp());
                        cycle = cycle + 3;
                        break;
                case 0xc6 :                                             // DEC zp
                        bytebuffer = (readzpm() - 1) & 0xff;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle = cycle + 5;
                        break;
                case 0xc7 :                                             // SMB4 zp
                        bytebuffer    = readzpm() | 0x10;
                        memorywrite(address, bytebuffer);
                        cycle = cycle + 5;
                        break;
                case 0xc8 :                                             // INY
                        yreg  = (yreg + 1) & 0xff;
                        cpuflagssetnz(yreg);
                        cycle = cycle + 2;
                        break;
                case 0xc9 :                                             // CMP #
                        cmp(memoryread(pc));
                        incpc;
                        cycle = cycle + 2;
                        break;
                case 0xca :                                             // DEX
                        xreg  = (xreg - 1) & 0xff;
                        cpuflagssetnz(xreg);
                        cycle = cycle + 2;
                        break;
                case 0xcb :
                        if (cputype == CPU_6502) {
                          cpuillegal(0xcb);                             // illegal 6502
                        }
                        else {                                          // NOP 65C02
                          cycle = cycle + 2;
                        }
                        break;
                case 0xcc :                                             // CPY abs
                        cpy(readabs());
                        cycle = cycle + 4;
                        break;
                case 0xcd :                                             // CMP abs
                        cmp(readabs());
                        cycle = cycle + 4;
                        break;
                case 0xce :                                             // DEC abs
                        bytebuffer    = (readabsm() - 1) & 0xff;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle = cycle + 6;
                        break;
                case 0xcf :
                  if (cputype == CPU_65SC02) {
                    if (readzp() & 0x10) {              /* BBS4 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      pc        = (pc + 1) & 0xffff;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    break;
                  }
                case 0xd0 :                                             // BNE
                        if ( zflag == 0) {
                          bytebuffer= memoryread(pc);
                          if ( bytebuffer > 0x7f) {
                            pc  = (pc - 255 + bytebuffer) & 0xffff;
                          }
                          else {
                            pc  = (pc + 1 + bytebuffer) & 0xffff;
                          }
                          cycle = cycle + 3;
                        }
                        else {
                          pc    = (pc + 1) & 0xffff;
                          cycle = cycle + 2;
                        }
                        break;
                case 0xd1 :                                             // CMP (zp),y
                        cmp(readindzpy());
                        cycle = cycle + 5;
                        break;
                case 0xd2 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0xd2);                   /* illegal 6502 */
                  }
                  else {                                /* CMP (zp) */
                    cmp(readindzp());
                    cycle = cycle + 5;
                  }
                  break;
                case 0xd3 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0xd3);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0xd4 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0xd4);               /* illegal SKB 6502 */
                    incpc;
                    cycle = cycle + 4;
                  }
                  else {                                /* NOP4 65C02 */
                    incpc;
                    cycle = cycle + 4;
                  }
                  break;
                case 0xd5 :                                             // CMP zp,x
                        cmp(readzpx());
                        cycle = cycle + 4;
                        break;
                case 0xd6 :                                             // DEC zp,x
                        bytebuffer = (readzpxm() - 1) & 0xff;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 6;
                        break;
                case 0xd7 :                                             // SMB5 zp
                        bytebuffer = readzpm() | 0x20;
                        memorywrite(address, bytebuffer);
                        cycle = cycle + 5;
                        break;
                case 0xd8 :                                             // CLD
                        dflag = 0;
                        cycle = cycle + 2;
                        break;
                case 0xd9 :                                             // CMP abs,y
                        cmp(readabsy());
                        cycle = cycle + 4;
                        break;
                case 0xda :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0xda);               /* illegal NOP 6502 */
                    cycle = cycle + 2;
                  }
                  else {                                /* PHX 65C02 */
                    cpupushstack(xreg);
                    cycle = cycle + 3;
                  }
                  break;
                case 0xdb :
                  if (cputype == CPU_6502) {
                    cpuillegal(0xdb);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0xdc :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0xdc);         /* illegal SKW 6502 */
                          pc = (pc + 2) & 0xffff;
                          cycle = cycle + 4;
                        }
                        else {                          /* NOP4 65C02 */
                          pc    = (pc + 2) & 0xffff;
                          cycle = cycle + 4;
                        }
                        break;
                case 0xdd :                                             // CMP abs,x
                        cmp(readabsx());
                        cycle = cycle + 4;
                        break;
                case 0xde :                                             // DEC abs,x
                        bytebuffer    = (readabsxm() - 1) & 0xff;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle         = cycle + 7;
                        break;
                case 0xdf :
                  if (cputype == CPU_65SC02) {
                    if (readzp() & 0x20) {              /* BBS5 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      pc        = (pc + 1) & 0xffff;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    break;
                  }
                case 0xe0 :                                             // CPX #
                        cpx(memoryread(pc));
                        incpc;
                        cycle   = cycle + 2;
                        break;
                case 0xe1 :                                             // SBC (zp,x)
                        sbc(readindzpx());
                        cycle   = cycle + 5;
                        break;
                case 0xe2 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0xe2);               /* illegal SKB 6502 */
                    incpc;
                    cycle = cycle + 2;
                  }
                  else {                                /* NOP2 65C02 */
                    incpc;
                    cycle = cycle + 2;
                  }
                  break;
                case 0xe3 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0xe3);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0xe4 :                                             // CPX zp
                        cpx(readzp());
                        cycle   = cycle + 3;
                        break;
                case 0xe5 :                                             // SBC zp
                        sbc(readzp());
                        cycle   = cycle + 3;
                        break;
                case 0xe6 :                                             // INC zp
                        bytebuffer      = (readzpm() + 1) & 0xff;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 5;
                        break;
                case 0xe7 :                                             // SMB6 zp
                        bytebuffer      = readzpm() | 0x40;
                        memorywrite(address, bytebuffer);
                        cycle = cycle + 5;
                        break;
                case 0xe8 :                                             // INX
                        xreg    = (xreg + 1) & 0xff;
                        cpuflagssetnz(xreg);
                        cycle   = cycle + 2;
                        break;
                case 0xe9 :                                             // SBC #
                        sbc(memoryread(pc));
                        pc      = (pc + 1) & 0xffff;
                        cycle   = cycle + 2;
                        break;
                case 0xea :                                             // NOP
                        cycle   = cycle + 2;
                        break;
                case 0xeb :
                  if (cputype == CPU_6502) {
                    cpuillegal(0xeb);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0xec :                                             // CPX abs
                        cpx(readabs());
                        cycle   = cycle + 4;
                        break;
                case 0xed :                                             // SBC abs
                        sbc(readabs());
                        cycle   = cycle + 4;
                        break;
                case 0xee :                                             // INC abs
                        bytebuffer = (readabsm() + 1) & 0xff;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 6;
                        break;
                case 0xef :
                  if (cputype == CPU_65SC02) {
                    if (readzp() & 0x40) {              /* BBS6 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      incpc;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    break;
                  }
                case 0xf0 :                             /* BEQ */
                  if ( zflag != 0 ) {
                    bytebuffer= memoryread(pc);
                    if ( bytebuffer > 0x7f ) {
                      pc        = (pc - 255 + bytebuffer) & 0xffff;
                    }
                    else {
                      pc        = (pc + 1 + bytebuffer) & 0xffff;
                    }
                    cycle       = cycle + 3;
                  }
                  else {
                    incpc;
                    cycle       = cycle + 2;
                  }
                  break;
                case 0xf1 :                                             // SBC (zp),y
                        sbc(readindzpy());
                        cycle   = cycle + 5;
                        break;
                case 0xf2 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0xf2);                   /* illegal KIL 6502 */
                  }
                  else {                                /* SBC (zp) 65C02 */
                    sbc(readindzp());
                    cycle = cycle + 5;
                  }
                  break;
                case 0xf3 :
                  if (cputype == CPU_6502) {
                    cpuillegal(0xf3);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0xf4 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0xf4);               /* illegal SKB 6502 */
                    incpc;
                    cycle = cycle + 4;
                  }
                  else {                                /* NOP4 65C02 */
                    incpc;
                    cycle = cycle + 4;
                  }
                  break;
                case 0xf5 :                                             // SBC zp,x
                        sbc(readzpx());
                        cycle = cycle + 4;
                        break;
                case 0xf6 :                                             // INC zp,x
                        bytebuffer = (readzpxm() + 1) & 0xff;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 6;
                        break;
                case 0xf7 :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0xf7);               /* illegal INS zp,x 6502 */
                    bytebuffer  = (readzpxm() + 1) & 0xff;
                    memorywrite(address, bytebuffer);
                    sbc(bytebuffer);
                    cycle       = cycle + 6;
                  }
                  else {
                    if (cputype == CPU_65C02) {
                      incpc;                            /* NOP2 65C02 correct? */
                      cycle     = cycle + 6;
                    }
                    else {
                      bytebuffer    = readzpm() | 0x80; /* SMB7 zp 65SC02 */
                      memorywrite(address, bytebuffer);
                      cycle = cycle + 5;
                    }
                  }
                  break;
                case 0xf8 :                                             // SED
                        dflag   = 1;
                        cycle   = cycle + 2;
                        break;
                case 0xf9 :                                             // SBC abs,y
                        sbc(readabsy());
                        cycle = cycle + 4;
                        break;
                case 0xfa :
                        if (cputype == CPU_6502) {
                          cpushowillegal(0xfa);                         // illegal NOP 6502
                          cycle = cycle + 2;
                        }
                        else {                                          // PLX 65C02
                          xreg  = cpupullstack();
                          cpuflagssetnz(xreg);
                          cycle = cycle + 4;
                        }
                        break;
                case 0xfb :
                  if (cputype == CPU_6502) {
                    cpuillegal(0xfb);                   /* illegal 6502 */
                  }
                  else {                                /* NOP 65C02 */
                    cycle = cycle + 2;
                  }
                  break;
                case 0xfc :
                  if (cputype == CPU_6502) {
                    cpushowillegal(0xfc);               /* illegal SKW 6502 */
                    pc = (pc + 2) & 0xffff;
                    cycle = cycle + 4;
                  }
                  else {                                /* NOP4 65C02 */
                    pc = (pc + 2) & 0xffff;
                    cycle = cycle + 4;
                  }
                  break;
                case 0xfd :                                             // SBC abs,x
                        sbc(readabsx());
                        cycle = cycle + 4;
                        break;
                case 0xfe :                                             // INC abs,x
                        bytebuffer = (readabsxm() + 1) & 0xff;
                        cpuflagssetnz(bytebuffer);
                        memorywrite(address, bytebuffer);
                        cycle   = cycle + 7;
                        break;
                case 0xff :
                  if (cputype == CPU_65SC02) {
                    if (readzp() & 0x80) {              /* BBS7 65SC02 */
                      bytebuffer= memoryread(pc);
                      if ( bytebuffer > 0x7f) {
                        pc      = (pc - 255 + bytebuffer) & 0xffff;
                      }
                      else {
                        pc      = (pc + 1 + bytebuffer) & 0xffff;
                      }
                      cycle     = cycle + 5;
                    }
                    else {
                      pc        = (pc + 1) & 0xffff;
                      cycle     = cycle + 3;
                    }
                    break;
                  }
                  else {
                    pc  = (pc + 2) & 0xffff;            /* NOP3 65C02 */
                    cycle = cycle + 3;
                    break;
                  }
              } // switch

          {
            register unsigned int delay;
            for (delay=cpudelay; delay; delay--);
          }
        } // do
        while (traceflag != 1);
        stateflags = stateflags | CPU_STATETRACE;

        setmessage("CPU stopped because of trace mode");
        cpulight(0);
        return stateflags;

// --> ifndef CPU_CYCLEEXACT
#endif
    } // cpuline


// --> ifndef CPU_ASM
#endif

/*--------------------------------------*/


#ifdef DEF_INC_ASMLIB_H
#ifdef DEF_INC_DAPPLE_H

     void cpumenu(void) {
        unsigned int  menukeyboard;
        unsigned int  menuwindow;
        unsigned char key;
        unsigned int  update;
        unsigned char waitstates[32];
        unsigned char filepath[260];

        if (cpuregswindow) {
          channelout(cpuregswindow, 12);
#ifdef EMUZ80
          if (!cpuflag) {
#endif
            cpuwriteregs(cpuregswindow);
#ifdef EMUZ80
          }
          else {
            z80writeregs(cpuregswindow);
          }
#endif
        }
        if (!windowaddio( -1, -1, WINDOWXSIZE, WINDOWYSIZE, -1, 1,
"!\
ECPU Options;\
GCPU Optionen;\
", 0, &menukeyboard, &menuwindow)) {
          update = 1;
          do {
            if (update) {
              channelout(menuwindow, 12);               /* clear window */
              stringwritemessage(menuwindow,
"!\
E\r[ESC] - Quit\r\r;\
G\r[ESC] - Verlasse Men\r\r;\
");

              stringwrite(menuwindow, "\r[1] - CPU 6502:   ");
              channelout(menuwindow, cpugettype()==CPU_6502 ?   '*':' ');
              stringwrite(menuwindow, "\r[2] - CPU 65C02:  ");
              channelout(menuwindow, cpugettype()==CPU_65C02 ?  '*':' ');
              stringwrite(menuwindow, "\r[3] - CPU 65SC02: ");
              channelout(menuwindow, cpugettype()==CPU_65SC02 ? '*':' ');

              stringwritemessage(menuwindow,
"!\
E\rPlease note: Differences between processors have not been\rfully implemented yet.;\
G\rBitte beachten: Unterschiede zwischen Prozessoren wurden noch\rnicht vollstndig implementiert.;\
");
              stringwritemessage(menuwindow,
"!\
E\r\r[M] - 65x02 Mode: ;\r\
G\r\r[M] - 65x02 Modus: ;\r\
;");
              if (cpugetlinecycle() <= 33) {
                stringwrite(menuwindow, "0.5 Mhz");
              }
              else {
                if (cpugetlinecycle() <= 65) {
                  stringwrite(menuwindow, "1 Mhz");
                }
                else {
                  if (cpugetlinecycle() <= 130) {
                    stringwrite(menuwindow, "2 Mhz");
                  }
                  else {
                    stringwrite(menuwindow, "Fast Mode");
                  }
                }
              }
#ifdef EMUZ80
              if (z80active) {
                stringwritemessage(menuwindow,
"!\
E\r\r[Z] - Z80 card in Slot 4;\r\
G\r\r[Z] - Z80-Karte in Slot 4;\r\
;");
              }
              else {
                stringwritemessage(menuwindow,
"!\
E\r\r[Z] - No Z80 card installed;\r\
G\r\r[Z] - Keine Z80-Karte installiert;\r\
;");
              }
#endif
              stringwritemessage(menuwindow,
"!\
E\r\r[A] - Assemble Source File (65c02 Assembler 0.1 Beta);\
G\r\r[A] - Assembliere Quelldatei (65c02 Assembler 0.1 Beta);\
;");

              stringwritemessage(menuwindow,
"!\
E\r[B] - Set Breakpoint (;\
G\r[B] - Setze Breakpoint (;\
;");

              if (cpugetbreakpoint() != 0xffffffff) {
                card32tohexd(cpugetbreakpoint(), waitstates);
                stringwrite(menuwindow, waitstates);
                stringwrite(menuwindow, ")");
              }
              else {
                stringwritemessage(menuwindow,
"!\
ENo breakpoint set);\
GKein Breakpoint gesetzt);\
;");
              }
              if ( cpugettracemode() == 0 ) {
                stringwritemessage(menuwindow,
"!\
E\r[T] - No tracemode activated;\
G\r[T] - Kein Tracemodus aktiviert;\
;");
              }
              else {
                if ( cpugettracemode() == 1) {
                  stringwritemessage(menuwindow,
"!\
E\r[T] - Single step tracemode activated;\
G\r[T] - Einzelschritt Tracemodus aktiviert;\
;");
                }
                else {
                  stringwritemessage(menuwindow,
"!\
E\r[T] - Frame tracemode activated;\
G\r[T] - Frame-Tracemodus aktiviert;\
;");
                }
              }

              stringwritemessage(menuwindow,
"!\
E\r[D] - 65x02 Debugger;\
G\r[D] - 65x02 Debugger;\
;");

              stringwritemessage(menuwindow,
"!\
E\r[I] - Information about BRK and illegal opcodes: ;\
G\r[I] - Informationen ber BRK und illegale Opcodes: ;\
;");

              stringwriteyesno(menuwindow, cpumessageflag);

              stringwrite(menuwindow, "\r[P] - Pause ");
              if (cpugetstate() & CPU_STATEHALT) {
                stringwritemessage(menuwindow,
"!\
E(CPU paused);\
G(CPU angehalten);\
;");
              }
              else {
                stringwritemessage(menuwindow,
"!\
E(CPU running);\
G(CPU luft);\
;");
              }

              stringwritemessage(menuwindow,
"!\
E\r[R] - Reset computer\
\r[C] - Clear memory and reset computer;\
G\r[R] - Reset Computer\
\r[C] - Lsche Speicher und reset Computer;\
;");

              stringwritemessage(menuwindow,
"!\
E\r[W] - Wait states (;\
G\r[W] - Wartezyklen (;\
;");
              card32tohexd(inicpudelay, waitstates);
              stringwrite(menuwindow, waitstates);
              stringwrite(menuwindow, ")");

              update = 0;
              screenupdate = 1;
            }
            do {
              taskswitch();
              if (windowgetclose(menuwindow)) {
                key = 27;
              }
              else {
                key = (unsigned char)channelin(menukeyboard);
              }
            }
            while ((key == 0) && (!exitprogram));
            switch (key) {
              case '1' :
                if (appletype & APPLEIIC) {
                  stringwritemessage(menuwindow,
"!\
E\r\rThe Apple//c needs at least a 65C02 processor.\r;\
G\r\rDer Apple//c bentigt mindestens einen 65C02 Prozessor.\r;\
;");
                  windowpresskey(menukeyboard, menuwindow);
                }
                else {
                  if (appletype & LASER128) {
                    stringwritemessage(menuwindow,
"!\
E\r\rThe Laser 128 needs at least a 65C02 processor.\r;\
G\r\rDer Laser 128 bentigt mindestens einen 65C02 Prozessor.\r;\
;");
                    windowpresskey(menukeyboard, menuwindow);
                  }
                  else {
                    cpusettype(CPU_6502);
                  }
                }
                update = 1;
                break;
              case '2' :
                cpusettype(CPU_65C02);
                update = 1;
                break;
              case '3' :
                cpusettype(CPU_65SC02);
                update = 1;
                break;
              case 'a' :
              case 'A' :
                if (fileselectmenu(
"!\
EAssemble Source File;\
EAssembliere Source Datei;\
",
"!\
ESource File (*.S, *.ASM, *.A65, *.TXT);\
GSource File (*.S, *.ASM, *.A65, *.TXT);\
",
                                   filepath, callingpath, "._s_", 0)) {
                  assembler(memram, sizeof(memram), filepath);  /* assemble source code */
                  virtsetmode();
                  virtcacheinit();
                }
                update = 1;
                break;
              case 'b' :
              case 'B' :
                stringwritemessage(menuwindow,
"!\
E\r\rPlease enter new breakpoint value:\r\
(<ESC> keeps current value,\r values > $ffff will clear the breakpoint)\r>;\
G\r\rBitte geben Sie den neuen Breakpointwert ein:\r\
(<ESC> behlt alten Wert bei,\r Werte > $ffff lschen den Breakpoint)\r>;\
;");
                update = card32readhex(menukeyboard, menuwindow, cpugetbreakpoint());
                if (update != cpugetbreakpoint()) {
                  cpusetbreakpoint(update);
#ifdef EMUZ80
                  z80setbreakpoint(update);
#endif
                }
                update = 1;
                break;
              case 'd' :
              case 'D' :
                inidebugflag = 1;
                ramdebugger(memram, sizeof(memram));
                inidebugflag = 0;
                virtsetmode();
                virtcacheinit();
                update = 1;
                break;
              case 'i' :
              case 'I' :
                cpumessageflag = !cpumessageflag;
                update = 1;
                break;
              case 'm' :
              case 'M' :
                if (cpugetlinecycle() > 130) {
                  cpusetdelay(inicpudelay*2);
                  cpusetlinecycle(65/2);
                }
                else {
                  if (cpugetlinecycle() < 65) {
                    cpusetdelay(inicpudelay);
                    cpusetlinecycle(65);
                  }
                  else {
                    if (cpugetlinecycle() == 65) {
                      cpusetdelay(inicpudelay/2);
                      cpusetlinecycle(65*2);
                    }
                    else {
                      cpusetdelay(0);
                      cpusetlinecycle(65*8);
                    }
                  }
                }
                drivefastmode = 0;
                update = 1;
                break;
              case 'p' :
              case 'P' :
                if (cpugetstate() & CPU_STATEHALT) {
                  cpuclearstate(CPU_STATEHALT);
                }
                else {
                  cpusetstate(CPU_STATEHALT);
                }
                update = 1;
                break;
              case 't' :
              case 'T' :
                update = cpugettracemode() + 1;
                if (update > 2) {
                  update = 0;
                }
                cpusettracemode(update);
                update = 1;
                break;
              case 'w' :
              case 'W' :
                stringwritemessage(menuwindow,
"!\
E\r\rPlease enter new delay value:\r(<ESC> keeps current value)\r>;\
G\r\rBitte geben Sie den neuen Verzgerungswert ein:\r(<ESC> behlt alten Wert bei)\r>;\
;");
                update = card32readhex(menukeyboard, menuwindow, inicpudelay);
                if (inicpudelay != update) {
                  inicpudelay = update;
                  cpusetdelay(inicpudelay);
                  cpusetlinecycle(65);
                  drivefastmode = 0;
                }
                update = 1;
                break;
#ifdef EMUZ80
              case 'z' :
              case 'Z' :
                z80active = !z80active;
                update = 1;
                break;
#endif
              case 'r' :                // taken from memory.c
              case 'R' :
                memoryresetmenu();
                break;
              case 'c' :                // taken from memory.c
              case 'C' :
                 memoryclearresetmenu();
                break;
            } // switch (key)
          }
          while ((key != 27) && (key != 32) && (key != 13) && (!exitprogram));
          channelclose(menukeyboard);
          channelclose(menuwindow);
#ifdef EMUZ80
          if (!z80active) {
            if (cpuflag) {
              applereset();
            }
          }
#endif
          messageflag = 1;
        }
      } // cpumenu


// --> #ifdef DEF_INC_DAPPLE_H
#endif

// --> #ifdef DEF_INC_ASMLIB_H
#endif




// --> #ifndef DEF_INC_CPU65C02.C
#endif
