#include <alloc.h>
#include <mem.h>
#include <time.h>

#pragma intrinsic memcpy

#include "system.h"
#include "ram.h"
#include "video.h"

/**********************************
 *                                *
 * AUXILIARY MEMORY SOFT SWITCHES *
 * Auxiliary Slot 64K 80-Col Card *
 *                                *
 **********************************/

BYTE EightyStore = 0; /* 80Store */

BYTE ReadAuxSlotIO( WORD Address )
{
    BYTE v;

    v = 0xff;

    switch ( Address )
    {
    case 0x13:
        /* RdRAMRd: bit 7 clear = read main 48K, set = read aux 48K */
        v = ( MainRSeg == AuxSeg ) ? 0x8d : 0x0d;
        break;
    case 0x14:
        /* RdRAMWrt: bit 7 clear = write main 48K, set = write aux 48K */
        v = ( MainWSeg == AuxSeg ) ? 0x8d : 0x0d;
        break;
    case 0x16:
        /* RdAltZP: bit 7 clear = main page 0 & 1, set = aux page 0 & 1 */
        v = ( ZPSRSeg == AuxSeg ) ? 0x8d : 0x0d;
        break;
    case 0x18:
        /* Rd80Store */
        v = ( EightyStore ) ? 0x8d : 0x0d;
        break;
    case 0x55:
        /* Store to Page 1X */
        break;
    }
    return v;
}

#pragma argsused
void WriteAuxSlotIO( WORD Address, BYTE Data )
{
    switch ( Address )
    {
    case 0x00:
        /* 80Store Off: Enable RAMRd and RAMWrt */
        EightyStore = 0;
        break;
    case 0x01:
        /* 80Store On: */
        EightyStore = 1;
        break;
    case 0x02:
        /* RAMRd Off: Read main 48K RAM */
        if ( !EightyStore )
        {
            MainRSeg = MainSeg;
        }
        break;
    case 0x03:
        /* RAMRd On: Read aux 48K RAM */
        if ( !EightyStore )
        {
            MainRSeg = AuxSeg;
        }
        break;
    case 0x04:
        /* RAMWrt Off: Write main 48K RAM */
        if ( !EightyStore )
        {
            MainWSeg = MainSeg;
        }
        break;
    case 0x05:
        /* RAMWrt On: Write aux 48K RAM */
        if ( !EightyStore )
        {
            MainWSeg = AuxSeg;
        }
        break;
    case 0x08:
        /* AltZP Off: use main bank page 0 & 1 */
        /* if bank RAM is active, make sure it is also switched */
        if ( BankRSeg  == CurBankSeg  ) BankRSeg  = MainSeg;
        if ( BankWSeg  == CurBankSeg  ) BankWSeg  = MainSeg;
        if ( Bank2RSeg == CurBank2Seg ) Bank2RSeg = MainBankSeg;
        if ( Bank2WSeg == CurBank2Seg ) Bank2WSeg = MainBankSeg;
        /* now set current segments */
        CurBankSeg  = MainSeg;
        CurBank2Seg = MainBankSeg;
//        CurZPSSeg   = MainSeg;
        ZPSRSeg = MainSeg;
        ZPSWSeg = MainSeg;
        break;
    case 0x09:
        /* AltZP On: use aux bank page 0 & 1 */
        /* if bank RAM is active, make sure it is also switched */
        if ( BankRSeg  == CurBankSeg  ) BankRSeg  = AuxSeg;
        if ( BankWSeg  == CurBankSeg  ) BankWSeg  = AuxSeg;
        if ( Bank2RSeg == CurBank2Seg ) Bank2RSeg = AuxBankSeg;
        if ( Bank2WSeg == CurBank2Seg ) Bank2WSeg = AuxBankSeg;
        /* now set current segments */
        CurBankSeg  = AuxSeg;
        CurBank2Seg = AuxBankSeg;
//        CurZPSSeg   = AuxSeg
        ZPSRSeg = AuxSeg;
        ZPSWSeg = AuxSeg;
        break;
    case 0x55:
        /* Store to Page 1X */
        ReadAuxSlotIO( Address );
        break;
    }
}

/******************
 *                *
 * VIDEO HANDLING *
 *                *
 ******************/

/* Hi-Res Colour Lookup Tables */
/* Given an Apple bit pattern and a colour of 0-3, return an EGA mask
   to be used to plot that colour */
unsigned char HiResTable[ 128 ][ 4 ][ 2 ];

/* Setup the Hi-Res colour lookup tables */
void InitHiresLookupTables( void )
{
    int byte, colour, odd, bit, col;
    unsigned char data, T1;

    for ( byte = 0; byte < 128; byte++ )
    {
        for ( colour = 0; colour < 4; colour++ )
        {
            for ( odd = 0; odd < 2; odd++ )
            {
                /* Build lookup table */
                data = 0;
                byte <<= 1;
                for ( bit = 0; bit < 7; bit++ )
                {
                    T1 = (unsigned char)( ( byte >> bit ) & 7 );
                    /* Check if black */
                    if ( ( T1 & 2 ) == 0 )
                    {
                        col = 0; /* Black */
                    }
                    else
                    {
                        /* Check if white, mask of 5 = 101 which checks for
                           adjacent bits */
                        if ( T1 & 5 )
                        {
                            col = 3; /* White */
                        }
                        else
                        {
                            /* Coloured */
                            if ( !odd && ( bit & 1 ) == 0 ) col = 2; /* even */
                            if ( !odd && ( bit & 1 ) != 0 ) col = 1; /* odd  */
                            if (  odd && ( bit & 1 ) == 0 ) col = 1; /* odd  */
                            if (  odd && ( bit & 1 ) != 0 ) col = 2; /* even */
                            #ifdef CRAP
                            if ( odd )
                            {
                                /* Odd column */
                                col = 1;
                            }
                            else
                            {
                                /* Even column */
                                col = 2;
                            }
                            #endif
                        }
                    }
                    /* Colour matches */
                    if ( col == colour )
                    {
                        data |= ( 1 << ( 7 - bit ) );
                    }
                }
                byte >>= 1;
                /* Set lookup table */
                HiResTable[ byte ][ colour ][ odd ] = data;
            }
        }
    }
}

/******************************
 *                            *
 * EGA TEXT/GRAPHICS ROUTINES *
 *                            *
 ******************************/

#define SETREG( INDEX,VALUE ) \
  asm mov dx, 0x3ce; \
  asm mov al, INDEX; \
  asm out dx, al   ; \
  asm mov dx, 0x3cf; \
  asm mov al, VALUE; \
  asm out dx, al   ;

#define WRITEMODE( MODE )   SETREG( 0x05, MODE )
#define SETMASK( MASK )     SETREG( 0x08, MASK )
#define SETLOGIC( LOGIC )   SETREG( 0x03, LOGIC )
#define SETCOLOUR( COLOUR ) SETREG( 0x00, COLOUR )

#define SET_PALETTE( REG, COLOUR ) \
  asm mov dx, 0x3da;  \
  asm in  al, dx;     \
  asm mov dx, 0x3c0;  \
  asm mov al, REG;    \
  asm out dx, al;     \
  asm mov al, COLOUR; \
  asm out dx, al;     \
  asm mov al, 0x20;   \
  asm out dx, al;

unsigned char far *GraphicsPage1 = (unsigned char far *)0xA0002000L;
unsigned char far *GraphicsPage2 = (unsigned char far *)0xA0004000L;
unsigned char far *GraphicsScreen;
unsigned int  far *    TextScreen = (unsigned int far *)0xB8000000L;

char Font[ 2048 ];

void interrupt ( *OldInt1C )( void );

/* This handles the FLASH characters */
void interrupt PaletteFlip()
{
    static int count = 0;

    count++;
    if ( count == 6 )
    {
        /* Wait for vertical retrace */
Again1:
        asm mov dx, 0x3da;
        asm in  al, dx;
        asm and al, 0x08;
        asm je  Again1;
        /* Set flashing black to white */
        SET_PALETTE( 0x0e, 0x3f );
        /* Set flashing white to black */
        SET_PALETTE( 0x0f, 0x00 );
    }
    if ( count == 12 )
    {
        count = 0;
        /* Wait for vertical retrace */
Again2:
        asm mov dx, 0x3da;
        asm in  al, dx;
        asm and al, 0x08;
        asm je  Again2;
        /* Set flashing black to black */
        SET_PALETTE( 0x0e, 0x00 );
        /* Set flashing white to white */
        SET_PALETTE( 0x0f, 0x3f );
    }
    OldInt1C();
}

/* Set the EGA font to that stored in Font */
void SetFont( void )
{
    int FntSeg, FntOff, row;

    /* Fix character 223 (upper block) */
    for ( row = 0; row < 8; row++ )
    {
        Font[ 223 * 8 + row ] = ( row < 4 ) ? 0xff : 0x00;
    }
    /* Set EGA font */
    FntSeg = FP_SEG( (char far *)Font );
    FntOff = FP_OFF( (char far *)Font );
    asm push es;
    asm push bp;
    asm mov  ax, FntSeg;
    asm mov  bx, FntOff;
    asm mov  es, ax;
    asm mov  bp, bx;
    asm mov  dx, 0x0000; /* start from ASCII 0 */
    asm mov  cx, 0x0100; /* 256 characters */
    asm mov  bx, 0x0800; /* 8 bytes/character */
    asm mov  ax, 0x1100;
    asm int  0x10;
    asm pop  bp;
    asm pop  es;
}

/* Set the EGA palette to better match the Apple palette */
void SetPalette( void )
{
    unsigned char Palette[ 16 ] =
    {
        0, 61, 9, 5, 2, 7, 11, 48, 38, 47, 26, 62, 19, 63, 0, 63
    };
    unsigned char colour, entry;
    int loop;

    /* Set palette */
    for ( loop = 0; loop < 16; loop++ )
    {
        entry = Palette[ loop ];
        colour = loop;
        SET_PALETTE( colour, entry );
    }
}

/* Enable intense colour instead of flashing */
void EnableIntensity( void )
{
    asm mov bl, 0x00;
    asm mov ax, 0x1003;
    asm int 0x10;
}

/* Turn the cursor on or off */
void Cursor( int state )
{
    unsigned int set = state ? 0x0607 : 0x2000;

    asm mov ah, 0x0f;
    asm int 0x10;
    asm mov ah, 0x01;
    asm mov cx, set;
    asm int 0x10;
}

void InitVideo( void )
{
    int fd, loop, row, bit;
    unsigned char Char, RevChar;
    int FntSeg, FntOff;

    InitHiresLookupTables();
    /* Copy ROM 8x8 character set */
    asm push bp;
    asm mov  ax, 0x1130;
    asm mov  bh, 0x03;
    asm int  0x10;
    asm mov  ax, es;
    asm mov  bx, bp;
    asm pop  bp;
    asm mov  FntSeg, ax;
    asm mov  FntOff, bx;
    _fmemcpy( (void far *)Font, MK_FP( FntSeg, FntOff ), 2048 );
    /* Load character set */
    fd = Open( "charset.bin", O_RDONLY );
    if ( fd != -1 )
    {
        /* Could open the font file - read and set the EGA text font */
        for ( loop = 0; loop < 96; loop++ )
        {
            for ( row = 0; row < 8; row++ )
            {
                /* Get apple line */
                Read( fd, (char far *)&Char, 1 );
                /* Reverse it */
                RevChar = 0;
                for ( bit = 0; bit < 8; bit++ )
                {
                    if ( Char & ( 1 << bit ) )
                    {
                        RevChar |= ( 1 << ( 8 - bit ) );
                    }
                }
                /* Modify EGA font */
                Font[ ( loop + 32 ) * 8 + row ] = RevChar >> 2;
            }
        }
        Close( fd );
    }
    /* Setup palette */
    SetPalette();
    /* Disable default palette loading */
    asm mov bl, 0x31;
    asm mov ax, 0x1201;
    asm int 0x10;
    /* Set 200 line mode */
    asm mov bl, 0x30;
    asm mov ax, 0x1200;
    asm int 0x10;
    /* Need to change to 40x25 (Mode 1) (no clearing) */
    asm mov ax, 0x0081;
    asm int 0x10;
    EnableIntensity();
    /* Set the EGA alphanumeric font */
    SetFont();
    /* Hide cursor */
    Cursor( 0 );
    /* Setup colour flasher */
    OldInt1C = GetVect( 0x1c );
    SetVect( 0x1c, PaletteFlip );
    /* Setup EGA */
    GraphicsScreen = GraphicsPage1;
    /* Clear display memories */
    _fmemset( (void far *)0xb8000000L, 0, 0x0800 );
    WRITEMODE( 0 );
    SETLOGIC( 0 );
    SETREG( 1, 0x0f );
    SETMASK( 0xff );
    SETCOLOUR( 0 );
    _fmemset( (void far *)0xa0000000L, 0, 0x6000 );
    /* Reset EGA */
    WRITEMODE( 0 );
    SETREG( 0, 0 );
    SETREG( 1, 0 );
    SETLOGIC( 0 );
    SETMASK( 0xff );
}

void ResetVideo( void )
{
    /* Reset EGA */
    WRITEMODE( 0 );
    SETREG( 0, 0 );
    SETREG( 1, 0 );
    SETLOGIC( 0 );
    SETMASK( 0xff );
    /* Enable blinking */
    asm mov bl, 0x01;
    asm mov ax, 0x1003;
    asm int 0x10;
    /* Disconnect colour flasher */
    SetVect( 0x1c, OldInt1C );
    /* Enable default palette loading */
    asm mov bl, 0x31;
    asm mov ax, 0x1200;
    asm int 0x10;
    /* Set 400 scan lines */
    asm mov bl, 0x30;
    asm mov ax, 0x1202;
    asm int 0x10;
    /* Text video mode 80x25x16 */
    asm mov ax,0x0003;
    asm int 0x10;
    Cursor( 1 );
}

/*****************************
 *                           *
 * APPLE SCREEN I/O ROUTINES *
 *                           *
 *****************************/

int Graphics = 0, Mixed = 0, Page2 = 0, HiRes = 0, Col80 = 0, AltChar = 0;

/* MS-byte=attribute,LS-byte=ASCII offset */
unsigned int Data2Screen[ 8 ] =
{
    0xd040, 0xd020, 0xfe40, 0xfe20, 0x0d40, 0x0d20, 0x0d40, 0x0d60
};
unsigned char Apple2EGA[ 16 ] =
{
    0, 1, 2, 3, 4, 5, 2, 6, 7, 8, 5, 9, 10, 11, 12, 13
};

void DrawCharacter( int X, int Y, char ch, unsigned char fg, unsigned char bg )
{
    /* EGA font is 8 pixels wide, but HGR modes use font 7 pixels wide */
    char far *screenptr = (char far *)( GraphicsScreen + Y * 40 + ( X >> 3 ) );
    register int pixelno = X & 7;
    char *charptr = (char *)( Font + ( (int)ch << 3 ) );
    register unsigned char rowpattern, mask;
    int rowcount = 8;

    WRITEMODE( 0 );
    SETLOGIC( 0 );
    SETREG( 1, 0x0f );

    while ( rowcount-- )
    {
        /* LEFT SIDE OF CHARACTER */
        rowpattern = *charptr << 1; /* first two columns of the font are blank */
        charptr++;
        /* FOREGROUND */
        SETCOLOUR( fg );
        mask = ( rowpattern & 0xfe ) >> pixelno;
        SETMASK( mask );
        *screenptr |= 0;
        /* BACKGROUND */
        SETCOLOUR( bg );
        mask = ( ~rowpattern & 0xfe ) >> pixelno;
        SETMASK( mask );
        *screenptr |= 0;
        /* RIGHT SIDE OF CHARACTER */
        /* FOREGROUND */
        SETCOLOUR( fg );
        mask = ( rowpattern & 0xfe ) << ( 8 - pixelno );
        SETMASK( mask );
        *( screenptr + 1 ) |= 0;
        /* BACKGROUND */
        SETCOLOUR( bg );
        mask = ( ~rowpattern & 0xfe ) << ( 8 - pixelno );
        SETMASK( mask );
        *( screenptr + 1 ) |= 0;

        screenptr += 40;
    }

    WRITEMODE( 0 );
    SETREG( 0, 0 );
    SETREG( 1, 0 );
    SETLOGIC( 0 );
    SETMASK( 0xff );
}

/* NOTE: to re-map the Apple checkerboard character to the PC checkerboard
   character, replace the lines:
     *ptr=(Data&0x1f)+Data2Screen[Data>>5];
   with:
     *ptr=(Data==255)?0x0db1:(Data&0x1f)+Data2Screen[Data>>5];
   and copy CHARSET.IBM over CHARSET.BIN.
   To go back to the Apple checkerboard, restore the lines to their original
   state and copy CHARSET.APL over CHARSET.BIN.
*/
void UpdateTextLoRes( WORD Address, BYTE Data )
{
    register unsigned int X, Y, T1, fg, bg;
    register unsigned int far *ptr;
    register unsigned int data, d2s;

    T1 = Address & 0x7f;
    if ( T1 < 120 )
    {
        X = T1 % 40;
        Y = T1 / 40 + ( Address >> 7 ) * 3; /* Get row # */
        Y = Y / 3 + ( Y % 3 << 3 );      /* Convert to staggered rows */
        ptr = TextScreen + 40 * Y + X;

        if ( !Graphics )
        {
            /* Text Mode */
            *ptr = ( Data & 0x1f ) + Data2Screen[ Data >> 5 ];
        }
        else
        {
            /* Graphics Mode */
            if ( !HiRes )
            {
                /* Lo-Res Mode */
                if ( Mixed && Y > 19 )
                {
                    /* Mixed and in bottom part - still text */
                    *ptr = ( Data & 0x1f ) + Data2Screen[ Data >> 5 ];
                }
                else
                {
                    /* Not mixed or in bottom part - graphics */
                    data  = Apple2EGA[ Data & 0x0f ];      /* top/foreground */
                    data |= Apple2EGA[ Data >> 4   ] << 4; /* bottom/background */
                    data <<= 8;
                    data |= 223;
                    *ptr = data;
                }
            }
            else
            {
                /* Hi-ResMode */
                if ( Mixed && Y > 19 )
                {
                    /* Update graphical text */
                    d2s = Data2Screen[ Data >> 5 ];
                    switch ( d2s >> 8 )
                    {
                    case 0xd0: /* inverse */
                        fg = 0;
                        bg = 13;
                        break;
                    case 0xfe: /* flashing */
                        fg = 14;
                        bg = 15;
                        break;
                    case 0x0d: /* normal */
                        fg = 13;
                        bg = 0;
                        break;
                    }
                    DrawCharacter( X * 7, Y * 8,
                                   ( Data & 0x1f ) + ( d2s & 0xff ),
                                   fg, bg );
                }
            }
        }
    }
}

BYTE ReadVideoLo1( WORD Address )
{
    BYTE v;

    if ( EightyStore )
    {
        if ( Page2 )
        {
            v = peekb( AuxSeg, Address );
        }
        else
        {
            v = peekb( MainSeg, Address );
        }
    }
    else
    {
        v = peekb( MainRSeg, Address );
    }
    return v;
}

void WriteVideoLo1( WORD Address, BYTE Data )
{
    if ( EightyStore )
    {
        if ( Page2 )
        {
            pokeb( AuxSeg, Address, Data );
        }
        else
        {
            pokeb( MainSeg, Address, Data );
            GraphicsScreen = GraphicsPage1;
            if ( !Page2 )
            {
                UpdateTextLoRes( Address & 0x3ff, Data ); /* -0x400 */
            }
        }
    }
    else
    {
        pokeb( MainWSeg, Address, Data );
        if ( MainWSeg == MainSeg )
        {
            GraphicsScreen = GraphicsPage1;
            if ( !Page2 )
            {
                UpdateTextLoRes( Address & 0x3ff, Data ); /* -0x400 */
            }
        }
    }
}

BYTE ReadVideoLo2( WORD Address )
{
    return peekb( MainRSeg, Address );
}

void WriteVideoLo2( WORD Address, BYTE Data )
{
    pokeb( MainWSeg, Address, Data );
    if ( MainWSeg == MainSeg )
    {
        GraphicsScreen = GraphicsPage2;
        if ( Page2 )
        {
            UpdateTextLoRes( Address & 0x3ff, Data ); /* -0x800 */
        }
    }
}

void UpdateHiRes( WORD Address, BYTE Data )
{
    WORD BaseAddress;
    register unsigned char Y, T1, row;
    register int X, bit;
    unsigned long buf;
    unsigned char mask;
    BYTE Data1, Data3;

    WRITEMODE( 0 );
    SETLOGIC( 0 );
    SETREG( 1, 0x0f );

    /* &0x7f to mask off top (unseen) bit */
    Data1 = ReadByte( Address + 1 );
    Data3 = ReadByte( Address - 1 );
    buf  = ( Data1 & 0x7f ) << 14;
    buf |= ( Data  & 0x7f ) <<  7;
    buf |=   Data3 & 0x7f;

    BaseAddress = Address & 0x1fff;
    row = ( BaseAddress & 0x1c00 ) >> 10; /* Row 0-7 within rows 0-23 */
    BaseAddress &= 0x3ff; /* Address now looks like text screen */
    T1 = BaseAddress & 0x7f;
    if ( T1 < 120 )
    {
        X = T1 % 40;                /* X=0-39 */
        Y = T1 / 40 + ( BaseAddress >> 7 ) * 3; /* Get row # */
        Y = Y / 3 + ( Y % 3 << 3 );      /* Convert to staggered rows */
        if ( Mixed && Y > 19 )
        {
            goto Exit;
        }
        Y = Y * 8 + row;           /* Convert 0-23 -> 0-191 */

        /* Mask off info that is not on the screen */
        if ( X ==  0 ) buf &= 0x001fff80UL; /* Mask off left 7 bits */
        if ( X == 39 ) buf &= 0x00003fffUL; /* Mask off right 7 bits */

        X *= 7; /* 7 display bits per column */

        /* EGA pixel positions = X*7,Y - X*7+6,Y */
        /* i.e. from base X,Y are offset from X 0-6 pixels */
        X--;
        buf >>= 4;
        for ( bit = 0; bit <= 8; bit++ )
        {
            buf >>= 1;
            T1 = (unsigned char)( buf & 7 );
            /* Check if black */
            if ( ( T1 & 2 ) == 0 )
            {
                SETCOLOUR( 0 ); /* Black */
            }
            else
            {
                /* Check if white, mask of 5 = 101 which checks for adjacent bits */
                if ( T1 & 5 )
                {
                    SETCOLOUR( 13 ); /* White */
                }
                else
                {
                    /* Coloured */
                    if ( ( X + bit ) & 1 )
                    {
                        /* Odd column */
                        if ( ( bit == 8            && ( Data1 & 0x80 ) ) ||
                             ( bit == 0            && ( Data3 & 0x80 ) ) ||
                             ( bit >  0 && bit < 8 && ( Data  & 0x80 ) ) )
                        {
                            /* Orange */
                            SETCOLOUR( 8 );
                        }
                        else
                        {
                            /* Green */
                            SETCOLOUR( 10 );
                        }
                    }
                    else {
                        /* Even column */
                        if ( (bit == 8            && ( Data1 & 0x80 ) ) ||
                             (bit == 0            && ( Data3 & 0x80 ) ) ||
                             (bit >  0 && bit < 8 && ( Data  & 0x80 ) ) )
                        {
                            /* Blue */
                            SETCOLOUR( 2 );
                        }
                        else
                        {
                            /* Purple */
                            SETCOLOUR( 3 );
                        }
                    }
                }
            }
            /* putpixel */
            mask = 0x80 >> ( ( X + bit ) & 7 );
            SETMASK( mask );
            GraphicsScreen[ Y * 40 + ( ( X + bit ) >> 3 ) ] |= 0;
        }
    }

Exit:

    WRITEMODE( 0 );
    SETREG( 0, 0 );
    SETREG( 1, 0 );
    SETLOGIC( 0 );
    SETMASK( 0xff );
}

#define SETPAGEOFFSET( PGOFF ) \
  asm mov dx, 0x3d4;           \
  asm mov al, 0x0c;            \
  asm out dx, al;              \
  asm mov dx, 0x3d5;           \
  asm mov al, PGOFF;           \
  asm out dx, al;

BYTE ReadVideoHi1( WORD Address )
{
    BYTE v;

    if ( EightyStore && HiRes )
    {
        if ( Page2 )
        {
            v = peekb( AuxSeg, Address );
        }
        else
        {
            v = peekb( MainSeg, Address );
        }
    }
    else
    {
        v = peekb( MainRSeg, Address );
    }
    return v;
}

void WriteVideoHi1( WORD Address, BYTE Data )
{
    if ( EightyStore && HiRes )
    {
        if ( Page2 )
        {
            pokeb( AuxSeg, Address, Data );
        }
        else
        {
            pokeb( MainSeg, Address, Data );
            GraphicsScreen = GraphicsPage1;
            UpdateHiRes( Address, Data );
        }
    }
    else
    {
        pokeb( MainWSeg, Address, Data );
        if ( MainWSeg == MainSeg )
        {
            GraphicsScreen = GraphicsPage1;
            UpdateHiRes( Address, Data );
        }
    }
}

BYTE ReadVideoHi2( WORD Address )
{
    return peekb( MainRSeg, Address );
}

void WriteVideoHi2( WORD Address, BYTE Data )
{
    pokeb( MainWSeg, Address, Data );
    if ( MainWSeg == MainSeg )
    {
        GraphicsScreen = GraphicsPage2;
        UpdateHiRes( Address, Data );
    }
}

BYTE ReadVideoControl( WORD Address )
{
    BYTE v;
    int i, base;
    int redrawtext = 0;

    v = 0xff;

    switch ( Address )
    {
    case 0x1a:
        /* RdText */
        v = Graphics ? 0x0d : 0x8d;
        break;
    case 0x1b:
        /* RdMixed */
        v = Mixed ? 0x8d : 0x0d;
        break;
    case 0x1c:
        /* RdPage2 */
        v = Page2 ? 0x8d : 0x0d;
        break;
    case 0x1d:
        /* RdHires */
        v = HiRes ? 0x8d : 0x0d;
        break;
    case 0x1e:
        /* RdAltChar */
        v = AltChar ? 0x8d : 0x0d;
        break;
    case 0x1f:
        /* Rd80Col */
        v = Col80 ? 0x8d : 0x0d;
        break;
    case 0x50:
        /* Select graphics mode */
        if ( !Graphics && HiRes )
        {
            /* Need to change to 320x200x16 (Mode D) (no clearing) */
            asm mov ax, 0x008d;
            asm int 0x10;
            EnableIntensity();
            WRITEMODE( 0 );
            SETLOGIC( 0 );
            SETREG( 1, 0x0f );
            if ( Page2 )
            {
                SETPAGEOFFSET( 0x40 );
            }
            else
            {
                SETPAGEOFFSET( 0x20 );
            }
        }
        Graphics = 1;
        if ( !HiRes || Mixed )
        {
            redrawtext = 1;
        }
        break;
    case 0x51:
        /* Select Text Mode */
        if ( Graphics )
        {
            redrawtext = 1;
            if ( HiRes )
            {
                /* Reset EGA */
                WRITEMODE( 0 );
                SETREG( 0, 0 );
                SETREG( 1, 0 );
                SETLOGIC( 0 );
                SETMASK( 0xff );
                /* Need to change to 40x25 (Mode 1) (no clearing) */
                asm mov ax, 0x0081;
                asm int 0x10;
                EnableIntensity();
                SetFont();
                Cursor( 0 );
                SETPAGEOFFSET( 0 );
            }
        }
        Graphics = 0;
        break;
    case 0x52:
        /* Select Full-Screen Graphics */
        if ( Mixed )
        {
            if ( !HiRes )
            {
                redrawtext = 1;
            }
            if ( HiRes && Graphics )
            {
                /* Draw bottom graphics */
                int BaseAddr[ 4 ] =
                {
                    0x0250, 0x02d0, 0x0350, 0x03d0
                };
                int line, row, col, addr, PageBase;
                int addr1, addr2;

                Mixed = 0;
                PageBase = Page2 ? 0x4000 : 0x2000;
                for ( line = 0; line < 4; line++ )
                {
                    addr1 = PageBase + BaseAddr[ line ];
                    for ( row = 0x0000; row < 0x2000; row += 0x0400 )
                    {
                        addr2 = addr1 + row;
                        for ( col = 0; col < 40; col++ )
                        {
//                            addr = PageBase + BaseAddr[ line ] + row + col;
                            addr = addr2 + col;
                            WriteByte( addr, ReadByte( addr ) );
                        }
                    }
                }
            }
        }
        Mixed = 0;
        break;
    case 0x53:
        /* Select Graphics Plus Text */
        if ( !Mixed )
        {
          redrawtext = 1;
        }
        Mixed = 1;
        break;
    case 0x54:
        /* Select Screen Page 1 */
        if ( Page2 )
        {
            if ( HiRes && Graphics )
            {
                SETPAGEOFFSET( 0x20 );
                if ( Mixed )
                {
                    redrawtext = 1;
                }
            }
            else
            {
                redrawtext = 1;
            }
        }
        Page2 = 0;
        break;
    case 0x55:
        /* Select Screen Page 2 */
        if ( !Page2 )
        {
            if ( !EightyStore )
            {
                if ( HiRes && Graphics )
                {
                    SETPAGEOFFSET( 0x40 );
                    if ( Mixed )
                    {
                        redrawtext = 1;
                    }
                }
                else
                {
                    redrawtext = 1;
                }
            }
        }
        Page2 = 1;
        break;
    case 0x56:
        /* Select Low-Resolution Graphics */
        if ( HiRes && Graphics )
        {
            /* Reset EGA */
            WRITEMODE( 0 );
            SETREG( 0, 0 );
            SETREG( 1, 0 );
            SETLOGIC( 0 );
            SETMASK( 0xff );
            /* Need to change to 40x25 (Mode 1) (no clearing) */
            asm mov ax, 0x0081;
            asm int 0x10;
            EnableIntensity();
            redrawtext = 1;
            SetFont();
            Cursor( 0 );
            SETPAGEOFFSET( 0 );
        }
        HiRes = 0;
        break;
    case 0x57:
        /* Select High-Resolution Graphics */
        if ( !HiRes && Graphics )
        {
            /* Need to change to 320x200x16 (Mode D) (no clearing) */
            asm mov ax, 0x008d;
            asm int 0x10;
            EnableIntensity();
            WRITEMODE( 0 );
            SETLOGIC( 0 );
            SETREG( 1, 0x0f );
            if ( Page2 )
            {
                SETPAGEOFFSET( 0x40 );
            }
            else
            {
                SETPAGEOFFSET( 0x20 );
            }
        }
        HiRes = 1;
        break;
    case 0x5e:
        /* Select Double High-Resolution Graphics */
        break;
    case 0x5f:
        /* Clear Double High-Resolution Graphics */
        break;
    case 0x7f:
        /* RdDHiRes */
        v = 0x7f; /* currently not supported: hence always off */
        break;
    }
    if ( redrawtext )
    {
        base = Page2 ? 0x800 : 0x400;
        for ( i = 0; i < 0x400; i++ )
        {
            WriteByte( base + i, ReadByte( base + i ) );
        }
    }
    return v;
}

#pragma argsused
void WriteVideoControl( WORD Address, BYTE Data )
{
    switch ( Address )
    {
    case 0x0c:
        /* 80Col Off */
        Col80 = 0;
        /* make sure we're in 40 col video mode */
        break;
    case 0x0d:
        /* 80Col On */
        Col80 = 1;
        /* make sure we're in 80 col video mode */
        break;
    case 0x0e:
        /* AltChar Off */
        AltChar = 0;
        break;
    case 0x0f:
        /* AltChar On */
//        AltChar = 1;
        break;
    case 0x50:
    case 0x51:
    case 0x52:
    case 0x53:
    case 0x54:
    case 0x55:
    case 0x56:
    case 0x57:
    case 0x5e:
    case 0x5f:
        ReadVideoControl( Address );
        break;
    }
}
