#define Uses_TListViewer
#define Uses_TPoint
#define Uses_TRect
#define Uses_TScrollBar
#define Uses_TView
#include <tv.h>

#include <string.h>

#include "hierarch.h"

THierarchyItem::THierarchyItem(int ic) :
  isCollapsed(ic)
{
  Ancestor=0;
  Descendant=0;
  Prev=0;
  Next=0;
  Data=0;
}

THierarchyItem::~THierarchyItem()
{
  Unlink();
  while (Descendant) delete Descendant;
  delete Data;
}

void THierarchyItem::DeleteDescendants()
{
  while (Descendant) delete Descendant;
  isCollapsed=0;
}

// If the item to decend from already has a descendant, then this item will
// be inserted at the *bottom* of the list of descendants
void THierarchyItem::DescendFrom(THierarchyItem *i)
{
  THierarchyItem *ptr;

  if (!i) return;
  Unlink();
  if (i->Descendant) {
    ptr=i->Descendant;
    while (ptr->Next) ptr=ptr->Next;
    InsertAfter(ptr);
  }
  else {
    i->Descendant=this;
    Ancestor=i;
  }
}

// Insert this item after (or 'below') the given item
void THierarchyItem::InsertAfter(THierarchyItem *i)
{
  if (!i) return;
  Link(i,i->Next);
}

// Insert this item before (or 'above') the given item
void THierarchyItem::InsertBefore(THierarchyItem *i)
{
  if (!i) return;
  Link(i->Prev,i);
}

// Places this item between the items pointed to by p (the previous item
// 'above') and n (the next item 'below')
void THierarchyItem::Link(THierarchyItem *p,THierarchyItem *n)
{
  Unlink();
  Prev=p;
  Next=n;
  if (p) {
    p->Next=this;
    Ancestor=p->Ancestor; // Doesn't matter which ancestor (prev or next) as
  }                       // long as it's one of them
  if (n) {
    n->Prev=this;
    Ancestor=n->Ancestor;
  }
  // If this is the new top of descendants, point ancestor to it
  if (!Prev && Ancestor) Ancestor->Descendant=this;
}

// Unlink the current item and all its' decendants from the hierarchy.
// The item can then be treated as the root item of a hierarchy of it's own.
void THierarchyItem::Unlink()
{
  // If the ancestor is referring to this item, then fix it to point to
  // the previous item, if it exists, or the next item if there is no
  // previous item;
  if (Ancestor)
    if (Ancestor->Descendant==this)
      Ancestor->Descendant=(Prev)?Prev:Next;
  // Fix the pointers of the previous and next items to skip over this item
  if (Prev) Prev->Next=Next;
  if (Next) Next->Prev=Prev;
  // Un-link this item from the hierarchy
  Ancestor=0;
  Prev=0;
  Next=0;
}

void THierarchyItem::SetData(void *d)
{
  if (!Data) Data=new char[DataSize()];
  if (!Data) return;
  memmove(Data,d,DataSize());
}

void THierarchyItem::GetData(void *d)
{
  if (!Data) return;
  if (d) memmove(d,Data,DataSize());
}

void *THierarchyItem::GetData()
{
  return Data;
}

void THierarchyItem::HideDescendants()
{
  if (Descendant) isCollapsed=1;
  else isCollapsed=0;
}

void THierarchyItem::ShowDescendants()
{
  isCollapsed=0;
}

int THierarchyItem::IsCollapsed()
{
  return isCollapsed;
}

/******************************
 *                            *
 * TEXT STRING HIERARCHY ITEM *
 *                            *
 ******************************/

TTextHierarchyItem::TTextHierarchyItem(char *text,int ic) :
  THierarchyItem(ic)
{
  SetData(text);
}

short TTextHierarchyItem::DataSize()
{
  return (Data) ? strlen((char*)Data)+1 : 0;
}

void TTextHierarchyItem::SetData(void *d)
{
  if (Data) delete Data;
  Data = new char[strlen((char*)d)+1];
  strcpy((char*)Data,(char*)d);
}

void TTextHierarchyItem::GetData(void *d)
{
  strcpy((char*)d,(char*)Data);
}

/*************************************
 *                                   *
 * SORTED TEXT STRING HIERARCHY ITEM *
 *                                   *
 *************************************/

TSortedTextHItem::TSortedTextHItem(char *text) :
  TTextHierarchyItem(text)
{
}

int TSortedTextHItem::compare(THierarchyItem *i1,THierarchyItem *i2)
{
  return strcmp((char*)i1->GetData(),(char*)i2->GetData());
}

void TSortedTextHItem::InsertAfter(THierarchyItem *i)
{
  // since the items are always sorted, just insert anywhere and let
  // InsertBefore look after the order.
  InsertBefore(i);
}

void TSortedTextHItem::InsertBefore(THierarchyItem *i)
{
  // Ignore InsertBefore/InsertAfter requests and always insert the item in
  // the correct sorted order.
  // Start at a known position: the top
  while (i->Prev) i=i->Prev;
  // Now search for the correct insertion point
  // i will be left either immediately before or after insertion point
  while (i->Next && compare(i,this)<0) i=i->Next;
  // figure out which direction to insert it
  if (compare(i,this)<0 && !i->Next) {
    TTextHierarchyItem::InsertAfter(i);
    return;
  }
  TTextHierarchyItem::InsertBefore(i);
}

/*********************************
 *                               *
 * TURBO VISION HIERARCHY VIEWER *
 *                               *
 *********************************/

THierarchyViewer::THierarchyViewer(TRect& bounds,
                                   TScrollBar *aHScrollBar,
                                   TScrollBar *aVScrollBar,
                                   int iw,
                                   THierarchyItem *r) :
  TListViewer(bounds,1,aHScrollBar,aVScrollBar),
  itemWidth(iw),
  Root(r)
{
  selected=0;
  calcSize(Root,hierarchyHeight,hierarchyWidth);
  setRange(hierarchyHeight);
}

// calculate the current height and width of the given hierarchy
void THierarchyViewer::calcSize(THierarchyItem *i,int& height,int& width)
{
  THierarchyItem *ptr;
  short curWidth=0;

  height=0;
  width =0;

  if (!i) return;

  height=1;
  width =1;

  ptr=i;

  // work through hierarchy going across, then down
  // at each item, go right if possible, otherwise go down, otherwise back
  // up and go down
  while (ptr) {
    if (ptr->Descendant && !ptr->IsCollapsed()) {
      height++;
      // go right
      ptr=ptr->Descendant;
      curWidth++;
      if (curWidth > width) width=curWidth;
    }
    else {
      if (ptr!=i && ptr->Next) {
        // go down
        ptr=ptr->Next;
        height++;
      }
      else {
        // back up and go down
        while (ptr!=i && ptr->Ancestor && !ptr->Next) ptr=ptr->Ancestor;
        if (ptr!=i && ptr->Next) {
          height++;
          ptr=ptr->Next;
        }
        else ptr=0;
      }
    }
  }
}

THierarchyItem *THierarchyViewer::getHierarchyItem(short item)
{
  // returns the item'th hierarchy item taking into account hidden items
  // item 0 is the root item.
  THierarchyItem *ptr=Root;
  short p=0;

  // work through hierarchy going across, then down
  // at each item, go right if possible, otherwise go down, otherwise back
  // up and go down
  while (ptr && p<item) {
    if (ptr->Descendant && !ptr->IsCollapsed()) {
      // go right
      ptr=ptr->Descendant;
      p++;
    }
    else {
      if (ptr->Next) {
        // go down
        ptr=ptr->Next;
        p++;
      }
      else {
        // back up and go down
        while (ptr->Ancestor && !ptr->Next) ptr=ptr->Ancestor;
        if (ptr->Next) {
          p++;
          ptr=ptr->Next;
        }
        else ptr=0;
      }
    }
  }
  return ptr;
}

void THierarchyViewer::getText(char *dest,short item,short maxLen)
{
  char linebuf[256];
  char itembuf[81];
  char linkbuf[81];
  THierarchyItem *ptr;
  char *cptr;

  // Get text associated with the given item
  ptr=getHierarchyItem(item);
  if (!ptr) {
    *dest=0; // nothing to show!
    return;
  }
  getHierarchyItemText(itembuf,ptr,80);

  // Create tree links. It is built in reverse order and reversed when
  // finished by the non-ANSI function strrev().
  *linkbuf=0;
  cptr=linkbuf;
  *cptr++=']';
  if (ptr->Descendant) {
    if (ptr->IsCollapsed()) *cptr++='+';
    else *cptr++='-';
  }
  else *cptr++=' ';
  *cptr++='[';
  *cptr++=(ptr->Next)?195:192;
  // now work back up the tree and put in decending tree links
  while (ptr->Ancestor) {
    *cptr++=' ';
    ptr=ptr->Ancestor;
    *cptr++ = (ptr->Next) ? 179 : ' '; // (179) or space
  }
  cptr--;
  *cptr=0;
  strrev(linkbuf);

  // Build line buffer from links and item text
  strcpy(linebuf,linkbuf);
  strcat(linebuf,itembuf);

  // Truncate buffer before copying to destination
  *(linebuf+maxLen)=0;
  if (dest) strcpy(dest,linebuf);
}

THierarchyItem *THierarchyViewer::newHierarchy(THierarchyItem *r)
{
  THierarchyItem *oldRoot=Root;
  Root=r;
  calcSize(Root,hierarchyHeight,hierarchyWidth);
  setRange(hierarchyHeight);
  drawView();
  return oldRoot;
}

void THierarchyViewer::selectItem(short item)
{
  int h,w;

  // Toggle collapsed state
  THierarchyItem *ptr=getHierarchyItem(item);
  if (ptr->IsCollapsed()) {
    ptr->ShowDescendants();
    calcSize(ptr,h,w);
    if (focused+h > topItem+size.y) {
      // bottom of newly exposed hierarchy is below the bottom of the view:
      // scroll the view up so that it is visible
      topItem+=(focused+h)-(topItem+size.y);
      // make sure the focused item is still visible
      if (focused<topItem) topItem=focused;
    }
  }
  else ptr->HideDescendants();
  newHierarchy(Root);
}
