#include "ram.h"
#include "cpu.h"
#include "massstor.h"
#include "system.h"
#include "diags.h"

#include "diskclnt.h"

/* drive light colours */
#define DRV_RD_COL   0x7a /* green on brown */
#define DRV_WR_COL   0x87 /* brown on orange */
#define DRV_OFF_COL  0x77 /* brown on brown */
/* drive light characters */
#define DRV_RD_CHAR  0xdf /* upper inverse block */
#define DRV_WR_CHAR  0xdf /* upper inverse block */
#define DRV_OFF_CHAR 0xdb /* full inverse block */

/* NOTE: in theory ProDOS can support 4 volumes in slots 5 & 6. However,
   it never seems to perform STATUS or FORMAT calls for the 2 extra units.
   Hence, my mass-storage routines supply ProDOS with incorrect block
   numbers, and won't format correctly. So extra volumes are disabled for
   now. */

/* #define MS_DMA to create DMA (direct memory access) version
   #undef  MS_DMA to create slow I/O (but more standard) version
                  where the card does not access memory
   Make sure the correct version of MASSSTOR.ASM has been assembled. */
#define MS_DMA

/*
;Mass-storage device I/O
; $0 - Writing the ProDOS command here will perform that command.
;      It is important to setup the other locations first.
;      0=Status,1=Read,2=Write,3=Format
; $1 - Unit number $0-$3
; $2 - ProDOS buffer low byte
; $3 - ProDOS buffer high byte
; $4 - Block low byte
; $5 - Block high byte
; $6 - Data register
*/

#define PS_OK    0x00
#define PS_IOERR 0x27
#define PS_NODEV 0x28
#define PS_WRPRO 0x2b

int MassStorSlot;

WORD MaxBlock = 2048; //280;

char VolumeFN[ 4 ][ 80 ]; /* volume file names */
int VolumeFH[ 4 ];        /* volume file handles */
long VolumeSize[ 4 ];     /* size of volumes in bytes */
int WritePro[ 4 ];        /* if a volume is write-protected */
int ValidBlock[ 4 ];      /* if current block buffer contains valid data */
WORD Block[ 4 ];          /* block # in buffer */
BYTE Buffer[ 4 ][ 512 ];  /* 512 byte buffer for each unit */

BYTE Unit;            /* current unit */
WORD ProDOSBuf;       /* Apple memory location to ProDOS buffer */
WORD Byte[ 4 ];       /* current byte offset in buffer */
BYTE DataReg;         /* current value of data register */
BYTE Status;          /* current status of device */

#define STATUS_OFF 0
#define STATUS_RD  1
#define STATUS_WR  2

/* Updates light for current unit based on command */
/* cmd:0=nothing,1=reading,2=writing */
void UpdateMassStorStatus( int cmd )
{
  GotoXY( ( MassStorSlot - 1 ) * 5, 24 );
  SetAttr( 0x0d );
  PutChar( 'M' );
  GotoXY( ( MassStorSlot - 1 ) * 5 + Unit + 1, 24 );
  switch ( cmd )
  {
  case STATUS_OFF:
      SetAttr( DRV_OFF_COL );
      PutChar( DRV_OFF_CHAR );
      break;
  case STATUS_RD:
      SetAttr( DRV_RD_COL );
      PutChar( DRV_RD_CHAR );
      break;
  case STATUS_WR:
      SetAttr( DRV_WR_COL );
      PutChar( DRV_WR_CHAR );
      break;
  }
}

void InitMassStor( int slot )
{
    int i;
    unsigned int attr;

    MassStorSlot = slot;

    for ( i = 0; i < 4; i++ )
    {
        VolumeFN[ i ][ 0 ] = 0;
    }

    strcpy( VolumeFN[ 0 ], "prodos1.dsk" );
    strcpy( VolumeFN[ 1 ], "prodos2.dsk" );

    for ( i = 0; i < 4; i++ )
    {
        Unit=i;
        UpdateMassStorStatus( STATUS_OFF );
        attr = GetAttrib( VolumeFN[ i ] );
        if ( attr == 0xffff )
        {
            attr = 0;
        }
        WritePro[ i ] = ( attr & FA_RDONLY ) ? 1 : 0;
        VolumeFH[ i ] = Open( VolumeFN[ i ], WritePro[ i ] ? O_RDONLY : O_RDWR );
        if ( VolumeFH[ i ] > 0 )
        {
            VolumeSize[ i ] = FileSize( VolumeFH[ i ] );
        }
        else
        {
            VolumeSize[ i ] = 0L;
        }
        ValidBlock[ i ] = 0;
    }
    Status = 0;
    Unit = 0;
}

void ShutdownMassStor( void )
{
    int i;

    for ( i = 0; i < 4; i++ )
    {
        if ( VolumeFH[ i ] > 0 )
        {
            Close( VolumeFH[ i ] );
            if ( VolumeSize[ i ] == 0L )
            {
                /* nothing in the file - might as well delete it */
                DeleteFile( VolumeFN[ i ] );
            }
        }
        VolumeFH[ i ] = 0;
    }
}

BYTE ReadMassStorIO( WORD Address )
{
    BYTE v;

    v = 0xff;
    switch ( Address & 0x0f )
    {
    case 0x01:
        v = Unit;
        break;
    case 0x02:
        #ifdef MS_DMA
        v = ProDOSBuf & 0xff;
        #else
        v = Byte[ Unit ] & 0xff;
        #endif
        break;
    case 0x03:
        #ifdef MS_DMA
        v = ProDOSBuf >> 8;
        #else
        v = Byte[ Unit ] >> 8;
        #endif
        break;
    case 0x04:
        v = Block[ Unit ] & 0xff;
        break;
    case 0x05:
        v = Block[ Unit ] >> 8;
        break;
    case 0x06:
        v = DataReg;
        break;
    case 0x07:
        v = Status;
        break;
    }
    return v;
}

/* this does not actually erase any data - it only attempts to create a
   volume of the size given in MaxBlock */
void FormatVolume( void )
{
    if ( WritePro[ Unit ] ) {
        Status = PS_WRPRO;
    }
    else {
        if ( Seek( VolumeFH[ Unit ], 512L * (long)MaxBlock -1L, SEEK_SET ) == -1L )
        {
            Status = PS_IOERR;
        }
        else
        {
            if ( Write( VolumeFH[ Unit ], Buffer[ Unit ], 1 ) == 0 )
            {
                Status = PS_IOERR;
            }
            else
            {
                VolumeSize[ Unit ] = FileSize( VolumeFH[ Unit ] );
            }
        }
    }
}

void ReadBlock( void )
{
    #ifdef MS_DMA
    int bufptr;
    #endif

    if ( !ValidBlock[ Unit ] )
    {
        /* get block from disk */
        if ( VolumeFH[ Unit ] < 1 )
        {
            Status = PS_NODEV;
        }
        else
        {
            #ifndef REGULAR_IO
            if ( Seek( VolumeFH[ Unit ], 512L * (long)Block[ Unit ], SEEK_SET ) == -1L ||
                VolumeSize[ Unit ] == 0L )
            {
                Status = PS_IOERR;
            }
            else
            {
                if ( Read( VolumeFH[ Unit ], Buffer[ Unit ], 512 ) == 0 )
                {
                    Status = PS_IOERR;
                }
                else
                {
                    ValidBlock[ Unit ] = 1;
                    Byte[ Unit ] = 0;
                }
            }
            #endif
            #ifdef REMOTE_IO
            RmtReadBlock( 6, 1, Block[ Unit ], Buffer[ Unit ] );
            #endif
        }
    }
    if ( Status == PS_OK )
    {
        #ifdef MS_DMA
        for ( bufptr = 0; bufptr < 512; bufptr++ )
        {
            WriteByte( ProDOSBuf + bufptr, Buffer[ Unit ][ bufptr ] );
        }
        #else
        DataReg = Buffer[ Unit ][ Byte[ Unit ] ];
        Byte[ Unit ]++;
        if ( Byte[ Unit ] == 0x200 )
        {
            Byte[ Unit ] = 0;
        }
        #endif
    }
}

void WriteBlock( void )
{
    #ifdef MS_DMA
    int bufptr;
    #endif

    #ifdef MS_DMA
    for ( bufptr = 0; bufptr < 512; bufptr++ )
    {
        Buffer[ Unit ][ bufptr ] = ReadByte( ProDOSBuf + bufptr );
    }
    Byte[ Unit ] = 0;
    #else
    Buffer[ Unit ][ Byte[ Unit ] ] = DataReg;
    Byte[ Unit ]++;
    if ( Byte[ Unit ] == 0x200 )
    {
        Byte[ Unit ] = 0;
    }
    #endif
    /* If not a valid filehandle, then no device error */
    if ( VolumeFH[ Unit ] < 1 )
    {
        Status = PS_NODEV;
        return;
    }
    /* If write-protected, then write-protected error */
    if ( WritePro[ Unit ] )
    {
        Status = PS_WRPRO;
        return;
    }
    /* Only write when the buffer is full */
    if ( Byte[ Unit ] == 0 )
    {
        /* Try to seek to the blocks location in the file */
        if ( Seek( VolumeFH[ Unit ], 512L * (long)Block[ Unit ], SEEK_SET ) == -1L )
        {
            /* Error */
            Status = PS_IOERR;
            return;
        }
        /* If the file is newly created (zero-length) then it needs to be
           formatted first */
        if ( VolumeSize[ Unit ] == 0L )
        {
            /* Check if the volume should be formatted */
//          if ( ShouldFormat )
//          {
//              FormatVolume();
//          }
        }
        /* Make sure out current location is inside the file somewhere */
        if ( Tell( VolumeFH[ Unit ] ) >= VolumeSize[ Unit ] )
        {
            /* No - error */
            Status = PS_IOERR;
            return;
        }
        /* Everything appears to be OK up until now - write the buffer */
        if ( Write( VolumeFH[ Unit ], Buffer[ Unit ], 512 ) == 0 )
        {
            /* Error on write */
            Status = PS_IOERR;
        }
    }
}

void WriteMassStorIO( WORD Address, BYTE Data )
{
    Status = PS_OK;
    switch ( Address & 0x0f )
    {
    case 0x00:
        switch ( Data )
        {
        case 0x00:
            /* Status */
            /* Set # blocks */
            Block[ Unit ] = ( VolumeSize[ Unit ] == 0L ) ?
                            MaxBlock : (WORD)( VolumeSize[ Unit ] / 512L );
            break;
        case 0x01:
            UpdateMassStorStatus( STATUS_RD );
            ReadBlock();
            break;
        case 0x02:
            UpdateMassStorStatus( STATUS_WR );
            WriteBlock();
            break;
        case 0x03:
            UpdateMassStorStatus( STATUS_WR );
            FormatVolume();
            break;
        }
        UpdateMassStorStatus( STATUS_OFF );
        break;
    case 0x01:
        Unit = Data;
        break;
    case 0x02:
        #ifdef MS_DMA
        ProDOSBuf = ( ProDOSBuf & 0xff00 ) | Data;
        #else
        Byte[ Unit ] = ( Byte[ Unit ] & 0xff00 ) | Data;
        #endif
        break;
    case 0x03:
        #ifdef MS_DMA
        ProDOSBuf = ( ProDOSBuf & 0x00ff ) | ( Data << 8 );
        #else
        Byte[ Unit ] = ( Byte[ Unit] & 0x00ff ) | ( Data << 8 );
        #endif
        break;
    case 0x04:
        Block[ Unit ] = ( Block[ Unit ] & 0xff00 ) | Data;
        ValidBlock[ Unit ] = 0;
        break;
    case 0x05:
        Block[ Unit ] = ( Block[ Unit ] & 0x00ff ) | ( Data << 8 );
        ValidBlock[ Unit ] = 0;
        break;
    case 0x06:
        DataReg = Data;
        break;
  }
}
