#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <dir.h>

struct A2ICommsParms {
  /* parallel port I/O */
  unsigned long presenddelay;
  unsigned long senddelay;
  unsigned long rectimeout;
} parms;

void InitA2IParms(struct A2ICommsParms *parms)
{
  parms->presenddelay=60000UL;   /* microsecond length of phase 1 */
  parms->senddelay=100UL;      /* microsecond length of phases 2 & 3 */
  parms->rectimeout=0x3ffffUL; /* # retries for each nibble receive phase */
}

/*******************************************
 *                                         *
 * ~1microsecond resolution delay routines *
 *                                         *
 * Thank you Borland!                      *
 *                                         *
 *******************************************/

/* NOTE: readtimer, timer_init, and usdelay are slight modifications of the
   functions found in the C Run-Time Library file DELAY.CAS */

static unsigned long multiplier;

static void near dummy (void) {}

/* returns complement of counter in timer 0. Complement is used so the
   reported count appears to go up (the timer counts down). */
static unsigned near readtimer(void)
{
  asm pushf                    /* Save interrupt flag */
  asm cli                      /* Disable interrupts */
  asm mov  al,0                /* Latch timer 0 */
  asm out  0x43,al
      dummy();                 /* Waste some time */
  asm in   al,0x40             /* Counter --> bx */
  asm mov  bl,al               /* LSB in BL */
      dummy();                 /* Waste some time */
  asm in   al,0x40
  asm mov  bh,al               /* MSB in BH */
  asm not  bx                  /* Need ascending counter */
  asm popf                     /* Restore interrupt flag */
  return( _BX );
}

/* Figure out what mode timer 0 is in. It should be either mode 2 or 3. In
   mode 3 it counts down by two twice and should always be even. Mode 2
   counts can be either odd or even. If the count is sampled 100 times and
   an odd value appears, then take it as mode 2. */
static void timer_init(void)
{
  int i;

  /* assume mode 3 */
  multiplier = 2L;
  for (i = 0; i < 100; i++)
    if ((readtimer() & 1) == 0) {  /* readtimer() returns complement */
      /* mode 2 */
      multiplier = 1L;
      return;
    }
}

/* usdelay (microsecond delay) - wait for specified number of microseconds */
void _CType usdelay(unsigned long microseconds)
{
  unsigned long stop;
  unsigned cur, prev;

  stop = (prev = readtimer()) + (microseconds*1000L/838L*multiplier);

  while ((cur = readtimer()) < stop) {
    if (cur < prev) {     /* Check for timer wraparound */
      if (stop < 0x10000L) break;
      stop -= 0x10000L;
    }
    prev = cur;
  }
}

/************
 *          *
 * BYTE I/O *
 *          *
 ************/

void SendByte(struct A2ICommsParms *parms,unsigned char byte)
{
  /* lower clock */
  outportb(0x378,0);
  /* pre-send delay */
  usdelay(parms->presenddelay);
  /* send low nibble with clock raised */
  outportb(0x378,(byte&0x0f)|0x10);
  /* send delay */
  usdelay(parms->senddelay);
  /* send high nibble with clock lowered */
  outportb(0x378,(byte>>4)&0x0f);
  /* send delay */
  usdelay(parms->senddelay);
}

/**********************
 *                    *
 * CRC-CCITT Routines *
 *                    *
 **********************/

unsigned int CRCTable[16]={
  0x0000,0x1081,0x2102,0x3183,0x4204,0x5285,0x6306,0x7387,
  0x8408,0x9489,0xa50a,0xb58b,0xc60c,0xd68d,0xe70e,0xf78f
};

void InitCRC(unsigned int *CRC)
{
  *CRC=0xffff;
}

/* Residue = 0xf0b8 */
void CalcCRC(unsigned int *CRC,int Ch)
{
  int idx;
  idx=((*CRC^(Ch>>0))&0x000f);
  *CRC=((*CRC>>4)&0x0fff)^CRCTable[idx];
  idx=((*CRC^(Ch>>4))&0x000f);
  *CRC=((*CRC>>4)&0x0fff)^CRCTable[idx];
}

void PreSendCRC(unsigned int *CRC)
{
  *CRC=~*CRC;
}

/*****************
 *               *
 * FILE TRANSFER *
 *               *
 *****************/

void TransFile(char *filename,unsigned int startaddr)
{
  FILE *fp;
  long filelen;
  int ch,loop;
  unsigned int CRC;
  char name[MAXFILE+MAXEXT];
  char ext[MAXEXT];

  fp=fopen(filename,"rb");
  if (fp==NULL) {
    printf("File open error\n");
    return;
  }
  fseek(fp,0L,SEEK_END);
  filelen=ftell(fp);
  fseek(fp,0L,SEEK_SET);
  InitCRC(&CRC);
  /* transmit file name */
  fnsplit(filename,NULL,NULL,name,ext);
  strcat(name,ext);
  strupr(name);
  for (loop=0;loop<=strlen(name);loop++) {
    /* <= causes terminating 0 to be sent as well */
    SendByte(&parms,name[loop]);
  }
  printf("Filename=%s\n",name);
  /* transmit start address */
  SendByte(&parms,startaddr%256);
  SendByte(&parms,startaddr/256%256);
  printf("Address =%d\n",startaddr);
  /* transmit length */
  SendByte(&parms,filelen%256);
  SendByte(&parms,filelen/256%256);
  printf("Length  =%ld\n",filelen);
  /* transmit data */
  for (loop=0;loop<filelen;loop++) {
    ch=fgetc(fp);
    SendByte(&parms,ch);
    CalcCRC(&CRC,ch);
    printf("%5d\b\b\b\b\b",loop+1);
  }
  PreSendCRC(&CRC);
  SendByte(&parms,CRC&0xff);
  SendByte(&parms,CRC>>8);
  printf("\n");
  fclose(fp);
}

main(int argc,char *argv[])
{
  if (argc < 3) {
    printf("Usage:\n\n  IBM2APL <filename> <start address>\n\n");
    return 1;
  }

  timer_init();
  InitA2IParms(&parms);

  TransFile(argv[1],(unsigned int)strtol(argv[2],NULL,0));
  return 0;
}
