/*
 * Appleblossom - Portable Open-Source Apple IIe Emulator
 * Copyright (C) 2005 Jonathan Bettencourt (jonrelay)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include "a2vars.h"

/* a2iou.c */
extern unsigned char a2_iou_peek(unsigned short addr);
extern void a2_iou_poke(unsigned short addr, unsigned char ch);

void a2_mmu_reset(void)
{
	A2_80STORE = 0; 
	A2_RDAUXRAM = 0;
	A2_WRAUXRAM = 0;
	A2_INTROM = 0;
	A2_AUXZP = 0;
	A2_SLOT3ROM = 0;
	A2_PAGE2 = 0;
	A2_RDBSRAM = 0;
	A2_WRBSRAM = 1;
	A2_RWB2RAM = 0;
	A2_IOUDIS = 0;
	A2_IOSELECT = 0;
	A2_C069CNT = 0;
	A2_LASTRW = 0;
	A2_DBLREAD = 0;
}

void a2_clear_memory(void)
{
	unsigned short i;
	for (i=0; i<0xC000; i++) {
		A2_MAIN[i] = A2_AUX[i] = 0;
	}
	for (i=0; i<0x4000; i++) {
		A2_BS_RAM[i] = A2_ABS_RAM[i] = 0;
	}
}

int a2_read_rom(char * fn)
{
	FILE * f;
	int fl;
	f = fopen(fn, "rb");
	if (f) {
		fseek(f, 0, SEEK_END);
		fl = ftell(f);
		fseek(f, 0, SEEK_SET);
		if (fl < 0x4000) {
			fread(&A2_ROM[0x4000-fl], 1, fl, f);
		} else {
			if (fl > 0x4000) {
				fseek(f, (fl-0x4000), SEEK_SET);
			}
			fread(A2_ROM, 1, 0x4000, f);
		}
		fclose(f);
		return 1;
	}
	return 0;
}

void a2_read_periph_rom(char * fn, int card)
{
	FILE * f;
	f = fopen(fn, "rb");
	if (f) {
		fread(&A2_PERIPH_ROM[card*0x100], 1, 0x100, f);
		fclose(f);
	}
}

void a2_read_periph_xrom(char * fn, int card)
{
	FILE * f;
	f = fopen(fn, "rb");
	if (f) {
		fread(&A2_PERIPH_XROM[card*0x800], 1, 0x800, f);
		fclose(f);
	}
}

void a2_clear_periph_rom(void)
{
	unsigned short i;
	for (i=0; i<0x800; i++) {
		A2_PERIPH_ROM[i] = 0;
	}
	for (i=0; i<0x4000; i++) {
		A2_PERIPH_XROM[i] = 0;
	}
}

/* Useful Debugging Routines */

void a2_dump_memory(char * fn)
{
	FILE * f;
	f = fopen(fn, "wb");
	if (f) {
		fwrite(A2_MAIN, 1, 0xC000, f);
		fwrite(A2_BS_RAM, 1, 0x4000, f);
		fwrite(A2_AUX, 1, 0xC000, f);
		fwrite(A2_ABS_RAM, 1, 0x4000, f);
		fwrite(A2_ROM, 1, 0x4000, f);
		fwrite(A2_PERIPH_ROM, 1, 0x800, f);
		fwrite(A2_PERIPH_XROM, 1, 0x4000, f);
		fclose(f);
	}
}

void a2_dump_switches(char * fn)
{
	FILE * f;
	f = fopen(fn, "w");
	if (f) {
		fprintf(f, "%s = %d\n", "A2_KEYBOARD", A2_KEYBOARD);
		fprintf(f, "%s = %d\n", "A2_CASSETTE_OUT", A2_CASSETTE_OUT);
		fprintf(f, "%s = %d\n", "A2_CASSETTE_IN", A2_CASSETTE_IN);
		fprintf(f, "%s = %d\n", "A2_SPEAKER", A2_SPEAKER);
		fprintf(f, "%s = %d\n", "A2_STROBE", A2_STROBE);
		fprintf(f, "%s = %d\n", "A2_ANN0", A2_ANN0);
		fprintf(f, "%s = %d\n", "A2_ANN1", A2_ANN1);
		fprintf(f, "%s = %d\n", "A2_ANN2", A2_ANN2);
		fprintf(f, "%s = %d\n", "A2_ANN3", A2_ANN3);
		fprintf(f, "%s = %d\n", "A2_PB0", A2_PB0);
		fprintf(f, "%s = %d\n", "A2_PB1", A2_PB1);
		fprintf(f, "%s = %d\n", "A2_PB2", A2_PB2);
		fprintf(f, "%s = %d\n", "A2_PDL0_VALUE", A2_PDL0_VALUE);
		fprintf(f, "%s = %d\n", "A2_PDL1_VALUE", A2_PDL1_VALUE);
		fprintf(f, "%s = %d\n", "A2_PDL2_VALUE", A2_PDL2_VALUE);
		fprintf(f, "%s = %d\n", "A2_PDL3_VALUE", A2_PDL3_VALUE);
		fprintf(f, "%s = %d\n", "A2_PDL0_TIMER", A2_PDL0_TIMER);
		fprintf(f, "%s = %d\n", "A2_PDL1_TIMER", A2_PDL1_TIMER);
		fprintf(f, "%s = %d\n", "A2_PDL2_TIMER", A2_PDL2_TIMER);
		fprintf(f, "%s = %d\n", "A2_PDL3_TIMER", A2_PDL3_TIMER);
		fprintf(f, "%s = %d\n", "A2_VBL", A2_VBL);
		fprintf(f, "%s = %d\n", "A2_80VIDEO", A2_80VIDEO);
		fprintf(f, "%s = %d\n", "A2_ALTCHAR", A2_ALTCHAR);
		fprintf(f, "%s = %d\n", "A2_GRAPHICS", A2_GRAPHICS);
		fprintf(f, "%s = %d\n", "A2_MIXED", A2_MIXED);
		fprintf(f, "%s = %d\n", "A2_HIRES", A2_HIRES);
		fprintf(f, "%s = %d\n", "A2_PAGE2", A2_PAGE2);
		fprintf(f, "%s = %d\n", "A2_80STORE", A2_80STORE);
		fprintf(f, "%s = %d\n", "A2_RDAUXRAM", A2_RDAUXRAM);
		fprintf(f, "%s = %d\n", "A2_WRAUXRAM", A2_WRAUXRAM);
		fprintf(f, "%s = %d\n", "A2_INTROM", A2_INTROM);
		fprintf(f, "%s = %d\n", "A2_AUXZP", A2_AUXZP);
		fprintf(f, "%s = %d\n", "A2_SLOT3ROM", A2_SLOT3ROM);
		fprintf(f, "%s = %d\n", "A2_RDBSRAM", A2_RDBSRAM);
		fprintf(f, "%s = %d\n", "A2_WRBSRAM", A2_WRBSRAM);
		fprintf(f, "%s = %d\n", "A2_RWB2RAM", A2_RWB2RAM);
		fprintf(f, "%s = %d\n", "A2_IOUDIS", A2_IOUDIS);
		fprintf(f, "%s = %d\n", "A2_IOSELECT", A2_IOSELECT);
		fprintf(f, "%s = %d\n", "A2_C069CNT", A2_C069CNT);
		fprintf(f, "%s = %d\n", "A2_LASTRW", A2_LASTRW);
		fprintf(f, "%s = %d\n", "A2_DBLREAD", A2_DBLREAD);
		fclose(f);
	}
}

/* a2iou.c */

extern unsigned char a2_iou_peek(unsigned short addr);
extern void a2_iou_poke(unsigned short addr, unsigned char ch);

/* Peek and Poke */

unsigned char a2_mmu_peek(unsigned short addr)
{
	if (addr < 0x200) {
		if (A2_AUXZP) {
			return A2_AUX[addr];
		} else {
			return A2_MAIN[addr];
		}
	} else if (addr < 0xC000) {
		if (( A2_80STORE && ((addr>=0x400 && addr<0x800) || (A2_HIRES && A2_GRAPHICS && addr>=0x2000 && addr<0x4000)) )?A2_PAGE2:A2_RDAUXRAM) {
			return A2_AUX[addr];
		} else {
			return A2_MAIN[addr];
		}
	} else if (addr >= 0xE000) {
		if (A2_RDBSRAM) {
			return ( (A2_AUXZP)?(A2_ABS_RAM[addr - 0xC000]):(A2_BS_RAM[addr - 0xC000]) );
		} else {
			return A2_ROM[addr - 0xC000];
		}
	} else if (addr >= 0xD000) {
		if (A2_RDBSRAM) {
			if (A2_RWB2RAM) {
				return ( (A2_AUXZP)?(A2_ABS_RAM[addr - 0xC000]):(A2_BS_RAM[addr - 0xC000]) );
			} else {
				return ( (A2_AUXZP)?(A2_ABS_RAM[addr - 0xD000]):(A2_BS_RAM[addr - 0xD000]) );
			}
		} else {
			return A2_ROM[addr - 0xC000];
		}
	} else if (addr >= 0xC800) {
		if (addr == 0xCFFF) {
			A2_IOSELECT = 0;
		}
		if (A2_INTROM || A2_IOSELECT == 0 || A2_IOSELECT == 3) {
			return A2_ROM[addr - 0xC000];
		} else {
			return A2_PERIPH_XROM[addr - 0xC800 + (A2_IOSELECT * 0x800)];
		}
	} else if (addr >= 0xC100) {
		A2_IOSELECT = ((addr & 0x0700) >> 8);
		if (  A2_INTROM || ( (!A2_SLOT3ROM) && ((addr & 0xFF00) == 0xC300) )  ) {
			return A2_ROM[addr - 0xC000];
		} else {
			return A2_PERIPH_ROM[addr - 0xC000];
		}
	} else {
		A2_DBLREAD = (A2_LASTRW == addr);
		A2_LASTRW = addr;
		switch (addr) {
		case 0xC011: return (A2_RWB2RAM*128); break;
		case 0xC012: return (A2_RDBSRAM*128); break;
		case 0xC013: return (A2_RDAUXRAM*128); break;
		case 0xC014: return (A2_WRAUXRAM*128); break;
		case 0xC015: return (A2_INTROM*128); break;
		case 0xC016: return (A2_AUXZP*128); break;
		case 0xC017: return (A2_SLOT3ROM*128); break;
		case 0xC018: return (A2_80STORE*128); break;
		case 0xC01C: return (A2_PAGE2*128); break;
		case 0xC054: A2_PAGE2 = 0; break;
		case 0xC055: A2_PAGE2 = 1; break;
		case 0xC07E: return (A2_IOUDIS*128); break;
		
		case 0xC04F: case 0xC069:
			/*
			 * This is an emulator ID byte thing. It's actually $C04F;
			 * I don't know where I got $C069 from.
			 *   0xFE is Bernie II the Rescue
			 *   0x16 is Sweet16
			 *   0x4B is KEGS
			 *   0xAB is Appleblossom
			 * If an emulator doesn't support this it returns 0 or random garbage.
			 */
			switch (A2_C069CNT) {
			case 1:
				A2_C069CNT++;
				return 0xAB;
				break;
			case 2:
				A2_C069CNT++;
				return 0x10; /* version number - 1.0 */
				break;
			case 3:
				A2_C069CNT=0;
				return 0x6A; /* code branch - 'j' for original branch (Jonathan Bettencourt's) */
				break;
			default:
				A2_C069CNT=0;
				break;
			}
			break;
		
		case 0xC080: case 0xC084:
			A2_RDBSRAM = 1; A2_RWB2RAM = 1; A2_WRBSRAM = 0;
			break;
		case 0xC081: case 0xC085:
			if (A2_DBLREAD) { A2_RDBSRAM = 0; A2_RWB2RAM = 1; A2_WRBSRAM = 1; }
			else { A2_RDBSRAM = 0; A2_RWB2RAM = 1; A2_WRBSRAM = 0; }
			break;
		case 0xC082: case 0xC086:
			A2_RDBSRAM = 0; A2_RWB2RAM = 1; A2_WRBSRAM = 0;
			break;
		case 0xC083: case 0xC087:
			if (A2_DBLREAD) { A2_RDBSRAM = 1; A2_RWB2RAM = 1; A2_WRBSRAM = 1; }
			else { A2_RDBSRAM = 1; A2_RWB2RAM = 1; A2_WRBSRAM = 0; }
			break;
		case 0xC088: case 0xC08C:
			A2_RDBSRAM = 1; A2_RWB2RAM = 0; A2_WRBSRAM = 0;
			break;
		case 0xC089: case 0xC08D:
			if (A2_DBLREAD) { A2_RDBSRAM = 0; A2_RWB2RAM = 0; A2_WRBSRAM = 1; }
			else { A2_RDBSRAM = 0; A2_RWB2RAM = 0; A2_WRBSRAM = 0; }
			break;
		case 0xC08A: case 0xC08E:
			A2_RDBSRAM = 0; A2_RWB2RAM = 0; A2_WRBSRAM = 0;
			break;
		case 0xC08B: case 0xC08F:
			if (A2_DBLREAD) { A2_RDBSRAM = 1; A2_RWB2RAM = 0; A2_WRBSRAM = 1; }
			else { A2_RDBSRAM = 1; A2_RWB2RAM = 0; A2_WRBSRAM = 0; }
			break;
		default:
			return a2_iou_peek(addr);
		}
	}
	return 0;
}

void a2_mmu_poke(unsigned short addr, unsigned char ch)
{
	A2_DBLREAD = (A2_LASTRW == addr);
	A2_LASTRW = addr;
	if (addr < 0x200) {
		if (A2_AUXZP) {
			A2_AUX[addr] = ch;
		} else {
			A2_MAIN[addr] = ch;
		}
	} else if (addr < 0xC000) {
		if (( A2_80STORE && ((addr>=0x400 && addr<0x800) || (A2_HIRES && A2_GRAPHICS && addr>=0x2000 && addr<0x4000)) )?A2_PAGE2:A2_WRAUXRAM) {
			A2_AUX[addr] = ch;
		} else {
			A2_MAIN[addr] = ch;
		}
	} else if (addr >= 0xE000) {
		if (A2_WRBSRAM) {
			if (A2_AUXZP) {
				A2_ABS_RAM[addr - 0xC000] = ch;
			} else {
				A2_BS_RAM[addr - 0xC000] = ch;
			}
		} /* else ROM - don't write */
	} else if (addr >= 0xD000) {
		if (A2_WRBSRAM) {
			if (A2_RWB2RAM) {
				if (A2_AUXZP) {
					A2_ABS_RAM[addr - 0xC000] = ch;
				} else {
					A2_BS_RAM[addr - 0xC000] = ch;
				}
			} else {
				if (A2_AUXZP) {
					A2_ABS_RAM[addr - 0xD000] = ch;
				} else {
					A2_BS_RAM[addr - 0xD000] = ch;
				}
			}
		} /* else ROM - don't write */
	} else if (addr >= 0xC100) {
		if (addr == 0xCFFF) {
			A2_IOSELECT = 0;
		}
		/* ROM - don't write */
	} else {
		A2_DBLREAD = (A2_LASTRW == addr);
		A2_LASTRW = addr;
		switch (addr) {
		case 0xC000: A2_80STORE = 0; break;
		case 0xC001: A2_80STORE = 1; break;
		case 0xC002: A2_RDAUXRAM = 0; break;
		case 0xC003: A2_RDAUXRAM = 1; break;
		case 0xC004: A2_WRAUXRAM = 0; break;
		case 0xC005: A2_WRAUXRAM = 1; break;
		case 0xC006: A2_INTROM = 0; break;
		case 0xC007: A2_INTROM = 1; break;
		case 0xC008: A2_AUXZP = 0; break;
		case 0xC009: A2_AUXZP = 1; break;
		case 0xC00A: A2_SLOT3ROM = 0; break;
		case 0xC00B: A2_SLOT3ROM = 1; break;
		case 0xC054: A2_PAGE2 = 0; break;
		case 0xC055: A2_PAGE2 = 1; break;
		case 0xC07E: A2_IOUDIS = 1; break;
		case 0xC07F: A2_IOUDIS = 0; break;
		
		case 0xC04F: case 0xC069:
			A2_C069CNT=1;
			break;
		
		case 0xC080: case 0xC084:
			A2_RDBSRAM = 1; A2_RWB2RAM = 1; A2_WRBSRAM = 0;
			break;
		case 0xC081: case 0xC085:
			if (A2_DBLREAD) { A2_RDBSRAM = 0; A2_RWB2RAM = 1; A2_WRBSRAM = 1; }
			else { A2_RDBSRAM = 0; A2_RWB2RAM = 1; A2_WRBSRAM = 0; }
			break;
		case 0xC082: case 0xC086:
			A2_RDBSRAM = 0; A2_RWB2RAM = 1; A2_WRBSRAM = 0;
			break;
		case 0xC083: case 0xC087:
			if (A2_DBLREAD) { A2_RDBSRAM = 1; A2_RWB2RAM = 1; A2_WRBSRAM = 1; }
			else { A2_RDBSRAM = 1; A2_RWB2RAM = 1; A2_WRBSRAM = 0; }
			break;
		case 0xC088: case 0xC08C:
			A2_RDBSRAM = 1; A2_RWB2RAM = 0; A2_WRBSRAM = 0;
			break;
		case 0xC089: case 0xC08D:
			if (A2_DBLREAD) { A2_RDBSRAM = 0; A2_RWB2RAM = 0; A2_WRBSRAM = 1; }
			else { A2_RDBSRAM = 0; A2_RWB2RAM = 0; A2_WRBSRAM = 0; }
			break;
		case 0xC08A: case 0xC08E:
			A2_RDBSRAM = 0; A2_RWB2RAM = 0; A2_WRBSRAM = 0;
			break;
		case 0xC08B: case 0xC08F:
			if (A2_DBLREAD) { A2_RDBSRAM = 1; A2_RWB2RAM = 0; A2_WRBSRAM = 1; }
			else { A2_RDBSRAM = 1; A2_RWB2RAM = 0; A2_WRBSRAM = 0; }
			break;
		default:
			a2_iou_poke(addr, ch);
		}
	}
}
