/*
 * Appleblossom - Portable Open-Source Apple IIe Emulator
 * Copyright (C) 2005 Jonathan Bettencourt (jonrelay)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "a2vars.h"

/* a2disk.c */
extern unsigned char a2_disk_memacc(unsigned short addr, unsigned char ch);

/* a2hdisk.c */
extern unsigned char a2_hdisk_peek(unsigned short a);
extern void a2_hdisk_poke(unsigned short a, unsigned char ch);

/* a2serial.c */
extern unsigned char a2_serial_peek(unsigned short a);
extern void a2_serial_poke(unsigned short a, unsigned char ch);

/* a2clock.c */
extern void a2_read_rtc_input(void);
extern void a2_read_rtc_prodos(void);
extern void a2_rtc_install_driver(void);

/* Functions for Apple II Device I/O */

void a2_iou_reset(void)
{
	A2_KEYBOARD = 0;
	A2_80VIDEO = 0;
	A2_ALTCHAR = 0;
	A2_GRAPHICS = 0;
	A2_MIXED = 0;
	A2_HIRES = 0;
	A2_PAGE2 = 0;
	A2_CASSETTE_OUT = 0;
	A2_CASSETTE_IN = 0;
	A2_SPEAKER = 0;
	A2_STROBE = 0;
	A2_ANN0 = 0;
	A2_ANN1 = 0;
	A2_ANN2 = 1;
	A2_ANN3 = 1;
	A2_PB0 = 0;
	A2_PB1 = 0;
	A2_PB2 = 0;
	A2_PDL0_VALUE = 0;
	A2_PDL1_VALUE = 0;
	A2_PDL2_VALUE = 0;
	A2_PDL3_VALUE = 0;
	A2_PDL0_TIMER = 1;
	A2_PDL1_TIMER = 1;
	A2_PDL2_TIMER = 1;
	A2_PDL3_TIMER = 1;
	A2_PDL_CLOCKSTART = A2_CLOCK;
	A2_VBL = 0;
}

void a2_key_down(char ch)
{
	A2_KEYBOARD = (ch | 0x80);
}

void a2_pb_down(int i)
{
	switch (i) {
	case 0: A2_PB0 = 1; break;
	case 1: A2_PB1 = 1; break;
	case 2: A2_PB2 = 1; break;
	}
}

void a2_pb_up(int i)
{
	switch (i) {
	case 0: A2_PB0 = 0; break;
	case 1: A2_PB1 = 0; break;
	case 2: A2_PB2 = 0; break;
	}
}

void a2_set_paddle(int i, int v)
{
	switch (i) {
	case 0: A2_PDL0_VALUE = v; break;
	case 1: A2_PDL1_VALUE = v; break;
	case 2: A2_PDL2_VALUE = v; break;
	case 3: A2_PDL3_VALUE = v; break;
	}
}

void a2_cassette_input(void)
{
	A2_CASSETTE_IN = !A2_CASSETTE_IN;
}

void a2_reset_strobe(void)
{
	A2_STROBE = 0;
}

int a2_speaker_output(void) { return A2_SPEAKER; }
int a2_strobe_output(void) { return A2_STROBE; }
int a2_cassette_output(void) { return A2_CASSETTE_OUT; }

int a2_annunciator(int i)
{
	switch (i) {
	case 0: return A2_ANN0; break;
	case 1: return A2_ANN1; break;
	case 2: return A2_ANN2; break;
	case 3: return A2_ANN3; break;
	}
}

void a2_set_vbl(int i)
{
	A2_VBL = i;
}

void a2_iou_poll(void)
{
	int cnt = A2_CLOCK - A2_PDL_CLOCKSTART;
	if (cnt > (A2_PDL0_VALUE * 11)) { A2_PDL0_TIMER = 0; }
	if (cnt > (A2_PDL1_VALUE * 11)) { A2_PDL1_TIMER = 0; }
	if (cnt > (A2_PDL2_VALUE * 11)) { A2_PDL2_TIMER = 0; }
	if (cnt > (A2_PDL3_VALUE * 11)) { A2_PDL3_TIMER = 0; }
}

/* Peek and Poke */

unsigned char a2_iou_peek(unsigned short addr)
{
	switch (addr) {
	case 0xC000: return A2_KEYBOARD; break;
	case 0xC010: A2_KEYBOARD &= 0x7F; return 0; break;
	case 0xC019: return A2_VBL*128; break;
	case 0xC01A: return (!A2_GRAPHICS)*128; break;
	case 0xC01B: return A2_MIXED*128; break;
	case 0xC01C: return A2_PAGE2*128; break;
	case 0xC01D: return A2_HIRES*128; break;
	case 0xC01E: return A2_ALTCHAR*128; break;
	case 0xC01F: return A2_80VIDEO*128; break;
	case 0xC020: A2_CASSETTE_OUT = !A2_CASSETTE_OUT; break;
	case 0xC030: A2_SPEAKER = !A2_SPEAKER; break;
	case 0xC040: A2_STROBE = 1; break;
	case 0xC050: A2_GRAPHICS = 1; break;
	case 0xC051: A2_GRAPHICS = 0; break;
	case 0xC052: A2_MIXED = 0; break;
	case 0xC053: A2_MIXED = 1; break;
	case 0xC054: A2_PAGE2 = 0; break;
	case 0xC055: A2_PAGE2 = 1; break;
	case 0xC056: A2_HIRES = 0; break;
	case 0xC057: A2_HIRES = 1; break;
	case 0xC058: A2_ANN0 = 0; break;
	case 0xC059: A2_ANN0 = 1; break;
	case 0xC05A: A2_ANN1 = 0; break;
	case 0xC05B: A2_ANN1 = 1; break;
	case 0xC05C: A2_ANN2 = 0; break;
	case 0xC05D: A2_ANN2 = 1; break;
	case 0xC05E: A2_ANN3 = 0; break;
	case 0xC05F: A2_ANN3 = 1; break;
	case 0xC060: return A2_CASSETTE_IN*128; break;
	case 0xC061: return A2_PB0*128; break;
	case 0xC062: return A2_PB1*128; break;
	case 0xC063: return A2_PB2*128; break;
	case 0xC064: return A2_PDL0_TIMER*128; break;
	case 0xC065: return A2_PDL1_TIMER*128; break;
	case 0xC066: return A2_PDL2_TIMER*128; break;
	case 0xC067: return A2_PDL3_TIMER*128; break;
	case 0xC070:
		A2_PDL0_TIMER = 1;
		A2_PDL1_TIMER = 1;
		A2_PDL2_TIMER = 1;
		A2_PDL3_TIMER = 1;
		A2_PDL_CLOCKSTART = A2_CLOCK;
		break;
	case 0xC07F: return (!A2_ANN3)*128; break;
	default:
		switch (addr & 0xFFF0) {
		case 0xC090: /* slot 1 */
		case 0xC0A0: /* slot 2 */
			return a2_serial_peek(addr);
			break;
		case 0xC0E0: /* slot 6 */
			return a2_disk_memacc(addr, 0);
			break;
		case 0xC0F0: /* slot 7 */
			return a2_hdisk_peek(addr);
			break;
		}
		break;
	}
	return 0;
}

void a2_iou_poke(unsigned short addr, unsigned char ch)
{
	A2_DBLREAD = (A2_LASTRW == addr)?1:0;
	A2_LASTRW = addr;
	switch (addr) {
	case 0xC00C: A2_80VIDEO = 0; break;
	case 0xC00D: A2_80VIDEO = 1; break;
	case 0xC00E: A2_ALTCHAR = 0; break;
	case 0xC00F: A2_ALTCHAR = 1; break;
	case 0xC010: A2_KEYBOARD &= 0x7F; break;
	case 0xC020: A2_CASSETTE_OUT = !A2_CASSETTE_OUT; break;
	case 0xC030: A2_SPEAKER = !A2_SPEAKER; break;
	case 0xC040: A2_STROBE = 1; break;
	case 0xC050: A2_GRAPHICS = 1; break;
	case 0xC051: A2_GRAPHICS = 0; break;
	case 0xC052: A2_MIXED = 0; break;
	case 0xC053: A2_MIXED = 1; break;
	case 0xC054: A2_PAGE2 = 0; break;
	case 0xC055: A2_PAGE2 = 1; break;
	case 0xC056: A2_HIRES = 0; break;
	case 0xC057: A2_HIRES = 1; break;
	case 0xC058: A2_ANN0 = 0; break;
	case 0xC059: A2_ANN0 = 1; break;
	case 0xC05A: A2_ANN1 = 0; break;
	case 0xC05B: A2_ANN1 = 1; break;
	case 0xC05C: A2_ANN2 = 0; break;
	case 0xC05D: A2_ANN2 = 1; break;
	case 0xC05E: A2_ANN3 = 0; break;
	case 0xC05F: A2_ANN3 = 1; break;
	case 0xC070:
		A2_PDL0_TIMER = 1;
		A2_PDL1_TIMER = 1;
		A2_PDL2_TIMER = 1;
		A2_PDL3_TIMER = 1;
		A2_PDL_CLOCKSTART = A2_CLOCK;
		break;
	case 0xC077:
		/* This is an I/O memory location used just by my emulator. */
		switch (ch) {
		case 0xC1:
			a2_rtc_install_driver();
			break;
		case 0xCC:
			a2_read_rtc_input();
			break;
		case 0xCF:
			a2_read_rtc_prodos();
			break;
		}
		break;
	default:
		switch (addr & 0xFFF0) {
		case 0xC090: /* slot 1 */
		case 0xC0A0: /* slot 2 */
			a2_serial_poke(addr, ch);
			break;
		case 0xC0E0: /* slot 6 */
			a2_disk_memacc(addr, ch);
			break;
		case 0xC0F0: /* slot 7 */
			a2_hdisk_poke(addr, ch);
			break;
		}
		break;
	}
}
