/*
 *  Apple II emulator originally by Alexander Jean-Claude Bottema (C) 1994
 *
 *  SVGA frontend to the emulator.
 *
 *  The routines in this file were moved from the misc.c file,
 *  originally by Alexander Jean-Claude Bottema.
 *
 *  $Id: svideo.c,v 1.4 1998/08/25 03:00:02 chernabog Exp $
 *
 *  MODIFICATION HISTORY
 *   v0.3 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jan 1997.
 *   v0.4 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jun 1997.
 *   v0.5 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Feb 1998.
 *   v0.6 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Aug 1998.
 *     This code has nothing to do with my employer, GTE Internetworking,
 *     BBN Technologies.  It was written completely on my own time and on
 *     my own machine.
 *
 */

#include <signal.h>
#include <vga.h>
#include <vgakeyboard.h>
#include <string.h>
#include <stdio.h>
#include <linux/kd.h>/* for ioctls */
#include <termios.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include "video.h"
#include "apple2.h"
#include "misc.h"
#include "keys.h"

/* I need at least 560x192 for 80col.  it seems that direct linear
 * framebuffer access to anything in the realm of 640x480x256 not
 * standard with vga/svga?
 * Memory sizes are hardcode here.  ugly ugly ugly.
 */

unsigned char	odd_colors[2] = {COLOR_LIGHT_PURPLE, COLOR_LIGHT_BLUE};
unsigned char	even_colors[2] = {COLOR_LIGHT_GREEN, COLOR_LIGHT_RED};

static unsigned char	*svga_GM;			/* SVGA base address of graphic area */
static unsigned char	vga_mem_page_0[64000];		/* page0 framebuffer */
static unsigned char	vga_mem_page_1[64000];		/* page1 framebuffer */


/* -------------------------------------------------------------------------
    c_setpage(p):    Set SVGA 64k page and update GM
                     (Graph Memory base address)
   ------------------------------------------------------------------------- */
void c_setpage(int p)
{
    svga_GM = vga_getgraphmem();
    GM = (crnt_visual_svga_page == p) ? svga_GM :
	((p == 0) ? vga_mem_page_0 : vga_mem_page_1);
    crnt_active_svga_page = p;
}

/* -------------------------------------------------------------------------
    c_setscreen(p):    Switch to screen page p
   ------------------------------------------------------------------------- */
void c_setscreen(int m)
{
    if (m != crnt_visual_svga_page) {
	memcpy( ((crnt_visual_svga_page == 0) ?
		vga_mem_page_0 : vga_mem_page_1),
		svga_GM, 64000 );
	memcpy( svga_GM, ((m == 0) ? vga_mem_page_0 : vga_mem_page_1),
		64000 );
	GM = (crnt_active_svga_page == m) ? svga_GM :
	    ((crnt_active_svga_page == 0) ?
	     vga_mem_page_0 : vga_mem_page_1);
	crnt_visual_svga_page = m;
    }
}

/* -------------------------------------------------------------------------
    c_initialize_colors():    Initialize color palette
   ------------------------------------------------------------------------- */

static void c_initialize_colors()
{

    static unsigned int col2[ 3 ] = { 27, 40, 62 };

    int				i, j;

    /* align the palette for hires graphics */
    for (i = 0; i < 8; i++)
	for (j = 0; j < 3; j++)
	    vga_setpalette( j+i*3+32, (i & 1) ? col2[ j ] : 0,
                                      (i & 2) ? col2[ j ] : 0,
                                      (i & 4) ? col2[ j ] : 0 );

    vga_setpalette( COLOR_FLASHING_BLACK, 0, 0, 0 );
    vga_setpalette( COLOR_FLASHING_WHITE, 63, 63, 63 );

    /* dhires colors are bass-ackwards because it's easier for me to
       deal with the endianness. */
    vga_setpalette( 0x0, 0, 0, 0 ); 		/* Black 		*/
    vga_setpalette( 0x1, 0, 0, 160 );		/* Magenta <med blue>	*/
    vga_setpalette( 0x2, 0, 100, 0 );		/* Brown <dark green>	*/
    vga_setpalette( 0x3, 28, 134, 255 );	/* Orange <dodger blue>	*/
    vga_setpalette( 0x4, 165, 21, 10 );		/* Dark Green <brown>	*/
    vga_setpalette( 0x5, 165, 42, 42);		/* Gray2 <light gray>	*/
    vga_setpalette( 0x6, 0, 255, 0 );		/* Green <Green>	*/
    vga_setpalette( 0x7, 160, 255, 160 );	/* Yellow <Aqua>	*/
    vga_setpalette( 0x8, 112, 192, 140 );	/* Dark Blue <magenta>	*/
    vga_setpalette( 0x9, 105, 205, 170 );	/* Purple <Purple>	*/
    vga_setpalette( 0xa, 90, 90, 90 );		/* Gray1 <dark gray>	*/
    vga_setpalette( 0xb, 131, 111, 250 );	/* Pink <slateblue>	*/
    vga_setpalette( 0xc, 255, 198, 203 );	/* Medium Blue <Orange>	*/
    vga_setpalette( 0xd, 255, 167, 165 );	/* Light Blue <Pink>	*/
    vga_setpalette( 0xe, 255, 255, 0 );		/* Aqua <Yellow>	*/
    vga_setpalette( 0xf, 255, 255, 255 );	/* White		*/

    /* lores colors (<<4) */
    vga_setpalette(0x10, 112, 192, 140 );	/* Dark Blue <magenta>	*/
    vga_setpalette(0x20, 0,	0,	160);/*Dark Blue*/
    vga_setpalette(0x30, 105, 205, 170 );	/* Purple <Purple>	*/
    vga_setpalette(0x40, 0,	100,	0);/*Dark Green*/
    vga_setpalette(0x50, 165, 42, 42);		/* Gray2 <light gray>	*/
    vga_setpalette(0x60, 28,	134,	255);/*Med Blue*/
    vga_setpalette(0x70, 65,	255,	255);/*Light Blue*/
    vga_setpalette(0x80, 165, 21, 10 );		/* Dark Green <brown>	*/
    vga_setpalette(0x90, 255,	66,	0);/*Orange*/
    vga_setpalette(0xa0, 90,	90,	90);/*Gray2*/
    vga_setpalette(0xb0, 255, 167, 165 );	/* Light Blue <Pink>	*/
    vga_setpalette(0xc0, 0,	255,	0);/*Green*/
    vga_setpalette(0xd0, 255,	255,	0);/*Yello*/
    vga_setpalette(0xe0, 160,	255,	160);/*Aqua*/
    vga_setpalette(0xf0, 255,	255,	255);/*White*/
}

static void c_poll_keyboard_input() {
    keyboard_update();/*svgalib calls event handler*/
    c_periodic_update();
    signal(SIGVTALRM, c_poll_keyboard_input);/* reinstall */
}

/* -------------------------------------------------------------------------
    c_initialize_keyboard()
   ------------------------------------------------------------------------- */

static void c_initialize_keyboard()
{
    if (keyboard_init()) {
    	printf("Error: Could not switch to RAW keyboard mode.\n");
	exit(-1);
    }

    signal( SIGVTALRM, c_poll_keyboard_input );
    signal( SIGALRM, c_shutdown_video );/*HACK - wtf?*/

    keyboard_translatekeys(DONT_CATCH_CTRLC);
/*    alarm( 120 ); */

    c_keyboard_on();
}

static struct termios origtermios;
static int origkbmode = -1;

void c_keyboard_on()
{
    static struct itimerval tval, old;
    if (keyboard_init()) {
	printf("Error: Could not switch to RAW keyboard mode.\n");
	exit(-1);
    }

    tval.it_interval.tv_sec = 0L;
    tval.it_interval.tv_usec = TIMER_DELAY;
    tval.it_value.tv_sec = 0L;
    tval.it_value.tv_usec = TIMER_DELAY;

    setitimer( ITIMER_VIRTUAL, &tval, &old );/*set up virtual timer*/
    keyboard_seteventhandler( c_read_raw_key );
}

void c_keyboard_off()
{
    static struct itimerval tval, old;
    static int kbmode;
    keyboard_close();
    /* HACK: it seems that keyboard_close() doesn't always reset the keyboard
     * state to normal ASCII mode.
     */
    if (ioctl(fileno(stdin), KDGKBMODE, &kbmode)) {
	printf("Cannot get keyboard mode!\n");
	c_shutdown_video();
    }
    if (kbmode == K_MEDIUMRAW) {
	if (ioctl(fileno(stdin), KDSKBMODE, origkbmode)) {
	    printf("Cannot set keyboard mode!\n");
	    c_shutdown_video();
	}
    }

    tval.it_interval.tv_sec = 0L;
    tval.it_interval.tv_usec = 0L;
    tval.it_value.tv_sec = 0L;
    tval.it_value.tv_usec = 0L;

    setitimer( ITIMER_VIRTUAL, &tval, &old );
}

void c_flash_cursor(int on) {
    if (!on) {
	vga_setpalette( COLOR_FLASHING_BLACK, 0, 0, 0 );
	vga_setpalette( COLOR_FLASHING_WHITE, 63, 63, 63 );
    } else {
	vga_setpalette( COLOR_FLASHING_BLACK, 63, 63, 63 );
	vga_setpalette( COLOR_FLASHING_WHITE, 0, 0, 0 );	
    }
}

/* -------------------------------------------------------------------------
    c_initialize_video()
   ------------------------------------------------------------------------- */
void c_initialize_video() {
    /* I'm forcing VGA mode since it seems to be supported by
     * the lowest common denominator cards.
     */
    printf("Using standard VGA 320x200 mode.\n");
    printf("Press RETURN to continue...");
    getchar();

    tcgetattr(fileno(stdin), &origtermios);
    if (ioctl(fileno(stdin), KDGKBMODE, &origkbmode)) {
	printf("Cannot get keyboard mode!\n");
	c_shutdown_video();
    }

    vga_setchipset( VGA );
    if (vga_init()) {
	printf("Cannot initialize svgalib!\n");
	c_shutdown_video();
    }

    signal(SIGTERM, c_shutdown_video );
    signal(SIGSEGV, c_shutdown_video );
    signal(SIGBUS, c_shutdown_video );
    signal(SIGILL, c_shutdown_video );
    signal(SIGALRM, c_shutdown_video );

    vga_setmode( G320x200x256 );

/*    vga_claimvideomemory( 131072 );*/
    c_initialize_keyboard();
    svga_GM = GM = vga_getgraphmem();
    c_initialize_colors();
}

void c_shutdown_video() {
    vga_setmode(TEXT);
    keyboard_close();
    exit( 0 );
}

#define my_pixel(x, y, c) GM[(y)*320+(x)]=(c)

void c_clear_video() {
    int		x, y;

    c_setpage( 1 );

    for (y = 0; y < 200; y++)
	for (x = 0; x < 320; x++)
            my_pixel( x, y, 0 );

    c_setpage( 0 );

    for (y = 0; y < 200; y++)
	for (x = 0; x < 320; x++)
            my_pixel( x, y, 0 );
}

static int _mygetch() {
    static int ch;
    return (read(fileno(stdin), &ch, 1) == 1) ? ch : -1;
}

/* -------------------------------------------------------------------------
 * c_mygetch():
 * 	Get next menu character (keyboard should be in cooked mode).
 * 	This busy loops regardless of what block is.
 * 	return: -1 (no input) or input character
 * ------------------------------------------------------------------------- */

int c_mygetch(int block) {
    static int ch;
    ch = _mygetch();
    if (ch == 27) {
	ch = _mygetch();
	if (ch == '[') {
	    ch = _mygetch();
	    switch (ch) {
		case 'A':
		    return kUP;
		case 'B':
		    return kDOWN;
		case 'C':
		    return kRIGHT;
		case 'D':
		    return kLEFT;
		case '6': 
		    ch = _mygetch();
		    if (ch == '~')
			return kPGDN;
		    break;
		case '5': 
		    ch = _mygetch();
		    if (ch == '~')
			return kPGUP;
		    break;
		case '1': 
		    ch = _mygetch();
		    if (ch == '~')
			return kHOME;
		    break;
		case '4': 
		    ch = _mygetch();
		    if (ch == '~')
			return kEND;
		    break;
	    }
	}
	else
	    return kESC;
    }
    return ch;
}


/*
 * this is a noop since svga version has direct (mostly) framebuffer access.
 */
void c_video_refresh() {}
