/*
 *  Apple II debugger routines
 *
 *  by Aaron Culliney - chernabog@baldmountain.bbn.com - (C) 1998
 *
 *  debugger.c - Main debugger command routines.
 *
 *  $Id: debugger.c,v 1.14 1998/08/23 17:12:51 chernabog Exp $
 *
 *   v0.3 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jan 1997.
 *   v0.4 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jun 1997.
 *   v0.5 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Feb 1998.
 *   v0.6 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Aug 1998.
 *	This code has nothing to do with my employer, GTE Internetworking,
 *	BBN Technologies.  It was written completely on my own time and on
 *	my own machine.
 *
 */

#ifdef DEBUGGER
#include "debug.h"
#include "apple2.h"
#include "misc.h"
#include "keys.h"
#include "video.h"
#include "disk.h"
#include "interface.h"

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>


/* debugger globals */
static unsigned char screen[SCREEN_Y][SCREEN_X] =
    { "||||||||||||||||||||||||||||||||||||||||",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "||||||||||||||||||||||||||||||||||||||||" };

static char command_buf[BUF_Y][BUF_X];	/* command line prompt */
char lexbuf[BUF_X+2];			/* comman line to be flex'ed */

int debug_EA;				/* debugger's effective addrs */
int debug_SP;				/* debugger's stack pointer */
unsigned char current_opcode;
unsigned char debug_X;			/* debugger's X register */
unsigned char debug_Y;			/* debugger's Y register */
unsigned char debug_A;			/* debugger's accumulator */
unsigned char debug_F;			/* debugger's flag register */

int op_breakpoints[256];		/* opcode breakpoints */

unsigned char watch_write;              /* watchpoint read/write access? */
unsigned char watch_olddata;            /* old byte */
unsigned char watch_data;               /* new byte */
int watch_addrs;                        /* address of tripped watchpoint */

/* in debug.l */
extern int debuglex();			/* yylex() */
extern void init_lex(char *buf, int size);

/* -------------------------------------------------------------------------
    c_get_current_rambank (addrs) - return the current ram bank for addrs.
	returns 0 = bank 0
		1 = bank 1
   ------------------------------------------------------------------------- */
int c_get_current_rambank(int addrs) {
#ifdef APPLE_IIE
    if ((addrs >= 0x200) && (addrs < 0xD000)) {


	/* SLOTROM */
	if ((addrs >= 0xC100) && (addrs < 0xD000)) {
	    /* expansion rom */
	    if ((addrs >= 0xC800) && (addrs < 0xD000))
		return (c8rom_flag & 0x80)>>7;

	    /* if SLOTCXROM, then internal rom (regardless of
               SLOTC3ROM setting).  HACK is this correct? */
	    if (cxrom_flag & 0x80)
		return (cxrom_flag & 0x80)>>7;

	    /* slot 3 rom */
	    if ((addrs >= 0xC300) && (addrs < 0xC400)) {
		return (c3rom_flag & 0x80)>>7;
	    }

	    return 0;	/* peripheral rom */
	}

	/* text page 1 with 80STORE on */
	if (((addrs >= 0x400) && (addrs < 0x800)) && 
	    eighty_store_flag)
	{
	    return vmode_page2;
	}

	/* hires page 1 with 80STORE and HIRES on */
	if (((addrs >= 0x2000) && (addrs < 0x4000)) && 
	    eighty_store_flag && vmode_hires)
	{
	    return vmode_page2;
	}

	/* otherwise return RAMRD flag */
	return (ramrd_flag & 0x80)>>7;
    }

    /* executing in ALTZP space. */
    return (altzp_flag & 0x80)>>7;

#else
    return 0;
#endif
}

/* -------------------------------------------------------------------------
    get_current_opcode () - returns the opcode from the address that
        the PC is currently reading from.
	returns 0 = bank 0
		1 = bank 1
   ------------------------------------------------------------------------- */
unsigned char get_current_opcode() {
    int bank = c_get_current_rambank(debug_PC);
    int lcbank = 0;

    /* main RAM */
    if (debug_PC < 0xD000) {
	return apple_ii_64k[bank][debug_PC];
    }

    /* LC RAM */
    if (debug_PC >= 0xE000) {
	if (language_card_read)
	    return language_card[bank][debug_PC-0xE000];
	else
	    return apple_ii_64k[bank][debug_PC];
    }

    /* LC BANK RAM */
    if (language_current_bank)
	lcbank = 0x1000;

    if (language_card_read)
	return language_banks[bank][debug_PC-0xD000+lcbank];
    else
	return apple_ii_64k[bank][debug_PC];
}

/* -------------------------------------------------------------------------
    dump_mem () - hexdump of memory to debug console
	we DO NOT wrap the display : 0xffff -> 0x0 (programs can't wrap)
   ------------------------------------------------------------------------- */

void dump_mem(int addrs, int len, int lc, int do_ascii, int rambank) {
    int i, j, mod, end;
    unsigned char op;
    int orig_addrs = addrs;		/* address for display */

    /* check which rambank */
    if (rambank == -1) {
	rambank = c_get_current_rambank(addrs);
    }

    if (!lc && language_card_read && (addrs >= 0xd000))	/* read lc anyway */
	lc = language_current_bank+1;

    if ((addrs < 0) || (addrs > 0xffff)) {
	addrs = debug_PC;
	orig_addrs = addrs;
    }
    if (lc) {
	orig_addrs = addrs;
	if ((addrs >= 0xd000) && (addrs <= 0xffff)) addrs -= 0xd000;
	if ((addrs < 0) || (addrs > 0x2fff)) addrs = 0;
    }

    if ((len < 1) || (len > 256)) len = 256;
    if (do_ascii && (len > 128)) len = 128;


    /* save hexdump in second_buf */
    end = (lc) ? 0x3000 : 0x10000;
    for (i = num_buffer_lines-1, j = 0; ((j < len) && (addrs+j < end)); j++) {

	mod = j % (16 >> do_ascii);

	if (lc) {
	    op = (addrs+j >= 0x1000) ? language_card[rambank][(addrs+j)-0x1000]
		: (lc == 1) ? language_banks[rambank][addrs+j]
		: language_banks[rambank][0x1000+addrs+j] ;
	}
	else op = apple_ii_64k[rambank][addrs+j];

	if (!mod) {
	    if (++i) {
		for (mod=0; mod<BUF_X; mod++)
		    if (second_buf[i-1][mod] == '\0')
			second_buf[i-1][mod] = ' ';
	    }
	    memset(second_buf[i], ' ', BUF_X); mod = 0;
	    sprintf(second_buf[i], "%04X:%02X", orig_addrs+j, op);
	    if (do_ascii) sprintf(second_buf[i]+23, "%c",
				  ((op&0x7f) > 31) ? (op&0x7f) : '.');
	    continue;
	}
	sprintf(second_buf[i]+5+mod*2, "%02X", op);
	if (do_ascii) sprintf(second_buf[i]+23+mod, "%c",
			      ((op&0x7f) > 31) ? (op&0x7f) : '.');
    }
    for (mod=0; mod<BUF_X; mod++)
	if (second_buf[i][mod] == '\0') second_buf[i][mod] = ' ';
    num_buffer_lines = i + 1;
}


/* -------------------------------------------------------------------------
    search_mem () - search memory for bytes
   ------------------------------------------------------------------------- */
void search_mem(char *hexstr, int lc, int rambank) {
    int i = 0, j = 0, end, op;
    static char scratch[3];
    unsigned char byte;

    end = (lc) ? 0x3000 : 0x10000;

    /* check which rambank for debug_PC */
    if (rambank == -1) {
	rambank = c_get_current_rambank(debug_PC);
    }

    /* iterate over memory */
    for (i = 0; i < end; i++) {
	strncpy(scratch, hexstr+j, 2);		/* extract a byte */
	byte = (unsigned char) strtol(scratch, (char**)NULL, 16);

	if (lc) {
	    op = (i >= 0x1000) ? language_card[rambank][i-0x1000]
		: (lc == 1) ? language_banks[rambank][i]
		            : language_banks[rambank][0x1000+i] ;
	}
	else op = apple_ii_64k[rambank][i];

	if (byte == op) {		/* matched byte? */
	    ++j;				/* increment */
	    if (!isxdigit(*(hexstr+j))) {	/* end of bytes? */
		/* then we found a match */
		sprintf(second_buf[num_buffer_lines], "%04X: %s",
			i-(j>>1), hexstr);
		num_buffer_lines = (num_buffer_lines + 1) % (BUF_Y-2);
		j = 0; continue;
	    }
	    ++j;
	    if (!isxdigit(*(hexstr+j))) {	/* end of bytes? */
		/* then we found a match */
		sprintf(second_buf[num_buffer_lines], "%04X: %s",
			i-(j>>1)+1, hexstr);
		num_buffer_lines = (num_buffer_lines + 1) % (BUF_Y-2);
		j = 0; continue;
	    }
	    continue;
	}
	j = 0;
    }
}


/* -------------------------------------------------------------------------
    set_mem () - write to memory.  we use the do_write_memory routine
    to "safely" set memory...
   ------------------------------------------------------------------------- */
void set_mem(int addrs, char *hexstr) {
    static char scratch[3];

    if ((addrs < 0) || (addrs > 0xffff)) {
	sprintf(second_buf[num_buffer_lines++], "invalid address");
	return;
    }

    while (*hexstr) {
	strncpy(scratch, hexstr, 2);
	watch_data = (unsigned char) strtol(scratch, (char**)NULL, 16);
	debug_scratch = addrs;

	/* call the set_memory routine, which knows how to route the
           request */
	do_write_memory();

	++hexstr;
	if (!*hexstr) break;
	++hexstr;
	if (++addrs > 0xffff) return;  /* don't overwrite memory */
    }
}


/* -------------------------------------------------------------------------
    set_lc_mem () - specifically write to apple II language card RAM memory
   ------------------------------------------------------------------------- */
void set_lc_mem(int addrs, int lcbank, char *hexstr) {
    static char scratch[3];

    if ((addrs >= 0xd000) && (addrs <= 0xffff)) addrs -= 0xd000;
    if ((addrs < 0) || (addrs > 0x2fff)) {
	sprintf(second_buf[num_buffer_lines++], "invalid LC address");
	return;
    }

    while (*hexstr) {
	strncpy(scratch, hexstr, 2);
	watch_data = (unsigned char) strtol(scratch, (char**)NULL, 16);
	watch_addrs = (addrs < 0x1000) ? lcbank-1 : -1; /* -1, 0, 1 */
	debug_scratch = addrs; debug_scratch += 0xd000;

	/* call the set_memory routine */
	do_write_lc();

	++hexstr;
	if (!*hexstr) break;
	++hexstr;
	if (++addrs > 0x2fff) return;
    }
}

/* -------------------------------------------------------------------------
    bload () - bload file data into emulator.  this is essentially the
    same as the set_mem routine.  we use the do_write_memory routine to
    "safely" set memory...
   ------------------------------------------------------------------------- */
void bload(FILE *f, char *name, int addrs) {
    unsigned char *hexstr = NULL;
    int len = -1;

    if ((addrs < 0) || (addrs > 0xffff)) {
	sprintf(second_buf[num_buffer_lines++], "invalid address");
	return;
    }

    while ((len = fread(temp, 1, 4096, f))) {
	hexstr = temp;
	for (; len > 0; len--) {
	    watch_data = *hexstr;
	    debug_scratch = addrs;

	    /* call the set_memory routine, which knows how to route
	       the request */
	    do_write_memory();

	    ++hexstr;
	    if (++addrs > 0xffff) return;  /* don't overwrite memory */
	}
    }
    sprintf(second_buf[num_buffer_lines++], "bloaded: %s", name);
}


/* -------------------------------------------------------------------------
    disasm () - disassemble instructions
	we DO NOT wrap the display : 0xffff -> 0x0
   ------------------------------------------------------------------------- */

void disasm(int addrs, int len, int lc, int rambank) {
    static char fmt[64];
    unsigned char op;
    char arg1, arg2;
    int i, j, k, end, orig_addrs = addrs;

    /* check which rambank for debug_PC */
    if (rambank == -1) {
	rambank = c_get_current_rambank(addrs);
    }

    /* read lc anyway (not implicitly specified) */
    if (!lc && language_card_read && (addrs >= 0xd000))
	lc = language_current_bank+1;

    /* handle invalid address request */
    if ((addrs < 0) || (addrs > 0xffff)) {
	addrs = debug_PC;
	orig_addrs = addrs;
    }

    /* disassembling from language card */
    if (lc) {
	if ((addrs >= 0xd000) && (addrs <= 0xffff)) addrs -= 0xd000;
	if ((addrs < 0) || (addrs > 0x2fff)) addrs = 0;
    }

    if (len > BUF_Y - 2) len = BUF_Y - 2 - num_buffer_lines;

    /* save hexdump in second_buf */
    end = (lc) ? 0x3000 : 0x10000;
    for (i = num_buffer_lines, j = addrs, k=orig_addrs;
	 ((i<len) && (j<end)); i++, j++)
    {

	if (lc)
	    op = (j >= 0x1000) ? language_card[rambank][j-0x1000]
	        : (lc == 1) ? language_banks[rambank][j]
		            : language_banks[rambank][0x1000+j];
	else
	    op = apple_ii_64k[rambank][j];

	switch ((int)(*opcodes)[op].numargs) {
	case 0:					/* no args */
	    sprintf(second_buf[i], "/%02X/%04X: %02X      %s",
		    rambank, k++, op, (*opcodes)[op].fmt);
	    break;

	case 1:					/* 1 arg */
	    if (k == 0xffff) {
		num_buffer_lines = i;
		return;
	    }

	    if (lc)
		arg1 = (j >= 0x1000) ? language_card[rambank][++j-0x1000]
	            : (lc == 1) ? language_banks[rambank][++j]
		    : language_banks[rambank][++j+0x1000];
	    else
		arg1 = apple_ii_64k[rambank][++j];

	    sprintf(fmt, "/%02X/%04X: %02X%02X    %s",
		    rambank, k, op, (unsigned char)arg1, (*opcodes)[op].fmt);
	    sprintf(second_buf[i], fmt, (unsigned char)arg1);
	    k+=2;
	    break;

	case 2:					/* 2 args */
	    if (k >= 0xfffe) {
		num_buffer_lines = i;
		return;
	    }

	    if (lc) {
		arg1 = (j >= 0x1000) ? language_card[rambank][++j-0x1000]
	            : (lc == 1) ? language_banks[rambank][++j]
		    : language_banks[rambank][++j+0x1000];
		arg2 = (j >= 0x1000) ? language_card[rambank][++j-0x1000]
	            : (lc == 1) ? language_banks[rambank][++j]
		    : language_banks[rambank][++j+0x1000];
	    }
	    else {
		arg1 = apple_ii_64k[rambank][++j];
		arg2 = apple_ii_64k[rambank][++j];
	    }

	    sprintf(fmt, "/%02X/%04X: %02X%02X%02X  %s",
		    rambank, k, op, (unsigned char)arg1, (unsigned char)arg2,
		    (*opcodes)[op].fmt);
	    sprintf(second_buf[i], fmt, (unsigned char)arg2,
		    (unsigned char)arg1);
	    k+=3;
	    break;

	case 3:					/* special branch */
	    if (k == 0xffff) {
		num_buffer_lines = i;
		return;
	    }

	    if (lc)
		arg1 = (j >= 0x1000) ? language_card[rambank][++j-0x1000]
	            : (lc == 1) ? language_banks[rambank][++j]
	            : language_banks[rambank][++j+0x1000];
	    else
		arg1 = apple_ii_64k[rambank][++j];

	    sprintf(fmt, "/%02X/%04X: %02X%02X    %s",
		    rambank, k, op, (unsigned char)arg1, (*opcodes)[op].fmt);
	    if (arg1 < 0) {
	      sprintf(second_buf[i], fmt,
		      k + arg1 + 2, (unsigned char)((~arg1) + 1));
	      second_buf[i][29] = '-';		/* branch back */
	    }
	    else {
	      sprintf(second_buf[i], fmt,
		      k + arg1 + 2, (unsigned char)arg1);
	      second_buf[i][29] = '+';		/* branch ahead */
	    }
	    k+=2;
	    break;

	default:				/* shouldn't happen */
	    sprintf(second_buf[i], "args to opcode incorrect!");
	    break;
	}
    }
    num_buffer_lines = i;
}

/* -------------------------------------------------------------------------
    show_regs () - shows 6502 registers
   ------------------------------------------------------------------------- */

void show_regs() {
    sprintf(second_buf[num_buffer_lines++], "PC = %04X EA = %04X SP = %04X",
	    debug_PC, debug_EA, debug_SP);
    sprintf(second_buf[num_buffer_lines++],
	    "X = %02X Y = %02X A = %02X F = %02X",
	    debug_X, debug_Y, debug_A, debug_F);

    memset(second_buf[num_buffer_lines], ' ', BUF_X);
    if (debug_F & C_Flag) second_buf[num_buffer_lines][0]='C';
    if (debug_F & X_Flag) second_buf[num_buffer_lines][1]='X';
    if (debug_F & I_Flag) second_buf[num_buffer_lines][2]='I';
    if (debug_F & V_Flag) second_buf[num_buffer_lines][3]='V';
    if (debug_F & B_Flag) second_buf[num_buffer_lines][4]='B';
    if (debug_F & D_Flag) second_buf[num_buffer_lines][5]='D';
    if (debug_F & Z_Flag) second_buf[num_buffer_lines][6]='Z';
    if (debug_F & N_Flag) second_buf[num_buffer_lines][7]='N';
    
    ++num_buffer_lines;
}

/* -------------------------------------------------------------------------
    will_branch () = will instruction branch?
		-1 - n/a
		0  - no it won't
		>0  - yes it will
   ------------------------------------------------------------------------- */
static int will_branch() {

    unsigned char op = get_current_opcode();
    
    switch (op) {
    case 0x10:				/* BPL */
      return (int) !(debug_F & N_Flag);
    case 0x30:				/* BMI */
      return (int) (debug_F & N_Flag);
    case 0x50:				/* BVC */
      return (int) !(debug_F & V_Flag);
    case 0x70:				/* BVS */
      return (int) (debug_F & V_Flag);
#ifdef APPLE_IIE
    case 0x80:				/* BRA */
      return 1;
#endif
    case 0x90:				/* BCC */
      return (int) !(debug_F & C_Flag);
    case 0xb0:				/* BCS */
      return (int) (debug_F & C_Flag);
    case 0xd0:				/* BNE */
      return (int) !(debug_F & Z_Flag);
    case 0xf0:				/* BEQ */
      return (int) (debug_F & Z_Flag);
    }

    return -1;
}


/* -------------------------------------------------------------------------
    set_halt () = set a breakpoint or watchpoint in memory
	type = points to "watchpoints" or "breakpoints" array
   ------------------------------------------------------------------------- */
void set_halt(int *type, int addrs) {
    int i;

    for (i = 0; i < MAX_BRKPTS; i++) {
	if (type[i] == -1) {
	    type[i] = addrs;
	    sprintf(second_buf[num_buffer_lines++], "set at %04X", addrs);
	    return;
	}
    }
    sprintf(second_buf[num_buffer_lines++], "too many!");
}

/* -------------------------------------------------------------------------
    clear_halt () = unset a critical breakpoint or watchpoint in memory
	type = points to "watchpoints" or "breakpoints" array
	pt = (pt - 1) into type.  0 indicates clear all.
   ------------------------------------------------------------------------- */
void clear_halt(int *type, int pt) {
    int i;

    if (!pt) {		/* unset all */
	for (i = 0; i < MAX_BRKPTS; i++)
	    type[i] = -1;
	return;
    }
    type[pt-1] = -1;	/* unset single */
}

/* -------------------------------------------------------------------------
    set_halt_opcode () = set a breakpoint on a particular opcode.
   ------------------------------------------------------------------------- */
void set_halt_opcode(unsigned char opcode) {
    op_breakpoints[opcode] = 1;
}

/* -------------------------------------------------------------------------
    clear_halt_opcode () = unset an opcode breakpoint.
   ------------------------------------------------------------------------- */
void clear_halt_opcode(unsigned char opcode) {
    op_breakpoints[opcode] = 0;
}

#ifdef APPLE_IIE
/* -------------------------------------------------------------------------
    set_halt_65c02 () = set a breakpoint on all 65c02 instructions.
    assumes that you are in //e mode...
   ------------------------------------------------------------------------- */
void set_halt_65c02() {
    set_halt_opcode((uchar)0x02); set_halt_opcode((uchar)0x04);
    set_halt_opcode((uchar)0x0C); set_halt_opcode((uchar)0x12);
    set_halt_opcode((uchar)0x14); set_halt_opcode((uchar)0x1A);
    set_halt_opcode((uchar)0x1C); set_halt_opcode((uchar)0x32);
    set_halt_opcode((uchar)0x34); set_halt_opcode((uchar)0x3A);
    set_halt_opcode((uchar)0x3C); set_halt_opcode((uchar)0x52);
    set_halt_opcode((uchar)0x5A); set_halt_opcode((uchar)0x64);
    set_halt_opcode((uchar)0x72); set_halt_opcode((uchar)0x74);
    set_halt_opcode((uchar)0x7A); set_halt_opcode((uchar)0x7C);
    set_halt_opcode((uchar)0x80); set_halt_opcode((uchar)0x89);
    set_halt_opcode((uchar)0x92); set_halt_opcode((uchar)0x9C);
    set_halt_opcode((uchar)0x9E); set_halt_opcode((uchar)0xB2);
    set_halt_opcode((uchar)0xD2); set_halt_opcode((uchar)0xDA);
    set_halt_opcode((uchar)0xF2); set_halt_opcode((uchar)0xFA);
}

/* -------------------------------------------------------------------------
    clear_halt_65c02 () = clear all 65c02 instructions
   ------------------------------------------------------------------------- */
void clear_halt_65c02() {
    clear_halt_opcode((uchar)0x02); clear_halt_opcode((uchar)0x04);
    clear_halt_opcode((uchar)0x0C); clear_halt_opcode((uchar)0x12);
    clear_halt_opcode((uchar)0x14); clear_halt_opcode((uchar)0x1A);
    clear_halt_opcode((uchar)0x1C); clear_halt_opcode((uchar)0x32);
    clear_halt_opcode((uchar)0x34); clear_halt_opcode((uchar)0x3A);
    clear_halt_opcode((uchar)0x3C); clear_halt_opcode((uchar)0x52);
    clear_halt_opcode((uchar)0x5A); clear_halt_opcode((uchar)0x64);
    clear_halt_opcode((uchar)0x72); clear_halt_opcode((uchar)0x74);
    clear_halt_opcode((uchar)0x7A); clear_halt_opcode((uchar)0x7C);
    clear_halt_opcode((uchar)0x80); clear_halt_opcode((uchar)0x89);
    clear_halt_opcode((uchar)0x92); clear_halt_opcode((uchar)0x9C);
    clear_halt_opcode((uchar)0x9E); clear_halt_opcode((uchar)0xB2);
    clear_halt_opcode((uchar)0xD2); clear_halt_opcode((uchar)0xDA);
    clear_halt_opcode((uchar)0xF2); clear_halt_opcode((uchar)0xFA);
}
#endif

/* -------------------------------------------------------------------------
    at_haltpt () - tests if at haltpt
	returns 0 = no breaks or watches
		1 = one break or watchpoint fired
		n = two or more breaks and/or watches fired
   ------------------------------------------------------------------------- */
int at_haltpt() {
    int i;

    /* check op_breakpoints */
    unsigned char op = get_current_opcode();
    if (op_breakpoints[op])
	sprintf(second_buf[num_buffer_lines++],
		"stopped at %04X bank %d instruction %02X",
		debug_PC, c_get_current_rambank(debug_PC), op);

    for (i = 0; i < MAX_BRKPTS; i++) {

	if (debug_PC == breakpoints[i]) {
	    sprintf(second_buf[num_buffer_lines++], "stopped at %04X bank %d",
		    breakpoints[i], c_get_current_rambank(debug_PC));
	}
    }
    for (i = 0; i < MAX_BRKPTS; i++) {
	if (watch_addrs == watchpoints[i]) {
	    if (watch_write) {
		sprintf(second_buf[num_buffer_lines++],
			"wrote: %04X: %02X",
			watchpoints[i], watch_data);
	    }
	    else {
		sprintf(second_buf[num_buffer_lines++],
			"read: %04X", watchpoints[i]);
	    }
	    watch_addrs = -666;	/* invalid address */
	}
    }
    return num_buffer_lines;	/* 0 indicates nothing happened */
}

/* -------------------------------------------------------------------------
    show_breakpts () - show breakpoints and watchpoints
   ------------------------------------------------------------------------- */
void show_breakpts() {
    int i=num_buffer_lines, k;

    for (k = 0; k < MAX_BRKPTS; k++) {
	if ((breakpoints[k] >= 0) && (watchpoints[k] >= 0)) {
	    sprintf(second_buf[i++], "break %02d at %04X  watch %02d at %04X",
		    k+1, breakpoints[k], k+1, watchpoints[k]);
	}
	else if (breakpoints[k] >= 0) {
	    sprintf(second_buf[i++], "break %02d at %04X",
		    k+1, breakpoints[k]);
	}
	else if (watchpoints[k] >= 0) {
	    memset(second_buf[i], ' ', BUF_X);
	    sprintf(second_buf[i++]+16, "  watch %02d at %04X",
		    k+1, watchpoints[k]);
	}
    }
    num_buffer_lines = i;
}

/* -------------------------------------------------------------------------
    show_opcode_breakpts () - show opcode breakpoints
   ------------------------------------------------------------------------- */
void show_opcode_breakpts() {
    int i=num_buffer_lines, k;

    sprintf(second_buf[i++], "    0 1 2 3 4 5 6 7 8 9 A B C D E F");
    sprintf(second_buf[i++], "   |-------------------------------|");
    for (k = 0; k < 0x10; k++) {
	sprintf(second_buf[i++],
		"  %X|%s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s|", k,
		op_breakpoints[k     ] ? "x" : " ",
		op_breakpoints[k+0x10] ? "x" : " ",
		op_breakpoints[k+0x20] ? "x" : " ",
		op_breakpoints[k+0x30] ? "x" : " ",
		op_breakpoints[k+0x40] ? "x" : " ",
		op_breakpoints[k+0x50] ? "x" : " ",
		op_breakpoints[k+0x60] ? "x" : " ",
		op_breakpoints[k+0x70] ? "x" : " ",
		op_breakpoints[k+0x80] ? "x" : " ",
		op_breakpoints[k+0x90] ? "x" : " ",
		op_breakpoints[k+0xA0] ? "x" : " ",
		op_breakpoints[k+0xB0] ? "x" : " ",
		op_breakpoints[k+0xC0] ? "x" : " ",
		op_breakpoints[k+0xD0] ? "x" : " ",
		op_breakpoints[k+0xE0] ? "x" : " ",
		op_breakpoints[k+0xF0] ? "x" : " ");
    }
    sprintf(second_buf[i++], "   |-------------------------------|");

    num_buffer_lines = i;
}

/* -------------------------------------------------------------------------
    show_lc_info () - show language card info
   ------------------------------------------------------------------------- */
void show_lc_info() {
    int i = num_buffer_lines;
    sprintf(second_buf[i++], "lc bank = %d", language_current_bank+1);
    (language_card_write) ? sprintf(second_buf[i++], "write LC")
	                  : sprintf(second_buf[i++], "LC write protected");
    (language_card_read)  ? sprintf(second_buf[i++], "read LC")
	                  : sprintf(second_buf[i++], "read ROM");
    sprintf(second_buf[i++], "second = %d", language_card_second);
    num_buffer_lines = i;
}

void show_misc_info() {
    int i = num_buffer_lines;
    sprintf(second_buf[i++], "TEXT (%04X): %s", SW_TEXT + (vmode_text & 1),
	    (vmode_text & 1)
	    ? "on" : "off");
    sprintf(second_buf[i++], "MIXED (%04X): %s", SW_MIXED + (vmode_mixed & 1),
	    (vmode_mixed & 1)
	    ? "on" : "off");
    sprintf(second_buf[i++], "PAGE2 (%04X): %s", SW_PAGE2 + (vmode_page2 & 1),
	    (vmode_page2 & 1)
	    ? "on" : "off");
    sprintf(second_buf[i++], "HIRES (%04X): %s", SW_HIRES + (vmode_hires & 1),
	    (vmode_hires & 1)
	    ? "on" : "off");
#ifdef APPLE_IIE
    sprintf(second_buf[i++], "80STORE (%04X): %s", SW_80STORE +
	    ((eighty_store_flag & 0x80)>>7), (eighty_store_flag & 0x80)
	    ? "on" : "off");
    sprintf(second_buf[i++], "RAMRD (%04X): %s", SW_RAMRD +
	    ((ramrd_flag & 0x80)>>7), (ramrd_flag & 0x80)
	    ? "on" : "off");
    sprintf(second_buf[i++], "RAMWRT (%04X): %s", SW_RAMWRT +
	    ((ramwrt_flag & 0x80)>>7), (ramwrt_flag & 0x80)
	    ? "on" : "off");
    sprintf(second_buf[i++], "ALTZP (%04X): %s", SW_ALTZP +
	    ((altzp_flag & 0x80)>>7), (altzp_flag & 0x80)
	    ? "on" : "off");
    sprintf(second_buf[i++], "80COL (%04X): %s", SW_80COL +
	    ((eighty_col_flag & 0x80)>>7), (eighty_col_flag & 0x80)
	    ? "on" : "off");
    sprintf(second_buf[i++], "ALTCHAR (%04X): %s", SW_ALTCHAR +
	    ((altchar_flag & 0x80)>>7), (altchar_flag & 0x80)
	    ? "on" : "off");
    sprintf(second_buf[i++], "SLOTC3ROM (%04X): %s", SW_SLOTC3ROM -/*anomaly*/
	    ((c3rom_flag & 0x80)>>7), (c3rom_flag & 0x80)
	    ? "on" : "off");
    sprintf(second_buf[i++], "SLOTCXROM (%04X): %s", SW_SLOTCXROM +
	    ((cxrom_flag & 0x80)>>7), (cxrom_flag & 0x80)
	    ? "on" : "off");
    sprintf(second_buf[i++], "DHIRES (%04X): %s", SW_DHIRES +
	    ((dhires_flag & 0x80)>>7), (dhires_flag & 0x80)
	    ? "on" : "off");
    sprintf(second_buf[i++], "IOUDIS (%04X): %s", SW_IOUDIS +
	    ((ioudis_flag & 0x80)>>7), (ioudis_flag & 0x80)
	    ? "on" : "off");
/*     sprintf(second_buf[i++], "RDVBLBAR: %s", (SLOTCXROM & 0x80) */
/* 	    ? "on" : "off"); */

#endif
    num_buffer_lines = i;
}

/* -------------------------------------------------------------------------
    show_disk_info () - disk II info
   ------------------------------------------------------------------------- */
void show_disk_info() {
    static char tmp[32];
    int i = num_buffer_lines, len = 0, off = 19;

    /* generic information */
    sprintf(second_buf[i++], "drive %s", (drive_6) ? "B" : "A");
    sprintf(second_buf[i++], "motor %s", (motor_6) ? "off" : "on");
    sprintf(second_buf[i++], "%s", (ddrw_6) ? "write" : "read");
    sprintf(second_buf[i++], "byte = %02X", disk_byte_6);
    if (!nibblized_6[drive_6]) {
	sprintf(second_buf[i++], "volume = %d", volume_6);
	sprintf(second_buf[i++], "checksum = %d", checksum_6);
    }

    sprintf(second_buf[i++], "-------------------------------------");

    /* drive / image specific information */
    memset(second_buf[i], ' ', BUF_X);
    if ((len = strlen(file_name_6[0]))) {
	while ((--len) && (file_name_6[0][len] != '/'));
	strncpy(tmp, file_name_6[0] + len + 1, 31);
	*(second_buf[i] + sprintf(second_buf[i], "%s", tmp)) = ' ';
    }

    if ((len = strlen(file_name_6[1]))) {
	while ((--len) && (file_name_6[1][len] != '/'));
	strncpy(tmp, file_name_6[1] + len + 1, 31);
	sprintf(second_buf[i]+off, "%s", tmp);
    }

    memset(second_buf[++i], ' ', BUF_X);
    *(second_buf[i] + sprintf(second_buf[i], "%s %d bytes",
	(nibblized_6[0]) ? ".nib" : ".dsk", (int)file_size_6[0])) = ' ';
    sprintf(second_buf[i++]+off, "%s %d bytes",
	    (nibblized_6[1]) ? ".nib" : ".dsk", (int)file_size_6[1]);

    memset(second_buf[i], ' ', BUF_X);
    *(second_buf[i] + sprintf(second_buf[i], "write %s",
	    (protected_6[0]) ? "protected" : "enabled")) = ' ';
    sprintf(second_buf[i++]+off, "write %s",
	    (protected_6[1]) ? "protected" : "enabled");

    memset(second_buf[i], ' ', BUF_X);
    *(second_buf[i] + sprintf(second_buf[i], "phase %d %s", phase_6[0],
	    (phase_change_6[0]) ? "(new)" : "")) = ' ';
    sprintf(second_buf[i++]+off, "phase %d %s", phase_6[1],
	    (phase_change_6[1]) ? "(new)" : "");

    memset(second_buf[i], ' ', BUF_X);
    if (!nibblized_6[0]) {
	*(second_buf[i] + sprintf(second_buf[i], "sector %d",
				  sector_6[0])) = ' ';
	if (nibblized_6[1]) ++i;
    }
    if (!nibblized_6[1])
	sprintf(second_buf[i++]+off, "sector %d", sector_6[1]);

    num_buffer_lines = i;
}

/* -------------------------------------------------------------------------
    clear_debugger_screen () - clears the screen of graphics artifacts.
   ------------------------------------------------------------------------- */
void clear_debugger_screen() {
    int i;
    c_setpage( 0 );
    c_setscreen( 0 );
    c_interface_textcolor( COLOR_LIGHT_RED, 0 );
    for (i = 0; i < 24; i++)
	c_interface_print(0, i, screen[ i ] );
}

/* -------------------------------------------------------------------------
    end_step () - finish a stepping command
	display the next instruction, and tell whether it will branch
   ------------------------------------------------------------------------- */
void end_step() {
    int branch;

    clear_debugger_screen();
    disasm(debug_PC, 1, 0, -1);		/* show next instruction */
    branch = will_branch();		/* test if it will branch */
    if (branch == -1) return;		/* n/a */
    sprintf(second_buf[num_buffer_lines++], "%s",
	    (branch) ? "will branch" : "will not branch");
}

/* -------------------------------------------------------------------------
    c_do_step () - step into or step over commands
   ------------------------------------------------------------------------- */
void c_do_step(int step_count) {
    char ch;
    unsigned char op;
    int step_frame = 0;

    /* do step while step_count AND no breaks AND no keypress */
    do {
	op = get_current_opcode();

	if (step_next && (op == 0x20)) {
	    do {
		op = get_current_opcode();
		if (op == 0x20) ++step_frame;/* JSR */
		if (op == 0x60) --step_frame;/* RTS */
		do_step();
	    } while (((ch = c_mygetch(0)) == -1) && !at_haltpt() && step_frame);
	}
	else do_step();			/* step one instruction */
    } while (--step_count && !at_haltpt() && (c_mygetch(0) == -1));

    end_step();				/* print location */
}

/* -------------------------------------------------------------------------
    display_help ()
	show quick reference command usage
   ------------------------------------------------------------------------- */
void display_help() {
    /*                     "|||||||||||||||||||||||||||||||||||||" */
    int i = num_buffer_lines;
    sprintf(second_buf[i++], "d{is} {lc1|lc2} {/bank/addr} {+}{len}");
    sprintf(second_buf[i++], "m{em} {lc1|lc2} {/bank/addr} {+}{len}");
    sprintf(second_buf[i++], "a{sc} {lc1|lc2} {/bank/addr} {+}{len}");
    sprintf(second_buf[i++], "r{egs}                               ");
    sprintf(second_buf[i++], "<addr> {lc1|lc2} : <byteseq>         ");
    sprintf(second_buf[i++], "(s{tep} | n{ext}) {len}              ");
    sprintf(second_buf[i++], "f{inish}                             ");
    sprintf(second_buf[i++], "u{ntil}                              ");
    sprintf(second_buf[i++], "g{o} {addr}                          ");
    sprintf(second_buf[i++], "sea{rch} {lc1|lc2} {bank} <bytes>    ");
    sprintf(second_buf[i++], "(b{reak} | w{atch}) {addr}           ");
    sprintf(second_buf[i++], "b{reak} op <byte>                    ");
    sprintf(second_buf[i++], "(c{lear} | i{gnore}) {num}           ");
    sprintf(second_buf[i++], "c{lear} op <byte>                    ");
    sprintf(second_buf[i++], "(sta{tus} | op{codes})               ");
    sprintf(second_buf[i++], "(l{ang} | d{rive} | vm)              ");
    sprintf(second_buf[i++], "bsave <filename> </bank/addr> <len>  ");
    sprintf(second_buf[i++], "bload <filename> <addr>              ");
    sprintf(second_buf[i++], "fr{esh}                              ");
    sprintf(second_buf[i++], "(h{elp} | ?)                         ");
    num_buffer_lines = i;
}


/* -------------------------------------------------------------------------
    do_debug_command ()
	perform a debugger command
   ------------------------------------------------------------------------- */

void do_debug_command() {
    int i = 0, j = 0, k = 0;
    
    /* reset key local vars */
    exception_flag &= ~DebugStepSig;
    step_next = 0;
    num_buffer_lines = 0;

    /* call lex to perform the command.*/
    strncpy(lexbuf, command_line + PROMPT_X, BUF_X);
    init_lex(lexbuf, BUF_X+2);
    debuglex();

    /* set up to copy results into main buffer */
    if (num_buffer_lines >= PROMPT_Y) {
	k = BUF_Y - PROMPT_Y;
    } else {
	/* scroll buffer */
	for (i = 0, j = 0; i < PROMPT_Y - num_buffer_lines; i++, j = 0) {
	    memcpy(command_buf[i], command_buf[num_buffer_lines+1+i], BUF_X);
	    while ((j < BUF_X) && (command_buf[i][j] != '\0')) j++;
	    memset (command_buf[i] + j, ' ', BUF_X - j);
	    command_buf[i][BUF_X - 1] = '\0';
	}
    }
    
    /* copy the debug results into debug console window. change '\0's
       to ' 's and cap with a single '\0' */
    while (i < PROMPT_Y) {
	j = 0;
	memcpy(command_buf[i], second_buf[k++], BUF_X);
	while ((j < BUF_X) && (command_buf[i][j] != '\0')) ++j;
	memset(command_buf[i] + j, ' ', BUF_X - j);
	command_buf[i++][BUF_X - 1] = '\0';
    }

    /* new prompt */
    memset(command_line, ' ', BUF_X);
    command_line[0] = '>';
    command_line[BUF_X - 1] = '\0';

    /* display the new information */
    c_interface_textcolor(COLOR_LIGHT_GREEN, 0);
    for (i=0; i<BUF_Y; i++) {
	c_interface_print(1, 1+i, command_buf[i]);
    }

    /* reset before we return */
    num_buffer_lines = 0;
    exception_flag &= ~DebugStepSig;
}

/* -------------------------------------------------------------------------
    c_do_debugging()
	main debugging console
   ------------------------------------------------------------------------- */

void c_do_debugging() {

    static char lex_initted = 0;

    int	i;
    int ch;
    int command_pos = PROMPT_X;

    exception_flag &= ~DebugStepSig;

    *opcodes = (apple_mode == 0) ? opcodes_6502 :
	       (apple_mode == 1) ? opcodes_undoc :
				   opcodes_65c02;

    /* initialize the buffers */
    for (i=0; i<BUF_Y; i++) {
	memset(command_buf[i], '\0', BUF_X);
	memset(second_buf [i], '\0', BUF_X);
    }
    memset(command_line, ' ', BUF_X);
    command_line[0] = '>';
    command_line[BUF_X - 1] = '\0';

    /* initialize the lexical scanner */
    if (!lex_initted) {
	memset(lexbuf, '\0', BUF_X+2);
	/*init_lex(lexbuf, BUF_X+2);*/
	lex_initted = 1;
    }

    c_setpage( 0 );
    c_setscreen( 0 );
    c_interface_translate_screen( screen );
    c_interface_textcolor( COLOR_LIGHT_RED, 0 );
    for (i = 0; i < 24; i++)
	c_interface_print(0, i, screen[ i ] );

    for (;;) {
	/* print command line */
	c_interface_textcolor(COLOR_LIGHT_GREEN, 0);
	c_interface_print(1, 1+PROMPT_Y, command_line);

	/* highlight cursor */
	c_interface_textcolor(COLOR_LIGHT_GREEN, COLOR_MEDIUM_BLUE);
	(command_line[command_pos])
	    ? c_interface_print_char(1+command_pos, 1+PROMPT_Y,
				     command_line[command_pos])
	    : c_interface_print_char(1+command_pos, 1+PROMPT_Y, 
				     command_line[command_pos]);
	c_interface_textcolor(COLOR_LIGHT_GREEN, 0);

	while ((ch = c_mygetch(1)) == -1) ;

	if (ch == kESC) {
	    exception_flag &= ~DebugStepSig;
	    for (i = 0xC030; i < 0xC040; i++)
		table_read_memory[i] = table_write_memory[i] =
		    (sound_mode && soundAllowed) ? read_speaker_toggle_pc : ram_nop;
	    c_interface_exit();
	    return;
	}
	else {
	    /* backspace */
	    if ((ch == 127 || ch == 8) &&
		(command_pos > PROMPT_X)) {
		command_line[--command_pos] = ' ';
	    }
	    /* return */
	    else if (ch == 13) {
		command_line[command_pos] = '\0';
		do_debug_command();
		command_pos = PROMPT_X;
	    }
	    /* normal character */
	    else if ((ch >= ' ') && (ch < 127) &&
		     (command_pos < PROMPT_END_X)) {
		command_line[command_pos++] = ch;
	    }
	}
    }
}

#endif /* DEBUGGER */
