#ifdef RCSID
static char RCSid[] =
"$Header: d:/tads/tads2/RCS/linf.c 1.6 96/10/14 16:11:33 mroberts Exp $";
#endif

/* Copyright (c) 1991 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  linf.c - line source File implementation
Function
  Implementation of file line source
Notes
  None
Modified
  04/25/93 JEras         - use new os_locate() to find include files
  08/14/91 MJRoberts     - creation
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "os.h"
#include "std.h"
#include "err.h"
#include "mch.h"
#include "tok.h"
#include "linf.h"
#include "dbg.h"

/* initialize a pre-allocated linfdef, skipping debugger page setup */
void linfini2(mctx, linf, filename, flen, fp)
mcmcxdef *mctx;
linfdef  *linf;                                /* linfdef already allocated */
char     *filename;            /* fully qualified filename - no path search */
int       flen;                                       /* length of filename */
osfildef *fp;                                    /* handle to the open file */
{
    /* set up method pointers */
    linf->linflin.lingetp = linfget;
    linf->linflin.linclsp = linfcls;
    linf->linflin.linppos = linfppos;
    linf->linflin.linglop = linfglop;
    linf->linflin.linwrtp = linfwrt;
    linf->linflin.lincmpp = linfcmp;
    linf->linflin.linactp = linfact;
    linf->linflin.lindisp = linfdis;
    linf->linflin.lintellp = linftell;
    linf->linflin.linseekp = linfseek;
    linf->linflin.linreadp = linfread;
    linf->linflin.linpaddp = linfpadd;
    linf->linflin.linqtopp = linfqtop;
    linf->linflin.lingetsp = linfgets;
    linf->linflin.linnamp = linfnam;
    linf->linflin.linfindp = linffind;
    linf->linflin.lingotop = linfgoto;
    linf->linflin.linofsp = linfofs;
    linf->linflin.linrenp = linfren;

    /* set up instance data */
    linf->linflin.linbuf = linf->linfbuf;
    linf->linflin.linflg = 0;
    memcpy(linf->linfnam, filename, (size_t)flen);
    linf->linfnam[flen] = '\0';
    linf->linfbuf[0] = '\0';
    linf->linffp = fp;
    linf->linfnum = 0;
    linf->linflin.linlln = 4;     /* OPCLINE operand is seek offset in file */
    linf->linfmem = mctx;                    /* save memory manager context */
    linf->linfcrec = 0;                  /* no debugger records written yet */
}

/* initialize a file line source object */
linfdef *linfini(mctx, ec, filename, flen, path)
mcmcxdef *mctx;
errcxdef *ec;
char     *filename;
int       flen;
tokpdef  *path;
{
    int       i;
    objnum   *objp;
    linfdef  *linf;
    osfildef *fp;
    char      fbuf[OSFNMAX + 1];
    tokpdef   fakepath;
    int       len;
    
    if (!path)
    {
        path = &fakepath;
        fakepath.tokpnxt = (tokpdef *)0;
        fakepath.tokplen = 0;
    }

    /* search through the path list */
    for ( ; path ; path = path->tokpnxt)
    {
        char last;

        /* prefix the current path */
        if (len = path->tokplen)
        {
            memcpy(fbuf, path->tokpdir, (size_t)len);
            last = fbuf[len - 1];
            if (last == OSPATHCHAR ||
                (OSPATHALT && strchr(OSPATHALT, last)))
                /* do nothing */ ;
            else
            {
                /* append path separator character */
                fbuf[len++] = OSPATHCHAR;
            }
        }

        /* add the filename and null-terminate */
        memcpy(fbuf + len, filename, (size_t)flen);
        fbuf[len + flen] = '\0';
        
        /* attempt to open this file */
        if (fp = osfoprs(fbuf)) break;
    }

    /* if no file opened yet, search tads path - if not found, return error */
    if (!fp
        && (!os_locate(filename, flen, (char *)0, fbuf, sizeof(fbuf))
        || !(fp = osfoprs(fbuf))))
        return((linfdef *)0);
    
    /* allocate a linfdef based on the filename length */
    len = (int)strlen(fbuf);
    linf = (linfdef *)mchalo(ec, (ushort)(sizeof(linfdef) + flen
                                          + len + 1), "linfini");

    /* do the basic initialization */
    linfini2(mctx, linf, filename, flen, fp);

    memcpy(linf->linfnam + flen + 1, fbuf, (size_t)len);
    linf->linfnam[flen + 1 + len] = '\0';
    
    /* set all debugger pages to not-yet-allocated */
    for (i = LINFPGMAX, objp = linf->linfpg ; i ; ++objp, --i)
        *objp = MCMONINV;

    return(linf);
}

int linfget(lin)
lindef *lin;
{
#   define  linf ((linfdef *)lin)
    char   *p;
    int     l;

    /* remember seek position of start of current line */
    linf->linfseek = osfpos(linf->linffp);
    
    /* get a line from the file, and quit if at eof */
    if (!osfgets(linf->linfbuf, (size_t)(sizeof(linf->linfbuf) - 1),
                 linf->linffp))
        return(TRUE);                                 /* error, end of file */

    p = lin->linbuf = linf->linfbuf; /* make sure buffer pointer is correct */

    /* remove newline from string if there is one */
    l = strlen(p);
    lin->linflg &= ~LINFMORE;         /* presume end of line is with us now */
    if (l)
    {
        if (p[l-1] == '\n' || p[l-1] == '\r')
            --l;                      /* don't count newline in line length */
        else
            lin->linflg |= LINFMORE;  /* indicate that there's more to come */
        
        if (p[l-1] == '\r' || p[l-1] == 26) --l;           /* remove \r, ^Z */
    }
    lin->linlen = l;
    
    if (!(lin->linflg & LINFMORE))         /* if not a continuation line... */
        ++(linf->linfnum);                 /* increment current line number */

    LINFDEBUG(printf("%s\n", linf->linfbuf);)
    
    return(FALSE);

#   undef  linf
}

/* make printable string from position in file (for error reporting) */
void linfppos(lin, buf, buflen)
lindef  *lin;
char    *buf;
uint     buflen;
{
    VARUSED(buflen);
    
    sprintf(buf, "%s(%d): ", ((linfdef *)lin)->linfnam,
            ((linfdef *)lin)->linfnum);
}

/* close line source */
void linfcls(lin)
lindef *lin;
{
    osfcls(((linfdef *)lin)->linffp);
}

/* generate operand of OPCLINE (source-line debug) instruction */
void linfglop(lin, buf)
lindef *lin;
uchar  *buf;
{
    oswp4(buf, ((linfdef *)lin)->linfseek);   /* save seek position of line */
}

/* save line source information to binary (.gam) file; TRUE ==> error */
int linfwrt(lin, fp)
lindef   *lin;
osfildef *fp;
{
#   define  linf ((linfdef *)lin)
    char    buf[128];
    uint    pgcnt;
    uchar  *objp;
    mcmon  *objn;

    buf[0] = lin->linid;
    buf[1] = strlen(linf->linfnam);
    oswp4(buf + 2, linf->linfcrec);
    memcpy(buf + 6, linf->linfnam, (size_t)buf[1]);
    if (osfwb(fp, buf, (int)(buf[1] + 6))) return(TRUE);
    
    /* write the debug source pages */
    if (!linf->linfcrec) return(FALSE);          /* no debug records at all */
    pgcnt = 1 + ((linf->linfcrec - 1) >> 10);     /* figure number of pages */
    
    for (objn = linf->linfpg ; pgcnt ; ++objn, --pgcnt)
    {
        objp = mcmlck(linf->linfmem, *objn);
        if (osfwb(fp, objp, (1024 * DBGLINFSIZ))) return(TRUE);
        mcmunlck(linf->linfmem, *objn);
    }
    
    return(FALSE);

#   undef  linf
}

/* load a file-line-source from binary (.gam) file */
int linfload(fp, dbgctx, ec, path)
osfildef *fp;
dbgcxdef *dbgctx;
errcxdef *ec;
tokpdef  *path;
{
    linfdef *linf;
    uchar    buf[128];
    uint     pgcnt;
    uchar   *objp;
    mcmon   *objn;
    
    /* read the source's description from the file */
    if (osfrb(fp, buf, 6)
        || osfrb(fp, buf + 6, (int)buf[1]))
        return(TRUE);
    
    /* initialize the linfdef */
    if (!(linf = linfini(dbgctx->dbgcxmem, ec, buf + 6, (int)buf[1], path)))
    {
        errlog1(ec, ERR_NOSOURC, ERRTSTR, errstr(ec, buf+6, (int)buf[1]));
        return(TRUE);
    }
    
    /* close the file - don't hold all files open */
    osfcls(linf->linffp);
    
    /* link into debug line source chain */
    linf->linflin.linnxt = dbgctx->dbgcxlin;
    dbgctx->dbgcxlin = &linf->linflin;
    linf->linflin.linid = buf[0];
    linf->linfcrec = osrp4(buf + 2);
    
    /* make sure the max line id is set above current line */
    if (buf[0] >= dbgctx->dbgcxfid) dbgctx->dbgcxfid = buf[0] + 1;
    
    /* allocate and read the debug source pages */
    if (!linf->linfcrec) return(FALSE);          /* no debug records at all */
    pgcnt = 1 + ((linf->linfcrec - 1) >> 10);     /* figure number of pages */
    
    for (objn = linf->linfpg ; pgcnt ; ++objn, --pgcnt)
    {
        objp = mcmalo(linf->linfmem, (ushort)(1024 * DBGLINFSIZ), objn);
        if (osfrb(fp, objp, (1024 * DBGLINFSIZ))) return(TRUE);
        mcmunlck(linf->linfmem, *objn);
    }
    
    return(FALSE);
}

/* add a debugger line record for the current line being compiled */
void linfcmp(lin, buf)
lindef *lin;
uchar  *buf;
{
    uint    pg;
    uchar  *objptr;
#   define  linf ((linfdef *)lin)

    /* figure out which page to use, and lock it */
    pg = linf->linfcrec >> 10;                     /* 2^10 records per page */
    if (pg >= LINFPGMAX)
        errsig(linf->linfmem->mcmcxgl->mcmcxerr, ERR_MANYDBG);
    if (linf->linfpg[pg] == MCMONINV)
        objptr = mcmalo(linf->linfmem, (ushort)(1024 * DBGLINFSIZ),
                        &linf->linfpg[pg]);
    else
        objptr = mcmlck(linf->linfmem, linf->linfpg[pg]);
    
    /* write the record to the appropriate offset within the page */
    memcpy(objptr + (linf->linfcrec & 1023) * DBGLINFSIZ, buf,
           (size_t)DBGLINFSIZ);
    
    /* increment counter of line records so far */
    ++(linf->linfcrec);
    
    /* done with page - touch it and unlock it */
    mcmtch(linf->linfmem, linf->linfpg[pg]);
    mcmunlck(linf->linfmem, linf->linfpg[pg]);
    
#   undef linf
}

/*
 *   Renumber an existing object.  Searches through all line records for
 *   any with the given object number, and changes the number to the new
 *   number if found.  
 */
void linfren(lin, oldnum, newnum)
lindef *lin;
objnum  oldnum;
objnum  newnum;
{
#   define  linf ((linfdef *)lin)
    uint    pgcnt;
    uchar  *objp;
    mcmon  *pgobjn;
    int     i;
    int     pgtot;
    int     tot;

    /* figure the number of pages - if no lines, stop now */
    tot = linf->linfcrec;
    if (!linf->linfcrec) return;
    pgcnt = 1 + ((tot - 1) >> 10);

    /* scan each page */
    for (pgobjn = linf->linfpg ; pgcnt ; ++pgobjn, --pgcnt, tot -= 1024)
    {
        /* lock the page */
        objp = mcmlck(linf->linfmem, *pgobjn);

        /* figure the number on this page */
        pgtot = (tot > 1024 ? 1024 : tot);

        /* scan each record on this page */
        for (i = 0 ; i < pgtot ; ++i, objp += DBGLINFSIZ)
        {
            /* check this one */
            if (osrp2(objp) == oldnum)
            {
                /* it matches - renumber it */
                oswp2(objp, newnum);
            }
        }

        /* done with the page - touch it and unlock it */
        mcmtch(linf->linfmem, *pgobjn);
        mcmunlck(linf->linfmem, *pgobjn);
    }

#   undef  linf
}


/* find the nearest line record to a file seek location */
void linffind(lin, buf, objp, ofsp)
lindef *lin;
char   *buf;
objnum *objp;
uint   *ofsp;
{
#   define  linf ((linfdef *)lin)
    uint    pg;
    uchar  *objptr;
    uchar  *bufptr;
    long    first;
    long    last;
    long    cur;
    ulong   seekpos;
    ulong   curpos;
    objnum  objn;
    uint    ofs;

    /* get desired seek position out of buffer */
    seekpos = osrp4(buf);
    
    /* run a binary search for the indicated line record */
    first = 0;
    last = linf->linfcrec - 1;
    
    for (;;)
    {
        if (first > last)
        {
            /* return the most recent record found - it's closest */
            *objp = objn;
            *ofsp = ofs;
            return;
        }
        cur = first + (last - first)/2;
        pg = cur >> 10;
        
        /* get object + offset corresponding to current source line */
        objptr = mcmlck(linf->linfmem, linf->linfpg[pg]);
        bufptr = objptr + ((cur & 1023) * DBGLINFSIZ);
        objn = osrp2(bufptr);
        ofs = osrp2(bufptr + 2);
        mcmunlck(linf->linfmem, linf->linfpg[pg]);
        
        /* read user data out of the object's OPCLINE record */
        objptr = mcmlck(linf->linfmem, (mcmon)objn);
        bufptr = objptr + ofs + 5;
        curpos = osrp4(bufptr);
        mcmunlck(linf->linfmem, (mcmon)objn);

        /* see what we have */
        if (curpos == seekpos)
        {
            *objp = objn;
            *ofsp = ofs;
            return;
        }
        else if (curpos < seekpos)
            first = (cur == first ? first + 1 : cur);
        else
            last = (cur == last ? last - 1 : cur);
    }
    
#   undef linf
}

/* disactivate line source under debugger - close file */
void linfdis(lin)
lindef *lin;
{
#   define linf ((linfdef *)lin)

    if (linf->linffp)
    {
        osfcls(linf->linffp);
        linf->linffp = (osfildef *)0;
    }
    
#   undef linf
}

/* activate line source under debugger - open file */
void linfact(lin)
lindef *lin;
{
    char   *fname;
#   define  linf ((linfdef *)lin)

    fname = linf->linfnam;                               /* get name buffer */
    fname += strlen(fname) + 1;                  /* point to full path name */
    linf->linffp = osfoprs(fname);                         /* open the file */

#   undef linf
}

/* get current seek position */
void linftell(lin, pos)
lindef *lin;
uchar  *pos;
{
#   define  linf ((linfdef *)lin)
    long    seekpos;

    seekpos = ftell(linf->linffp);
    oswp4(pos, seekpos);

#   undef linf
}

/* seek to a new position */
void linfseek(lin, pos)
lindef *lin;
uchar  *pos;
{
#   define  linf ((linfdef *)lin)
    long    seekpos;
    
    seekpos = osrp4(pos);
    osfseek(linf->linffp, seekpos, OSFSK_SET);

#   undef linf
}

/* read bytes - fread-style interface */
int linfread(lin, buf, siz)
lindef *lin;
uchar  *buf;
uint    siz;
{
#   define  linf ((linfdef *)lin)

    return(fread(buf, 1, siz, linf->linffp));

#   undef linf
}

/* add a signed delta to a seek position */
void linfpadd(lin, pos, delta)
lindef *lin;
uchar  *pos;
long    delta;
{
#   define  linf ((linfdef *)lin)
    long    seekpos;

    seekpos = osrp4(pos);
    seekpos += delta;
    if (seekpos < 0) seekpos = 0;
    oswp4(pos, seekpos);

#   undef linf
}

/* query whether we're at top of file */
int linfqtop(lin, pos)
lindef *lin;
uchar  *pos;
{
#   define  linf ((linfdef *)lin)

    return(osrp4(pos) == 0);

#   undef linf
}

/* read one line at current position - fgets-style interface */
int linfgets(lin, buf, siz)
lindef *lin;
uchar  *buf;
uint    siz;
{
#   define  linf ((linfdef *)lin)

    return(fgets(buf, siz, linf->linffp) != 0);

#   undef linf
}

/* get name of line source */
void linfnam(lin, buf)
lindef *lin;
char   *buf;
{
#   define  linf ((linfdef *)lin)

    strcpy(buf, linf->linfnam);

#   undef linf
}

/* go to top/bottom of line source */
void linfgoto(lin, where)
lindef *lin;
int     where;
{
#   define  linf ((linfdef *)lin)

    osfseek(linf->linffp, 0L, where);

#   undef linf
}

/* return current seek offset within source */
long linfofs(lin)
lindef *lin;
{
#   define linf ((linfdef *)lin)

    return(ftell(linf->linffp));
    
#   undef  linf
}
