#ifdef RCSID
static char RCSid[] =
"$Header: c:/tads/tads2/RCS/DBGRUN.C 1.6 94/11/06 13:07:10 mroberts Exp $";
#endif

/* Copyright (c) 1992 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  dbgrun.c - functions needed when actually debugging
Function
  This module contains functions actually used while debugging, as
  opposed to functions needed in the compiler and/or runtime to set
  up debugging without activating an interactive debugger session.
  These functions are in a separate file to reduce the size of the
  runtime and compiler, which don't need to link these functions.
Notes
  None
Modified
  04/20/92 MJRoberts     - creation
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "dbg.h"
#include "tio.h"
#include "obj.h"
#include "prp.h"
#include "tok.h"
#include "run.h"
#include "tok.h"
#include "prs.h"
#include "os.h"
#include "linf.h"

/* forward declaration for static function */
static int dbgcompile(/*_ dbgcxdef *, char *, dbgfdef *, objnum * _*/);


/* format a display of where execution is stopped */
void dbgwhere(ctx, p)
dbgcxdef *ctx;
char     *p;
{
    dbgfdef *f;

    f = &ctx->dbgcxfrm[ctx->dbgcxfcn - 1];
    if (f->dbgftarg == MCMONINV)
        p += dbgnam(ctx, p, TOKSTBIFN, f->dbgfbif);
    else
        p += dbgnam(ctx, p,
                    (f->dbgfself == MCMONINV ? TOKSTFUNC : TOKSTOBJ),
                    (int)f->dbgftarg);

    if (f->dbgfself != MCMONINV && f->dbgfself != f->dbgftarg)
    {
        memcpy(p, "<self=", (size_t)6);
        p += 6;
        p += dbgnam(ctx, p, TOKSTOBJ, (int)f->dbgfself);
        *p++ = '>';
    }
    if (f->dbgfprop)
    {
        *p++ = '.';
        p += dbgnam(ctx, p, TOKSTPROP, (int)f->dbgfprop);
    }
    if (f->dbgfself == MCMONINV || f->dbgfbif)
    {
        *p++ = '(';
        *p++ = ')';
    }
    *p = '\0';
}

/*
 *   Evaluate breakpoint condition; returns TRUE if breakpoint condition
 *   was non-nil, FALSE if it was nil or had no value. 
 */
static int dbgbpeval(ctx, bp, fr)
dbgcxdef *ctx;
dbgbpdef *bp;
dbgfdef  *fr;
{
    uchar   *objptr;
    runsdef *oldsp;
    runsdef  val;
    int      err;
                
    objptr = mcmlck(ctx->dbgcxmem, bp->dbgbpcond);
            
    ERRBEGIN(ctx->dbgcxerr)
            
    oldsp = ctx->dbgcxrun->runcxsp;
    runexe(ctx->dbgcxrun, objptr, fr->dbgfself, bp->dbgbpcond,
           (prpnum)0, 0);
            
    ERRCATCH(ctx->dbgcxerr, err)
        /* no error recover - just proceed as normal */
        ;
    ERREND(ctx->dbgcxerr)

    /* done with condition code; unlock it */
    mcmunlck(ctx->dbgcxmem, bp->dbgbpcond);
            
    /* if condition evaluated to non-nil, break now */
    if (ctx->dbgcxrun->runcxsp != oldsp)
    {
        runpop(ctx->dbgcxrun, &val);
        if (val.runstyp == DAT_NIL)
            return(FALSE);              /* value was nil - condition failed */
        else
            return(TRUE);              /* value was non-nil - condition met */
    }
    else
        return(FALSE);       /* expression returned no value - treat as nil */
}

/* single-step interrupt */
void dbgss(ctx, ofs, instr, err)
dbgcxdef *ctx;
uint      ofs;                                     /* offset of line record */
int       instr;
int       err;                        /* error (0 if not catching an error) */
{
    dbgbpdef *bp;
    int       i;
    dbgfdef  *fr = &ctx->dbgcxfrm[ctx->dbgcxfcn - 1];
    int       brk;
    int       bphit = 0;
    voccxdef *vcx = ctx->dbgcxrun->runcxvoc;
    
    /* don't re-enter debugger - return if running from tdb */
    if (ctx->dbgcxflg & DBGCXFIND) return;
    
    if (instr == OPCBP)
    {
        /* look up breakpoint, and make sure 'self' is correct */
        for (i = 0, bp = ctx->dbgcxbp ; i < DBGBPMAX ; ++bp, ++i)
        {
            if ((bp->dbgbpflg & DBGBPFUSED)
                && !(bp->dbgbpflg & DBGBPFDISA)
                && (bp->dbgbpself == fr->dbgfself
                    || bp->dbgbpself == MCMONINV
		    || bifinh(vcx, vocinh(vcx, fr->dbgfself),
			      bp->dbgbpself))
                && bp->dbgbptarg == fr->dbgftarg
                && bp->dbgbpofs + 3 == ofs)
                break;
        }
        if (i == DBGBPMAX) return;        /* no such breakpoint - ignore it */
        
        /* if this is a conditional breakpoint, make sure condition is true */
        if (!(ctx->dbgcxflg & DBGCXFSS) && (bp->dbgbpflg & DBGBPFCOND)
            && !dbgbpeval(ctx, bp, fr))
            return;
            
        bphit = i + 1;
    }
    else
    {
        if (!err && !(ctx->dbgcxflg & DBGCXFSS) && (ctx->dbgcxflg & DBGCXFGBP))
        {
            /*
             *   Not single-stepping, but one or more global breakpoints
             *   are in effect.  for each global breakpoint, eval
             *   condition; if any are true, stop execution, otherwise
             *   resume execution.  
             */
            for (brk = FALSE, i = 0, bp = ctx->dbgcxbp ; i < DBGBPMAX ;
                 ++bp, ++i)
            {
                if ((bp->dbgbpflg & DBGBPFUSED)
                    && !(bp->dbgbpflg & DBGBPFDISA)
                    && bp->dbgbptarg == MCMONINV
                    && dbgbpeval(ctx, bp, fr))
                {
                    brk = TRUE;
                    bp->dbgbpflg |= DBGBPFDISA;             /* auto disable */
                    bphit = i + 1;
                    break;
                }
            }
        }
        else
            brk = FALSE;
        
        if (!brk && !err)
        {
            if (!(ctx->dbgcxflg & DBGCXFSS))
                return;               /* not single-stepping - keep running */

            if ((ctx->dbgcxflg & DBGCXFSO) && ctx->dbgcxdep > ctx->dbgcxsof)
                return;  /* stepping over, haven't returned yet; don't stop */
        }
    }
    
    /* call user-interface main command processor routine */
    ctx->dbgcxflg |= DBGCXFIND;
    dbgucmd(ctx, bphit, err);
    ctx->dbgcxflg &= ~DBGCXFIND;
}

/* activate debugger; returns TRUE if no debugger is present */
int dbgstart(ctx)
dbgcxdef *ctx;
{
    if (!ctx || !(ctx->dbgcxflg & DBGCXFOK)) return(TRUE);
    ctx->dbgcxflg |= DBGCXFSS;                 /* activate single-step mode */
    return(FALSE);
}

/* save a text string in the dbgcxname buffer; TRUE ==> can't store it */
static int dbgnamsav(ctx, nam, ofsp)
dbgcxdef *ctx;
char     *nam;
uint     *ofsp;
{
    size_t len = strlen(nam) + 1;
    
    if (ctx->dbgcxnam
        && ctx->dbgcxnamf + len < ctx->dbgcxnams)
    {
        memcpy(ctx->dbgcxnam + ctx->dbgcxnamf, nam, len);
        *ofsp = ctx->dbgcxnamf;
        ctx->dbgcxnamf += len;
        return(FALSE);
    }
    else
        return(TRUE);                     /* uanble to store - return error */
}

/* delete a string from the dbgcxname area, adjusting bp/wx offsets */
static void dbgnamdel(ctx, ofs)
dbgcxdef *ctx;
uint      ofs;                              /* offset of name to be deleted */
{
    int       i;
    dbgbpdef *bp;
    dbgwxdef *wx;
    uint      delsiz;
    
    /* if no name text is being stored, we need do nothing */
    if (!ctx->dbgcxnam) return;
    
    /* compute size of area to be deleted */
    delsiz = strlen(ctx->dbgcxnam + ofs) + 1;
    
    /* go through breakpoints, moving text if necessary */
    for (i = DBGBPMAX, bp = ctx->dbgcxbp ; i ; ++bp, --i)
    {
        if ((bp->dbgbpflg & DBGBPFUSED) && (bp->dbgbpflg & DBGBPFNAME)
            && bp->dbgbpnam > ofs)
            bp->dbgbpnam -= delsiz;
    }
    
    /* do the same for the watch expressions */
    for (i = DBGWXMAX, wx = ctx->dbgcxwx ; i ; ++wx, --i)
    {
        if ((wx->dbgwxflg & DBGWXFUSED) && (wx->dbgwxflg & DBGWXFNAME)
            && wx->dbgwxnam > ofs)
            wx->dbgwxnam -= delsiz;
    }
    
    /* now actually remove the string from the dbgcxname area */
    if (ctx->dbgcxnamf - ofs - delsiz)
        memmove(ctx->dbgcxnam + ofs, ctx->dbgcxnam + ofs + delsiz,
                (size_t)(ctx->dbgcxnamf - ofs - delsiz));
    ctx->dbgcxnamf -= delsiz;
}

/* set a breakpoint at an object + offset location */
int dbgbpat(ctx, target, self, ofs, bpnum, bpname, toggle, cond)
dbgcxdef *ctx;
objnum    target;
objnum    self;
uint      ofs;
int      *bpnum;
char     *bpname;
int       toggle;
char     *cond;
{
    uchar    *objp;
    int       done;
    int       i;
    dbgbpdef *bp;
    int       err;
    objnum    condobj;
    
    /* find a slot for the new breakpoint */
    for (i = 0, bp = ctx->dbgcxbp ; i < DBGBPMAX ; ++bp, ++i)
        if (!(bp->dbgbpflg & DBGBPFUSED)) break;
    *bpnum = i + 1;              /* tell the caller which breakpoint we set */
        
    /* if this is a global breakpoint, on OPCBP is needed */
    if (target == MCMONINV)
    {
        /* compile the condition, with no local frame */
        dbgfdef fr;
        
        fr.dbgffr = 0;
        fr.dbgftarg = MCMONINV;
        if (err = dbgcompile(ctx, cond, &fr, &condobj))
            return(err);
        
        bp->dbgbpflg  = DBGBPFUSED | DBGBPFCOND;
        bp->dbgbpself =
        bp->dbgbptarg = MCMONINV;
        bp->dbgbpofs  = 0;
        bp->dbgbpcond = condobj;
        if (!dbgnamsav(ctx, bpname, &bp->dbgbpnam))
            bp->dbgbpflg |= DBGBPFNAME;
        
        ctx->dbgcxflg |= DBGCXFGBP;
        return(0);
    }

    /* lock the object */
    objp = mcmlck(ctx->dbgcxmem, (mcmon)target);
    
    /* skip any ENTER, CHKARGC, or FRAME instructions */
    for (done = FALSE ; !done ; )
    {
        switch(*(objp + ofs))
        {
        case OPCENTER:
            ofs += 3;
            break;
        case OPCCHKARGC:
            ofs += 2;
            break;
        case OPCFRAME:
            ofs += 1 + osrp2(objp + ofs + 1);
            break;
        default:
            done = TRUE;
            break;
        }
    }
    
    /* see what kind of instruction we've found */
    if (*(objp + ofs) == OPCBP)
    {
        if (toggle && (!cond || !*cond))
        {
            for (i = 0, bp = ctx->dbgcxbp ; i < DBGBPMAX ; ++bp, ++i)
            {
                if ((bp->dbgbpflg & DBGBPFUSED)
                    && bp->dbgbptarg == target
                    && bp->dbgbpofs == ofs
                    && bp->dbgbpself == self)
                {
                    err = dbgbpdel(ctx, i+1);
                    goto return_error;
                }
            }
        }
        else err = 0;
    }
    else if (*(objp + ofs) != OPCLINE)
        err = ERR_BPNOTLN;
    else if (i == DBGBPMAX)
        err = ERR_MANYBP;
    else
    {
        *(objp + ofs) = OPCBP;
        mcmtch(ctx->dbgcxmem, (mcmon)target);
        err = 0;
    }
    

    /* check for a condition attached to the breakpoint */
    if (cond && *cond)
    {
        dbgfdef fr;

        fr.dbgffr = osrp2(objp + ofs + 2);
        fr.dbgftarg = target;
        err = dbgcompile(ctx, cond, &fr, &condobj);
    }
    else
        condobj = MCMONINV;

    if (!err)
    {
        bp->dbgbpflg = DBGBPFUSED;
        bp->dbgbpself = self;
        bp->dbgbptarg = target;
        bp->dbgbpofs = ofs;
        
        if (cond && *cond)
        {
            bp->dbgbpcond = condobj;
            bp->dbgbpflg |= DBGBPFCOND;
        }
    }

return_error:
    mcmunlck(ctx->dbgcxmem, (mcmon)target);
    
    /* free condition object if we have one */
    if (err && condobj != MCMONINV) mcmfre(ctx->dbgcxmem, condobj);
    
    /* store breakpoint name for displaying to user */
    if (!err && !dbgnamsav(ctx, bpname, &bp->dbgbpnam))
        bp->dbgbpflg |= DBGBPFNAME;

    return(err);
}

/* set a breakpoint at a symbolic address: function or object.property */
int dbgbpset(ctx, addr, bpnum)
dbgcxdef *ctx;
char     *addr;
int      *bpnum;
{
    char     *p;
    char      psav;
    dbgbpdef *bp;
    int       i;
    char     *tok1;
    char      tok1sav;
    char     *tok1end;
    char     *tok2;
    toksdef   sym1;
    toksdef   sym2;
    toktdef  *tab;
    uint      hsh1;
    uint      hsh2;
    objnum    objn;
    uint      ofs;
    uchar    *objp;
    prpdef   *prp;
    dattyp    typ;
    int       err;
    int       done;
    objnum    target;
    char     *end;
    char      buf1[40];
    char      buf2[40];
    
    /* determine what kind of address we have */
    for (p = addr ; *p && isspace(*p) ; ++p);
    for (tok1 = addr = p ; *p && TOKISSYM(*p) ; ++p);
    
    /* see if the very first thing is "when" */
    if (!strnicmp(tok1, "when ", (size_t)5))
    {
        for (end = tok1 + 5 ; isspace(*end) ; ++end);
        return(dbgbpat(ctx, MCMONINV, MCMONINV, 0, bpnum, addr, FALSE, end));
    }
    
    /* see if we have a second token */
    for (tok1end = p ; isspace(*p) ; ++p);
    if (*p == '.')
    {
        for (++p ; isspace(*p) ; ++p);
        for (tok2 = p ; TOKISSYM(*p) ; ++p);
        psav = *p;
        *p = '\0';
        end = (psav ? p + 1 : p);
    }
    else
    {
        tok2 = (char *)0;
        end = tok1end;
        if (*end) ++end;
        p = (char *)0;
    }
    
    tok1sav = *tok1end;
    *tok1end = '\0';
    
    /* look up the symbols */
    tab = (toktdef *)ctx->dbgcxtab;
    if (ctx->dbgcxprs->prscxtok->tokcxflg & TOKCXCASEFOLD)
    {
	strcpy(buf1, tok1);
	strlwr(buf1);
	tok1 = buf1;
    }
    hsh1 = tokhsh(tok1);
    if (!(*tab->toktfsea)(tab, tok1, (int)strlen(tok1), hsh1, &sym1))
        return(ERR_BPSYM);
    objn = sym1.toksval;
    
    if (tok2)
    {
        /* we have "object.property" */
	if (ctx->dbgcxprs->prscxtok->tokcxflg & TOKCXCASEFOLD)
	{
	    strcpy(buf2, tok2);
	    strlwr(buf2);
	    tok2 = buf2;
	}
        hsh2 = tokhsh(tok2);
        if (!(*tab->toktfsea)(tab, tok2, (int)strlen(tok2), hsh2, &sym2))
            return(ERR_BPSYM);
        if (sym1.tokstyp != TOKSTOBJ) return(ERR_BPOBJ);
        
        /* we need to look up the property */
        ofs = objgetap(ctx->dbgcxmem, objn, (prpnum)sym2.toksval,
                       &target, FALSE);
        if (!ofs) return(ERR_BPNOPRP);
        
        /* make sure the property is code */
        objp = mcmlck(ctx->dbgcxmem, (mcmon)target);
        prp = objofsp((objdef *)objp, ofs);
        typ = prptype(prp);
        ofs = ((uchar *)prpvalp(prp)) - objp;
        mcmunlck(ctx->dbgcxmem, (mcmon)target);
        
        if (typ != DAT_CODE) return(ERR_BPPRPNC);
    }
    else
    {
        /* we have just "function" */
        if (sym1.tokstyp != TOKSTFUNC) return(ERR_BPFUNC);
        ofs = 0;                   /* functions always start at offset zero */
        target = objn;     /* for function, target is always same as object */
        objn = MCMONINV;               /* there is no "self" for a function */
    }
    
    /* undo our changes to the string text */
    if (p) *p = psav;
    *tok1end = tok1sav;
    
    /* check for a "when" expression */
    if (*end)
    {
        while (isspace(*end)) ++end;
        if (*end)
        {
            if (strnicmp(end, "when ", (size_t)5))
                return(ERR_EXTRTXT);
            for (end += 5 ; isspace(*end) ; ++end);
        }
    }

    /* set a breakpoint at this object + offset */
    return(dbgbpat(ctx, target, objn, ofs, bpnum, addr, FALSE, end));
}

/* delete a breakpoint */
int dbgbpdel(ctx, bpnum)
dbgcxdef *ctx;
int       bpnum;
{
    dbgbpdef *bp = &ctx->dbgcxbp[--bpnum];
    dbgbpdef *bp2;
    uchar    *objp;
    int       i;
    uint      ofs;
    uint      delsiz;
    int       bpset;
    
    /* make sure it's a valid breakpoint */
    if (bpnum < 0 || bpnum >= DBGBPMAX || !(bp->dbgbpflg & DBGBPFUSED))
        return(ERR_BPNSET);
    
    /* see if we now have NO breakpoints set on this location */
    for (bpset = FALSE, i = DBGBPMAX, bp2 = ctx->dbgcxbp ; i ; ++bp2, --i)
    {
        if (bp != bp2 && (bp2->dbgbpflg & DBGBPFUSED)
            && bp2->dbgbptarg == bp->dbgbptarg
            && bp2->dbgbpofs  == bp->dbgbpofs)
        {
            bpset = TRUE;
            break;
        }
    }
    
    /* if no other bp's here, convert the OPCBP back into an OPCLINE */
    if (!bpset)
    {
        if (bp->dbgbptarg == MCMONINV)
        {
            /* no more global breakpoints - delete global BP flag */
            ctx->dbgcxflg &= ~DBGCXFGBP;
        }
        else
        {
            /* convert the OPCBP back into OPCLINE */
            objp = mcmlck(ctx->dbgcxmem, (mcmon)bp->dbgbptarg);
            *(objp + bp->dbgbpofs) = OPCLINE;
            mcmtch(ctx->dbgcxmem, (mcmon)bp->dbgbptarg);
            mcmunlck(ctx->dbgcxmem, (mcmon)bp->dbgbptarg);
        }
    }

    bp->dbgbpflg &= ~DBGBPFUSED;     /* breakpoint slot is no longer in use */
    if (bp->dbgbpflg & DBGBPFNAME)
        dbgnamdel(ctx, bp->dbgbpnam);       /* delete name from text buffer */

    /* free condition object, if one has been allocated */
    if (bp->dbgbpflg & DBGBPFCOND)
    {
        mcmfre(ctx->dbgcxmem, bp->dbgbpcond);
        bp->dbgbpflg &= ~DBGBPFCOND;
    }
    
    return(0);
}

/* list breakpoints using user callback */
void dbgbplist(ctx, dispfn, dispctx)
dbgcxdef  *ctx;
void     (*dispfn)(/*_ dvoid *ctx, char *str _*/);
dvoid     *dispctx;
{
    dbgbpdef *bp = ctx->dbgcxbp;
    int       i;
    char     *p;
    char      buf[10];

    /* if we're not recording names, there's nothing we can do */
    if (!ctx->dbgcxnam) return;
    
    for (i = 0 ; i < DBGBPMAX ; ++bp, ++i)
    {
        if ((bp->dbgbpflg & DBGBPFUSED) && (bp->dbgbpflg & DBGBPFNAME))
        {
            p = ctx->dbgcxnam + bp->dbgbpnam;
            sprintf(buf, "%d: ", i + 1);
            (*dispfn)(dispctx, buf, (int)strlen(buf));
            (*dispfn)(dispctx, p, (int)strlen(p));
            
            if (bp->dbgbpflg & DBGBPFDISA)
                (*dispfn)(dispctx, " [disabled]", 11);
            
            (*dispfn)(dispctx, "\n", 1);
        }
    }
}

/* enable/disable a breakpoint */
int dbgbpdis(ctx, bpnum, disable)
dbgcxdef *ctx;
int       bpnum;
int       disable;                    /* TRUE ==> disable, FALSE ==> enable */
{
    dbgbpdef *bp = &ctx->dbgcxbp[--bpnum];
    
    /* make sure it's a valid breakpoint */
    if (bpnum < 0 || bpnum >= DBGBPMAX || !(bp->dbgbpflg & DBGBPFUSED))
        return(ERR_BPNSET);
    
    if (disable) bp->dbgbpflg |= DBGBPFDISA;
    else bp->dbgbpflg &= ~DBGBPFDISA;
    return(0);
}

/*
 *   Call user callback with lindef data for each breakpoint; global
 *   breakpoints are NOT passed to user callback, since they don't
 *   correspond to source locations.
 */
void dbgbpeach(ctx, fn, fnctx)
dbgcxdef  *ctx;
void     (*fn)(/*_ dvoid *fnctx, int id, uchar *buf, uint bufsiz _*/);
dvoid     *fnctx;
{
    int       i;
    dbgbpdef *bp = ctx->dbgcxbp;
    uchar    *p;
    uint      len;

    for (i = 0 ; i < DBGBPMAX ; ++bp, ++i)
    {
        if ((bp->dbgbpflg & DBGBPFUSED)
            && bp->dbgbptarg != MCMONINV)
        {
            p = mcmlck(ctx->dbgcxmem, (mcmon)bp->dbgbptarg);
            p += bp->dbgbpofs + 1;
	    len = *p - 4;
	    p += 3;
            (*fn)(fnctx, (int)(*p), p + 1, len);
            mcmunlck(ctx->dbgcxmem, (mcmon)bp->dbgbptarg);
        }
    }
}


/* ====================== debug line source routines ====================== */
int dbglinget(lin)
lindef *lin;
{
    if (lin->linflg & LINFEOF)
        return(TRUE);          /* expression has only one line - now at EOF */
    else
    {
        lin->linflg |= LINFEOF;
        return(FALSE);
    }
}

void dbglincls(lin)
lindef *lin;
{
    VARUSED(lin);
}

/* never called */
void dbglinpos(lin, buf, bufl)
lindef *lin;
char   *buf;
uint    bufl;
{
    VARUSED(lin);
    VARUSED(buf);
    VARUSED(bufl);
}

/* never called */
void dbglinglo(lin, buf)
lindef *lin;
uchar  *buf;
{
    VARUSED(lin);
    VARUSED(buf);
}

/* never called */
int dbglinwrt(lin, fp)
lindef   *lin;
osfildef *fp;
{
    VARUSED(lin);
    VARUSED(fp);
    return(TRUE);
}

/* never called */
void dbglincmp(lin, buf)
lindef *lin;
uchar  *buf;
{
    VARUSED(lin);
    VARUSED(buf);
}

/* maximum number of symbols that can be mentioned in an eval expression */
#define DBGMAXSYM 30

struct toktfdef
{
    toktdef   toktfsc;
    mcmcxdef *toktfmem;
    dbgcxdef *toktfdbg;
    dbgfdef  *toktffr;                       /* frame to be used for symbol */
};
typedef struct toktfdef toktfdef;

/* search debug frame tables for a symbol */
int dbgtabsea(tab, name, namel, hash, ret)
toktdef *tab;
char    *name;
int      namel;
int      hash;
toksdef *ret;
{
    dbgfdef  *fr;
    uchar    *objp;
    toktfdef *ftab = (toktfdef *)tab;
    mcmcxdef *mctx = ftab->toktfmem;
    dbgcxdef *ctx = ftab->toktfdbg;
    uint      len;
    uint      symlen;
    uint      symval;
    uint      ofs;
    uint      encofs;
    uchar    *framep;

    fr = ftab->toktffr;
    
    /* if there is no local context, we obviously can't find a symbol */
    if (fr->dbgftarg == MCMONINV) return(FALSE);
    
    /* lock object and get its frame pointer */
    objp = mcmlck(mctx, fr->dbgftarg);

    for (ofs = fr->dbgffr ; ofs ; ofs = encofs)
    {
        /* look through this frame table */
        framep = objp + ofs;
        len = osrp2(framep) - 4;                 /* get length out of table */
        encofs = osrp2(framep + 2);         /* get enclosing frame's offset */
        framep += 4;                                     /* skip the header */
            
        while (len)
        {
            symval = osrp2(framep);
            framep += 2;
            symlen = *framep++;
            len -= 3 + symlen;
                
            if (symlen == namel && !memcmp(name, framep, (size_t)namel))
            {
                /* unlock the object and set up the toksdef for return */
                mcmunlck(mctx, fr->dbgftarg);
                ret->toksval = symval;
                ret->tokstyp = TOKSTLOCAL;
                ret->tokslen = namel;
                ret->toksfr  = ofs;
                return(TRUE);
            }
            
            /* advance past this symbol */
            framep += symlen;
        }
    }

    /* nothing found - unlock object and return */
    mcmunlck(mctx, fr->dbgftarg);
    return(FALSE);
}

/* compile an expression; returns error if one occurs */
static int dbgcompile(ctx, expr, fr, objnp)
dbgcxdef *ctx;
char     *expr;
dbgfdef  *fr;
objnum   *objnp;
{
    lindef    lin;
    ushort    oldflg;
    int       err;
    prscxdef *prs = ctx->dbgcxprs;
    toktfdef  tab;
    mcmon     objn;
    
    /* set up a lindef to provide the source line to the compiler */
    prs->prscxtok->tokcxlin = &lin;
    lin.lingetp = dbglinget;
    lin.linclsp = dbglincls;
    lin.linppos = dbglinpos;
    lin.linglop = dbglinglo;
    lin.linwrtp = dbglinwrt;
    lin.lincmpp = dbglincmp;
    lin.linpar = (lindef *)0;
    lin.linnxt = (lindef *)0;
    lin.linid = 0;
    lin.linbuf = expr;
    lin.linflg = LINFNOINC;
    lin.linlen = strlen(expr);
    lin.linlln = 0;
    
    /* set emit frame object */
    prs->prscxemt->emtcxfrob = fr->dbgftarg;
    
    /* set up special frame-searching symbol table */
    tab.toktfsc.toktfadd = 0;      /* we'll never add symbols to this table */
    tab.toktfsc.toktfsea = dbgtabsea;        /* we will, however, search it */
    tab.toktfsc.toktnxt  = prs->prscxtok->tokcxstab;
    tab.toktfsc.tokterr  = prs->prscxerr;
    tab.toktfmem = ctx->dbgcxrun->runcxmem;
    tab.toktfdbg = ctx;
    tab.toktffr  = fr;
    
    /* presume no error and no object allocated */
    err = 0;

    /* allocate an object for emitting the code */
    prs->prscxemt->emtcxptr = mcmalo(prs->prscxmem, (ushort)512, &objn);
    prs->prscxemt->emtcxobj = objn;
    prs->prscxemt->emtcxofs = 0;

    ERRBEGIN(ctx->dbgcxerr)

    /* set the parsing context to generate no debug records */
    oldflg = prs->prscxflg;
    prs->prscxflg &= ~(PRSCXFLIN + PRSCXFLCL);
    
    /* set topmost symbol table to the debug frame table */
    prs->prscxtok->tokcxstab = &tab.toktfsc;

    /* now parse the expression and generate code for it */
    prsrstn(prs);
    prs->prscxtok->tokcxlen = 0;           /* purge anything left in buffer */
    toknext(prs->prscxtok);
    prsxgen(prs);
    emtop(prs->prscxemt, OPCDBGRET);             /* make certain it returns */
    
    ERRCATCH(ctx->dbgcxerr, err)
        /* unlock and free the object used for holding the pcode */
        mcmunlck(prs->prscxmem, objn);
        mcmfre(prs->prscxmem, objn);

        /* restore original parsing flags and symbol table */
        prs->prscxflg = oldflg;
        prs->prscxtok->tokcxstab = tab.toktfsc.toktnxt;
    
        return(err);
    ERREND(ctx->dbgcxerr)
        
    /* restore original parsing flags and symbol table */
    prs->prscxflg = oldflg;
    prs->prscxtok->tokcxstab = tab.toktfsc.toktnxt;
    
    mcmunlck(prs->prscxmem, objn);
    *objnp = objn;
    return(0);
}

/* evaluate an expression */
int dbgeval(ctx, expr, dispfn, dispctx, level)
dbgcxdef  *ctx;
char      *expr;
void     (*dispfn)(/*_ dvoid *ctx, char *str, int strl _*/);
dvoid     *dispctx;
int        level;
{
    runsdef  *oldsp;
    runsdef   val;
    objnum    objn;
    uchar    *objptr;
    int       err;

    /* compile the expression */
    if (err = dbgcompile(ctx, expr, &ctx->dbgcxfrm[level], &objn))
        return(err);

    ERRBEGIN(ctx->dbgcxerr)

    objptr = mcmlck(ctx->dbgcxmem, objn);

    /* execute the generated code */
    oldsp = ctx->dbgcxrun->runcxsp;  /* remember sp so we can detect change */
    runexe(ctx->dbgcxrun, objptr, ctx->dbgcxfrm[level].dbgfself,
           objn, (prpnum)0, 0);

    /* if the expression left a value on the stack, display it */
    if (ctx->dbgcxrun->runcxsp != oldsp)
    {
        runpop(ctx->dbgcxrun, &val);
        dbgpval(ctx, &val, dispfn, dispctx, TRUE);
        (*dispfn)(dispctx, "\n", 1);
    }

    ERRCATCH(ctx->dbgcxerr, err)
        /* do nothing - just continue to clean up and return the error */
    ERREND(ctx->dbgcxerr)

    /* free the object that we used to hold the compiled code */
    mcmunlck(ctx->dbgcxmem, objn);
    mcmfre(ctx->dbgcxmem, objn);

    /* return the error status, if any */
    return(err);
}

/* Set a watchpoint; returns error if any */
int dbgwxset(ctx, expr, wxnum, level)
dbgcxdef *ctx;
char     *expr;
int      *wxnum;
int       level;
{
    dbgwxdef *wx;
    int       i;
    objnum    objn;
    int       err;
    uint      oldflg;
    
    /* scan off leading spaces in expression text */
    while (isspace(*expr)) ++expr;
    
    /* get a free watch slot */
    for (i = 0, wx = ctx->dbgcxwx ; i < DBGWXMAX ; ++wx, ++i)
        if (!(wx->dbgwxflg & DBGWXFUSED)) break;

    if (i == DBGWXMAX) return(ERR_MANYWX);
    *wxnum = i + 1;
    
    /* compile the expression */
    oldflg = ctx->dbgcxprs->prscxflg;                   /* save parse flags */
    ctx->dbgcxprs->prscxflg |= PRSCXFWTCH;     /* note this is a watchpoint */
    err = dbgcompile(ctx, expr, &ctx->dbgcxfrm[level], &objn);
    ctx->dbgcxprs->prscxflg = oldflg;                  /* restore old flags */
    if (err) return(err);
    
    /* record the watch data */
    wx->dbgwxobj = objn;
    wx->dbgwxflg = DBGWXFUSED;
    wx->dbgwxself = ctx->dbgcxfrm[level].dbgfself;
    if (!dbgnamsav(ctx, expr, &wx->dbgwxnam)) wx->dbgwxflg |= DBGWXFNAME;
    
    return(0);
}

/* delete a watch expression */
int dbgwxdel(ctx, wxnum)
dbgcxdef *ctx;
int       wxnum;
{
    dbgwxdef *wx = &ctx->dbgcxwx[--wxnum];
    
    /* make sure it's valid */
    if (wxnum < 0 || wxnum > DBGWXMAX || !(wx->dbgwxflg & DBGWXFUSED))
        return(ERR_WXNSET);
    
    mcmfre(ctx->dbgcxmem, wx->dbgwxobj);         /* delete pcode for object */
    wx->dbgwxflg &= ~DBGWXFUSED;    /* watch expression is no longer in use */
    if (wx->dbgwxflg & DBGWXFNAME)
        dbgnamdel(ctx, wx->dbgwxnam);  /* delete expr text from name buffer */

    return(0);
}

/* update all watch expressions */
void dbgwxupd(ctx, dispfn, dispctx)
dbgcxdef  *ctx;
void     (*dispfn)(/*_ dvoid *dispctx, char *txt, int len _*/);
dvoid     *dispctx;
{
    int       i;
    dbgwxdef *wx;
    int       err;
    uchar    *objptr;
    runsdef  *oldsp;
    runsdef   val;
    char      buf[50];
    int       first = TRUE;
    
    /* suppress all output while processing watchpoints */
    outwx(1);
    
    for (i = 0, wx = ctx->dbgcxwx ; i < DBGWXMAX ; ++wx, ++i)
    {
        if (!(wx->dbgwxflg & DBGWXFUSED)) continue;
        
        /* display watch number and expression text */
        if (!first) (*dispfn)(dispctx, "\n", 1);
        else first = FALSE;
        
        sprintf(buf, "[%d] %s = ", i + 1, ctx->dbgcxnam + wx->dbgwxnam);
        (*dispfn)(dispctx, buf, (int)strlen(buf));

        /* lock the object containing the compiled expression */
        objptr = mcmlck(ctx->dbgcxmem, wx->dbgwxobj);

        ERRBEGIN(ctx->dbgcxerr)
            
        oldsp = ctx->dbgcxrun->runcxsp;
        runexe(ctx->dbgcxrun, objptr, wx->dbgwxself, wx->dbgwxobj, 
               (prpnum)0, 0);

        if (ctx->dbgcxrun->runcxsp != oldsp)
        {
            runpop(ctx->dbgcxrun, &val);
            dbgpval(ctx, &val, dispfn, dispctx, FALSE);
        }
        else
            (*dispfn)(dispctx, "<no value>", 10);
        
        ERRCATCH(ctx->dbgcxerr, err)
            if (err == ERR_INACTFR)
                (*dispfn)(dispctx, "<not available>", 15);
            else
            {
                sprintf(buf, "<error %d>", err);
                (*dispfn)(dispctx, buf, (int)strlen(buf));
            }
        ERREND(ctx->dbgcxerr)
                
        mcmunlck(ctx->dbgcxmem, wx->dbgwxobj);
    }
    
    /* stop suppressing output */
    outwx(0);
}


/*
 *   Find a base pointer, given the object+offset of the frame.  If the
 *   frame is not active, this routine signals ERR_INACTFR; otherwise, the
 *   bp value for the frame is returned. 
 */
runsdef *dbgfrfind(ctx, frobj, frofs)
dbgcxdef *ctx;
objnum    frobj;
uint      frofs;
{
    int      i;
    dbgfdef *f;
    mcmon    objn;
    uchar   *objptr;
    uint     ofs;
    
    /* search stack */
    for (i = ctx->dbgcxfcn, f = &ctx->dbgcxfrm[i-1] ; i ; --f, --i)
    {
        /* ignore this stack level if we have the wrong object */
        objn = f->dbgftarg;
        if (frobj != objn) continue;
        
        /* search nested frames within this stack level */
        objptr = mcmlck(ctx->dbgcxmem, objn);
        
        for (ofs = f->dbgffr ; ofs ; )
        {
            if (ofs == frofs)
            {
                /* object and offset match - this is the correct frame */
                mcmunlck(ctx->dbgcxmem, objn);
                return(f->dbgfbp);
            }
            
            /* this isn't it; move on to the enclosing frame */
            ofs = osrp2(objptr + ofs + 2);
        }
        
        /* done with this object - unlock it */
        mcmunlck(ctx->dbgcxmem, objn);
    }
    
    /* couldn't find the frame at all - signal INACTIVE error */
    errsig(ctx->dbgcxerr, ERR_INACTFR);
    NOTREACHEDV(runsdef *);
}

/*
 *   Switch to a new current lindef, closing the file referenced by the
 *   current lindef.  This lets us keep only one file open at a time when
 *   running under the stand-alone source debugger, which reads the source
 *   from disk files.  
 */
void dbgswitch(linp, newlin)
lindef **linp;
lindef  *newlin;
{
    if (*linp) lindisact(*linp);
    if (newlin) linactiv(newlin);
    *linp = newlin;
}

