/*
 * fastmem.c
 *
 * Memory related functions (fast version without virtual memory)
 *
 */

/* Hacked for the unix port to support INFOCOM_PATH, patch by
   Mike Phillips (mike@lawlib.wm.edu).  Thanks, Mike! */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "frotz.h"

#define MAX_UNDO_SLOTS 20

#ifdef __MSDOS__

#include <alloc.h>

#define malloc(size)		farmalloc (size)
#define realloc(size,p)		farrealloc (size,p)
#define free(size)		farfree (size)

#endif /* __MSDOS__ */

#ifndef SEEK_SET
#define SEEK_SET 0
#endif

#ifndef SEEK_CUR
#define SEEK_CUR 1
#endif

#ifndef SEEK_END
#define SEEK_END 2
#endif

char save_name[MAX_FILE_NAME + 1] = DEFAULT_SAVE_NAME;
char auxilary_name[MAX_FILE_NAME + 1] = DEFAULT_AUXILARY_NAME;

zbyte *zmp = NULL;
zbyte *pcp = NULL;

static zbyte *undo_zmp[MAX_UNDO_SLOTS];
static zbyte *undo_stack[MAX_UNDO_SLOTS];

static FILE *story_fp;

static int undo_slots = 0;
static int undo_count = 0;
static int undo_valid = 0;

/*
 * restart_header
 *
 * Set all header fields which hold information about the interpreter.
 *
 */

void restart_header (void)
{
    int i;

    SET_BYTE (H_CONFIG, h_config)
    SET_WORD (H_FLAGS, h_flags)

    if (h_version >= V4) {
	SET_BYTE (H_INTERPRETER_NUMBER, h_interpreter_number)
	SET_BYTE (H_INTERPRETER_VERSION, h_interpreter_version)
	SET_BYTE (H_SCREEN_ROWS, h_screen_rows)
	SET_BYTE (H_SCREEN_COLS, h_screen_cols)
    }

    if (h_version >= V5) {
	SET_WORD (H_SCREEN_WIDTH, h_screen_width)
	SET_WORD (H_SCREEN_HEIGHT, h_screen_height)
	SET_BYTE (H_FONT_WIDTH, h_font_width)
	SET_BYTE (H_FONT_HEIGHT, h_font_height)
	SET_BYTE (H_DEFAULT_BACKGROUND, h_default_background)
	SET_BYTE (H_DEFAULT_FOREGROUND, h_default_foreground)
    }

    /* V6 games change the order of the font width and height entries
       in the header structure -- presumably a mistake in the standard. */

    if (h_version == V6) {
	SET_BYTE (H_FONT_HEIGHT, h_font_width)
	SET_BYTE (H_FONT_WIDTH, h_font_height)
    }

    SET_BYTE (H_STANDARD_HIGH, h_standard_high)
    SET_BYTE (H_STANDARD_LOW, h_standard_low)

    if (h_version == V6)
	for (i = 0; i < 8; i++)
	    z_storeb (H_USER_NAME, i, h_user_name[i]);

}/* restart_header */

/*
 * load_header
 *
 * Load header and copy header entries to global variables.
 *
 */

static void load_header (void)
{
    zword addr;
    int i;

    /* Allocate memory for story header */

    if ((zmp = (zbyte *) malloc (64)) == NULL)
	os_fatal ("Out of memory");

    /* Load header into memory */

    if (fread (zmp, 1, 64, story_fp) != 64)
	os_fatal ("Story file read error");

    /* Copy header fields to global variables */

    LOW_BYTE (H_VERSION, h_version)

    if (h_version < V1 || h_version > V8)
	os_fatal ("Unsupported Z-code version");

    LOW_BYTE (H_CONFIG, h_config)

    if (h_version == V3 && (h_config & CONFIG_BYTE_SWAPPED))
	os_fatal ("Byte swapped story file");

    LOW_WORD (H_RELEASE, h_release)
    LOW_WORD (H_RESIDENT_SIZE, h_resident_size)
    LOW_WORD (H_START_PC, h_start_pc)
    LOW_WORD (H_DICTIONARY, h_dictionary)
    LOW_WORD (H_OBJECTS, h_objects)
    LOW_WORD (H_GLOBALS, h_globals)
    LOW_WORD (H_DYNAMIC_SIZE, h_dynamic_size)
    LOW_WORD (H_FLAGS, h_flags)

    for (i = 0, addr = H_SERIAL; i < 6; i++, addr++)
	LOW_BYTE (addr, h_serial[i])

#ifndef AMIGA
    if (h_release == 47 || h_release == 49 || h_release == 51 || h_release == 57)
	beyond_zork_flag = (h_serial[0] == '8' && h_serial[1] == '7');
    if (h_release == 3)
	german_zork_flag = (h_serial[0] == '8' && h_serial[1] == '8');
#endif

    LOW_WORD (H_ABBREVIATIONS, h_abbreviations)
    LOW_WORD (H_FILE_SIZE, h_file_size)

    if (h_file_size != 0) {

	/* Calculate story file size in bytes */

	story_size = (long) 2 * h_file_size;

	if (h_version >= V4)
	    story_size *= 2;
	if (h_version >= V6)
	    story_size *= 2;

    } else {

	/* Some old games lack the file size entry */

	fseek (story_fp, 0, SEEK_END);
	story_size = ftell (story_fp);
	fseek (story_fp, 64, SEEK_SET);
    }

    LOW_WORD (H_CHECKSUM, h_checksum)
    LOW_WORD (H_ALPHABET, h_alphabet)
    LOW_WORD (H_FUNCTIONS_OFFSET, h_functions_offset)
    LOW_WORD (H_STRINGS_OFFSET, h_strings_offset)
    LOW_WORD (H_TERMINATING_KEYS, h_terminating_keys)
    LOW_WORD (H_MOUSE_TABLE, h_mouse_table)

}/* load_header */

/*
 * load_story
 *
 * Load the entire story file into memory.
 *
 */

static void load_story (void)
{
    long i;
    unsigned n;

    /* Allocate memory for story data */

    if ((zmp = (zbyte *) realloc (zmp, story_size)) == NULL)
	os_fatal ("Out of memory");

    /* Load story file into memory */

    n = 0xf000;

    for (i = 64; i < story_size; i += n) {

	SET_PC (i)

	if (story_size - i < 0xf000)
	    n = (unsigned) (story_size - i);

	if (fread (pcp, 1, n, story_fp) != n)
	    os_fatal ("Story file read error");
    }

}/* load_story */

/*
 * init_memory
 *
 * Allocate memory and load the story file.
 *
 */

void init_memory (void)
{
    char *path, *p,
         tmp[256];

    /* Open story file */

    if ((story_fp = fopen (story_name, "rb")) != NULL) {

        /* found story file in current directory, so
           Initialise Z-machine memory */

        load_header ();
        load_story ();
        return;
    }

    if ((path = getenv("INFOCOM_PATH")) == NULL)
        os_fatal ("Cannot open story file");

    p=strtok(path,":");
    while (p) {
        sprintf(tmp, "%s/%s", p, story_name);
        if ((story_fp = fopen (tmp, "rb")) != NULL) {

            /* Initialise Z-machine memory */

            load_header ();
            load_story ();
            return;
        }

        p = strtok(NULL, ":");

    }

    os_fatal("Cannot open story file");

}/* init_memory */

/*
 * init_undo
 *
 * Allocate memory for multiple undo. It is important not to occupy
 * all the memory available, because the IO interface might need
 * to allocate some memory during the game (e.g. for loading sounds
 * or pictures).
 *
 */

void init_undo (void)
{
    void *reserved_mem;

    reserved_mem = malloc (reserve_mem);

    if (option_undo_slots > MAX_UNDO_SLOTS)
	option_undo_slots = MAX_UNDO_SLOTS;

    for (undo_slots = 0; undo_slots < option_undo_slots; undo_slots++) {

	undo_zmp[undo_slots] = (zbyte *) malloc (h_dynamic_size);
	undo_stack[undo_slots] = (zbyte *) malloc (sizeof (stack));

	if (undo_zmp[undo_slots] != NULL && undo_stack[undo_slots] == NULL)
	    free (undo_zmp[undo_slots]);
	if (undo_zmp[undo_slots] == NULL && undo_stack[undo_slots] != NULL)
	    free (undo_stack[undo_slots]);

	if (undo_stack[undo_slots] == NULL || undo_zmp[undo_slots] == NULL)
	    break;
    }

    free (reserved_mem);

}/* init_undo */

/*
 * reset_memory
 *
 * Close the story file and deallocate memory.
 *
 */

void reset_memory (void)
{

    /* Close story file */

    fclose (story_fp);

    /* Free Z-machine memory */

    free (zmp);

}/* reset_memory */

/*
 * reset_undo
 *
 * Free the memory resources allocated for multiple undo.
 *
 */

void reset_undo (void)
{

    while (undo_slots-- != 0) {
	free (undo_zmp[undo_slots]);
	free (undo_stack[undo_slots]);
    }

}/* reset_undo */

/*
 * z_restart
 *
 * Restart game by re-loading dynamic memory and setting PC to the
 * start address of the program.
 *
 */

void z_restart (void)
{
    static int first_time = 1;
    long pc;

    /* Prepare the screen */

    restart_screen ();

    /* Randomize */

    seed_random (0);

    /* Reload dynamic memory */

    if (first_time == 0) {

	rewind (story_fp);

	if (fread (zmp, 1, h_dynamic_size, story_fp) != h_dynamic_size)
	    os_fatal ("Story file read error");

    } else first_time = 0;

    /* Initialise story header */

    restart_header ();

    /* Initialise PC, SP and FP */

    sp = STACK_SIZE;
    fp = STACK_SIZE;

    if (h_version != V6) {

	pc = h_start_pc;
	SET_PC (pc)

    } else z_call (1, &h_start_pc, 0);

}/* z_restart */

/*
 * get_default_name
 *
 * Read a default file name from the memory of the Z-machine and
 * copy it to an array.
 *
 */

static void get_default_name (char *default_name, zword addr)
{
    zbyte len;
    zbyte c;
    int i;

    if (addr != 0) {

	LOW_BYTE (addr, len)

	for (i = 0; i < len; i++) {
	    addr++;
	    LOW_BYTE (addr, c)
	    default_name[i] = c;
	}
	default_name[i] = 0;

	if (strchr (default_name, '.') == 0)
	    strcpy (default_name + i, ".aux");

    } else strcpy (default_name, auxilary_name);

}/* get_default_name */

/*
 * old_restore
 *
 * This function tries to load old save files (ie. uncompressed
 * files from Zip 2.00 to 2.04). It returns true if successful.
 *
 */

static int old_restore (FILE *gfp)
{
    long pc;
    long len;

    fseek (gfp, 0, SEEK_END);
    len = ftell (gfp);
    fseek (gfp, 0, SEEK_SET);

    if (len != sizeof (stack) + h_dynamic_size)
	return 0;

    fread (stack, 1, sizeof (stack), gfp);
    fread (zmp, 1, h_dynamic_size, gfp);

    sp = stack[0] + 4;
    fp = stack[stack[0] + 1] + 1;
    pc = stack[stack[0] + 3];
    pc = stack[stack[0] + 2] | (pc << 9);

    SET_PC (pc)

    fclose (gfp);

    return 1;

}/* old_restore */

/*
 * z_restore
 *
 * Restore game state from disk. Returns:
 *
 *    0 = restore failed
 *    2 = restore succeeded
 *
 * All arguments are optional. If these are given then only a part of
 * the dynamic memory is restored. When restore is used in this long
 * form, the return value is the number of bytes read.
 *
 */

void z_restore (int argc, zword table, zword bytes, zword name)
{
    char new_name[MAX_FILE_NAME + 1];
    char default_name[MAX_FILE_NAME + 1];
    FILE *gfp;
    long pc;
    zword release;
    zword checksum;
    zword addr;
    int success;
    int skip;
    int i;

    success = 0;

    if (argc != 0) {

	/* Get the file name */

	get_default_name (default_name, argc >= 3 ? name : 0);

	if (os_get_file_name (new_name, default_name, FILE_LOAD_AUX) == 0)
	    goto finished;

	/* Open auxilary file */

	if ((gfp = fopen (new_name, "rb")) == NULL)
	    goto finished;

	/* Load auxilary file */

	success = fread (zmp + table, 1, bytes, gfp);

	/* Close auxilary file */

	fclose (gfp);

	/* Success? */

	if (success != 0)
	    strcpy (auxilary_name, default_name);

    } else {

	/* Get the file name */

	if (os_get_file_name (new_name, save_name, FILE_RESTORE) == 0)
	    goto finished;

	/* Open game file */

	if ((gfp = fopen (new_name, "rb")) == NULL)
	    goto finished;

	/* This might be an old Zip save file */

	if (old_restore (gfp) == 0) {

	    /* Load game file */

	    release = fgetc (gfp) << 8;
	    release = fgetc (gfp) | release;
	    checksum = fgetc (gfp) << 8;
	    checksum = fgetc (gfp) | checksum;

	    if (release != h_release || checksum != h_checksum) {
		print_string ("Invalid save file\n");
		fclose (gfp);
		goto finished;
	    }

	    pc = fgetc (gfp);
	    pc = fgetc (gfp) | (pc << 8);
	    pc = fgetc (gfp) | (pc << 8);

	    SET_PC (pc)

	    sp = fgetc (gfp) << 8;
	    sp = fgetc (gfp) | sp;
	    fp = fgetc (gfp) << 8;
	    fp = fgetc (gfp) | fp;

	    for (i = sp; i < STACK_SIZE; i++) {
		stack[i] = fgetc (gfp) << 8;
		stack[i] = fgetc (gfp) | stack[i];
	    }

	    rewind (story_fp);

	    for (addr = 0; addr < h_dynamic_size; addr++) {
		skip = fgetc (gfp);
		for (i = 0; i < skip; i++)
		    zmp[addr++] = fgetc (story_fp);
		zmp[addr] = fgetc (gfp);
		fgetc (story_fp);
	    }

	    /* Check for errors and close game file */

	    if (ferror (gfp) != 0 || ferror (story_fp) != 0)
		os_fatal ("Error reading save file");

	    fclose (gfp);
	}

	/* Reset upper window (V3 only) */

	if (h_version == V3)
	    z_split_window (0);

	/* Initialise story header */

	restart_header ();

	/* Success */

	strcpy (save_name, new_name);
	success = 2;
    }

finished:

    if (h_version <= V3)
	branch (success);
    else
	store (success);

}/* z_restore */

/*
 * z_restore_undo
 *
 * Restore the current Z-machine state from memory. Returns:
 *
 *   -1 = feature unavailable
 *    0 = restore failed
 *    2 = restore succeeded
 *
 * Note: The "flag" parameter is set when the interpreter doesn't want
 * the function to return a value to the Z-machine. Furthermore, the
 * flag also indicates multiple undo: The last Z-machine state will be
 * removed from the stack such that future z_restore_undo calls refer
 * to earlier states.
 *
 */

void z_restore_undo (int flag)
{
    long pc;
    int success;

    /* Check if undo is available first */

    if (undo_slots == 0)

	success = -1;

    else if (undo_valid == 0)

	success = 0;

    else {

	/* Set the "undo_flag" */

	undo_flag = 1;

	/* Restore state from undo memory */

	if (undo_count == 0)
	    undo_count = undo_slots;

	memcpy (zmp, undo_zmp[undo_count - 1], h_dynamic_size);
	memcpy (stack, undo_stack[undo_count - 1], sizeof (stack));

	pc = stack[0];
	pc = stack[1] | (pc << 16);
	sp = stack[2];
	fp = stack[3];

	SET_PC (pc)

	/* Initialise story header */

	restart_header ();

	/* Adjust undo counters */

	if (flag != 0) {
	    undo_count--;
	    undo_valid--;
	}

	/* Success */

	success = 2;
    }

    /* Return success to the Z-machine */

    if (flag == 0)
	store (success);

}/* restore_undo */

/*
 * z_save
 *
 * Save game state to disk. Returns:
 *
 *    0 = save failed
 *    1 = save succeeded
 *
 * All arguments are optional. If these are given then only a part
 * of the dynamic memory is saved.
 *
 */

void z_save (int argc, zword table, zword bytes, zword name)
{
    char new_name[MAX_FILE_NAME + 1];
    char default_name[MAX_FILE_NAME + 1];
    FILE *gfp;
    long pc;
    zword addr;
    int success;
    int skip;
    int i;

    success = 0;

    if (argc != 0) {

	/* Get the file name */

	get_default_name (default_name, argc >= 3 ? name : 0);

	if (os_get_file_name (new_name, default_name, FILE_SAVE_AUX) == 0)
	    goto finished;

	/* Open auxilary file */

	if ((gfp = fopen (new_name, "wb")) == NULL)
	    goto finished;

	/* Write auxilary file */

	success = fwrite (zmp + table, bytes, 1, gfp);

	/* Close auxilary file */

	fclose (gfp);

	/* Success? */

	if (success != 0)
	    strcpy (auxilary_name, default_name);

    } else {

	/* Get the file name */

	if (os_get_file_name (new_name, save_name, FILE_SAVE) == 0)
	    goto finished;

	/* Open game file */

	if ((gfp = fopen (new_name, "wb")) == NULL)
	    goto finished;

	/* Write game file */

	fputc (hi (h_release), gfp);
	fputc (lo (h_release), gfp);
	fputc (hi (h_checksum), gfp);
	fputc (lo (h_checksum), gfp);

	GET_PC (pc)

	fputc ((int) (pc >> 16) & 0xff, gfp);
	fputc ((int) (pc >> 8) & 0xff, gfp);
	fputc ((int) (pc) & 0xff, gfp);

	fputc (hi (sp), gfp);
	fputc (lo (sp), gfp);
	fputc (hi (fp), gfp);
	fputc (lo (fp), gfp);

	for (i = sp; i < STACK_SIZE; i++) {
	    fputc (hi (stack[i]), gfp);
	    fputc (lo (stack[i]), gfp);
	}

	rewind (story_fp);

	for (addr = 0, skip = 0; addr < h_dynamic_size; addr++)
	    if (zmp[addr] != fgetc (story_fp) || skip == 255 || addr == h_dynamic_size - 1) {
		fputc (skip, gfp);
		fputc (zmp[addr], gfp);
		skip = 0;
	    } else skip++;

	/* Close game file and check for errors */

	if (fclose (gfp) == EOF || ferror (story_fp) != 0) {
	    print_string ("Error writing save file\n");
	    goto finished;
	}

	/* Success */

	strcpy (save_name, new_name);
	success = 1;
    }

finished:

    if (h_version <= V3)
	branch (success);
    else
	store (success);

}/* z_save */

/*
 * z_save_undo
 *
 * Save the current Z machine state in memory for a future undo. Returns:
 *
 *    -1 = feature unavailable
 *     0 = save failed
 *     1 = save succeeded
 *
 * Note: The "flag" parameter is set when the interpreter doesn't want
 * the function to return a value to the Z-machine.
 *
 */

void z_save_undo (int flag)
{
    long pc;
    int success;

    /* Check if undo is available first */

    if (undo_slots == 0)

	success = -1;

    else {

	/* Copy current state to undo memory */

	if (undo_count == undo_slots)
	    undo_count = 0;

	GET_PC (pc)

	stack[0] = pc >> 16;
	stack[1] = pc & 0xffff;
	stack[2] = sp;
	stack[3] = fp;

	memcpy (undo_zmp[undo_count], zmp, h_dynamic_size);
	memcpy (undo_stack[undo_count], stack, sizeof (stack));

	/* Adjust undo counters */

	if (++undo_count == undo_slots)
	    undo_count = 0;
	if (++undo_valid > undo_slots)
	    undo_valid = undo_slots;

	/* Success */

	success = 1;
    }

    /* Return success to the Z-machine */

    if (flag == 0)
	store (success);

}/* z_save_undo */

/*
 * z_verify
 *
 * Verify game ($verify verb). Add all bytes in story file except for
 * bytes in the story file header.
 *
 */

void z_verify (void)
{
    zword checksum;
    long i;

    checksum = 0;

    /* Sum all bytes in story file except header bytes */

    fseek (story_fp, 64, SEEK_SET);

    for (i = 64; i < story_size; i++)
	checksum += fgetc (story_fp);

    /* Branch if the checksum is equal */

    branch (checksum == h_checksum);

}/* z_verify */
