#charset "us-ascii"
/*
 *  Copyright (c) 2005 by Kevin Forchione. All rights reserved.
 *  Based on code provided by Michael J. Roberts.
 *
 *  This file is part of the TADS 3 Services Package
 *
 *  tsp_symbols.t
 *
 *  Provides a basic implementation for mapping from symbol to string
 *  and string to symbol using the TADS 3 Symbol Table generated at pre-
 *  compile time.
 */

#include "tads.h"
#include "t3.h"
#include "tsp_mod_registry.h"

RegisterModule

/* enumerators and macros for object types */
enum TypeObjectClass, TypeObjectNamed, TypeObjectInternal;
enum TypeObjectAnonymous, TypeObjectNested, TypeObjectDynamic;

property sortOrderGroup;

/* ------------------------------------------------------------------------ */
/*
 *  Library Pre-Initializer.  This object performs the following
 *  initialization operations immediately after compilation is completed:
 *
 *  Stash a reference to the global symbol table and build a reverse 
 *  symbol table.
 *
 *  This object is named so that other libraries and/or user code can
 *  create initialization order dependencies upon it.  
 */
Symbols: PreinitObject
{
    /*
     *  Map a string value to its symbol value. 
     */
    getSymbol(val)
    {
        if (dataType(val) == TypeSString)
        {
            /*
             *  Modifications to objects may be preceded by spaces.
             *  We might need to pad the beginning of the string.
             */
            if (toInteger(val) > 0)
            {
                local newStr = '';

                while (val.length() % 4 != 0)
                {
                    newStr = ' ' + val;
                    val = newStr;
                }
            }
            return symtab[val];
        }
        else 
        {
            if (isSymKeyPresent(val))
                return val;
            else return nil;   
        }
    }

    getSString(val)
    {
        local str;

        if (dataType(val) == TypeSString)
        {
            if (isStrKeyPresent(val))
                return val;
            else return '';
        }
        else
        {
            str = reverseSymtab[val];

            return (str ? str : '');
        }
    }

    isKeyPresent(val)
    {
        if (dataType(val) == TypeSString)
            return isStrKeyPresent(val);
        else
            return isSymKeyPresent(val);
    }

    isStrKeyPresent(val)
    {
        return symtab.isKeyPresent(val);
    }

    isSymKeyPresent(val)
    {
        return reverseSymtab.isKeyPresent(val);
    }

    /* 
     *  Returns one of the following enums for an object
     *      TypeObjectClass
     *      TypeObjectNamed
     *      TypeObjectAnonymous
     *      TypeObjectDynamic
     */
    getObjType(obj)
    {
        if (dataType(obj) == TypeObject)
            if (objtab.isKeyPresent(obj))
                return objtab[obj];
            else
                return TypeObjectDynamic;

        return nil;
    }

    /*
     *  Returns the sort order group for this object. Sort order group
     *  is assigned to statically defined TadsObject classes and objects.
     */
    getSortOrderGroup(obj)
    {
        if (obj.propDefined(&sortOrderGroup))
            return 0;
        else
            return obj.sortOrderGroup;
    }

    execute()
    {
        local cnt = 0;

        /*  store a reference to the global symbol table */
        symtab = t3GetGlobalSymbols();
        
        /*
         *  building a reverse mapping - in other words, building a
         *  LookupTable that uses symbol values as the keys and symbol
         *  strings as the associated values.
         */
        reverseSymtab = new LookupTable();
        symtab.forEachAssoc({str, sym: reverseSymtab[sym] = str});

        /*
         *  Build the object type table. This table will indicate
         *  whether a statically-defined object is a class, an 
         *  instance, or anonymously-defined.
         */
        objtab = new LookupTable();
        symtab.forEachAssoc(new function(str, sym)
        {
            if (dataType(sym) == TypeObject)
                if (sym.isClass())
                    if (toInteger(getSString(sym)) > 0)
                        objtab[sym] = TypeObjectInternal;
                    else
                        objtab[sym] = TypeObjectClass;
                else
                    objtab[sym] = TypeObjectNamed;
        });

        for (local obj = firstObj(TadsObject) ; obj != nil ; obj = nextObj(obj, TadsObject))
        {
            /*
             *  If it's in the objtab already, continue 
             */
            if (objtab.isKeyPresent(obj))
                continue;

            /*
             *  If obj has directly-defined property 
             *  &sourceTextOrder, then we check it's 
             *  lexicalParent to see if it's to be 
             *  classified as anonymously-defined or
             *  as a nested object.
             */
            if (obj.propDefined(&sourceTextOrder, PropDefDirectly))
                if (obj.lexicalParent == nil)
                    objtab[obj] = TypeObjectAnonymous;
                else
                    objtab[obj] = TypeObjectNested;
        }

        /*
         *  Add the sortOrderGroup property to the object
         */
        objtab.forEachAssoc(new function(obj, val)
        {
            if (obj.ofKind(TadsObject)
                && obj != TadsObject
                && getObjType(obj) is in (TypeObjectClass, TypeObjectNamed))
                obj.sortOrderGroup = ++cnt;
        });
    }

    /* string-to-symbol lookup table */
    symtab			= nil

    /* symbol-to-string lookup table */
    reverseSymtab	= nil

    /* object type lookup table */
    objtab          = nil
}

modify Object
{
	/*
	 *	Determine if the object has been dynamically-created 
     *  or anonymously defined.
	 */
	isDynamOrAnon()
	{
        return (Symbols.getObjType(self) is in (TypeObjectAnonymous,
            TypeObjectNested, TypeObjectDynamic));
	}

    /*
     *  Determine if the object has been anonymously-defined.
     */
    isAnonymous()
    {
        return (Symbols.getObjType(self) == TypeObjectAnonymous);
    }

    /*
     *  Determine if the object has been dynamically-created.
     */
    isDynamic()
    {
        return (Symbols.getObjType(self) == TypeObjectDynamic);
    }

    /*
     *  Determine if the object has been dynamically-created.
     */
    isNested()
    {
        return (Symbols.getObjType(self) == TypeObjectNested);
    }

    /*
     *  Determine if the object is an unnamed internal. This
     *  occurs when an object has been modified by the 'modify'
     *  keyword.
     */
    isUnnamedInternal()
    {
        return (Symbols.getObjType(self) == TypeObjectInternal);
    }
}